;;; -*- Mode: Emacs-Lisp; Syntax: Common-lisp; Base: 10; Package: SKY; -*-
;;; File: sky-mouse.el
;;; Author: Heinz Schmidt (hws@icsi.berkeley.edu)
;;; Copyright (C) 1990, International Computer Science Institute
;;;
;;; COPYRIGHT NOTICE: This code is provided "AS IS" WITHOUT ANY WARRANTY.
;;; It is subject to the terms of the GNU EMACS GENERAL PUBLIC LICENSE
;;; described in a file COPYING in the GNU EMACS distribution or to be obtained
;;; from Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;* FUNCTION: Mouse region commands for Emacs a la ZMacs under native
;;;*           Emacs x-mouse and Epoch. 
;;;*
;;;* SUMMARY:
;;;* mouse-mark-thing,mouse-yank-thing-to-point,mouse-toggle-focus-screen,
;;;* mouse-save-kill-yank,mouse-kill-region,mouse-set-point-force-kbd,
;;;* mouse-execute-kbd-macro,mouse-tags-search,mouse-fill-or-indent,
;;;* mouse-warp-to-point,scroll-point-to-mouse,help-with-mouse,
;;;* mouse-beep,mouse-ignore,mouse-quit-kbd,mouse-mark-thing-extend,
;;;* mouse-execute-at-point,mouse-execute-on-region,mouse-define,defmouse,
;;;* goto-ratio-linewise,goto-percent-linewise
;;;* 
;;;* INSTALLATION: cf. file INSTALL
;;;*
;;;* RELATED PACKAGES: 
;;;*     epoch-mouse-base.el     -- basic mouse under Epoch
;;;*     x-mouse-base.el         -- basic mouse under Emacs x-mouse
;;;*     poor-mans-mouse.el      -- kbd substitute for some of the cmds
;;;*                                for running dumb Emacs via modem
;;;*     epoch-help-patch.el     -- change help-for-help to provide 
;;;*                                mouse help, optional
;;;*
;;;*     Modifications to Epoch select-screen for focus support in
;;;*     epoch-mouse-base.el. Otherwise pure extensions.
;;;* 
;;;* CONCEPTS: 
;;;*  1. SKY: The idea of the ZMacs save/kill/yank is to have a single mouse
;;;*     command (mouse buttons are limited resources) allowing fast access to
;;;*     the kill-ring in any situation where the mouse is used for marking a
;;;*     region.  Basically a small save/kill/yank (sky) state machine runs
;;;*     through the transitions corresponding to M-w,C-w,C-y,M-y* (i.e.  save,
;;;*     kill, yank, yank-pop*). The machine is reset whenever the 'focus is
;;;*     shifted', i.e.  whenever a new region is chosen, in particular by
;;;*     mouse-set-point.
;;;*  2. Mark/Yank-Thing: A thing at point is defined according to the
;;;*     current syntax. A single mouse click can mark a thing.  For instance
;;;*     clicking to `{' in C-mode marks the corresponding expression.
;;;*     Subsequent clicks can be used to save/kill/yank or run through the
;;;*     kill-ring fast without moving the hand from mouse to keyboard.  Things
;;;*     can be copied to point with a single click by yank-thing-to-point
;;;*     (save them without marking and yank them to point). This is
;;;*     particularly useful across different windows.  Under Epoch we allow to
;;;*     fix the keyboard focus on a screen temporarily, so that the mouse can
;;;*     'go searching for interesting things' on other screens and spit them
;;;*     to the place where we are composing something, such as arguments for a
;;;*     command, definitions in a language mode etc. Focus fixing is
;;;*     implemented by a single command that toggles the state of focusing.
;;;*     According to my experience this makes focus fixing natural under the
;;;*     'focus follows mouse' mode, too.
;;;*  3. Command programming: In general commands are pairs of functions. The
;;;*     first function is the selection method allocated on the button down
;;;*     transition such as mouse-set-point or mouse-drag-point.  The second
;;;*     function allocated on the button up transition operates on the object
;;;*     selected by the first function. A high-level function mouse-define, cf.
;;;*     online doc, treats commands as pairs. For simplicity of mouse command
;;;*     programming, a number of selection methods have been preprogrammed,
;;;*     they are slighly different from the functions in Epoch mouse.el, to
;;;*     work with the region more consistently. For example, free-hand dragging
;;;*     is like movement (foward-char, forward-word etc) and changes the
;;;*     current point. M
;;;*  4. Kill-ring and X cut-buffer: The X cut-buffer is partly integrated with
;;;*     this. Whenever a region is copied or killed, it is also transferred to
;;;*     the X cut-buffer.  Pasting (of the X cut-buffer contents) is a separate
;;;*     operation and must be placed on the appropriate paste button according
;;;*     to the chosen window manager.
;;;*  5. Feedback: The commands provide uniform feedback. Regions are underlined
;;;*     if selected by the mouse. The mouse curor changes during the execution
;;;*     of a selection method to indicate the kind of selection method or the
;;;*     command proper that is going to be executed when selection is
;;;*     completed.  Under Epoch a command can be interrupted by moving the
;;;*     mouse out of the window and only then releasing the button(s). Under
;;;*     x-mouse, the commands provide feedback for region marking by flashing
;;;*     the cursor at the region ends.  
;;;*
;;;* HISTORY: 
;;;* Last edited: Mar  8 14:47 1992 (hws)
;;;*  Jun  4 10:34 1991 (hws): help-key is unbound in some Emacs distributions.
;;;*                           add a defvar here.
;;;*  May  2 01:53 1991 (hws): Allow fill comment in indented comment.
;;;*  Jan 30 20:16 1991 (hws): Finish scrolling.
;;;*  Jan 25 14:55 1991 (hws): By and by added multi-clicks.
;;;*  Jan 15 18:21 1991 (hws): Added mouse-define.
;;;*  Dec 22 20:10 1990 (hws): Avoid messages when yanking from and to minibuf.
;;;*  Dec 16 21:52 1990 (hws): Improve spacing for mouse-yank-thing-to-point;
;;;*                make thing be comment if looking-at comment-char.
;;;*  Dec 13 23:20 1990 (hws): Mouse-set-point-force-kbd treats focus properly
;;;*  Dec 12 10:02 1990 (hws): changed ...force-kbd to use top-level, **** HACK,
;;;*                better than keyboard-quit, doesn't leave a recursive level.
;;;*                but maybe it flips into other buffer (when?) and do we want
;;;*                to leave all levels behind or just interrupt the minibuffer?
;;;* 12/04/90 (hws) fix mouse-fill-or-indent for click in Mail mode.
;;;*                Add automatic install routine.
;;;* 12/03/90 (hws) fix mark-thing to work again under MacLayers/Emacs. 
;;;* 12/01/90 (hws) adapted mark-thing to use thing.el if available.
;;;* 12/01/90 (hws) move cut-buffer simulation here from .emacs.
;;;* 11/29/90 (hws) workaround the 'hanging redisplay bug'.
;;;* 11/28/90 (hws) fix focus to work under autoraise. Fix select-screen.
;;;*    rudimentary describe-key for mouse. have mouse down set point by
;;;*    default.
;;;* 11/27/90 (hws) add save/kill/yank machine a la Zmacs. Fix 
;;;*    mouse::set-point, and relatives to be more cooperative with s/k/y. 
;;;* 11/21/90 (hws) make sure thing-yank's work across Epoch screens; 
;;;*    allow temporary focusing; adapt to new Epoch 3.2 mouse handling.
;;;* 10/15/90 (hws) adapt mouse-region saving to cut-buffer ops under 
;;;*    Xview olwm.
;;;* 04/03/90 (hws) separated from .emacs and eiffel-mode.el.
;;;*    adapt to Epoch gwm stuff. 
;;;* 02/23/90 (hws) fixed to also work across Emacs windows.
;;;* 01/03/90 (hws) Allow X Emacs user to yank-compose code at point by 
;;;*    pointing to  different things successively a la Zmacs mark-thing,
;;;*    yank-thing-to-point.
;;;* Created: Wed Nov 28 12:20:30 1990
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;* WISH LIST:
;;;*
;;;* 1. REDISPLAY BUG: There seems to be a bug in epoch::redisplay-screen. The
;;;* workaround in this file calls redisplay exclicitly. But finally xdisp.c
;;;* should be fixed to be smarter with multi-screens.  The bug can be
;;;* reproduced as follows: Have 2 screens beside each other. Select two
;;;* different buffers in the different screens. Use the mouse to set point in
;;;* screen A to the end of a line (click behind the line) and to set point in
;;;* screen B someplace.  Move the mouse over screen A to select it, but don't
;;;* click to set-point. Type a character. Move the mouse back over screen B and
;;;* set the point a couple of times by clicking (without dragging).  The cursor
;;;* does not move although the (internal) point follows correctly as one can
;;;* see when typing any character in screen B.  Drag calls redisplay
;;;* explicitly. But subsequent clicks continue to leave the cursor behind.
;;;* Also any even-numbered iteration of type-to-other-screen-click-here
;;;* releases redisplay but the next iteration blocks it again.
;;;*
;;;* 2. REDISPLAY should be changed to provide for FEEDBACK ON CURRENT REGION
;;;* automatically. For instance drag-button could be known to redisplay or
;;;* something like this. Smart redisplay would reconsider region marking only
;;;* at the end of top-level commands and would interrupt redisplay as soon as
;;;* input processing is pending for kbd OR button handler (bypassing moves and
;;;* drags in this short-cut logic). Commands could only rely on mark and point
;;;* during execution. Motion, like now, would force redisplay to happen but
;;;* would not control drag-button. It would simply move Point. drag-button
;;;* would be renamed to region-button.
;;;*
;;;* 3. The logic for translating mouse coordinates into minbuf points does
;;;* not reflect that the first line has a prompt. Yanking to minibuffer,
;;;* pasting etc work fine, but mouse-set-point and marking are are offset
;;;* on the first line.
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(provide 'sky-mouse)

(require 'backquote)
(require 'mini-cl)			;use cl if this is not around
(require 'thing)			;in epoch/contrib, we assume now it
					;is there always

;;;
;;; Figure where we are running and require necessary stuff
;;;

(setq running-epoch (boundp 'epoch::version))
(setq running-x-mouse
      (if (not running-epoch)
	(or (memq 'x-mouse features)
	    ;; make sure we have x-mouse loaded under X, Emacs may load
	    ;; x-mouse after .emacs and only when running under X. 
	    ;; Don't require x-mouse without check, cannot be loaded
	    ;; when running Emacs from dialup or other dumb remote terminal.
	    (and (getenv "DISPLAY")
		 (intern-soft "x-get-default")
		 (not (condition-case nil
			  (x-get-default "Heiliger Bim-Bam")
			(error t)))))))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; GLOBALS
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(defvar sky-state 'nil "save-kill-yank state machine")
(defvar yank-n 0)			;feedback for user.
(defvar *print-mouse-message* t)	;don't show messages in minibuf when 
					;mouse is used there

(defvar multi-click-timeout 400
  "* Maximal temporal difference between two successive clicks to be considered
as part of a multi-click sequence. Measured in milliseconds.")

(defvar multi-click-outspace 2
  "* Maximal spatial distance between two successive clicks to be considered
as part of a multi-click sequence. Measured in pixels.")

(defvar multi-click-hint nil
  "* Don't set it! Used in mouse command programming. Interactive commands
can be combined with a selection method using mouse-define.
Before a combined mouse commmand is executed, the handler figures whether the
currently handled click belongs to a multi-click sequence. This is done by
comparing the down transition events. Buttons and modifier information must
agree, coordinate and time info can vary a little limited by multi-click-timeout
and multi-click-outspace. The value of multi-click-hint is nil if 
this is a single click or the first click in a multi-click sequence.
Else it is an integer in {2,...}. The same combined command is executed for each
click in a multi-click sequence. This is limiting but does not require delays in
command dispatching and can be used to make the mouse function more uniformly:
We recommend to design the second, third ... (click) case of a command as 
a logical extension to the first, second, ... one.
Restriction: Currently this is only an approximation to the real thing.
Epoch lisp does not see the timestamps of X events. Multi-click reasoning
thus reflects the times of event arrival at the handler. This may be confusing
on slow machines and/or front-ends working over the network, and also
if a command is time consuming.
You may consider to set multi-click-timeout to 0 in such cases or restrict
multi-click functionality to guaranteed fast commands.")
;;Moreover the event loop runs outside of Epoch lisp and the event queue
;;is invisible (the x-... stuff does not seem to work), so a handler cannot 
;;integrate events based on timeout.

;; all initial lies to avoid time-consuming control in the handler
(defvar last-button-down-time 0) 
(defvar last-button-down-x-pix 0)
(defvar last-button-down-y-pix 0)
(defvar last-button-down-code 0)

(defvar *mouse-installation-spec* nil)
(defvar mouse-save-excursion-info nil
  "* Don't set it! Used in mouse command programming. Interactive commands
can be combined with a selection method using mouse-define.
Before a combined mouse commmand is executed, the state of the current buffer
is saved here as a list (MARK POINT BUFFER WINDOW &optional SCREEN). 
SCREEN is nil if multi-screens are not supported under this Emacs.
Mouse command functions can use these values to achieve an effect similar
to save-excursion.")

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; THING COMMANDS a la Zmacs
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;
;;; GENERIC THING RELATED COMMANDS
;;;

(defun mouse-mark-thing (arg &optional no-feedback)
  "Determines the thing under the mouse and marks it. For thing syntax cf.
mark-thing."
  (interactive "d") 
  (mouse-set-point arg)
  (mark-thing  (point) no-feedback))

(defun mouse-mark-thing-extend (arg)
  "Determines the thing under the mouse and marks it. Then starts dragging.
For thing syntax cf. mark-thing."
  (interactive) 
  (mouse-mark-thing arg)
  (mouse-extend-drag-point arg))

;;; In the following lines of code there is some undo stuff that takes
;;; care that the nth multi-click execution of a command undoes effects
;;; of previous executions. The forms marked with (*MC*)
;;; can be deleted once multi-click timeout work under a later version
;;; of Epoch.
(defvar multi-click-undo-length 0)	;(*MC*)
(defun mouse-yank-thing-to-point (arg &rest ignore)
  "The thing under the mouse is copied to the kill ring and yanked to
point as if this was an excursion of point. The yanked thing can be
killed immediately by the save/kill/yank command.  Works across Emacs
windows and Epoch screens. Use mouse-toggle-focus-screen to set focus
to yank destination for a series of yanks from other screens and
mouse-warp-to-point to warp the mouse back to the point (usually on
neighbor keys for the same modifiers).
This works with multi-click selection across windows and in a restricted
form in the same window (previous clicks in a multi-click sequence may have
moved the thing under the mouse)."
  (interactive "d")
  ;; make sure this works also across different buffer windows. Save-excursion 
  ;; won't do it across screens. Didn't check in newest Epoch version.
  (unwind-protect
      (let* ((point-w (selected-window))
	     (p (point))
	     (minibuf (minibuffer-window))
	     (*print-mouse-message*
	      (if (not (or (eq minibuf point-w)
			   (and running-epoch (eq minibuf (caddr arg))))) t))
	     thing-first-syntax)
	;; a multi click can actually select the wrong thing if the previous
	;; yank into the source same buffer implied a scroll of the current line.
	;; or the yank was inserted before point. Why fix it? Wrapped lines make
	;; any fix too complex. Even a symbol yank before point can make thing
	;; move.
	;; Select thing but display only if non-standard (multi-click) selection.
	(mouse-mark-thing arg (not multi-click-hint))
	;;(*MC*)
	(cond (multi-click-hint		; we are in point window, point after kill!
	       (setq p (- p multi-click-undo-length))
	       (delete-region p (+ p multi-click-undo-length))
	       (pop kill-ring)))	
	;; don't concat to previous yank
	(setq last-command 'anything-but-yank
	      this-command 'mouse-yank-thing-to-point)
	(setq thing-first-syntax (char-syntax (char-after (mark))))
	(copy-region-as-kill (mark) (point))
	(select-window point-w)
	(goto-char p)
	;; don't lump successive yanks, space if right of word or symbol
	(if (cond ((equal (point) (point-min)) nil) ; at begin of something
		  ((memq (char-syntax (char-after (1- (point)))) '(?w ?_ ?\"))
		   ;; after word or symbol "bla"
		   (cond ((= thing-first-syntax ?w) t)
			 ((eq major-mode 'emacs-lisp-mode) t)
			 ((eq major-mode 'lisp-mode) t)
			 (t nil)))
		  ((= (char-syntax (char-after (1- (point)))) ?\))
		   (cond ((eq major-mode 'emacs-lisp-mode) t)
			 ((eq major-mode 'lisp-mode) t)
			 (t nil))))
	    (insert " "))
	(yank)
	(setq multi-click-undo-length (- (point) p))
	(redisplay-screen)		; 'hanging redisplay bug'; remove when fixed
	(sky-state-saved)		; allow save/kill/yank kill continuation
	(setq last-command 'yank)	; allow subsequent yank-pop's
	)))

;;; Set-focus, i.e. kbp input stays with current screen (or window in
;;; plain emacs). 
(defun mouse-toggle-focus-screen (arg &rest ignore)
  "Toggles whether the screen pointed to with the mouse holds the focus.
The first toggle click on a screen sets the focus. The next releases it.
Clicking to another screen moves the focus."
  (interactive "d")
  (let ((scr (nth 3 arg)))
    ;; must set point to understand where we are.
    ;; but do not support it on the minibuffer for now;
    ;; it works ... but commands like top-level select another screen
    ;; without going through select-screen ...
    (if (eq scr (minibuf-screen))
	(progn (beep) (message "Come on. Minibuffer focuses automatically.")) 
      (progn
	(if (eq scr current-focus-screen)
	    (release-kbd-focus)
	  (set-kbd-focus scr))
	(mouse-set-point arg)))
    ))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; SAVE/KILL/YANK MACHINE a la Zmacs
;;;
;;; Given a region, throw a series of save/kill/yank requests to a buffer.
;;; The first one saves, the next kills, then yank back the kill (if this was
;;; not what what you wanted), subsequent ones yank-pop.
;;; Any other region related commands reset the machine, i.e. leave the buffer
;;; region-less, so that the next s/k/y request can yank the top of the 
;;; kill ring. Usually the saved region. 
;;; Yank-to-point additionally allows to fix the insertion point and stream 
;;; into it a series of kills from different sources (like the procedure calls
;;; and body templates when procedures are composed).
;;; Since this logic is target oriented (the target point is fixed)
;;; and the visual marking provides a clear feedback, the logic
;;; nicely complements the cut buffer button logic of Epoch which
;;; is source oriented (the source point is fixed). If the keys are
;;; kept somewhat independent one gets quickly used to it without
;;; much danger of confusion and starts to like the fast way 
;;; things can changed reliably. 
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 
(defun sky-state-reset ()		; forget all region related state
  (setq sky-state nil
	yank-n 0))

(defun sky-state-saved (&optional mess)
  ;; move into saved state
  (setq sky-state 'saved       ; allow save/kill/yank to kill away immediately
	this-command 'sky
	yank-n 0) 
  (if *print-mouse-message* (message "Saved")))

(defun sky-state-yanked ()  
  (mark-region (region-beginning) (region-end))
  (setq last-command 'yank
	sky-state 'yanked
	yank-n (1+ yank-n))
  (if *print-mouse-message* (message (format "Yank %s" yank-n))))

;;; interact properly with the global save kill yank state.
;;; this-command, last-command
(defun mouse-save-kill-yank (arg)
  "Save/Kill/Yank operates on a currently marked region. The first click
saves (copies to kill-ring). The next click kills the region. A following
click yanks back the killed region and successive clicks then yank-pop from
the kill ring (exchanging the current region with the kill-ring top).
This sequence is interrupted by any other command.
If there is no region to begin with, the top of the kill-ring is yanked
to point and the yank-pop sequence starts.
Programs use sky-state-reset to reset the save/kill/yank machine."
  (mark-region (region-beginning) (region-end))
  (interactive "d")
  (let ((*print-mouse-message*
	 (if (not (and running-epoch (eq (caddr arg) (minibuffer-window)))) t)))
    (if (and sky-state 
	     ;; under Emacs x-mouse these are 'always' flush-mouse-queues
	     (if running-epoch (eq this-command 'sky)
	       t))
	;; parallel Emacs state check means emacs commands
	;; can also reset the sky-machine.
	(cond ((eq sky-state 'saved) 
	       ;; was just previously saved, don't kill, but
	       ;; make sure that X get's the right picuture.
	       (store-cut-buffer (mark) (point))
	       (delete-region (mark) (point))
	       (setq sky-state 'killed)
	       (setq yank-n 0)
	       (if *print-mouse-message* (message "Killed")))
	      ((eq sky-state 'killed) 
	       (yank) (sky-state-yanked))
	      ((eq sky-state 'yanked)
	       (if running-x-mouse (setq last-command 'yank))
	       (yank-pop 1) (sky-state-yanked))
	      (t (sky-start-save-kill-yank)))
      ;; else reset machine
      (sky-start-save-kill-yank)))
  (setq this-command 'sky))

(defun sky-start-save-kill-yank ()
  (cond ((/= (mark) (point))		; there's a region
	 (copy-region-as-kill (mark) (point))
	 (store-cut-buffer (mark) (point))
	 (sky-state-saved))
	(t (yank) (sky-state-yanked))))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; THE BASIC THING based on thing.el
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(defun mark-thing (point &optional no-feedback)
  "Gets the thing containing point (first arg) and makes it the current 
region. Optionally provides visual feedback for thing (second arg NIL).
For thing syntax cf. thing.el."
  (let* ((thing 
	  (save-excursion 
	    (goto-char point)
	    (cond ((and comment-start (looking-at comment-start))
		   (thing-comment point))
		  (multi-click-hint (multi-click-thing-boundaries point))
		  (t (thing-boundaries point)))))
	 (from (car thing))
	 (to (cdr thing)))
    (set-mark from) (goto-char to)
    (if (not no-feedback) (mark-region from (point)))))

;;; thing extension for multi-clicks, cf. MOUSE-TUTORIAL
(defvar thing-multi-click-boundary-alist
  ;; make sure we get the same larger picture wherever we click 
  '((?w thing-symbol thing-get-line thing-sentence thing-paragraph) 
    (?_ thing-symbol thing-get-line thing-sentence thing-paragraph)
    (?. thing-symbol thing-get-line thing-sentence thing-paragraph)
    (?  thing-get-line thing-get-line thing-sentence)
    ;; the first click resets to thing here.
    (?\( thing-backward-up thing-backward-up thing-backward-up thing-backward-up)
    (?\) thing-backward-up thing-backward-up thing-backward-up thing-backward-up))
  "* An alist associating thing selection functions to multi clicks on a specific
character syntax.")

(defun multi-click-thing-boundaries (here)
  (let* ((s (char-syntax (char-after here)))
	 (selection (nth (- multi-click-hint 2)
			 (cdr (assq s thing-multi-click-boundary-alist)))))
    (or (and selection (funcall selection here))
	(thing-boundaries here))))
  
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; MISCELLANEOUS MOUSE COMMANDS 
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

;;;
;;; FREE-HAND Kill (like Emacs x-mouse default ctrl-Middle).
;;;

(defun mouse-kill-region (arg &rest ignore)
  "Kills the current region. The killed text is also saved in the X cut buffer
to be yanked to other application."
  (interactive "d")
  (mouse-end-drag-point arg)
  (store-cut-buffer (mark) (point))
  (kill-region (mark) (point))
  )

;;;
;;; FORCIBLE SET-POINT, also moves kbd input to point (exit minbuf if)
;;;

(defun mouse-set-point-force-kbd (arg)
  "Sets point to mouse if possible and interrupts minibuffer interaction to
redirect kbd input to point. Does NOT release the kbd focus if this
is set. Used in a series of mouse commands that require type-in like
space/return between successive yanks."
  (interactive "d") 
  (mouse-set-point arg)
  ;;(top-level)   ;; recenters screen or selects other buffer sometimes?
  (condition-case error	(throw 'search-done t) (no-catch nil))
  (keyboard-quit) ;; if no isearch, exit anyway
  )

;;; 
;;; MACRO MOUSE  ( not Mouse Macros, what a pity !)
;;;

(defun mouse-execute-kbd-macro (arg &rest ignore)
  "Execute keyboard macro at point under mouse."
  (interactive "d")
  (mouse-set-point arg)
  (call-last-kbd-macro))

;;;
;;; MOUSE TAGS SEARCH
;;;
;;; Function arglist and documentation; assuming we have setup tags
;;; table.  When typing things one usually ends up in a syntactically
;;; incomplete arglist when a description can be of help. Also after
;;; yanking calls one is close with the mouse either at the source or
;;; at the destination.  Put this function close to the button codes
;;; for yank-to-point.
(defun mouse-tags-search (arg)
  "Get thing under mouse according to the current mode syntax and -- assuming 
it is a name -- lookup its documentation via tags search."
  (interactive "d")
  (mouse-mark-thing arg)
  (let ((name (buffer-string (region-beginning) (region-end))))
    (tags-search name)))

;;;
;;; MOUSE INDENT OR FILL
;;; a versatile command for language modes. Allows to forget the difference
;;; between m-g, m-q, m-c-q and friends.

(defun mouse-fill-or-indent (arg &rest ignore) 
  "Fills or indents the currently selected region determined by a click
or a free-hand drag.  With a free-hand drag the command applies to the
lines of the marked region. With a click it applies to the current
paragraph in text mode and mail mode but to the current thing in any
other mode (which is assumed to be a language mode).  When the region
starts with COMMENT-START it is filled otherwise it is indented. A
paragraph is filled and a thing is indented.

Indent: For indentation the function indent-region is called whatever
this happens to be in the current mode.  (It works on things since
these are marked before command execution).

Fill: uses fill-region for regions and fill-paragraph for paragraphs.
For a region, the region beginning defines the fill-prefix. It extends
from the COMMENT-START to the first white space and includes
subsequent white space so that indentation is preserved. The
fill-prefix is set only temporarily so that different styles of
comments can be formatted successively. Paragraphs are filled with
fill-prefix set to \"\" so that it makes a difference whether you
click to or mark a paragraph.

Cf. the documentation of fill-region and fill-paragraph for more details
of filling and the use of fill-prefix."
  (interactive "d")
  (mouse-end-drag-point arg)
  (unwind-protect
      (let* ((p (region-beginning))
	     (old-pref fill-prefix)	;save 
	     ;; get fill-prefix if we are at in a comment
	     (fillp (or (save-excursion 
			  (goto-char p)
			  (when  (progn (beginning-of-line)
					     (skip-chars-forward " \t")
					     (and comment-start (looking-at comment-start)))
				 (forward-char (length comment-start))
				 (re-search-forward "[ \t]")
				 (while (looking-at "[ \t]")
				   (forward-char 1))
				 (set-fill-prefix)
				 t))
			(equal mode-name "Text Mode")
			(equal mode-name "Mail"))))
	(cond ((= (mark) (point))
	       (cond (fillp	(setq fill-prefix "")
				(fill-paragraph nil))
		     (t (mark-thing (point))
			(indent-region (mark) (point) nil))))
	      (fillp (fill-region (mark) (point)))
	      (t (indent-region (mark) (point) nil)))
	(setq fill-prefix old-pref)
	(forget-region)
	)))

;;;
;;;  ERROR and NOOP
;;;

(defun mouse-beep (arg)
  (interactive)
  (beep))

(defun mouse-ignore (arg)
  (interactive))

(defun mouse-quit-kbd (arg)
  (interactive)
  (keyboard-quit))

(defun mouse-message (arg)
  "Shows what the mouse handler thinks is going in."
  (interactive)
  (message (format "Argument: %s, Multi-click: %s" arg multi-click-hint)))

(defun mouse-execute-at-point (arg)
  "Redefine this to run some emacs fn at point."
  (interactive)
  (beep)
  (message "Redefine fn mouse-execute-at-point to run here."))

(defun mouse-execute-on-region (arg)
  "Redefine this to run some emacs fn  on the current region." 
  (interactive)
  (beep)
  (mouse-end-drag-point arg)
  (message "Redefine fn mouse-execute-on-region to run on this region."))

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; ONLINE MOUSE LINE now that also supported under Emacs, common parts
;;; are moved here.

(defvar mouse-glyph 84)			;mouse glyph
(defvar button-glyph-list '(74 82 100)) ;left middlle right glyph
(defvar help-exit-
      "RET to exit, press or release modifiers and buttons to see more.")

(defun describe-mouse-event (handler downp btn controlp metap shiftp)
  (cond (handler 
	 (condition-case error (describe-function handler) (error nil))
	 (message 
	  (format "%s runs %s, %s" 
		  (mouse-button-string downp btn controlp metap shiftp t) 
		  handler
		  help-exit- )))
	(t (with-output-to-temp-buffer "*Help*"
	     (princ "Button transition undefined.\n")
	     (princ help-exit-) 
	     (message 
	      (format "%s undefined, (RET to exit)." 
		      (mouse-button-string downp btn controlp metap shiftp t) 
		      handler))))))

(defun mouse-button-string (downp btn controlp metap shiftp &optional visual)
  (if visual (cursor-glyph 
	      (if downp (nth btn button-glyph-list) mouse-glyph)))
  (setq btn (nth btn '("Left" "Middle" "Right")))
  (mapconcat 'identity
	     (append (if controlp (list "C"))
		     (if metap (list "M"))
		     (if shiftp (list "S"))
		     (list btn (if downp "Down" "Up")))
	     "-"))

;;;
;;; Generic help
;;; 

(defun help-with-mouse-tutorial (&optional ignore)
  "Select the learn-by-doing mouse tutorial."
  (interactive)
  (let ((file (expand-file-name "~/MOUSE-TUTORIAL")))
    (delete-other-windows)
    (if (get-file-buffer file)
	(switch-to-buffer (get-file-buffer file))
      (switch-to-buffer (create-file-buffer file))
      (setq buffer-file-name file)
      (setq default-directory (expand-file-name "~/"))
      (setq auto-save-file-name nil)
      (insert-file-contents *sky-mouse-tutorial*)
      (goto-char (point-min))
      (scroll-up 7))))

(define-key help-map "Z" 'help-with-mouse-tutorial)
(define-key help-map "z" 'help-with-mouse)

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; Generic multi-click reasoning must be based on different
;;; base functions in x-mouse and epoch. Called by handlers.

(defun provide-multi-click-hint (button-code time coords)
  (let ((downx (pop coords))
	(downy (pop coords)))
    (setq multi-click-hint
	  (if (and (< (- time last-button-down-time) multi-click-timeout)
		   (= button-code last-button-down-code)
		   (let ((dx (- downx last-button-down-x-pix))
			 (dy (- downy last-button-down-y-pix)))
		     ;; don't rely on having cl around.
		     (if (< dx 0) (setq dx (- 0 dx)))
		     (if (< dy 0) (setq dy (- 0 dy)))
		     (and (< dx multi-click-outspace)
			  (< dy multi-click-outspace)))
		   )
	      (if multi-click-hint (1+ multi-click-hint) 2)
	    nil)
	  last-button-down-time time
	  last-button-down-x-pix downx
	  last-button-down-y-pix downy
	  last-button-down-code button-code)))
    
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; High level command definition.
;;; Allow to separate mouse commands, appropriate selection method and
;;; installation. A high-level command is a pair 
;;; (sel,com) sel being installed on the button down transition, com
;;; on the up transition, such that sel can provide feedback and allows
;;; the user to exit from the command before executing com.
;;; Feedback is provided in terms of mouse cursor glyphs where available.
;;; Feedback is associated to the pair (sel,com) and is to be respected
;;; by the mouse handler and/or the selection methods implemented in the
;;; host mouse base, prefering
;;;          - explicitly registered glyph, then
;;;          - selection method feedback for sel in
;;;            '(:point :drag :extend :thing), then
;;;          - default cursor for mouse down buffer, then
;;;          - default cursor.
;;; in this order.
;;; Details cf. online documentation of mouse-define.
;;;
;;; Note: mouse-define wraps the command proper in two command functions that
;;; are installed on the down and up transition. The functions are small and not
;;; compiled to avoid requiring byte-compile always at runtime. 

(defun mouse-define (selection command &optional buttoncode feedback map)
  "SELECTION and COMMAND are two mouse commands that make up a `combined mouse
command'. Each of these is an interactive function of one argument, a list (PT
BUFFER WINDOW SCREEN) representing the abstract mouse event. 

The two parts of the combined mouse command are installed on the down and up
transition event for BUTTONCODE, the third argument when called from a program.
BUTTONCODE is a string like \"C-M-S-L\" of letters in the following
order with the following meaning:
C                     control modifier
M                     meta modifier
S                     shift modifier
Left,Middle,Right     the corresponding button (only one of them). 
This representation is translated by the host mouse base to the proper code.

FEEDBACK is a cursor glyph acceptable to the host mouse system or nil, meaning to
provide default feedback. Feedback will be provided before executing SELECTION
and until before executing COMMAND. For `busy' feedback (watch) use the macro
with-busy-feedback in the command body.

MAP is either the keyword symbol :local (the default) or :global and specifies
which command table to install the combined command.

Typically SELECTION is a selection method defining the object on which COMMAND
is going to work. Predefined selection methods are

 SELECTION                          COMMAND works 

mouse-set-point                     from point -- region is empty
mouse-drag-point                    on region defined by freehand drag
mouse-extend-drag-point             an extension of previous region
mouse-mark-thing                    a region defined by the language syntax

Each of these methods (and for that matter any new command starting with a call
to one of these) saves the current point, buffer and such for COMMANDs that want
to do excursions. Cf. variable mouse-save-excursion-info.
Dragging is only possible after mouse-drag-point and mouse-extend-drag-point.
And in this case COMMAND must start with a call to mouse-end-drag-point to
complete the dragging interaction.

For command programming cf. the related variables, mouse-save-excursion-info,
multi-click-hint, multi-click-timeout, multi-click-outspace."
  (interactive "CSelection: \nCMouse command: ")
  ;; get missing arguments
  (if (not map) (setq map ':local))	;x-mouse does not support distinction
  (cond ((not buttoncode)
	 (setq *mouse-installation-spec* (list selection command feedback map))
	 ;; host mouse base can do it by allowing user to click the mouse
	 (mouse-get-button-spec-and-install-command))
	(t  (mouse-define-internal selection command buttoncode
				       feedback map))))

;;; a little more readable from programs, nothing optional
(defun defmouse (map button selection command &optional feedback)
  "A non-interactive version of mouse-define to be called from programs. 
Less optional parameters. Checking instead of interactive prompts.
Arguments are MAP BUTTON SELECTION COMMAND &OPTIONAL FEEDBACK."
  (mouse-define selection command button feedback map))

(defvar :global ':global)		;make things a little more readable.
(defvar :local ':local)			;just in case CL users take doc as is.

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; SCROLLING as far as independent of mouse base.
;;;
;;; fast scrolls until someone comes up with Emacs scrollbars under
;;; olwm.
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(defun scroll-point-to-mouse (&optional arg)
  "Scroll such that saved point ends up as close as possible to current point."
  (interactive)
  (mouse-set-point arg)
  (let ((opoint (nth 1 mouse-save-excursion-info))
	(obuf (nth 2 mouse-save-excursion-info))
	(wp (window-point-line)))
    (when (not (eq obuf (current-buffer))) (switch-to-buffer obuf))
    (goto-char opoint)
    ;;scroll this point to where mouse was pointing (p)
    (recenter wp)))

;;; Make named mouse commands for easier reference, doc and arg compatibility.
;;; Arg is optional, wo we can invoke this from keyboard too.

(defun this-many-lines-down (&optional arg)
  "Scroll down that many lines point is away from window start measured in lines,
start line inclusive."
  (interactive)
  (scroll-down (1+ (window-point-line))))

(defun goto-ratio-linewise (&optional numerator denominator)
  "Scrolls to a fraction of the buffer measured in lines from the beginning
of the buffer. NUMERATOR and DENOMINATOR are positive numbers defining a
fraction in [0,1]."
  (interactive "nNumerator: \nnDenominator: ")
  (when (and (<= numerator denominator)
	     (>= numerator 0)
	     (> denominator 0))
	(let* ((buffer-height (count-lines (point-min) (point-max))))
	  (goto-line (if (< buffer-height 1000)
		       (/ (* numerator buffer-height) denominator)
		       (/ (* numerator (/ buffer-height 100)) 
			  (/ denominator 100)))))))

(defun goto-percent-linewise (&optional arg)
  "Scrolls to a percentage of buffer defined by the distance of point's line and
window top relative to window height. For instance, if point's line is 1/3 of
the current window (from top), then we scroll to 1/3 of the current buffer
measured vertically, i.e. in the number of lines."
  (interactive)
  (let ((permill (window-permill)))
    (message (format "V-scroll to %s/%s" permill 1000))
    (goto-ratio-linewise permill 1000)))

;;; Scrolling auxiliaries

(defun line-number (here) 
  (count-lines 1 (save-excursion (goto-char here) (beginning-of-line) (point))))
(defun window-point-line () (- (line-number (point)) (line-number (window-start))))
(defun window-permill ()
  ;; make this pixel calc. to get finer granularity once we see pixels in Epoch
    (/ (* 1000 (window-point-line))
       (- (window-height) 2)) ;not counting mode-line and last line 
       )

;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; LOAD THE RIGHT HOST MOUSE BASE
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(cond (running-epoch (require 'epoch-mouse-base "ep-sky-bas"))
      (running-x-mouse (require 'x-mouse "x-sky-bas"))
      (t (require 'poor-mans-mouse "poor-mouse")))

;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;; POLISH HELP (C-h) documentation
;;;~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 
(defvar help-key ?\C-h)

(defun help-for-help ()
  "You have typed \\[help-command], the help character.  Type a Help option:

A  command-apropos.   Give a substring, and see a list of commands
              (functions interactively callable) that contain
	      that substring.  See also the  apropos  command.
B  describe-bindings.  Display table of all key bindings.
C  describe-key-briefly.  Type a command key sequence;
	      it prints the function name that sequence runs.
F  describe-function.  Type a function name and get documentation of it.
I  info. The  info  documentation reader.
K  describe-key.  Type a command key sequence;
	      it displays the full documentation.
L  view-lossage.  Shows last 100 characters you typed.
M  describe-mode.  Print documentation of current major mode,
	      which describes the commands peculiar to it.
N  view-emacs-news.  Shows emacs news file.
S  describe-syntax.  Display contents of syntax table, plus explanations
T  help-with-tutorial.  Select the Emacs learn-by-doing tutorial.
V  describe-variable.  Type name of a variable;
	      it displays the variable's documentation and value.
W  where-is.  Type command name; it prints which keystrokes
	      invoke that command.
z  help-with-mouse. Press/release mouse buttons; it describes the 
              button and the command it runs.
Z  help-with-mouse-tutorial. Select a learn-by-doing tutorial for mouse commands.

C-c print Emacs copying permission (General Public License).
C-d print Emacs ordering information.
C-n print news of recent Emacs changes.
C-w print information on absence of warranty for GNU Emacs."
  (interactive)
  (message
 "A B C F I K L M N S T V W z Z C-c C-d C-n C-w.  Type \\[help-command] again for more help: ")
  (let ((char (read-char)))
    (if (or (= char help-key) (= char ??))
	(save-window-excursion
	  (switch-to-buffer "*Help*")
	  (erase-buffer)
	  (insert (documentation 'help-for-help))
	  (goto-char (point-min))
	  (while (memq char (list help-key ?? ?\C-v ?\ ?\177 ?\M-v))
	    (if (memq char '(?\C-v ?\ ))
		(scroll-up))
	    (if (memq char '(?\177 ?\M-v))
		(scroll-down))
	    (message "A B C F I K L M N S T V W Z z  C-c C-d C-n C-w%s: "
		     (if (pos-visible-in-window-p (point-max))
			 "" " or Space to scroll"))
	    (let ((cursor-in-echo-area t))
	      (setq char (read-char))))))
    (if (not (equal char ?Z)) (setq char (downcase char)))
    (let ((defn (cdr (assq char (cdr help-map)))))
      (if defn (call-interactively defn) (ding)))))




