%{

#ifndef lint
static char *RCSid = "$Id: yaccsrc.y,v 1.15 1993/05/07 21:31:04 anders Exp anders $";
#endif

/*
 *  The Regina Rexx Interpreter
 *  Copyright (C) 1992  Anders Christensen <anders@solan.unit.no>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version. 
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: yaccsrc.y,v $
 * Revision 1.15  1993/05/07  21:31:04  anders
 * Lots of changes to optimize. Constant parts of expressions are cached.
 * Compound variables are stored as a tree. And the initial part of an
 * analysis pass over the parse tree has been created.
 *
 * Revision 1.14  1993/02/09  17:03:44  anders
 * Renamed Str*() to Str_*(), to avoid probl. with case insensitivity
 * Fixed completely incorrect parsing of the ADDRESS clause.
 * Added support to handle initial whitespace on the first line
 * Added support to handle some error condition (e.g. missing then)
 * Fixed up syntax of NUMERIC, more resistent to errors.
 * Assignment can take an empty expression, CMS compatibility.
 * Added support for the explicit use of "=" in templates
 * Treats both bin and hex string as 'normal' strings after parsing.
 * Added support for indirect references to vars in drop.
 *
 * Revision 1.13  1992/07/24  04:00:25  anders
 * Added GPL. Fixed varius bugs. Several minor changes. Improved syntax.
 *
 * Revision 1.12  1992/04/25  18:43:20  anders
 * Fixed bug, lexsrc able to send HEXSTRINGs containing
 *    null characters.
 *
 * Revision 1.11  1992/04/25  13:18:30  anders
 * Converted to REXX strings
 *
 * Revision 1.10  1992/04/05  20:37:15  anders
 * Added copyright notice
 * Made changes to DROP, ADDRESS and TRACE
 *
 * Revision 1.9  1992/03/22  18:58:46  anders
 * Made strings allocated through stralloc() allign to 8 byte
 *    boundaries.
 *
 * Revision 1.8  1992/03/22  00:42:40  anders
 * Added #include <stdio.h> as this is not done in rexx.h any more.
 * Added #include <alloca.h> for use on SGI
 * Removed %expect 7 directive, which is not compatible with yacc
 *
 * Revision 1.7  1992/03/01  19:29:33  anders
 * Changed the support for templates, to fix a bug.
 *
 * Revision 1.6  1990/12/11  00:35:33  anders
 * Removed (another) bug related to line numbers (probably the last
 *     one!). Just an expression (used for external commands) does not
 *     have a 'reserved' word, and a special dummy rule had to be
 *     inserted.
 *
 * Revision 1.4  90/12/11  00:00:15  anders
 * Removed bug that assigned wrong line/character number to the 
 *     leave statement. There might still be some trouble with getting 
 *     the linenumbers of the statements correct.
 * 
 * Revision 1.3  90/12/10  18:41:09  anders
 * Removed bug: the linenumbers of the nop-statement were shifted one
 *     line up. The tline/tchar variables used instead of next{line,char}
 * 
 * Revision 1.2  90/12/10  04:10:45  anders
 * Removed bug which set linenumbers to incorrect values. They got 
 *     the (integer) value of a pointer.
 * 
 * Revision 1.1  90/08/08  02:28:52  anders
 * Initial revision
 * 
 * Revision 1.1  90/08/08  02:23:11  anders
 * Initial revision
 * 
 */

#include "rexx.h"
/* #if defined(SGI)
 * #include <alloca.h>
 * #endif
 */
#include <ctype.h>
#include <stdarg.h>
#include <stdio.h>
#include <assert.h>


#define YYSTYPE nodeptr

extern char retvalue[] ;
extern int retlength ;
extern int nextstart, nextline, tline, tstart ;
extern sysinfobox *systeminfo ;
extern nodeptr parseroot ;

void checkconst( nodeptr this ) ;
void transform( nodeptr this ) ;
nodeptr create_head( char *name ) ;
void checkout( void ) ;
nodeptr makenode( int type, int numb, ... ) ;
streng *stralloc( char *ptr ) ;
void checkdosyntax( nodeptr this ) ;
void newlabel( nodeptr this ) ;

%}

%token ADDRESS ARG COMMAND CALL DO TO BY FOR WHILE UNTIL EXIT IF THEN 
%token ELSE ITERATE INTERPRET LEAVE NOP NUMERIC PARSE EXTERNAL SOURCE
%token VAR VALUE WITH PROCEDURE EXPOSE PULL PUSH QUEUE RETURN SAY 
%token SELECT WHEN DROP OTHERWISE SIGNAL ON OFF ERROR SYNTAX HALT 
%token NOVALUE TRACE END NULLSTRING UPPER INTERMEDIATES LABELS 
%token ASSIGNMENTVARIABLE STATSEP FOREVER ALL COMMANDS ERRORS 
%token NORMAL NEGATIVE RESULTS SCAN DIGITS FORM FUZZ SCIENTIFIC
%token ENGINEERING BANGBANG BANGWHAT BANG WHATBANG WHATWHAT WHAT 
%token NOT CONCATENATE MODULUS GTE GT LTE LT DIFFERENT EQUALEQUAL
%token NOTEQUALEQUAL OFFSET SPACE EXP XOR PLACEHOLDER NOTREADY
%token CONSYMBOL SIMSYMBOL EXFUNCNAME INFUNCNAME LABEL DOVARIABLE
%token SYMBOL HEXSTRING STRING VERSION LINEIN WHATEVER NAME FAILURE
%token BINSTRING ENVIRONMENT

%start prog

%left '|' XOR
%left '&'
%nonassoc '=' '>' '<' DIFFERENT GTE GT LT LTE EQUALEQUAL NOTEQUALEQUAL
%left CONCATENATE SPACE CCAT
%left '+' '-'
%left '*' '/' '%' MODULUS
%left EXP
%left UMINUS UPLUSS NOT

%{ 
#ifdef NDEBUG
# define YYDEBUG 0
#else
# define YYDEBUG 1
#endif
%}

%%

prog         : whitespace nseps stats  { parseroot = $3 ; checkout() ; }
             | whitespace nseps        { parseroot = NULL ; checkout() ; }
             ;


stats        : statement stats         { $$->next = $2 ; }
             | statement               { $$->next = NULL ; }
             ;

nstats       : stats                   { $$ = $1 ; }
             |                         { $$ = NULL ; }
             ;

nseps        : seps
             | 
             ;

seps         : STATSEP seps
             | STATSEP 
             ;

statement    : address_stat
             | expr_stat
             | arg_stat
             | call_stat
             | do_stat
             | drop_stat
             | exit_stat
             | if_stat 
             | unexp_then
             | unexp_else
             | ipret_stat
             | iterate_stat
             | label_stat
             | leave_stat
             | nop_stat
             | numeric_stat
             | parse_stat
             | proc_stat
             | pull_stat
             | push_stat
             | queue_stat
             | return_stat
             | say_stat
             | select_stat
             | signal_stat
             | trace_stat
             | assignment
             ;

call         : CALL		       { $$ = makenode(X_CALL,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
do           : DO 		       { $$ = makenode(X_DO,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
exit         : EXIT		       { $$ = makenode(X_EXIT,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
if           : IF		       { $$ = makenode(X_IF,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
iterate      : ITERATE  	       { $$ = makenode(X_ITERATE,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
leave        : LEAVE		       { $$ = makenode(X_LEAVE,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
return       : RETURN		       { $$ = makenode(X_RETURN,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
say          : SAY		       { $$ = makenode(X_SAY,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
address      : ADDRESS                 { $$ = makenode(X_ADDR_N,0) ;
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
arg          : ARG                     { $$ = makenode(X_PARSE_ARG_U,0) ;
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
drop         : DROP                    { $$ = makenode(X_DROP,0) ;
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
interpret    : INTERPRET               { $$ = makenode(X_IPRET,0) ;
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
label        : LABEL                   { $$ = makenode(X_LABEL,0) ;
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
nop          : NOP                     { $$ = makenode(X_NULL,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; }
numeric      : NUMERIC                 { $$ = makenode(0,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
parse        : PARSE                   { $$ = makenode(0,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
proc         : PROCEDURE               { $$ = makenode(X_PROC,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
pull         : PULL                    { $$ = makenode(X_PULL,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
push         : PUSH                    { $$ = makenode(X_PUSH,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
queue        : QUEUE                   { $$ = makenode(X_QUEUE,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
select       : SELECT                  { $$ = makenode(X_SELECT,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
signal       : SIGNAL                  { $$ = makenode(X_SIG_LAB,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
when         : WHEN                    { $$ = makenode(X_WHEN,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
otherwise    : OTHERWISE               { $$ = makenode(X_OTHERWISE,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
trace        : TRACE                   { $$ = makenode(X_TRACE,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;
address_stat : address VALUE expr seps { $$ = $1 ;
                                         $$->type = X_ADDR_V ;
                                         $$->p[0] = $3 ; }
             | address seps            { $$ = $1 ;
                                         $$->type = X_ADDR_S ; }
             | address nvir nexpr seps { $$ = $1 ;
                                         $$->type = X_ADDR_N ;
                                         $$->p[0] = $3 ;
                                         $$->name = (streng *)$2 ; }
             | address expr seps       { $$ = $1 ;
                                         $$->type = X_ADDR_V ;
                                         $$->p[0] = $2 ; }
             ;

arg_stat     : arg templs seps         { $$ = $1 ;
                                         $$->p[0] = $2 ; }
             ;

call_stat    : call asymbol whitespace exprs seps { $$ = $1 ;
                                         $$->p[0] = $4 ;
                                         $$->name = (streng *) $2 ; }
             | call on c_action namespec seps  
                                       { $$ = $1 ;
                                         $$->type = X_CALL_SET ;
                                         $$->p[0] = $2 ;
                                         $$->name = (streng *)$4 ;
                                         $$->p[1] = $3 ; }
             | call off c_action seps  { $$ = $1 ;
                                         $$->type = X_CALL_SET ;
                                         $$->p[0] = $2 ;
                                         $$->p[1] = $3 ; }
             ;

whitespace   : SPACE
             | CCAT
	     |
             ; 

nothing      :                         { $$ = makenode(X_COMMAND,0) ; 
                                         $$->charnr = tstart ; 
				         $$->lineno = tline ; }
             ;

expr_stat    : nothing expr seps       { $1->p[0] = $2 ; 
                                         $$ = $1 ; } 
             ;

end          : END nsimsymb            { $$ = makenode(X_END,0) ;
                                         $$->name = (streng*)($2) ;
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } ;

do_stat      : do repetitor conditional seps nstats end seps 
                                       { $$ = $1 ;
                                         $$->p[0] = $2 ;
                                         $$->p[1] = $3 ;
                                         $$->p[2] = $5 ; 
                                         $$->p[3] = $6 ; 
                          if (($$->p[0])&&($$->p[0]->name)&&
                              ($$->p[3]->name)&&
                              (($$->p[3]->name->len != $$->p[0]->name->len)||
                               (strncmp($$->p[3]->name->value,
                                        $$->p[0]->name->value,
                                        $$->p[0]->name->len))))
                                           exiterror( ERR_UNMATCHED_END ) ;
                                       }
					    
             ;

repetitor    : dovar '=' expr tobyfor tobyfor tobyfor
                                       { $$ =makenode(X_REP,4,$3,$4,$5,$6) ;
                                         $$->name = (streng *)$1 ; 
                                         checkdosyntax($$) ; }
             | FOREVER                 { $$ = makenode(X_REP_FOREVER,0) ; }
             | expr                    { $1 = makenode(X_DO_FOR,1,$1) ;
                                         $$ = makenode(X_REP,2,NULL,$1) ; } 
             |                         { $$ = NULL ; }  
             ; 

nvir         : ENVIRONMENT             { $$ = (nodeptr)stralloc(retvalue) ; }
             ;

dovar        : DOVARIABLE              { $$ = (nodeptr)stralloc(retvalue) ; }
             ;

tobyfor      : TO expr                 { $$ = makenode(X_DO_TO,1,$2) ; }
             | FOR expr                { $$ = makenode(X_DO_FOR,1,$2) ; }
             | BY expr                 { $$ = makenode(X_DO_BY,1,$2) ; }
             |			       { $$ = NULL ; }
             ;

conditional  : WHILE expr              { $$ = makenode(X_WHILE,1,$2) ; }
             | UNTIL expr              { $$ = makenode(X_UNTIL,1,$2) ; }
             |                         { $$ = NULL ; }
             ;

drop_stat    : drop anyvars seps       { $$ = $1 ;
                                         $$->p[0] = $2 ; }
             ;

exit_stat    : exit nexpr seps         { $$ = $1 ;
                                         $$->p[0] = $2 ; }
             ;

if_stat      : if expr nseps THEN nseps statement  
                                       { $$ = $1 ;
                                         $$->p[0] = $2 ;
                                         $$->p[1] = $6 ; } 
             | if expr nseps THEN nseps statement ELSE nseps statement 
                                       { $$ = $1 ;
                                         $$->p[0] = $2 ;
                                         $$->p[1] = $6 ;
                                         $$->p[2] = $9 ; } 
             ;

unexp_then   : THEN                    { exiterror( ERR_THEN_UNEXPECTED ) ; }
             ;

unexp_else   : ELSE                    { exiterror( ERR_THEN_UNEXPECTED ) ; }
             ;

ipret_stat   : interpret expr seps     { $$ = $1 ;
                                         $$->p[0] = $2 ; }
             ;


iterate_stat : iterate simsymb seps    { $$ = $1 ; 
                                         $$->name = (streng *) $2 ; }
             | iterate seps            { $$ = $1 ; }
             ;

label_stat   : labelname nseps         { $$ = $1 ; 
                                         newlabel( $1 ) ; }
             ;

labelname    : label                   { $$ = $1 ;
                                         $$->name = stralloc(retvalue) ; }
             ;

leave_stat   : leave simsymb seps      { $$ = $1 ;
                                         $$->name = (streng *) $2 ; } 
             | leave seps              { $$ = $1 ; }
             ;

nop_stat     : nop seps                { $$ = $1 ; }
             ;

numeric_stat : numeric DIGITS nexpr seps { $$ = $1 ;
                                         $$->type = X_NUM_D ; 
                                         $$->p[0] = $3 ; }
             | numeric FORM form_expr seps  { $$ = $1 ;
                                         $$->type = X_NUM_F ; 
                                         $$->p[0] = $3 ; }
             | numeric FUZZ nexpr seps { $$ = $1 ;
                                         $$->type = X_NUM_FUZZ ; 
                                         $$->p[0] = $3 ; }
             ;

form_expr    : SCIENTIFIC              { $$ = makenode(X_NUM_SCI,0) ; } 
             | ENGINEERING             { $$ = makenode(X_NUM_ENG,0) ; }
             |                         { $$ = NULL ; }
             ;

parse_stat   : parse parse_param template seps
                                       { $$ = $1 ;
                                         $$->type = X_PARSE ; 
                                         $$->p[0] = $2 ;
                                         $$->p[1] = $3 ; }
             | parse UPPER parse_param template seps
                                       { $$ = $1 ;
                                         $$->type = X_PARSE_U ; 
                                         $$->p[0] = $3 ;
                                         $$->p[1] = $4 ; }
             | parse ARG templs seps   { $$ = $1 ;
                                         $$->type = X_PARSE_ARG ; 
                                         $$->p[0] = $3 ; }
             | parse UPPER ARG templs seps 
                                       { $$ = $1 ;
                                         $$->type = X_PARSE_ARG_U ; 
                                         $$->p[0] = $4 ; }
             ;

templs       : template ',' templs     { $$ = makenode(X_TMPLS,2,$1,$3) ; }
             | template                { $$ = $1 ; }
             ;

parse_param  : LINEIN                  { $$ = makenode(X_PARSE_EXT,0) ; }
             | EXTERNAL                { $$ = makenode(X_PARSE_EXT,0) ; }
             | NUMERIC                 { $$ = makenode(X_PARSE_NUM,0) ; }  
             | VERSION                 { $$ = makenode(X_PARSE_VER,0) ; }
             | PULL                    { $$ = makenode(X_PARSE_PULL,0) ; }
             | SOURCE                  { $$ = makenode(X_PARSE_SRC,0) ; }
             | VAR simsymb             { $$ = makenode(X_PARSE_VAR,0) ; 
                                         $$->name = (streng *) $2 ; }
             | VALUE nexpr WITH        { $$ = makenode(X_PARSE_VAL,1,$2) ; }
             ;

proc_stat    : proc seps               { $$ = $1 ; }
             | proc EXPOSE anyvars seps { $$ = $1 ; 
                                         $$->p[0] = $3 ; }
             ;

pull_stat    : pull template seps      { $$ = $1 ; 
                                         $$->p[0] = $2 ; }
             ;

push_stat    : push nexpr seps         { $$ = $1 ; 
                                         $$->p[0] = $2 ; }
             ;

queue_stat   : queue nexpr seps        { $$ = $1 ;
                                         $$->p[0] = $2 ; } 
             ;

return_stat  : return nexpr seps       { $$ = $1 ; 
                                         $$->p[0] = $2 ; }
             ;

say_stat     : say nexpr seps          { $$ = $1 ;
                                         $$->p[0] = $2 ; }
             ;

select_stat  : select seps when_stats otherwise_stat END seps
                                       { $$ = $1 ;
                                         $$->p[0] = $3 ;
                                         $$->p[1] = $4 ; }
             ;

when_stats   : when_stat when_stats    { $$->next = $2 ; }
             | when_stat               { $$ = $1 ; }
             ;

when_stat    : when expr nseps THEN nseps statement {
                                         $$ = $1 ;
                                         $$->p[0] = $2 ;
                                         $$->p[1] = $6 ; }
             ;

otherwise_stat : otherwise nseps nstats {
                                         $$ = $1 ; 
                                         $$->p[0] = $3 ; }
             |                         { $$ = makenode(X_NO_OTHERWISE,0) ; 
                                         $$->lineno = tline ;
                                         $$->charnr = tstart ; } 
             ;


signal_stat : signal VALUE expr seps   { $$ = $1 ;
                                         $$->type = X_SIG_VAL ; 
                                         $$->p[0] = $3 ; }
            | signal asymbol seps      { $$ = $1 ;
                                         $$->name = (streng *)$2 ; }
            | signal on s_action namespec seps  
                                       { $$ = $1 ;
                                         $$->type = X_SIG_SET ;
                                         $$->p[0] = $2 ;
                                         $$->name = (streng *)$4 ;
                                         $$->p[1] = $3 ; }
            | signal off s_action seps { $$ = $1 ;
                                         $$->type = X_SIG_SET ;
                                         $$->p[0] = $2 ;
                                         $$->p[1] = $3 ; }
            ;

namespec    : NAME asymbol             { $$ = $2 ; }
            |                          { $$ = NULL ; }
            ;

asymbol     : CONSYMBOL                { $$ = (nodeptr)stralloc(retvalue) ; }
            | SIMSYMBOL                { $$ = (nodeptr)stralloc(retvalue) ; }
            ;
 
on          : ON                       { $$ = makenode(X_ON,0) ; }
            ;

off         : OFF                      { $$ = makenode(X_OFF,0) ; }
            ;

c_action    : ERROR                    { $$ = makenode(X_S_ERROR,0) ; }
            | HALT                     { $$ = makenode(X_S_HALT,0) ; }
            | NOTREADY                 { $$ = makenode(X_S_NOTREADY,0) ; }
            | FAILURE                  { $$ = makenode(X_S_FAILURE,0) ; }
            ;

s_action    : c_action                 { $$ = $1 ; }
            | NOVALUE                  { $$ = makenode(X_S_NOVALUE,0) ; }
            | SYNTAX                   { $$ = makenode(X_S_SYNTAX,0) ; }

trace_stat  : trace VALUE expr seps    { $$ = $1 ;
                                         $$->p[0] = $3 ; }
            | trace whatever seps      { $$ = $1 ; 
                                         $$->name = (streng *) $2 ; }
            ;

whatever    : WHATEVER                 { $$ = (nodeptr)stralloc(retvalue) ; }
            ;


assignment  : ass_part nexpr seps      { $$ = $1 ;
                                         $$->p[1] = $2 ; 
                                         if (gettypeof($$->p[1])==1)
                                            $$->type = X_NASSIGN ; } 
            ;          

ass_part    : ASSIGNMENTVARIABLE       { $$ = makenode(X_ASSIGN,0) ; 
                                         $$->charnr = tstart ;
                                         $$->lineno = tline ;
                                         $$->p[0] = create_head( retvalue ); }
            ;


expr        : '(' expr ')'             { $$ = $2 ; } 
            | NOT expr                 { $$ = makenode(X_LOG_NOT,1,$2) ; }
            | expr '+' expr            { $$ = makenode(X_PLUSS,2,$1,$3) ; }
            | expr '=' expr            { $$ = makenode(X_EQUAL,2,$1,$3) ; 
                                         transform( $$ ) ; }
            | expr '-' expr            { $$ = makenode(X_MINUS,2,$1,$3) ; }
            | expr '*' expr            { $$ = makenode(X_MULT,2,$1,$3) ; }
            | expr '/' expr            { $$ = makenode(X_DEVIDE,2,$1,$3) ; }
            | expr '%' expr            { $$ = makenode(X_INTDIV,2,$1,$3) ; }
            | expr '|' expr            { $$ = makenode(X_LOG_OR,2,$1,$3) ; }
            | expr '&' expr            { $$ = makenode(X_LOG_AND,2,$1,$3) ; }
            | expr XOR expr            { $$ = makenode(X_LOG_XOR,2,$1,$3) ; }
            | expr EXP expr            { $$ = makenode(X_EXP,2,$1,$3) ; }
            | expr SPACE expr          { $$ = makenode(X_SPACE,2,$1,$3) ; }
            | expr GTE expr            { $$ = makenode(X_GTE,2,$1,$3) ; 
                                         transform( $$ ) ; }
            | expr LTE expr            { $$ = makenode(X_LTE,2,$1,$3) ; 
                                         transform( $$ ) ; }
            | expr GT expr             { $$ = makenode(X_GT,2,$1,$3) ; 
                                         transform( $$ ) ; }
            | expr MODULUS expr        { $$ = makenode(X_MODULUS,2,$1,$3) ; }
            | expr LT expr             { $$ = makenode(X_LT,2,$1,$3) ; 
                                         transform( $$ ) ; }
            | expr DIFFERENT expr      { $$ = makenode(X_DIFF,2,$1,$3) ; 
                                         transform( $$ ) ; }
            | expr EQUALEQUAL expr     { $$ = makenode(X_S_EQUAL,2,$1,$3) ; }
            | expr NOTEQUALEQUAL expr  { $$ = makenode(X_S_DIFF,2,$1,$3) ; }
            | symbtree                 { $$ = $1 ; }
            | CONSYMBOL                { $$ = makenode(X_STRING,0) ;
                                         $$->name = stralloc(retvalue) ; }
            | HEXSTRING                { $$ = makenode(X_STRING,0) ;
                                         $$->name = Str_make(retlength) ; 
                                         memcpy($$->name->value,retvalue,
                                                    $$->name->len=retlength); }
            | BINSTRING                { $$ = makenode(X_STRING,0) ;
                                         $$->name = Str_make(retlength) ; 
                                         memcpy($$->name->value,retvalue,
                                                    $$->name->len=retlength); }
            | STRING                   { $$ = makenode(X_STRING,0) ; 
                                         $$->name = stralloc(retvalue) ; }
            | function                 { $$ = $1 ; } 
            | '+' expr %prec UPLUSS    { $$ = makenode(X_U_PLUSS,1,$2) ; }
            | '-' expr %prec UMINUS    { $$ = makenode(X_U_MINUS,1,$2) ; }
            | expr CONCATENATE expr    { $$ = makenode(X_CONCAT,2,$1,$3) ; }
            | expr CCAT expr           { $$ = makenode(X_CONCAT,2,$1,$2) ; }
            ;

symbtree    : SIMSYMBOL                { $$ = create_head( retvalue ) ; }
            ;


function    : extfunc  exprs ')'       { $$ = makenode(X_EX_FUNC,1,$2) ; 
                                         $$->name = (streng *)$1 ; }
            | intfunc  exprs ')'       { $$ = makenode(X_IN_FUNC,1,$2) ; 
                                         $$->name = (streng *)$1 ; }
            ;

intfunc     : INFUNCNAME               { $$ = (nodeptr)stralloc(retvalue) ; }
            ; 

extfunc     : EXFUNCNAME               { $$ = (nodeptr)stralloc(retvalue) ; }
            ;

template    : pv solid template        { $$ =makenode(X_TPL_SOLID,3,$1,$2,$3);}
            | pv                       { $$ =makenode(X_TPL_SOLID,1,$1) ; } 
            ;

solid	    : '-' offset               { $$ = makenode(X_NEG_OFFS,0) ; 
                                         $$->name = (streng *) $2 ; }
            | '+' offset               { $$ = makenode(X_POS_OFFS,0) ;
                                         $$->name = (streng *) $2 ; }
            | offset                   { $$ = makenode(X_ABS_OFFS,0) ;
                                         $$->name = (streng *) $1 ; }
            | '=' offset               { $$ = makenode(X_ABS_OFFS,0) ;
                                         $$->name = (streng *) $1 ; }
            | '(' symbtree ')'	       { $$ = makenode(X_TPL_VAR,0) ;
                                         $$->p[0] = $2 ; }
            | string                   { $$ = makenode(X_TPL_MVE,0) ;
                                         $$->name = (streng *) $1 ; }
            ; 

offset      : OFFSET                   { $$ = (nodeptr)stralloc(retvalue) ; }
            ;

string      : STRING                   { $$ = (nodeptr) stralloc(retvalue) ; }
            | HEXSTRING                { streng *sptr = Str_make( retlength ) ;
                                         memcpy(sptr->value,retvalue,
                                                 sptr->len=retlength) ;
                                         $$ = (nodeptr) sptr ; }
            | BINSTRING                { streng *sptr = Str_make( retlength ) ;
                                         memcpy(sptr->value,retvalue,
                                                 sptr->len=retlength) ;
                                         $$ = (nodeptr) sptr ; }
            ;                     

pv          : PLACEHOLDER pv           { $$ = makenode(X_TPL_POINT,1,$2) ; }
            | symbtree pv              { $$ = makenode(X_TPL_SYMBOL,1,$2) ; 
                                         $$->p[1] = $1 ; }
            |                          { $$ = NULL ; }
            ;

exprs       : nexpr ',' exprs          { $$ = makenode(X_EXPRLIST,2,$1,$3) ; 
                                         checkconst( $$ ) ; } 

            | nexpr                    { $$ = makenode(X_EXPRLIST,1,$1) ; 
                                         checkconst( $$ ) ; } 
            ;

nexpr       : expr                     { $$ = $1 ; }
            |                          { $$ = NULL ; }
            ;

anyvars     : simsymb anyvars          { $$ = makenode(X_SIM_SYMBOL,1,$2) ; 
                                         $$->name = (streng *) $1 ; }
            | simsymb                  { $$ = makenode(X_SIM_SYMBOL,0) ; 
                                         $$->name = (streng *) $1 ; }
            | '(' simsymb ')' anyvars  { $$ = makenode(X_IND_SYMBOL,1,$4) ; 
                                         $$->name = (streng *) $2 ; }
            | '(' simsymb ')'          { $$ = makenode(X_IND_SYMBOL,0) ; 
                                         $$->name = (streng *) $2 ; }
            ;

simsymb     : SIMSYMBOL                { $$ = (treenode *) stralloc(retvalue);}
            ;               

nsimsymb    : simsymb                  { $$ = $1 ; }
            |                          { $$ = NULL ; }
            ;

%%

streng *stralloc( char *ptr )
{
   return Str_cre( ptr ) ;
}


static nodeptr *narray=NULL ;
static int narptr=0, narmax=0 ;

void checkin( nodeptr ptr )
{
   nodeptr *new ;

   if (!narray)
   {
      narray = Malloc( sizeof(nodeptr)* 100 ) ;
      narmax = 100 ;
      narptr = 0 ;
   }

   if (narptr==narmax)
   {
      new = Malloc( sizeof(nodeptr)*narmax*3 ) ;
      memcpy( new, narray, sizeof(nodeptr)*narmax ) ;
      narmax *= 3 ;
      Free( narray ) ;
      narray = new ;
   }

   narray[narptr++] = ptr ;
}

void purge( void )
{
   int i, type ;
   nodeptr this ;
 
   for (i=0; i<narptr; i++)
   {
      this = narray[i] ;
      type = this->type ;
      if (type == X_CON_SYMBOL || type == X_STRING)
         if (this->u.number)
         {
            Free( this->u.number->num ) ;
            Free( this->u.number ) ;
            this->u.number = NULL ;
         }
  
      if (type==X_SIM_SYMBOL || type==X_STEM_SYMBOL || type==X_HEAD_SYMBOL ||
          type==X_CTAIL_SYMBOL || type==X_VTAIL_SYMBOL )
         if (this->u.varbx)
         {
            detach( this->u.varbx ) ;
            this->u.varbx = NULL ;
         }
 
      if (this->type == X_CEXPRLIST)
         if (this->u.strng)
         { 
            Free_string( this->u.strng ) ;
            this->u.strng = NULL ;
         }

      Free( this ) ;
   }  
   narptr = 0 ;
}


void checkout( void )
{
   narptr = 0 ;
}


nodeptr makenode( int type, int numb, ... ) 
{
   nodeptr thisleave ;
   va_list argptr ;
   int i ;

#ifdef REXXDEBUG
   printf("makenode: making new node, type: %d\n",type) ;
#endif /* REXXDEBUG */ 

   thisleave=(nodeptr)Malloc(sizeof(*thisleave)) ;

   for (i=0;i!=5;i++) 
      thisleave->p[i] = NULL ;

   va_start( argptr, numb ) ;
   thisleave->type = type ;
   thisleave->lineno = -1 ;
   thisleave->now = 0 ;
   thisleave->unow = 0 ;
   thisleave->sec = 0 ;
   thisleave->usec = 0 ;
   thisleave->u.node = NULL ;  /* handles ->u.whatever */
/*   thisleave->value = NULL ; */
   thisleave->next = NULL ;
   thisleave->charnr = -1 ;
   thisleave->name = NULL ;
   for (i=0;i!=numb;i++) 
      thisleave->p[i]=va_arg(argptr, nodeptr) ; 

   va_end( argptr ) ;

   checkin( thisleave ) ;
   return( thisleave ) ;
}


void checkdosyntax( nodeptr this ) 
{
   if ((this->p[1]!=NULL)&&(this->p[2]!=NULL))
      if ((this->p[1]->type)==(this->p[2]->type))
         exiterror(27) ;
   if ((this->p[2]!=NULL)&&(this->p[3]!=NULL))
      if ((this->p[2]->type)==(this->p[3]->type))
         exiterror(27) ;
   if ((this->p[1]!=NULL)&&(this->p[3]!=NULL))
      if ((this->p[1]->type)==(this->p[3]->type))
         exiterror(27) ;
   return ;
}


void newlabel( nodeptr this ) 
{
   extern sysinfo systeminfo ;
   labelboxptr new ;

   assert( this ) ;

   new = (labelboxptr)Malloc(sizeof(labelbox)) ;

   new->next = NULL ;
   new->entry = this ;
   if (systeminfo->last!=NULL)
      systeminfo->last->next = new ;
   systeminfo->last = new ;
   if (systeminfo->first==NULL)
      systeminfo->first = new ;
}
 



void destroytree( nodeptr this ) 
{
   int i ;
   int type ;

   for (i=0; i<5; i++)
      if (this->p[i])
         destroytree( this->p[i] ) ;

   if (this->next)
      destroytree( this->next ) ;

   if (this->name)
      Free_string( this->name ) ; 

   type = this->type ;
   if (type == X_CON_SYMBOL || type == X_STRING)
      if (this->u.number)
      {
         Free( this->u.number->num ) ;
         Free( this->u.number ) ;
         this->u.number = NULL ;
      }

   if (type==X_SIM_SYMBOL || type==X_STEM_SYMBOL || type==X_HEAD_SYMBOL ||
       type==X_CTAIL_SYMBOL || type==X_VTAIL_SYMBOL )
      if (this->u.varbx)
      {
         detach( this->u.varbx ) ;
         this->u.varbx = NULL ;
      }

   if (this->type == X_CEXPRLIST)
      if (this->u.strng)
      {
         Free_string( this->u.strng ) ;
         this->u.strng = NULL ;
      }

   Free( this ) ;
}


void kill_lines( lineboxptr first ) 
{
   lineboxptr ptr ;

   for (;first;first=ptr)
   {
      ptr = first->next ;
      assert( first->line ) ;
      Free_string( first->line ) ;
      Free( first ) ;
   }

}


nodeptr create_tail( char *name )
{
   char *cptr ;
   nodeptr node ;
   int constant ;
   streng *tname ;

   if (!*name)
      return NULL ;

   cptr = name ;
   constant = isdigit(*cptr) || *cptr=='.' ;
   node = makenode( (constant) ? X_CTAIL_SYMBOL : X_VTAIL_SYMBOL, 0 ) ;
   for (;*cptr && *cptr!='.'; cptr++) ;
   node->name = Str_ncre( name, cptr-name ) ;
   
   if (*cptr)
   {
      node->p[0] = create_tail( ++cptr ) ;
      if (constant && node->p[0]->type==X_CTAIL_SYMBOL)
      {
         streng *first, *second ;
         nodeptr tptr ;

         first = node->name ;
         second = node->p[0]->name ;
         tname = Str_make( first->len + second->len + 1) ;
         memcpy( tname->value, first->value, first->len ) ;
         tname->value[first->len] = '.' ;
         memcpy( tname->value+first->len+1, second->value, second->len) ;
         tname->len = first->len + second->len + 1 ;

         Free_string( first ) ;
         Free_string( second ) ;
         node->name = tname ;
         tptr = node->p[0] ;
         node->p[0] = tptr->p[0] ;
         Free( tptr ) ;
      }
   } 
  
   return node ;
}

nodeptr create_head( char *name )
{
   char *cptr ;
   nodeptr node ;
   
   for (cptr=name; *cptr && *cptr!='.'; cptr++) ;
   node = makenode( X_SIM_SYMBOL, 0 ) ;
   node->name = Str_ncre( name, cptr-name+(*cptr=='.')) ;

   if (*cptr)
   {
      node->p[0] = create_tail( ++cptr ) ;
      node->type = (node->p[0]) ? X_HEAD_SYMBOL : X_STEM_SYMBOL ;
   }

   return node ;
}

#define IS_UNKNOWN   0
#define IS_A_NUMBER  1
#define IS_NO_NUMBER 2
#define IS_IRREG_NUMBER 10

int gettypeof( nodeptr this )
{
   switch(this->type)
   {
      case X_PLUSS:
      case X_MINUS:
      case X_MULT: 
      case X_U_PLUSS:
      case X_U_MINUS:
      case X_DEVIDE:
      case X_INTDIV: 
      case X_MODULUS:
      case X_EQUAL:
      case X_DIFF:
      case X_GTE:
      case X_GT:
      case X_LTE:
      case X_LT:
      case X_SEQUAL:
      case X_SDIFF:
      case X_SGTE:
      case X_SGT:
      case X_SLTE:
      case X_SLT:
      case X_NEQUAL:
      case X_NDIFF:
      case X_NGTE:
      case X_NGT:
      case X_NLTE:
      case X_NLT:
         return IS_A_NUMBER ;
 
 
      case X_SIM_SYMBOL:
         return 3 ;
     
      case X_HEAD_SYMBOL:
         return 4 ;

      case X_STRING:
      case X_CON_SYMBOL:
      {
         if (this->u.number)
            return IS_A_NUMBER ;

         this->u.number = is_a_descr( this->name ) ;
         if (this->u.number)
         {
            streng *stmp = str_norm( this->u.number, NULL ) ;
            if (Str_cmp(stmp,this->name))
            {
               Free_string( stmp ) ;
               return IS_UNKNOWN ;
            }
            Free_string( stmp ) ;
         }
         return (this->u.number) ? IS_A_NUMBER : IS_NO_NUMBER ;
      }
   }
   return IS_UNKNOWN ;
}



void transform( nodeptr this )
{
   int left, rght, type ;

   left = gettypeof( this->p[0] ) ;
   rght = gettypeof( this->p[1] ) ;
   type = this->type ;

   if (left==1 && rght==1)
   {
      if (type==X_EQUAL)
         this->type = X_NEQUAL ;
      else if (type==X_DIFF)
         this->type = X_NDIFF ;
      else if (type==X_GTE)
         this->type = X_NGTE ;
      else if (type==X_GT)
         this->type = X_NGT ;
      else if (type==X_LTE)
         this->type = X_NLTE ;
      else if (type==X_LT)
         this->type = X_NLT ;
   }
   else if (left==2 || rght==2)
   {
      if (type==X_EQUAL)
         this->type = X_SEQUAL ;
      else if (type==X_DIFF)
         this->type = X_SDIFF ;
      else if (type==X_GTE)
         this->type = X_SGTE ;
      else if (type==X_GT)
         this->type = X_SGT ;
      else if (type==X_LTE)
         this->type = X_SLTE ;
      else if (type==X_LT)
         this->type = X_SLT ;
   }     
   else
   {
      type = this->p[0]->type ;
      if (left==1 && (type==X_STRING || type==X_CON_SYMBOL))
         this->u.flags.lnum = 1 ;
      else if (left==3)
         this->u.flags.lsvar = 1 ;
      else if (left==4)
         this->u.flags.lcvar = 1 ;

      type = this->p[1]->type ;
      if (rght==1 && (type==X_STRING || type==X_CON_SYMBOL))
         this->u.flags.rnum = 1 ;
      else if (rght==3)
         this->u.flags.rsvar = 1 ;
      else if (rght==4)
         this->u.flags.rcvar = 1 ;
   }
}


int is_const( nodeptr this ) 
{
   if (!this)
      return 1 ;

   switch (this->type)
   {
      case X_STRING:
      case X_CON_SYMBOL:
         return 1 ;

      case X_U_PLUSS:
      case X_U_MINUS:
         return is_const( this->p[0] ) ;

      case X_PLUSS:
      case X_MINUS:
      case X_MULT: 
      case X_DEVIDE:
      case X_INTDIV: 
      case X_MODULUS:
      case X_EQUAL:
      case X_DIFF:
      case X_GTE:
      case X_GT:
      case X_LTE:
      case X_LT:
      case X_SEQUAL:
      case X_SDIFF:
      case X_SGTE:
      case X_SGT:
      case X_SLTE:
      case X_SLT:
      case X_NEQUAL:
      case X_NDIFF:
      case X_NGTE:
      case X_NGT:
      case X_NLTE:
      case X_NLT:
      case X_SPACE:
      case X_CONCAT:
         return is_const( this->p[0] ) && is_const( this->p[1] ) ;
   }
   return 0 ;
}


void checkconst( nodeptr this ) 
{
   assert( this->type == X_EXPRLIST ) ;
   if (is_const(this->p[0]))
   {
      if (this->p[0])
         this->u.strng = evaluate( this->p[0], NULL ) ;
      else
         this->u.strng = NULL ;

      this->type = X_CEXPRLIST ;
   }
}

