#ifndef lint
static char *RCSid = "$Id: tracing.c,v 1.9 1993/05/10 05:49:42 anders Exp anders $";
#endif

/*
 *  The Regina Rexx Interpreter
 *  Copyright (C) 1992  Anders Christensen <anders@solan.unit.no>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version. 
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Log: tracing.c,v $
 * Revision 1.9  1993/05/10  05:49:42  anders
 * Minor ajustments to kill warnings.
 *
 * Revision 1.8  1993/05/07  20:21:09  anders
 * Implemented more service routines to trace various 'datatypes' in order
 * to service the new functions in expr.c. Also implemented hooks for
 * applications to get trace output.
 *
 * Revision 1.7  1993/02/09  18:27:37  anders
 * Renamed Str*() to Str_*() to humor case insensitive machines
 * Fixed up compiler warnings.
 *
 * Revision 1.6  1992/07/24  03:45:30  anders
 * Added GPL. Fixed interactive trace.
 *
 * Revision 1.5  1992/04/25  13:18:30  anders
 * Converted to REXX strings
 *
 * Revision 1.4  1992/04/05  20:41:29  anders
 * Added copyright notice
 * Added routine to interpret and set tracing state.
 *
 * Revision 1.3  1992/03/22  01:14:39  anders
 * #include'd stdio.h
 * Commented out some external definitions, since some compilers
 *    didn't like the way it was written ... sigh...
 *
 * Revision 1.2  1991/06/03  02:58:49  anders
 * Changed definitions for pushcallstack() and popcallstack(), to
 *    make it possible to remove extra entries on return from function
 *    or subroutine.
 *
 * Revision 1.1  90/08/08  02:13:53  anders
 * Initial revision
 * 
 */

#include "rexx.h"
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <assert.h>
#include <string.h>

static int traceflag = 0 ;
static int lasttracedline = -1 ;   
static int ctrlcounter = 0 ;
static int notnow = 0 ;

static char tracestr[LINELENGTH+1] ;


extern sysinfo systeminfo ;

int pushcallstack( treenode *this )
{
   nodeptr *tmpptr ; 
   extern sysinfo systeminfo ;
   
   if (systeminfo->cstackcnt >= systeminfo->cstackmax)
   {
      assert( systeminfo->cstackcnt == systeminfo->cstackmax ) ;
      tmpptr = Malloc( (systeminfo->cstackmax*2+10)*sizeof(nodeptr)) ;
      if (systeminfo->callstack)
      {
         memcpy( tmpptr, systeminfo->callstack, 
                                 systeminfo->cstackcnt*sizeof(nodeptr) ) ;
         Free( systeminfo->callstack ) ;
      }
      systeminfo->callstack = tmpptr ;
      systeminfo->cstackmax *= 2 ;
      systeminfo->cstackmax += 10 ;
   }

   ctrlcounter++ ;      
   systeminfo->callstack[systeminfo->cstackcnt++] = this ;
   return systeminfo->cstackcnt ;
}


void popcallstack( int value )
{
   extern sysinfo systeminfo ;

   if (value>=0)
   {
      assert( systeminfo->cstackcnt >= value ) ;
      ctrlcounter -= systeminfo->cstackcnt - value ;
      systeminfo->cstackcnt = value ;
   }

   ctrlcounter-- ;
   --(systeminfo->cstackcnt) ;

   assert( systeminfo->cstackcnt >= 0 ) ;
   assert( ctrlcounter >= 0 ) ;
}


void printout( streng *message ) 
{
   int rc ;

   rc = HOOK_GO_ON ;
   if (systeminfo->hooks & HOOK_MASK(HOOK_STDERR))
      rc = hookup( HOOK_STDERR, message ) ;

   if (rc==HOOK_GO_ON)
   {
      write( fileno(stderr), message->value, message->len ) ;
      putc( 0x0a, stderr ) ;
   }
}




void traceerror( treenode *this, int RC )
{
   streng *message ;

   if (trace_stat=='N') 
      traceline(this, 'C') ; 

   if (trace_stat!='O')
   {
      message = Str_make( 20+sizeof(int)*3 ) ;
      sprintf(message->value, "       +++ RC=%d +++",RC) ; 
      message->len = strlen( message->value ) ;

      printout( message ) ;
      Free_string( message ) ;
   }
}


void tracecompound( streng *stem, int length, streng *index )
{
   extern int ctrlcounter, traceflag ;
   streng *message ;
  
   if ((traceflag)||(trace_stat!='I'))
      return ;
   
   stem = Str_ify(Str_dup(stem)) ;
   index = Str_ify(Str_dup(index)) ;
   message = Str_make( 1024 ) ;

   sprintf(tracestr,"       >C> %%%ds  \"%%.%ds.%%s\"\n",ctrlcounter,length) ;
   sprintf(message->value, tracestr,"",stem->value,index->value ) ; 

   message->len = strlen( message->value ) ;
   printout( message ) ;

   Free_string( message ) ;
   Free( stem ) ;
   Free( index ) ;
}


void starttrace( void ) 
{
   extern int notnow ;
   notnow = 1 ;
}

void tracemsg( void )
{
   streng *message ;
   int rc ;
   char *msg= "       +++ Interactive trace. TRACE OFF to end debug, ENTER to continue. +++" ;

   rc = HOOK_GO_ON ;
   message = Str_cre( msg ) ;
   printout( message ) ;
   Free_string( message ) ;
}


int intertrace( void )
{
   streng *str ;
   int retvalue, rc ;
   extern int notnow, traceflag ;
  
   if (traceflag)
      return 0 ;

   if (notnow==1) 
   {
      notnow = 2 ;
      return 0 ; 
   }
   else if (notnow==2) 
   {
      notnow = 0 ;
      tracemsg() ; 
   }

   traceflag = 1 ;
   retvalue = (-1) ;
   for (; retvalue<0; ) {
      rc = HOOK_GO_ON ;
      if (systeminfo->hooks & HOOK_MASK(HOOK_TRCIN))
         rc = hookup( HOOK_TRCIN, &str ) ;

      if (rc==HOOK_GO_ON)
         str = readkbdline() ;

      if (str->len==0) 
      {
         traceflag = 0 ;
         retvalue = 0 ;
      }

      if ((Str_len(str)==1)&&(str->value[0]=='='))
      {
         traceflag = 0 ;
         retvalue = 1 ;
      }
      else if (str->len)
         dointerpret( str ) ; }

   return retvalue ;
}


void tracenumber( num_descr *num, char type ) 
{
   extern proclevel currlevel ;
   extern int ctrlcounter, traceflag ;
   char tmpch ;
   streng *message, *tmpstr ;
   
   if (traceflag)
      return ;

   tmpch = currlevel->tracestat ;
   if ((tmpch=='I')||((tmpch=='R')&&(type!='.')))
   {
      tmpstr = str_norm( num, NULL ) ;
      message = Str_make( 1024 ) ;
      sprintf(tracestr,"       >%%c> %%%ds  \"%%s\"",ctrlcounter) ;
      sprintf(message->value,tracestr,type,"", Str_ify(tmpstr)->value) ; 
      message->len = strlen( message->value ) ;
      printout( message ) ;
      Free_string( message ) ;
      Free_string( tmpstr ) ;
   }
}


void tracebool( int value, char type ) 
{
   char tmpch ;
   extern proclevel currlevel ;
   streng *message ;

   if (traceflag)
      return ;

   tmpch = currlevel->tracestat ;
   if ((tmpch=='I') || ((tmpch=='R') && (type!='.')))
   {
      message = Str_make( 1024 ) ;
      sprintf(tracestr,"       >%%c> %%%ds  \"%%d\"",ctrlcounter ) ;
      sprintf(message->value,tracestr,type,"",value) ;
      message->len = strlen( message->value ) ;
      printout( message ) ;
      Free_string( message ) ;
   }
}


void tracevalue( streng *str, char type )
{
   extern proclevel currlevel ;
   extern int ctrlcounter, traceflag ;
/*   extern char tracestr[] ; */
   char tmpch ;
   streng *message ;

   if (traceflag)
      return ;

   tmpch = currlevel->tracestat ;
   if ((tmpch=='I')||((tmpch=='R')&&(type!='.'))) 
   {
      message = Str_make( 1024 ) ;
      sprintf(tracestr,"       >%%c> %%%ds  \"%%s\"",ctrlcounter) ;
      sprintf(message->value,tracestr,type,"", Str_ify(str)->value) ; 
      message->len = strlen( message->value ) ;
      printout( message ) ;
      Free_string( message ) ;
   }
}



void traceline( treenode *this, char tch ) 
{
   streng *srcstr ;
   extern int ctrlcounter, lasttracedline ;
   extern sysinfo systeminfo ;
   extern int traceflag ;
   
   if ((tch=='O')||(traceflag))
      return ; 

   if ((this->charnr<0)||(this->lineno<0)) 
      return ;

   if ((tch=='I')||(tch=='R')||(tch=='A')) goto trace ;
   if ((tch=='L')&&(this->type==X_LABEL)) goto trace ;
   if (tch=='C')
      if ((this->type==X_COMMAND) || 
          ((this->type==X_ADDR_N) && (this->p[0])))
         goto trace ;

   return ;

trace: {
       streng *message ;
       srcstr = getsourceline(this->lineno,this->charnr,
                      systeminfo->firstline, systeminfo->lastline) ; 

       message = Str_make( ctrlcounter+20+srcstr->len ) ;
       if (this->lineno==lasttracedline) {
          sprintf(tracestr,"       *-* %%%ds%%s",ctrlcounter) ;
          sprintf(message->value,tracestr,"", Str_ify(srcstr)->value) ; }
       else {
          sprintf(tracestr,"%%6d *-* %%%ds%%s",ctrlcounter) ;
          sprintf(message->value,tracestr,this->lineno,"", 
                                          Str_ify(srcstr)->value) ; }

       message->len = strlen( message->value ) ;
       printout( message ) ;
       lasttracedline = this->lineno ;
       Free_string( message ) ;
       Free(srcstr) ; }
}


void traceback( void )
{
   extern nodeptr currentnode /*, callstack[] */ ;
   extern int ctrlcounter ;
   static char fmt[20] ;
   extern sysinfo systeminfo ;
   sysinfo ss ;
   streng *message ;
   nodeptr ptr ;
   int i, j ;

   message = Str_make( 1024 ) ;
   if (currentnode)
   {
      sprintf(fmt,"%%6d +++ %%%ds%%s", ctrlcounter*3 ) ;
      sprintf(message->value,fmt,currentnode->lineno,"",
         Str_ify(getsourceline(currentnode->lineno,
                   currentnode->charnr, systeminfo->firstline,
                   systeminfo->lastline))->value) ;
      message->len = strlen( message->value ) ;
      printout( message ) ;
   }
   
   j = ctrlcounter-1 ;
   for (ss=systeminfo; ss; ss=ss->previous ) 
   {
      for (i=ss->cstackcnt-1;i>=0;i--) 
      {
         ptr = ss->callstack[i] ;
         if (!ptr)
            continue ;
         sprintf(fmt,"%%6d +++ %%%ds%%s", j*3 ) ;
         sprintf(message->value,fmt,ptr->lineno,"",
            Str_ify(getsourceline(ptr->lineno,ptr->charnr,
                                  ss->firstline, ss->lastline))->value) ; 

         message->len = strlen( message->value ) ;
         printout( message ) ;
      } 
   }
   Free_string( message ) ;
}



void set_trace_char( char ch )
{
   extern sysinfo systeminfo ;
   extern proclevel currlevel ;
 
   switch (ch=toupper(ch))
   {
      case '?':
         ch = systeminfo->interactive = !systeminfo->interactive ;
         if (ch)
            starttrace() ;
         break ;         

      case 'F':
      case 'E':
         ch = 'N' ;

      case 'A':
      case 'C':
      case 'I':
      case 'L':
      case 'N':
      case 'O':
      case 'R':
      case 'S':
         currlevel->tracestat=ch ;
         break ;

      default:
         exiterror( ERR_INVALID_TRACE ) ;
   }
   trace_stat = currlevel->tracestat ;
}



void set_trace( streng *setting )
{
   int cptr ;

   for (cptr=0; cptr<Str_len(setting); cptr++)
   {
      set_trace_char( setting->value[cptr] ) ;
      if (isalpha( setting->value[cptr])) 
         return ;
   }
}


