#ifndef lint
static char *RCSid = "$Id: rxiface.c,v 1.1 1993/05/07 21:37:12 anders Exp anders $";
#endif

/*
 *  The Regina Rexx Interpreter
 *  Copyright (C) 1993  Anders Christensen <anders@solan.unit.no>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version. 
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * $Id: rxiface.c,v 1.1 1993/05/07 21:37:12 anders Exp anders $
 */


#define INCL_RXSHV
#define INCL_RXSUBCOM
#define INCL_RXFUNC
#define INCL_RXSYSEXIT 


#include "rexxsaa.h"
#include "rxiface.h"
#include <limits.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <ctype.h>


struct EnvBox
{
   struct EnvBox *prev, *next ;
   char *EnvName ;
   unsigned char UserData[8] ;
   union {
     PFN EntryPnt ;
     RexxSubcomHandler *SubCom ;
   } u ;
} ;


static int ToRexx[2] = { -1, -1 } ;
static int ToAppl[2] = { -1, -1 } ;


static int interpreting=0 ;
static int RxProgPid=0 ;


void ExecuteFunction( void ) ;
static int StartupRexx( char *EnvName ) ;

#define MAP_TYPE(a) ((a)==RXCOMMAND ? RX_TYPE_COMMAND : \
                    (a)==RXFUNCTION ? RX_TYPE_FUNCTION : RX_TYPE_SUBROUTINE)


#define ERROR_NOERR   0
#define ERROR_NOMEM   1
#define ERROR_SYNCH   2
#define ERROR_NODATA  3
#define ERROR_COMM    4
#define ERROR_INCORR  5


static int ErrorStat=ERROR_NOERR ;
static void HandleError( int type ) 
{
   ErrorStat = type ;
}



static char *RecvData( int size, char *data ) 
{
   int left ;
   int rc ;
   char *cptr ;

   if (ErrorStat!=ERROR_NOERR)
      return NULL ;

   left = size ;
   cptr = data ;
   while (left)
   {
      errno = 0 ;
      rc = read( ToAppl[0], cptr, left ) ;
      if (rc!=left && errno)
      {
         HandleError( ERROR_NODATA ) ;
         return data ;
      }
      else
      {
         left -= rc ;
         cptr += rc ;
      }
   }

   return data ;
}   


/* 
 * Read a number from the communication channel.
 */
static int RecvNumber( void )
{
   int number ;

   if (!RecvData( sizeof(number), (char*)&number ))
      number = 0 ;

   return number ;
}


/* 
 * Read a string from the communication channel. The size of the string to 
 * read is the parameter, and a pointer to the string read is returned. 
 * The string will be null-terminated -- just in case -- but note that the 
 * string may also contain ASCII NUL characters (so don't count on this 
 * to be a real ASCIZ string unless you know for sure).
 * 
 * If the NULL pointer is returned, the routine was not able to allocate 
 * enough memory to hold the string to be read. 
 */
static char *RecvString( int size ) 
{
   char *cptr, *orig ;

   if (size<0)
      return NULL ;

   orig = malloc( size+1 ) ;
   if (!orig)
   {
      HandleError( ERROR_NOMEM ) ;
      return NULL ;
   }

   cptr = RecvData( size, orig ) ;
   if (cptr)
      cptr[size] = 0x00 ;   /* Add a terminating NUL, just in case */
   else
      free( orig ) ;

   return cptr ;
}


static char *RecvJunk( int size ) 
{
   static char junk[8192] ;
   char *cptr ;
   
   while (size>0)
   {
      if (size>8192)
         cptr = RecvData( 8192, junk ) ;
      else
         cptr = RecvData( size, junk ) ;
      size -= 8192 ;

      if (!cptr)
         return NULL ;
   }
   return junk ;   /* any valid addres */
}


/*
 * Takes an integer as parameter, and sends it on the communication 
 * channel. 
 */
static void SendNumber( int number ) 
{
   int rc ;

   if (!RxProgPid && StartupRexx( "" ))
      ErrorStat = ERROR_COMM ;

   if (ErrorStat!=ERROR_NOERR)
      return ;

   rc = write( ToRexx[1], &number, sizeof(number) ) ;
   if (rc!=sizeof(number))
      HandleError( ERROR_NODATA ) ;
}


/* 
 * Takes the specification of a string as input, and sends that string
 * on the communication channel. First the size of the string is sent, 
 * then the contents.
 */
static void SendString( int length, char *cptr ) 
{
   int rc ;

   if (!RxProgPid && StartupRexx( "" ))
      ErrorStat = ERROR_COMM ;

   if (!cptr)
   {
      SendNumber( RX_NO_STRING ) ;
      return ;
   }

   SendNumber( length ) ;
   if (ErrorStat!=ERROR_NOERR)
      return ;

   rc = write( ToRexx[1], cptr, length ) ;
   if (rc!=length)
      HandleError( ERROR_NODATA ) ;
}



/*
 * This routine will establish a communication channel, and start 
 * the Rexx interpreter. Far more logic can be introduced in order to 
 * search for the interpreter and other interesting things ...
 *
 * Returncodes:
 *   0 = OK 
 *   1 = Program already started
 *   2 = Unable to allocate sufficient resources
 *   3 = Environment name too long
 *   4 = Could not establish contact with the interpreter
 *   5 = Incorrect major version
 */
static int StartupRexx( char *EnvName )
{
   char ArgStr[256] ;

   if (RxProgPid)
      return 1 ;

   /* 
    * Why? Well, suppose StartRexx was called with an extremely long 
    * environment name, then the sprintf() would dance fandango on core
    * later in this routine, so just make sure it never happens. It 
    * probably won't
    */
   if (strlen((char*)EnvName)>MAXENVNAMELEN)
      return 3 ;

   if (pipe(ToRexx) || pipe(ToAppl))
   {
      HandleError( ERROR_COMM ) ;
      return 2 ;
   }

   if ((RxProgPid=fork()))
   {
      close( ToRexx[0] ) ;
      close( ToAppl[1] ) ;

      /* Need to check that channel was actually estabished */
      if (RecvNumber()!=RX_VERSION)
         return 4 ;
 
      if (RecvNumber() != RXVERSION_MAJ || RecvNumber() !=RXVERSION_MIN) 
         return 5 ;
   }
   else
   {
      close( ToRexx[1] ) ;
      close( ToAppl[0] ) ;

      sprintf( ArgStr, "-Cp%d,%d,%s", ToRexx[0], ToAppl[1], EnvName ) ;

      execlp( RXPROG, RXPROG, ArgStr, NULL ) ;
      exit( 0 ) ;
   }
     
   return 0 ;
}




/* 
 * The WaitForValue() funtion is called when a message has been sent 
 * to the interpreter, requesting the value of a variable. It will 
 * wait until that value is returned, and does not allow processing 
 * of other messages in the meantime, except for syncing.
 * 
 * It returns a pointer to the string read, or NULL if no string was
 * awailable. It may receive a parameter, which must be a pointer to 
 * an integer; than integer is set to the length of the string returned
 * (the terminating ASCII NUL excluded). The parameter may be a NULL
 * pointer if this value is of no interest. 
 */
/*
static char *WaitForValue( int *RetLength )
{
   int Code, Length ;
   char *Result ;

   while (1) 
   {
      Code = RecvNumber() ;
      if (Code==RX_VALUE)
         break ;
      else if (Code==RX_SYNCREQ)
         SendNumber( RX_SYNCACK) ;
      else
         HandleError( ERROR_SYNCH ) ;
   }

   / * 
    * If 'Length' is 0, a string of zero characters are returned. If 
    * 'Length' is -1, no string is returned. If 'Length' is 0, no problem
    * arises, since RecvString() allocates 1 bytes (it allocates one
    * byte extra for a terminating ASCII NUL).
    * /
   Length = RecvNumber() ;
   if (Length>=0)
      Result = RecvString(Length) ;
   else
      Result = NULL ;

   if (RetLength)
      *RetLength = Length>0 ? Length : 0 ;

   return Result ;
}
*/


int WaitForNumber( void )
{
   return RecvNumber() ;
}



static struct EnvBox *FirstEnv=NULL ;

/* 
 * Find a particular environment, and return a pointer to a struct
 * containing information about that environment. If it is not found,
 * a pointer to NULL is returned. 
 */
static struct EnvBox *FindEnv( char *Env )
{
   struct EnvBox *bptr ;

   assert( Env ) ;
   for (bptr=FirstEnv; bptr; bptr=bptr->next)
      if (!strcmp(bptr->EnvName,Env))
         return bptr ;

   return NULL ;
}


static void SubCommand( void ) {
   RXSTRING Cmd, Ret ;
   char result[DEFAULT_RETSTRING_LENGTH], *OldResult ;
   USHORT Flags ;
   int EnvLen, Length ;
   char *EnvNam, *Command ;
   struct EnvBox *Envir ;
   int rvalue, RCode ;

   EnvLen = RecvNumber() ;
   EnvNam = RecvString( EnvLen ) ;
   Length = RecvNumber() ;
   Command = RecvString( Length ) ;
            
   Envir = FindEnv( EnvNam ) ;
   if (Envir)
   {
      MAKERXSTRING( Cmd, Command, Length ) ;
      MAKERXSTRING( Ret, result, DEFAULT_RETSTRING_LENGTH ) ;
      OldResult = result ;

      rvalue = (*(Envir->u.SubCom))( &Cmd, &Flags, &Ret ) ;
 
      if (Flags==RXSUBCOM_OK)
         RCode = RXFLAG_OK ;
      else if (Flags==RXSUBCOM_ERROR)
         RCode = RXFLAG_ERROR ;
      else if (Flags==RXSUBCOM_FAILURE)
         RCode = RXFLAG_FAILURE ;
      else
         assert( 0 ) ;
   }
   else
   {
      RCode = RXFLAG_FAILURE ;
      Ret.strlength = 0 ;
   }

   SendNumber( RX_RETURN ) ;
   SendNumber( RCode ) ;

   if (Ret.strlength)
      SendString( Ret.strlength, Ret.strptr ) ;
   else
      SendString( 1, "0" ) ;

   free( Command ) ;
   free( EnvNam ) ;
   if (Ret.strlength && OldResult != Ret.strptr)
      free( Ret.strptr ) ; }


void SetEnvironment( int length, char *name ) {
   SendNumber( RX_ENVIR ) ;
   SendString( length, name ) ; }

static struct EnvBox *FirstExit=NULL ;

static struct EnvBox *FindExit( char *Env ) {
   struct EnvBox *bptr ;

   assert( Env ) ;
   for (bptr=FirstExit; bptr; bptr=bptr->next)
      if (!strcmp(bptr->EnvName,Env))
         return bptr ;

   return NULL ; }





struct ExitHandlers {
   RexxExitHandler *(Handlers[RXNOOFEXITS]) ;
   struct ExitHandlers *prev ; } *CurrentHandlers=NULL ;


static void ProcHook( void ) {
   EXIT ParBox ;
   int Code, rc, SubCode, MainCode ;
   int RLength ;
   char *RString ;

   Code = RecvNumber() ;
   RLength = RecvNumber() ;
   if (RLength<0)
   {
      RString = NULL ;
      RLength = 0 ;
   }
   else
      RString = RecvString( RLength ) ;

   switch (Code)
   {
      case RX_EXIT_STDERR:
      case RX_EXIT_STDOUT:
         ParBox.siosay.rxsio_string.strlength = RLength ;
         ParBox.siosay.rxsio_string.strptr = RString ;
         SubCode = (Code==RX_EXIT_STDOUT) ? RXSIOSAY : RXSIOTRC ;
         MainCode = RXSIO ;
         break ;

      case RX_EXIT_TRCIN:
         ParBox.siodtr.rxsiodtr_retc.strlength = 0 ;
         ParBox.siodtr.rxsiodtr_retc.strptr = NULL ;
         SubCode = RXSIODTR ;
         MainCode = RXSIO ;
         break ;

      case RX_EXIT_PULL:
         ParBox.siotrd.rxsiotrd_retc.strlength = 0 ;
         ParBox.siotrd.rxsiotrd_retc.strptr = NULL ;
         SubCode = RXSIOTRD ;
         MainCode = RXSIO ;
         break ;

      case RX_EXIT_INIT:
         MainCode = RXINI ;
         SubCode = RXINIEXT ;
         break ;
 
      case RX_EXIT_TERMIN:
         MainCode = RXTER ;
         SubCode = RXTEREXT ;
         break ;

      default:
         HandleError( ERROR_INCORR ) ;
   }

   assert( CurrentHandlers->Handlers[MainCode] ) ;
   rc = (*(CurrentHandlers->Handlers[MainCode]))(MainCode, SubCode,
&ParBox);
   assert( rc==RXEXIT_HANDLED || rc==RXEXIT_NOT_HANDLED ||
                                            rc==RXEXIT_RAISE_ERROR ) ;

   switch (Code)
   {
      case RX_EXIT_STDERR:
      case RX_EXIT_STDOUT:
         if (RString)
           free( RString ) ;
         RString = NULL ;
         break ;

     case RX_EXIT_INIT:
     case RX_EXIT_TERMIN:
         RString = NULL ;
         break ;

     case RX_EXIT_TRCIN:
         RLength = ParBox.siodtr.rxsiodtr_retc.strlength ;
         RString = ParBox.siodtr.rxsiodtr_retc.strptr ;
         break ;

     case RX_EXIT_PULL:
         RLength = ParBox.siotrd.rxsiotrd_retc.strlength ;
         RString = ParBox.siotrd.rxsiotrd_retc.strptr ;
         break ;

      default:
         HandleError( ERROR_INCORR ) ;
   }


   SendNumber( RX_RETURN ) ;
   if (rc==RXEXIT_HANDLED)
      rc = RX_HOOK_NOPE ;
   else if (rc==RXEXIT_NOT_HANDLED)
      rc = RX_HOOK_GO_ON ;
   else if (rc==RXEXIT_RAISE_ERROR)
      rc = RX_HOOK_ERROR ;

   SendNumber( rc ) ;
   if (RString)
      SendString( RLength, RString ) ;
   else
      SendNumber( RX_NO_STRING ) ;

}



LONG RexxStart(
   LONG		ArgCount,
   PRXSTRING 	ArgList,
   PSZ		ProgramName,
   PRXSTRING	Instore,
   PSZ		EnvName,
   LONG		CallType,
   PRXSYSEXIT	Exits,
   PLONG	ReturnCode,
   PRXSTRING	Result ) {
   int cnt, Code, RLength ;
   char *RString ;
   struct ExitHandlers *Handlers ;
   RexxExitHandler *handler ;
   struct EnvBox *EnvPtr ;
   LONG ResValue ;
   int drop_ipret=0 ;
   
   if (Instore)
   {
      if (Instore[1].strptr && Instore[1].strlength < sizeof(int))
         return RX_START_BADP ;
   }

   if (CallType!=RXCOMMAND && CallType!=RXFUNCTION && CallType!=RXSUBROUTINE)
      return RX_START_BADP ;

   if (CallType==RXCOMMAND && ArgCount>1)
      return RX_START_TOOMANYP ;

   if (!RxProgPid && StartupRexx( EnvName ))
      return RX_DIDNT_START ;

   if (!interpreting)
      drop_ipret = interpreting = 1 ;

   SendNumber( RX_EXECUTE ) ;
   SendString( strlen(ProgramName), ProgramName ) ;

   SendNumber( ArgCount ) ;
   for (cnt=0; cnt<ArgCount; cnt++)
      SendString( ArgList[cnt].strlength, ArgList[cnt].strptr ) ;

   SendNumber( MAP_TYPE( CallType )) ;

   Handlers = malloc( sizeof( struct ExitHandlers )) ;
   Handlers->prev = CurrentHandlers ;
   CurrentHandlers = Handlers ;
   for (cnt=0; cnt<RXNOOFEXITS; cnt++)
      CurrentHandlers->Handlers[cnt] = NULL ;

   for (; Exits && Exits->sysexit_code!=RXENDLST; Exits++ )
   {
      EnvPtr = FindExit( Exits->sysexit_name ) ;
      if (!EnvPtr)
         continue ;

      /* Sigh ... Definition requires some strange casting */
      handler = (RexxExitHandler*)(EnvPtr->u.EntryPnt) ;
      switch (Exits->sysexit_code)
      {
          case RXSIO:
             SendNumber( RX_EXIT_STDOUT ) ;
             SendNumber( RX_EXIT_STDERR ) ;
             SendNumber( RX_EXIT_TRCIN ) ;
             SendNumber( RX_EXIT_PULL ) ;
             /* This is just too easy ... must be handled better */
             CurrentHandlers->Handlers[RXSIO] = handler ;
             break ;

          case RXINI:
             SendNumber( RX_EXIT_INIT ) ;
             CurrentHandlers->Handlers[RXINI] = handler ;
             break ;
  
          case RXTER:
             SendNumber( RX_EXIT_TERMIN ) ;
             CurrentHandlers->Handlers[RXTER] = handler ;
             break ;

          default:
             assert( 0 ) ;
      }
   }
   SendNumber( RX_LASTHOOK ) ;
   if (EnvName)
      SendString( strlen(EnvName), EnvName ) ;
   else
      SendNumber( RX_NO_STRING ) ;

   if (Instore && Instore[1].strptr)
   {
      SendNumber( RX_TYPE_INSTORE ) ;
      SendNumber( *((int*)(Instore[1].strptr))) ;
   }
   else if (Instore && Instore[0].strptr)
   {
      SendNumber( RX_TYPE_SOURCE ) ;
      SendString( Instore[0].strlength, Instore[0].strptr ) ;
   }
   else if (Instore)
      SendNumber( RX_TYPE_MACRO ) ;
   else
      SendNumber( RX_TYPE_EXTERNAL ) ;

   while (1)
   {
      Code = RecvNumber() ;
      if (Code == RX_RETURN)
      {
         Code = RecvNumber() ;
         RLength = RecvNumber() ;
         RString = RecvString(RLength) ;
         break ;
      }
      else if (Code == RX_INSTORE)
      {
         Code = RecvNumber() ;
         assert( Instore ) ;
         if (Instore[1].strptr && Instore[1].strlength<sizeof(int))
         {
            free( Instore[1].strptr ) ;
            Instore[1].strptr = NULL ;
         }

         if (!Instore[1].strptr)
         {
            Instore[1].strptr = malloc(sizeof( int )) ;
            Instore[1].strlength = sizeof( int ) ;
         }

         *((int*)(Instore[1].strptr)) = Code ;
      }
      else if (Code == RX_DO_HOOK)
      {
         ProcHook() ;
      }
      else if (Code == RX_EXECFUNC)
         ExecuteFunction() ;
      else if (Code == RX_SUBCOM)
         SubCommand() ;
      else
         HandleError( ERROR_SYNCH ) ;
   }

   Handlers = CurrentHandlers ;
   CurrentHandlers = Handlers->prev ;
   free( Handlers ) ;

   ResValue = atoi( RString ) ;
   if (ReturnCode)
      *ReturnCode = ResValue ;

   if (Result)
   {
      if (!Result->strptr || Result->strlength>=RLength+1)
      {
         Result->strlength = RLength ;
         Result->strptr = RString ;
      }
      else
      {
         Result->strlength = RLength ;
         memcpy( Result->strptr, RString, RLength+1 ) ;
         free( RString ) ;
      }
   }

   if (drop_ipret)
      interpreting = 0 ;

   return Code ;
}



ULONG RexxRegisterSubcomExe( 
   PSZ EnvName,
   PFN EntryPoint,
   PUCHAR UserArea )
{
   int EnvLen ;
   struct EnvBox *NewBox ;

   /* 
    * Perform sanity check on the parameters; UserArea may be NULL 
    */
   if (!EnvName || !EntryPoint)
      return RXSUBCOM_BADTYPE ;

   if (FindEnv( EnvName ))
      return RXSUBCOM_NOTREG ;

   NewBox = malloc( sizeof( struct EnvBox )) ;
   if (!NewBox)
      return RXSUBCOM_NOEMEM ;

   EnvLen = strlen( EnvName ) ;
   NewBox->EnvName = malloc( EnvLen+1 ) ;
   if (!NewBox->EnvName)
   {
      free( NewBox ) ;
      return RXSUBCOM_NOEMEM ;
   }

   NewBox->prev = NULL ;
   NewBox->next = FirstEnv ;
   if (FirstEnv)
      FirstEnv->prev = NewBox ;
   FirstEnv = NewBox ;

   strcpy( NewBox->EnvName, EnvName ) ;
   NewBox->u.EntryPnt = EntryPoint ;
   if (UserArea)
      memcpy( NewBox->UserData, UserArea, 8 ) ;
   else
      memset( NewBox->UserData, 0x00, 8 ) ;

   return RXSUBCOM_OK ;      
}


ULONG RexxRegisterSubcomDll(
   PSZ EnvName,
   PSZ ModuleName,
   PFN EntryPoint,
   PUCHAR UserArea,
   ULONG DropAuth ) 
{
   /* not yet functional */
   return RXSUBCOM_NOTREG ;
}


ULONG RexxQuerySubcom(
   PSZ EnvName,
   PSZ ModuleName,
/* PUSHORT Flag, */   /* Who knows what this is used for ... */
   PUCHAR UserWord )
{
   int ret ;
   struct EnvBox *eptr ;

   if (!EnvName)
      return RXSUBCOM_BADTYPE ;

   if (ModuleName)
      return RXSUBCOM_BADTYPE ;   /* not yet functional */

   eptr = FindEnv( EnvName ) ;
   if (eptr)
   {
      ret = RXSUBCOM_OK ;
      if (UserWord)
         memcpy( UserWord, eptr->UserData, 8 ) ;
   }
   else
      ret = RXSUBCOM_NOTREG ;

   return ret ;
}
   
    
 
ULONG RexxDeregisterSubcom( 
   PSZ EnvName,
   PSZ ModuleName )
{
   struct EnvBox *OldBox ;

   if (!EnvName)
      return RXSUBCOM_BADTYPE ;

   if (ModuleName)
      return RXSUBCOM_BADTYPE ;  /* not yet functional */

   OldBox = FindEnv( EnvName ) ;
   if (OldBox)
   {
      if (OldBox->prev)
         OldBox->prev->next = OldBox->next ;
      if (OldBox->next)
         OldBox->next->prev = OldBox->prev ;
      if (FirstEnv==OldBox)
         FirstEnv = OldBox->prev ;
      
      free( OldBox->EnvName ) ;
      free( OldBox ) ;
      return RXSUBCOM_OK ;
   }

   return RXSUBCOM_NOTREG ;
}



int GobbleInto( RXSTRING *str, int max )
{
   int length ;

   length = RecvNumber() ;

   if (length == -1) 
      str->strptr = NULL ; 
   else if (str->strptr==NULL)
   {
      str->strptr = RecvString(length) ;
      str->strlength = length ;
      if  (str->strptr==NULL)
         return RXSHV_MEMFL ;
   }
   else
   {
      if (length>max)
      {
	  RecvData( max, str->strptr );
	  RecvJunk( length - max ) ;
	  str->strlength = max ;
          return RXSHV_TRUNC ;
       }
       else
       {
	  RecvData( length, str->strptr ) ;
	  str->strlength = length ;
       }
   }
   return RXSHV_OK ;
}



ULONG RexxVariablePool( 
   PSHVBLOCK RequestBlockList )
{
   int Code, RetCode ;
   PSHVBLOCK Req=RequestBlockList ;


   RetCode = 0 ;
   if (!RxProgPid || !interpreting)
      return RXSHV_NOAVL ;

   for (;Req;Req=Req->shvnext)
   {
      switch (Req->shvcode)
      {
         case RXSHV_SYDRO:
         case RXSHV_SYSET:
         {
            SendNumber( RX_SETSVAR ) ;
            SendString( Req->shvname.strlength, Req->shvname.strptr ) ;
            if (Req->shvcode==RXSHV_SYSET)
               SendString( Req->shvvalue.strlength, Req->shvvalue.strptr ) ;
            else
               SendNumber( RX_NO_STRING ) ;

            Code = WaitForNumber() ;
            Code = RecvNumber() ;

            Req->shvret = RXSHV_OK ;
            if (Code==RX_CODE_NOVALUE)
               Req->shvret |= RXSHV_NEWV ;
            else if (Code==RX_CODE_INVNAME)
               Req->shvret |= RXSHV_BADN ;
            else if (Code!=RXSHV_OK)
               HandleError( ERROR_SYNCH ) ;
            break ;
         }
         case RXSHV_SYFET:
         {
            SendNumber( RX_GETSVAR ) ;
            SendString( Req->shvname.strlength, Req->shvname.strptr ) ;

            Code = WaitForNumber() ;
            assert( Code == RX_VALUE ) ;

            Code = RecvNumber() ;         
            Req->shvret = RXSHV_OK ;
            if (Code==RX_CODE_NOVALUE)
               Req->shvret |= RXSHV_NEWV ;
            else if (Code==RX_CODE_INVNAME)
               Req->shvret |= RXSHV_BADN ;
            else if (Code!=RXSHV_OK)
               HandleError( ERROR_SYNCH ) ;

            Req->shvret |= GobbleInto( &(Req->shvvalue), Req->shvvaluelen) ;
            break ;
         }

         case RXSHV_PRIV:
         {
            char ctmp[32] ;   /* can hold ASCII repr. of an int */
            int Number, Length ;

            Req->shvret = RXSHV_OK ;
            if (Req->shvname.strlength==4 && Req->shvname.strptr &&
                !strncmp(Req->shvname.strptr, "PARM", 4 ))
            {
               SendNumber( RX_CODE_PARAM ) ;
               SendNumber( 0 ) ;
               Number = WaitForNumber() ;

               Number = RecvNumber() ;
               sprintf(ctmp, "%d", Number ) ;
               Length = strlen(ctmp) ;
               if (Req->shvvalue.strptr)
               { 
                  if (Req->shvvaluelen<Length)
                  {
                     Req->shvret |= RXSHV_TRUNC ;
                     Length = Req->shvvaluelen ;
                  }
                  memcpy(Req->shvvalue.strptr, ctmp, Length ) ;
                  Req->shvvalue.strlength = Length ;
               }
               else
               {
                  Req->shvvalue.strptr = malloc( Length ) ;
                  if (Req->shvvalue.strptr)
                  {
                     memcpy( Req->shvvalue.strptr, ctmp, Length ) ;
                     Req->shvvalue.strlength = Length ;
                     Req->shvvaluelen = Length ;
                  }
                  else
                     Req->shvret |= RXSHV_MEMFL ;
               }
               Number = RecvNumber() ;
	       assert( Number==RX_NO_STRING ) ;
            }
            else if (Req->shvname.strlength>=4 && Req->shvname.strptr &&
                !strncmp(Req->shvname.strptr, "PARM.", 5 ))
            {
               int cnt ;
               long sum ;

               sum = 0 ;
               for (cnt=5; cnt<Req->shvname.strlength; cnt++)
               {
                  if (!isdigit(Req->shvname.strptr[cnt]) || cnt>SHRT_MAX)
                  {
                     Req->shvret |= RXSHV_BADN ;
                     break ;
                  }
                  sum = sum*10 + Req->shvname.strptr[cnt]-'0' ;
               }
               if (sum<=0)
                  Req->shvret |= RXSHV_BADN ;

               if (!(Req->shvret & RXSHV_BADN))
               {
                  int length ;

                  SendNumber( RX_CODE_PARAM ) ;
                  SendNumber( sum ) ;
                  
                  Number = WaitForNumber() ;
                  length = RecvNumber() ;
                  length = RecvNumber() ;
                  if (length == RX_NO_STRING) 
                  {
                     Req->shvvalue.strptr = NULL ;
                  }
                  else if (Req->shvvalue.strptr==NULL)
                  {
                     Req->shvvalue.strptr = RecvString(length) ;
                     Req->shvvalue.strlength = length ;
                     if  (Req->shvvalue.strptr==NULL)
                        Req->shvret |= RXSHV_MEMFL ;
                  }
                  else
                  {
                     if (length>Req->shvvaluelen)
                     {
                        Req->shvret |= RXSHV_TRUNC ;
                        RecvData( Req->shvvaluelen, Req->shvvalue.strptr );
                        RecvJunk( length - Req->shvvaluelen ) ;
                        Req->shvvalue.strlength = Req->shvvaluelen ;
                     }
                     else
                     {
                        RecvData( length, Req->shvvalue.strptr ) ;
                        Req->shvvalue.strlength = length ;
                     }
                  }
               }
            }
            else 
            {
               int length ;
               if (Req->shvname.strptr)
               {
                  if (Req->shvname.strlength==7 &&
                         !memcmp(Req->shvname.strptr, "QUENAME", 7))
                  {
                     SendNumber( RX_CODE_QUEUE ) ;
                  }
                  else if (Req->shvname.strlength==7 &&
                         !memcmp(Req->shvname.strptr, "VERSION", 7))
                  {
                     SendNumber( RX_CODE_VERSION) ;
                  }
                  else if (Req->shvname.strlength==6 &&
                         !memcmp(Req->shvname.strptr, "SOURCE", 6))
                  {
                     SendNumber( RX_CODE_SOURCE ) ;
                  }
                  else
                     Req->shvret |= RXSHV_BADN ;

                  if (!Req->shvret | RXSHV_BADN)
                  {
                     Number = WaitForNumber() ;
                     length = RecvNumber() ;
                     length = RecvNumber() ;
                     if (length == -1) 
                        { } 
                     else if (Req->shvvalue.strptr==NULL)
                     {
                        Req->shvvalue.strptr = RecvString(length) ;
                        Req->shvvalue.strlength = length ;
                        if  (Req->shvvalue.strptr==NULL)
                           Req->shvret |= RXSHV_MEMFL ;
                     }
                     else
                     {
                        if (length>Req->shvvaluelen)
                        {
                           Req->shvret |= RXSHV_TRUNC ;
                           RecvData( Req->shvvaluelen, Req->shvvalue.strptr );
                           RecvJunk( length - Req->shvvaluelen ) ;
                           Req->shvvalue.strlength = Req->shvvaluelen ;
                        }
                        else
                        {
                           RecvData( length, Req->shvvalue.strptr ) ;
                           Req->shvvalue.strlength = length ;
                        }
                     }
                  }
               }
               else
                  Req->shvret |= RXSHV_BADN ;
            }
            break ;
         }
     
         case RXSHV_NEXTV:
         {
            int Items ;

            Req->shvret = RXSHV_OK ;
	    
            SendNumber(RX_NEXTVAR) ;
            Code = RecvNumber() ;
            Items = RecvNumber() ;

            assert( Code==RX_STRINGS ) ;
            assert( Items==0 || Items==2 ) ;

            if (Items==2)
            {
               Req->shvret |= GobbleInto( &(Req->shvname), Req->shvnamelen ) ;
               Req->shvret |= GobbleInto( &(Req->shvvalue), Req->shvvaluelen);
            }
            else
               Req->shvret |= RXSHV_LVAR ;

            break ;
         }

         default:
            Req->shvret = RXSHV_BADF ;
      }

      RetCode |= ( Req->shvret & 0x007f ) ;
   }

   return RetCode ;
}





ULONG RexxRegisterExitExe( 
   PSZ EnvName,
   PFN EntryPoint, 
   PUCHAR UserArea )
{
   int EnvLen ;
   struct EnvBox *NewBox ;

   /* 
    * Perform sanity check on the parameters; UserArea may be NULL 
    */
   if (!EnvName || !EntryPoint)
      return RXEXIT_BADTYPE ;

   EnvLen = strlen( EnvName ) ;
   if (EnvLen>MAXENVNAMELEN)
      return RXEXIT_NOTREG ;

   if (FindEnv( EnvName))
      return RXEXIT_NOTREG ;

   NewBox = malloc( sizeof( struct EnvBox )) ;
   if (!NewBox)
      return RXEXIT_NOEMEM ;

   NewBox->EnvName = malloc( strlen( EnvName)+1 ) ;
   if (!NewBox->EnvName)
   {
      free( NewBox ) ;
      return RXEXIT_NOEMEM ;
   }

   NewBox->prev = NULL ;
   NewBox->next = FirstExit ;
   if (FirstExit)
      FirstExit->prev = NewBox ;
   FirstExit = NewBox ;
   strcpy( NewBox->EnvName, EnvName ) ;
   if (UserArea)
      memcpy( NewBox->UserData, UserArea, 8 ) ;
   else
      memset( NewBox->UserData, 0x00, 8 ) ;
   NewBox->u.EntryPnt = EntryPoint ;

   return RXEXIT_OK ;      
}


int RexxDeregisterExit( 
   PSZ EnvName,
   PSZ ModuleName )
{
   struct EnvBox *OldBox ;

   if (!EnvName)
      return RXEXIT_BADTYPE ;

   if (ModuleName)
      return RXEXIT_BADTYPE ;

   OldBox = FindEnv( EnvName ) ;
   if (OldBox)
   {
      if (OldBox->prev)
         OldBox->prev->next = OldBox->next ;
      if (OldBox->next)
         OldBox->next->prev = OldBox->prev ;
      if (FirstExit==OldBox)
         FirstExit = OldBox->prev ;
      
      free( OldBox ) ;
      return RXEXIT_OK ;
   }

   return RXEXIT_NOTREG ;
}


   
void DosFreeMem( void *ptr ) 
{
   SendNumber( RX_DROP_INSTORE ) ;
   SendNumber( *((int*)(ptr)) ) ;
   WaitForNumber() ;
}
 


struct funcbox {
   struct funcbox *next, *prev ;
   PSZ name ;
   RexxFunctionHandler *entry ;
} *firstfunc=NULL ;


static struct funcbox *findfunc( char *name )
{
   struct funcbox *fptr ;

   for (fptr=firstfunc; fptr; fptr=fptr->prev) 
      if (!strcmp(name, fptr->name))
         return fptr ;

   return NULL ;
}


static int delfunc( char *name )
{
   struct funcbox *old ;
  
   old = findfunc( name ) ;
   if (!old)
      return RXFUNC_NOTREG ;

   free( old->name ) ;
   if (old==firstfunc)
      firstfunc = old->prev ;
   else 
      old->next->prev = old->prev ;

   if (old->prev)
      old->prev->next = old->next ;

   free( old ) ;
   return RXFUNC_OK ;
}


static int addfunc( PSZ name, RexxFunctionHandler *EntryPoint ) 
{
   struct funcbox *new ;

   if (findfunc( name ))
      return RXFUNC_DEFINED ;

   new = malloc( sizeof(struct funcbox )) ;
   if (!new)
      return RXFUNC_NOMEM ;

   new->name = malloc( strlen( name )+1 ) ;
   if (!new->name)
   {
      free( new ) ;
      return RXFUNC_NOMEM ;
   }

   strcpy( new->name, name ) ;
   new->entry = EntryPoint ;

   new->next = NULL ;
   new->prev = firstfunc ;
   if (firstfunc)
      firstfunc->next = new ;
   firstfunc = new ;

   return RXFUNC_OK ;
}



ULONG RexxRegisterFunctionExe( PSZ Name, PFN EntryPoint )
{
   int code ;

   code = addfunc( Name, (RexxFunctionHandler*)EntryPoint ) ;
   if (code)
      return code ;

   SendNumber( RX_ADDFUNC ) ;
   SendString( strlen(Name), Name ) ;

   code = WaitForNumber() ;
   assert( code==RX_RETVOID ) ;

   return RXFUNC_OK ;
}
      


ULONG RexxQueryFunction( PSZ Name )
{
   return (findfunc(Name)) ? RXFUNC_OK : RXFUNC_NOTREG ;
}


ULONG RexxDeregisterFunction( PSZ Name ) 
{
   int code ;

   SendNumber( RX_DELFUNC ) ;
   SendString( strlen(Name), Name ) ;

   code = WaitForNumber() ;
   assert( code==RX_RETVOID ) ;

   return delfunc( Name ) ;
}


void ExecuteFunction( void )
{
   char foo ;
   long nparams ;
   struct funcbox *fptr ;
   int i, length, rc ;
   char *name ;
   RXSTRING *params ;
   RXSTRING retstr ;
   char retdata[256] ;

   name = RecvString( RecvNumber() ) ;

   nparams = RecvNumber() ;
   params = malloc( sizeof(RXSTRING)*nparams ) ;
   for (i=0; i<nparams; i++)
   {
      length = RecvNumber() ;
      if (length==RX_NO_STRING)
      {
         params[i].strptr = NULL ;
         params[i].strlength = 0 ;
      }
      else if (length==0)
      {
         params[i].strptr = &foo ;
         params[i].strlength = 0 ;
      }
      else
      {
         assert( length>0 ) ;
         params[i].strptr = RecvString(length) ;
         params[i].strlength = length ;
      }
   }

   fptr = findfunc( name ) ;
   if (!fptr)
   {
      SendNumber( RX_RETURN ) ;
      SendNumber( RX_CODE_NOSUCH ) ;
      SendNumber( RX_NO_STRING ) ;
      return ;
   }

   retstr.strptr = retdata ;
   retstr.strlength = 256 ;
   rc = (*(fptr->entry))( name, nparams, params, "default", &retstr ) ;

   for (i=0; i<nparams; i++)
      if (params[i].strptr && params[i].strlength)
         free( params[i].strptr ) ;

   free( name ) ;
   free( params ) ;

   SendNumber( RX_RETURN ) ;
   SendNumber( rc ) ;

   if (!rc && retstr.strptr)
      SendString( retstr.strlength, retstr.strptr ) ;
   else
      SendNumber( RX_NO_STRING ) ;

   if (retstr.strptr != retdata)
      free( retstr.strptr ) ;
}

   
   





