/***********************************************************************/
/* PREFIX.C - Prefix commands.                                         */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1993 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Header: C:\THE\RCS\prefix.c 1.4 1993/09/01 16:26:56 MH Interim MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"
/*-------------------------- declarations -----------------------------*/
#ifdef PROTO
static int parse_prefix_command(int,char *,char *);
static int invalidate_prefix(int);
static int prefix_makecurr(int,int,long);
static int prefix_add(int,int,long);
static int prefix_duplicate(int,int,long);
static int prefix_copy(int,int,long);
static int prefix_move(int,int,long);
static int prefix_delete(int,int,long);
static int prefix_shift_left(int,int,long);
static int prefix_shift_right(int,int,long);
static int prefix_block_duplicate(int,int,long);
static int prefix_block_copy(int,int,long);
static int prefix_block_move(int,int,long);
static int prefix_block_delete(int,int,long);
static int prefix_block_shift_left(int,int,long);
static int prefix_block_shift_right(int,int,long);
void clear_pending_prefix_command(int ,LINE *);
static int find_bottom_ppc(int,int);
static int find_target_ppc(int *);
static long calculate_target_line(void);
#if !defined(NOREXX)
static int try_rexx_prefix_macro(int);
#endif
static char *substr(char *, char *, int, int);
#else
static int parse_prefix_command();
static int invalidate_prefix();
static int prefix_makecurr();
static int prefix_add();
static int prefix_duplicate();
static int prefix_copy();
static int prefix_move();
static int prefix_delete();
static int prefix_shift_left();
static int prefix_shift_right();
static int prefix_block_duplicate();
static int prefix_block_copy();
static int prefix_block_move();
static int prefix_block_delete();
static int prefix_block_shift_left();
static int prefix_block_shift_right();
void clear_pending_prefix_command();
static int find_bottom_ppc();
static int find_target_ppc();
static long calculate_target_line();
static char *substr();
#endif
/*---------------------------------------------------------------------*/
/* The following two are to specify the first and last items in the    */
/* linked list for prefix synonyms.                                    */
/*---------------------------------------------------------------------*/
LINE *first_prefix_synonym=NULL;
LINE *last_prefix_synonym=NULL;
/*---------------------------------------------------------------------*/

#define PPC_NO_TARGET        (-1)
#define PPC_NO_COMMAND       (-2)
#define PPC_TARGET_PREVIOUS  0
#define PPC_TARGET_FOLLOWING 1
/* the above two defines correspond to the position in the pc[] array  */
/* and should be changed if the position in pc[] array changes.        */

#define NUMBER_PREFIX_COMMANDS 17
 static PREFIX_COMMAND pc[NUMBER_PREFIX_COMMANDS] =
  {
   {(char *)"p",1,FALSE,FALSE,FALSE,FALSE,FALSE,FALSE,TRUE,NULL},
   {(char *)"f",1,FALSE,FALSE,FALSE,FALSE,FALSE,TRUE,FALSE,NULL},
   {(char *)"\"\"",2,TRUE,TRUE,FALSE,FALSE,FALSE,FALSE,FALSE,prefix_block_duplicate},
   {(char *)"cc",2,TRUE,TRUE,FALSE,TRUE,TRUE,FALSE,FALSE,prefix_block_copy},
   {(char *)"mm",2,TRUE,TRUE,FALSE,TRUE,TRUE,FALSE,FALSE,prefix_block_move},
   {(char *)"dd",2,TRUE,TRUE,FALSE,TRUE,FALSE,FALSE,FALSE,prefix_block_delete},
   {(char *)"<<",2,TRUE,TRUE,FALSE,TRUE,FALSE,FALSE,FALSE,prefix_block_shift_left},
   {(char *)">>",2,TRUE,TRUE,FALSE,TRUE,FALSE,FALSE,FALSE,prefix_block_shift_right},
   {(char *)"\"",1,TRUE,TRUE,FALSE,FALSE,FALSE,FALSE,FALSE,prefix_duplicate},
   {(char *)"c",1,TRUE,TRUE,TRUE,FALSE,TRUE,FALSE,FALSE,prefix_copy},
   {(char *)"m",1,TRUE,TRUE,TRUE,FALSE,TRUE,FALSE,FALSE,prefix_move},
   {(char *)"d",1,TRUE,TRUE,TRUE,FALSE,FALSE,FALSE,FALSE,prefix_delete},
   {(char *)"<",1,TRUE,TRUE,FALSE,FALSE,FALSE,FALSE,FALSE,prefix_shift_left},
   {(char *)">",1,TRUE,TRUE,FALSE,FALSE,FALSE,FALSE,FALSE,prefix_shift_right},
   {(char *)"i",1,TRUE,TRUE,FALSE,FALSE,FALSE,TRUE,FALSE,prefix_add},
   {(char *)"a",1,TRUE,TRUE,FALSE,FALSE,FALSE,TRUE,FALSE,prefix_add},
   {(char *)"/",1,TRUE,FALSE,FALSE,FALSE,FALSE,TRUE,TRUE,prefix_makecurr},
  };

char pending_prefix_command[PREFIX_WIDTH+1]="";
long prefix_current_line;
bool in_prefix_macro=FALSE;     /* indicate if processing prefix macro */
/*-------------------------- external data ----------------------------*/
extern VIEW_DETAILS *vd_current,*vd_first,*vd_mark;
/***********************************************************************/
#ifdef PROTO
int execute_prefix_commands(void)
#else
int execute_prefix_commands()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern char number_of_files;
/*--------------------------- local data ------------------------------*/
 register int i;
 int len,cmd_idx,rc,ppc_idx;
 char cmd[PREFIX_WIDTH+1];
 char mult[PREFIX_WIDTH+1];
 int first_pending = (-1);
 long long_mult=0L;
 long start_line,number_lines,target_line;
 int top_ppc,bot_ppc,target_ppc;
 int number_prefix_commands;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("prefix.c:  execute_prefix_commands");
#endif
 post_process_line(CURRENT_VIEW->focus_line);
/*---------------------------------------------------------------------*/
/* This variable is set here because post_process_line() actually sets */
/* CURRENT_VIEW->prefix_command_index.                                 */
/*---------------------------------------------------------------------*/
 number_prefix_commands = CURRENT_VIEW->prefix_command_index;
/*---------------------------------------------------------------------*/
/* For each pending prefix command for the current view, validate the  */
/* command and parameters.                                             */
/*---------------------------------------------------------------------*/
 for (i=0;i<number_prefix_commands;i++)
   {
/*---------------------------------------------------------------------*/
/* If a set point command, validate the name.                          */
/*---------------------------------------------------------------------*/
    if (CURRENT_VIEW->ppc[i].ppc_command[0] == '.')
      {
       if (isalpha(CURRENT_VIEW->ppc[i].ppc_command[1]))
         {
          if (execute_set_point(CURRENT_VIEW->ppc[i].ppc_command,CURRENT_VIEW->ppc[i].ppc_line_number,TRUE) != RC_OK)
            {
             invalidate_prefix(i);
             continue;
            }
          clear_pending_prefix_command(i,(LINE *)NULL);
          continue;
         }
       else
         {
          invalidate_prefix(i);
          continue;
         }
      }
/*---------------------------------------------------------------------*/
/* If an invalid prefix command from last time, clear it.              */
/*---------------------------------------------------------------------*/
    if (CURRENT_VIEW->ppc[i].ppc_command[0] == '?')
      {
       clear_pending_prefix_command(i,(LINE *)NULL);
       continue;
      }
/*---------------------------------------------------------------------*/
/* To get here we have 'normal' prefix command.                        */
/*---------------------------------------------------------------------*/
#if !defined(NOREXX)               /* if REXX support is in effect ... */
/*---------------------------------------------------------------------*/
/* first determine if the command is a prefix macro BEFORE looking for */
/* standard prefix commands.                                           */
/*---------------------------------------------------------------------*/
    rc = try_rexx_prefix_macro(i);
/*---------------------------------------------------------------------*/
/* If at this point there are no more files in the ring; we assume that*/
/* this was caused by exitting the last file in the ring from a prefix */
/* macro, exit and ignore any more prefix commands. This is messy !!!  */
/*---------------------------------------------------------------------*/
    if (number_of_files == 0)
      {
/*     invalidate_prefix(i);*/
#ifdef TRACE
       trace_return();
#endif
       return(RC_COMMAND_NO_FILES);
      }
    if (rc != RC_NOT_COMMAND)
       continue;
#endif
/*---------------------------------------------------------------------*/
/* if no prefix macro found for the prefix command, check to see if it */
/* is a standard prefix command.                                       */
/*---------------------------------------------------------------------*/
    if ((cmd_idx = parse_prefix_command(i,cmd,mult)) == PPC_NO_TARGET)
      {
       invalidate_prefix(i);
       continue;
      }
    if (cmd_idx == PPC_NO_COMMAND)
       continue;
/*---------------------------------------------------------------------*/
/* We now have a recognised command. We have to validate its parameters*/
/* and find any associated pending commands.                           */
/*---------------------------------------------------------------------*/
/*---------------------------------------------------------------------*/
/* Set the block_command flag for the current prefix command to the    */
/* appropriate value for the prefix command.                           */
/*---------------------------------------------------------------------*/
    CURRENT_VIEW->ppc[i].ppc_block_command = pc[cmd_idx].block_prefix_command;
/*---------------------------------------------------------------------*/
/* If the command does not allow parameters and there are parameters,  */
/* error.                                                              */
/*---------------------------------------------------------------------*/
    if (!pc[cmd_idx].multiples_allowed
    && strcmp(mult,"") != 0)
      {
       invalidate_prefix(i);
       continue;
      }
/*---------------------------------------------------------------------*/
/* If the command does allow parameters and there are no parameters,   */
/* default to 1.                                                       */
/*---------------------------------------------------------------------*/
    if (strcmp(mult,"") == 0)
       strcpy(mult,"1");
/*---------------------------------------------------------------------*/
/* If the command allows full targets, validate the target.            */
/*---------------------------------------------------------------------*/
    if (pc[cmd_idx].multiples_allowed)
      {
       if (pc[cmd_idx].full_target_allowed)
          long_mult = valid_target(mult,CURRENT_VIEW->ppc[i].ppc_line_number);
       else
          if (!valid_positive_integer(mult))
             long_mult = TARGET_ERROR;
          else
             long_mult = atol(mult);
      }
/*---------------------------------------------------------------------*/
/* If the target is invalid or not found, invalidate the command.      */
/*---------------------------------------------------------------------*/
    if (long_mult == TARGET_ERROR
    ||  long_mult == TARGET_NOT_FOUND)
      {
       invalidate_prefix(i);
       continue;
      }
    CURRENT_VIEW->ppc[i].ppc_cmd_param = long_mult;
   }
/*---------------------------------------------------------------------*/
/* For each valid pending  command for the current view, execute  the  */
/* command.                                                            */
/*---------------------------------------------------------------------*/
 strcpy(pending_prefix_command,"");
 for (i=0;i<number_prefix_commands;i++)
   {
    top_ppc = bot_ppc = target_ppc = (-1);
/*---------------------------------------------------------------------*/
/* Execute the function associated with the prefix command...          */
/*---------------------------------------------------------------------*/
    cmd_idx = CURRENT_VIEW->ppc[i].ppc_cmd_idx;
    long_mult = CURRENT_VIEW->ppc[i].ppc_cmd_param;
    if (cmd_idx != (-1)
    &&  pc[cmd_idx].function != NULL)
       rc = (*pc[cmd_idx].function)(i,cmd_idx,long_mult);
   }
/*---------------------------------------------------------------------*/
/* Now that we are here, we have to reset the count of the pending     */
/* prefix commands.                                                    */
/*---------------------------------------------------------------------*/
 ppc_idx = 0;
 strcpy(pending_prefix_command,"");
 for (i=0;i<number_prefix_commands;i++)
   {
    if (CURRENT_VIEW->ppc[i].ppc_line_number != (-1L))
       ppc_idx = i;
/*---------------------------------------------------------------------*/
/* If the heading display of pending prefix commands is blank, set it  */
/* to the command for which there is no bottom block or target command.*/
/*---------------------------------------------------------------------*/
    if (strcmp(pending_prefix_command,"") == 0
    && CURRENT_VIEW->ppc[i].ppc_cmd_idx != (-1))
       strcpy(pending_prefix_command,pc[CURRENT_VIEW->ppc[i].ppc_cmd_idx].cmd);
   }
 pre_process_line(CURRENT_VIEW->focus_line);
 show_page();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
static int parse_prefix_command(int ppc_idx,char *cmd,
                                char *mult)
#else
static int parse_prefix_command(ppc_idx,cmd,mult)
int ppc_idx;
char *cmd,*mult;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register int i,j;
 int len_pcmd;
 char pcmd[PREFIX_WIDTH+1];
 int pos,rc;
 LINE *curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("prefix.c:  parse_prefix_command");
#endif
/*---------------------------------------------------------------------*/
/* For each pending prefix command for the current view, execute the   */
/* appropriate command.                                                */
/*---------------------------------------------------------------------*/
 rc = PPC_NO_TARGET;
 strcpy(pcmd,CURRENT_VIEW->ppc[ppc_idx].ppc_command);/* get our own copy to pull apart */
 if (blank_field(pcmd))       /* if prefix command is blank, return */
   {
#ifdef TRACE
    trace_return();
#endif
    return(PPC_NO_COMMAND);
   }
 len_pcmd = strlen(pcmd);
/*---------------------------------------------------------------------*/
/* For each prefix synonym, determine if it exists somewhere in the    */
/* entered prefix command.                                             */
/*---------------------------------------------------------------------*/
 curr = first_prefix_synonym;
 while(curr != NULL)
   {
    pos = memfind(pcmd,curr->name,len_pcmd,strlen(curr->name),TRUE,FALSE,' ');
    if (pos == (-1))
      {
       curr = curr->next;
       continue;
      }
/*---------------------------------------------------------------------*/
/* Now that a match on synonym is made, determine the original prefix  */
/* command associated with that synonym...                             */
/*---------------------------------------------------------------------*/
    for (i=0;i<NUMBER_PREFIX_COMMANDS;i++)
      {
       if (strcmp(pc[i].cmd,curr->line) == 0)
         {
          strcpy(cmd,pc[i].cmd);
          for (j=0;j<strlen(curr->name);j++)
             *(pcmd+pos+j) = ' ';
          strtrunc(pcmd);
          strcpy(mult,pcmd);
          CURRENT_VIEW->ppc[ppc_idx].ppc_cmd_idx = i;
#ifdef TRACE
          trace_return();
#endif
          return(i);
         }
      }
/*---------------------------------------------------------------------*/
/* To get here we found a prefix synonym, but no matching original     */
/* command, so return an error.                                        */
/*---------------------------------------------------------------------*/
    CURRENT_VIEW->ppc[ppc_idx].ppc_cmd_idx = (-1);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* For each valid prefix command,  check to see if a valid command     */
/* exists somewhere in the entered prefix command.                     */
/*---------------------------------------------------------------------*/
 for (i=0;i<NUMBER_PREFIX_COMMANDS;i++)
   {
    pos = memfind(pcmd,pc[i].cmd,len_pcmd,pc[i].cmd_len,TRUE,FALSE,' ');
    if (pos == (-1))
       continue;
    strcpy(cmd,pc[i].cmd);
    for (j=0;j<pc[i].cmd_len;j++)
       *(pcmd+pos+j) = ' ';
    strtrunc(pcmd);
    strcpy(mult,pcmd);
    rc = i;
/*---------------------------------------------------------------------*/
/* Set a flag in ppc[] array to indicate which command is present.     */
/*---------------------------------------------------------------------*/
    CURRENT_VIEW->ppc[ppc_idx].ppc_cmd_idx = i;
    break;
   }
/*---------------------------------------------------------------------*/
/* If command not found, set a flag in ppc[] array to indicate command */
/* is invalid.                                                         */
/*---------------------------------------------------------------------*/
 if (rc == PPC_NO_TARGET)
    CURRENT_VIEW->ppc[ppc_idx].ppc_cmd_idx = (-1);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_makecurr(int ppc_idx,int cmd_idx,long number_lines)
#else
static int prefix_makecurr(ppc_idx,cmd_idx,number_lines)
int ppc_idx,cmd_idx;
long number_lines;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 long top_line = CURRENT_VIEW->ppc[ppc_idx].ppc_line_number;
/*--------------------------- processing ------------------------------*/
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 execute_makecurr(top_line);
 return(0);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_add(int ppc_idx,int cmd_idx,long number_lines)
#else
static int prefix_add(ppc_idx,cmd_idx,number_lines)
int ppc_idx,cmd_idx;
long number_lines;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int rc=(-1);
 long top_line = CURRENT_VIEW->ppc[ppc_idx].ppc_line_number;
/*--------------------------- processing ------------------------------*/
 if (top_line == CURRENT_FILE->number_lines+1)
    top_line--;
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 rc = insert_new_line((char *)"",0,number_lines,
                     top_line,FALSE,FALSE);
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_duplicate(int ppc_idx,int cmd_idx,long number_occ)
#else
static int prefix_duplicate(ppc_idx,cmd_idx,number_occ)
int ppc_idx,cmd_idx;
long number_occ;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int rc=(-1);
 long top_line = CURRENT_VIEW->ppc[ppc_idx].ppc_line_number;
/*--------------------------- processing ------------------------------*/
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 if (top_line != 0L
 &&  top_line != CURRENT_FILE->number_lines+1)
    rc = rearrange_line_blocks(COMMAND_DUPLICATE,SOURCE_PREFIX,top_line,top_line,top_line,(int)number_occ,CURRENT_VIEW,CURRENT_VIEW);
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_copy(int ppc_idx,int cmd_idx,long number_lines)
#else
static int prefix_copy(ppc_idx,cmd_idx,number_lines)
int ppc_idx,cmd_idx;
long number_lines;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 long bottom_line,target_line;
 long top_line = CURRENT_VIEW->ppc[ppc_idx].ppc_line_number;
 int rc=(-1);
/*--------------------------- processing ------------------------------*/
 if ((target_line = calculate_target_line()) == (-1L))
    return(-1L);
 bottom_line = top_line + number_lines - ((number_lines < 0L) ? (-1L) : 1L);
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 if (top_line != 0L
 &&  top_line != CURRENT_FILE->number_lines+1)
    rc = rearrange_line_blocks(COMMAND_COPY,SOURCE_PREFIX,top_line,bottom_line,target_line,1,CURRENT_VIEW,CURRENT_VIEW);
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_move(int ppc_idx,int cmd_idx,long number_lines)
#else
static int prefix_move(ppc_idx,cmd_idx,number_lines)
int ppc_idx,cmd_idx;
long number_lines;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 long bottom_line,target_line;
 long top_line = CURRENT_VIEW->ppc[ppc_idx].ppc_line_number;
 int rc=(-1);
/*--------------------------- processing ------------------------------*/
 if ((target_line = calculate_target_line()) == (-1L))
    return(-1L);
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 bottom_line = top_line + number_lines - ((number_lines < 0L) ? (-1L) : 1L);

 if (top_line != 0L
 &&  top_line != CURRENT_FILE->number_lines+1)
   {
    if ((rc = rearrange_line_blocks(COMMAND_MOVE_COPY_SAME,SOURCE_PREFIX,
         top_line,bottom_line,target_line,1,CURRENT_VIEW,CURRENT_VIEW)) != RC_OK)
       return(rc);
    if (target_line < top_line)
      {
       top_line += number_lines;
       target_line += number_lines;
       bottom_line += number_lines;
      }
    rc = rearrange_line_blocks(COMMAND_MOVE_DELETE_SAME,SOURCE_PREFIX,
         top_line,bottom_line,target_line,1,CURRENT_VIEW,CURRENT_VIEW);
   }
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_delete(int ppc_idx,int cmd_idx,long number_lines)
#else
static int prefix_delete(ppc_idx,cmd_idx,number_lines)
int ppc_idx,cmd_idx;
long number_lines;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int rc=(-1);
 long top_line = CURRENT_VIEW->ppc[ppc_idx].ppc_line_number;
 long bottom_line,target_line;
/*--------------------------- processing ------------------------------*/

 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 if (!TOF(top_line)
 &&  !BOF(top_line))
   {
    bottom_line = top_line + number_lines - ((number_lines < 0L) ? (-1L) : 1L);
    target_line = (number_lines < 0L) ? (bottom_line) : (top_line);
    rc = rearrange_line_blocks(COMMAND_DELETE,SOURCE_PREFIX,top_line,bottom_line,target_line,1,CURRENT_VIEW,CURRENT_VIEW);
   }
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_shift_left(int ppc_idx,int cmd_idx,long number_cols)
#else
static int prefix_shift_left(ppc_idx,cmd_idx,number_cols)
int ppc_idx,cmd_idx;
long number_cols;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 long top_line = CURRENT_VIEW->ppc[ppc_idx].ppc_line_number;
/*--------------------------- processing ------------------------------*/
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 if (top_line != 0L
 &&  top_line != CURRENT_FILE->number_lines+1)
   {
    execute_shift_command(TRUE,(int)number_cols,top_line,1L);
   }
 return(0);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_shift_right(int ppc_idx,int cmd_idx,long number_cols)
#else
static int prefix_shift_right(ppc_idx,cmd_idx,number_cols)
int ppc_idx,cmd_idx;
long number_cols;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 long start_line = CURRENT_VIEW->ppc[ppc_idx].ppc_line_number;
/*--------------------------- processing ------------------------------*/
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 if (start_line != 0L
 &&  start_line != CURRENT_FILE->number_lines+1)
   {
    execute_shift_command(FALSE,(int)number_cols,start_line,1L);
   }
 return(0);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_block_duplicate(int ppc_idx,int cmd_idx,long number_occ)
#else
static int prefix_block_duplicate(ppc_idx,cmd_idx,number_occ)
int ppc_idx,cmd_idx;
long number_occ;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int bottom_idx,rc;
 long top_line,bottom_line;
/*--------------------------- processing ------------------------------*/
 if ((bottom_idx = find_bottom_ppc(ppc_idx,cmd_idx)) == PPC_NO_TARGET)
    return(-1L);
 top_line = min(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);
 bottom_line = max(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);
 top_line = (top_line == 0L) ? 1L : top_line;
 bottom_line = (bottom_line == CURRENT_FILE->number_lines+1L) ? bottom_line-1L : bottom_line;
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 clear_pending_prefix_command(bottom_idx,(LINE *)NULL);
 rc = rearrange_line_blocks(COMMAND_DUPLICATE,SOURCE_PREFIX,top_line,bottom_line,bottom_line,(int)number_occ,CURRENT_VIEW,CURRENT_VIEW);
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_block_copy(int ppc_idx,int cmd_idx,long number_lines)
#else
static int prefix_block_copy(ppc_idx,cmd_idx,number_lines)
int ppc_idx,cmd_idx;
long number_lines;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int bottom_idx,rc;
 long top_line,bottom_line,target_line;
/*--------------------------- processing ------------------------------*/
 if ((bottom_idx = find_bottom_ppc(ppc_idx,cmd_idx)) == PPC_NO_TARGET)
    return(-1L);
 if ((target_line = calculate_target_line()) == (-1L))
    return(-1L);
 top_line = min(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);
 bottom_line = max(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);
 top_line = (top_line == 0L) ? 1L : top_line;
 bottom_line = (bottom_line == CURRENT_FILE->number_lines+1L) ? bottom_line-1L : bottom_line;
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 clear_pending_prefix_command(bottom_idx,(LINE *)NULL);
 rc = rearrange_line_blocks(COMMAND_COPY,SOURCE_PREFIX,top_line,bottom_line,target_line,1,CURRENT_VIEW,CURRENT_VIEW);
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_block_move(int ppc_idx,int cmd_idx,long number_lines)
#else
static int prefix_block_move(ppc_idx,cmd_idx,number_lines)
int ppc_idx,cmd_idx;
long number_lines;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int bottom_idx,rc;
 long top_line,bottom_line,target_line,num_lines;
/*--------------------------- processing ------------------------------*/
 if ((bottom_idx = find_bottom_ppc(ppc_idx,cmd_idx)) == PPC_NO_TARGET)
    return(-1L);
 if ((target_line = calculate_target_line()) == (-1L))
    return(-1L);
 top_line = min(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);
 bottom_line = max(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);
 top_line = (top_line == 0L) ? 1L : top_line;
 bottom_line = (bottom_line == CURRENT_FILE->number_lines+1L) ? bottom_line-1L : bottom_line;
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 clear_pending_prefix_command(bottom_idx,(LINE *)NULL);
 if ((rc = rearrange_line_blocks(COMMAND_MOVE_COPY_SAME,SOURCE_PREFIX,
           top_line,bottom_line,target_line,1,CURRENT_VIEW,CURRENT_VIEW)) != RC_OK)
    return(rc);
 if (target_line < top_line)
   {
    num_lines = bottom_line - top_line + 1L;
    top_line += num_lines;
    bottom_line += num_lines;
    target_line += num_lines;
   }
 rc = rearrange_line_blocks(COMMAND_MOVE_DELETE_SAME,SOURCE_PREFIX,
      top_line,bottom_line,target_line,1,CURRENT_VIEW,CURRENT_VIEW);
 return(rc);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_block_delete(int ppc_idx,int cmd_idx,long number_lines)
#else
static int prefix_block_delete(ppc_idx,cmd_idx,number_lines)
int ppc_idx,cmd_idx;
long number_lines;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int bottom_idx,rc;
 long top_line,bottom_line;
/*--------------------------- processing ------------------------------*/
 if ((bottom_idx = find_bottom_ppc(ppc_idx,cmd_idx)) == PPC_NO_TARGET)
    return(-1L);
 top_line = min(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);
 bottom_line = max(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);
 top_line = (top_line == 0L) ? 1L : top_line;
 bottom_line = (bottom_line == CURRENT_FILE->number_lines+1L) ? bottom_line-1L : bottom_line;
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 clear_pending_prefix_command(bottom_idx,(LINE *)NULL);
 rc = rearrange_line_blocks(COMMAND_DELETE,SOURCE_PREFIX,top_line,bottom_line,bottom_line,1,CURRENT_VIEW,CURRENT_VIEW);
 return(rc);
/*--------------------------- processing ------------------------------*/
}
/***********************************************************************/
#ifdef PROTO
static int prefix_block_shift_left(int ppc_idx,int cmd_idx,long number_cols)
#else
static int prefix_block_shift_left(ppc_idx,cmd_idx,number_cols)
int ppc_idx,cmd_idx;
long number_cols;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int bottom_idx;
 long top_line,bottom_line;
/*--------------------------- processing ------------------------------*/
 if ((bottom_idx = find_bottom_ppc(ppc_idx,cmd_idx)) == PPC_NO_TARGET)
    return(-1L);
 top_line = min(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);
 bottom_line = max(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);

 top_line = (top_line == 0L) ? 1L : top_line;
 bottom_line = (bottom_line == CURRENT_FILE->number_lines+1L) ? bottom_line-1L : bottom_line;
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 clear_pending_prefix_command(bottom_idx,(LINE *)NULL);

 execute_shift_command(TRUE,(int)number_cols,top_line,bottom_line-top_line+1L);

 return(0);
}
/***********************************************************************/
#ifdef PROTO
static int prefix_block_shift_right(int ppc_idx,int cmd_idx,long number_cols)
#else
static int prefix_block_shift_right(ppc_idx,cmd_idx,number_cols)
int ppc_idx,cmd_idx;
long number_cols;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int bottom_idx;
 long top_line,bottom_line;
/*--------------------------- processing ------------------------------*/
 if ((bottom_idx = find_bottom_ppc(ppc_idx,cmd_idx)) == PPC_NO_TARGET)
    return(-1L);
 top_line = min(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);
 bottom_line = max(CURRENT_VIEW->ppc[ppc_idx].ppc_line_number,CURRENT_VIEW->ppc[bottom_idx].ppc_line_number);

 top_line = (top_line == 0L) ? 1L : top_line;
 bottom_line = (bottom_line == CURRENT_FILE->number_lines+1L) ? bottom_line-1L : bottom_line;
 clear_pending_prefix_command(ppc_idx,(LINE *)NULL);
 clear_pending_prefix_command(bottom_idx,(LINE *)NULL);

 execute_shift_command(FALSE,(int)number_cols,top_line,bottom_line-top_line+1L);

 return(0);
}
/***********************************************************************/
#ifdef PROTO
static int invalidate_prefix(int ppc_idx)
#else
static int invalidate_prefix(ppc_idx)
int ppc_idx;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int len;
/*--------------------------- processing ------------------------------*/
 if (*(CURRENT_VIEW->ppc[ppc_idx].ppc_command) != '?')
   {
    len = strlen(CURRENT_VIEW->ppc[ppc_idx].ppc_command);
    meminschr(CURRENT_VIEW->ppc[ppc_idx].ppc_command,'?',0,PREFIX_WIDTH,len);
    *(CURRENT_VIEW->ppc[ppc_idx].ppc_command+len+1) = '\0';
   }
/*---------------------------------------------------------------------*/
/* ensure that there is no chance that a 'real' prefix command can be  */
/* executed.                                                           */
/*---------------------------------------------------------------------*/
 CURRENT_VIEW->ppc[ppc_idx].ppc_cmd_idx = (-1);

 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
void clear_pending_prefix_command(int ppc_idx,LINE *curr)
#else
void clear_pending_prefix_command(ppc_idx,curr)
int ppc_idx;
LINE *curr;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
/*---------------------------------------------------------------------*/
/* If the ppc_idx value is (-1), then do nothing.                      */
/*---------------------------------------------------------------------*/
 if (ppc_idx == (-1))
    return;
 strcpy(CURRENT_VIEW->ppc[ppc_idx].ppc_command,"");
 if (curr == (LINE *)NULL)
#ifdef USE_VOID
    curr = (LINE *)ll_find((void *)CURRENT_FILE->first_line,CURRENT_VIEW->ppc[ppc_idx].ppc_line_number);
#else
    curr = lll_find(CURRENT_FILE->first_line,CURRENT_VIEW->ppc[ppc_idx].ppc_line_number);
#endif
 curr->pre = (-1);
 CURRENT_VIEW->ppc[ppc_idx].ppc_line_number = (-1L);
 CURRENT_VIEW->ppc[ppc_idx].ppc_cmd_idx = (-1);
 CURRENT_VIEW->ppc[ppc_idx].ppc_block_command = FALSE;
 return;
}
/***********************************************************************/
#ifdef PROTO
static int find_bottom_ppc(int top_ppc,int top_cmd_idx)
#else
static int find_bottom_ppc(top_ppc,top_cmd_idx)
int top_ppc,top_cmd_idx;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register int i;
/*--------------------------- processing ------------------------------*/
 for (i=top_ppc+1;i<CURRENT_VIEW->prefix_command_index;i++)
   {
    if (CURRENT_VIEW->ppc[i].ppc_cmd_idx == (-1))
       continue;
    if (top_cmd_idx == CURRENT_VIEW->ppc[i].ppc_cmd_idx)
       return(i);
   }
 return(PPC_NO_TARGET);
}
/***********************************************************************/
#ifdef PROTO
static int find_target_ppc(int *target_idx)
#else
static int find_target_ppc(target_idx)
int *target_idx;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register int i;
/*--------------------------- processing ------------------------------*/
 for (i=0;i<CURRENT_VIEW->prefix_command_index;i++)
   {
    if (CURRENT_VIEW->ppc[i].ppc_cmd_idx == PPC_TARGET_PREVIOUS
    ||  CURRENT_VIEW->ppc[i].ppc_cmd_idx == PPC_TARGET_FOLLOWING)
      {
       *target_idx = i;
       return(CURRENT_VIEW->ppc[i].ppc_cmd_idx);
/*     return(i);*/
      }
   }
 return(PPC_NO_TARGET);
}
/***********************************************************************/
#ifdef PROTO
static long calculate_target_line(void)
#else
static long calculate_target_line()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 int target_idx,target_type;
 long target_line;
/*--------------------------- processing ------------------------------*/
 if ((target_type = find_target_ppc(&target_idx)) == PPC_NO_TARGET)
    return(-1L);
 target_line = CURRENT_VIEW->ppc[target_idx].ppc_line_number;
/*---------------------------------------------------------------------*/
/* If the target line is NOT top of file line and the target type is   */
/* PREVIOUS, subtract 1 from the target line.                          */
/*---------------------------------------------------------------------*/
 if (target_type == PPC_TARGET_PREVIOUS
 && !TOF(target_line))
    target_line--;
/*---------------------------------------------------------------------*/
/* If the target line is the bottom of file and the target type is     */
/* FOLLOWING, sutract 1 from the target line.                          */
/*---------------------------------------------------------------------*/
 if (target_type == PPC_TARGET_FOLLOWING
 && BOF(target_line))
    target_line--;
 clear_pending_prefix_command(target_idx,(LINE *)NULL);
 return(target_line);
}
#if !defined(NOREXX)
/***********************************************************************/
#ifdef PROTO
static int try_rexx_prefix_macro(int ppc_idx)
#else
static int try_rexx_prefix_macro(ppc_idx)
int ppc_idx;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern char *temp_cmd;
/*--------------------------- local data ------------------------------*/
 int ctr, parm_length, pmacro_rc, errnum = 0 ;
 bool found_left, found_right;
 long line_number;
 char pm_parms[(PREFIX_WIDTH*4)+1], t_area[PREFIX_WIDTH+1],
      parms[PREFIX_WIDTH+1], tmpstr[PREFIX_WIDTH+1], 
      orig_cmd[PREFIX_WIDTH+1];
/*--------------------------- processing ------------------------------*/
 strcpy(orig_cmd, CURRENT_VIEW->ppc[ppc_idx].ppc_command);

 found_right = FALSE;
 found_left = FALSE;
/*---------------------------------------------------------------------*/
/* Scan 'orig_cmd' from right to left looking for an embedded macro    */
/* file name.                                                          */
/*---------------------------------------------------------------------*/
 for (ctr = strlen(orig_cmd); ctr >= 0 && !found_right; ctr--)
   {
    substr(&tmpstr[0], orig_cmd, 1, ctr);
    strcpy(t_area, tmpstr);
/*---------------------------------------------------------------------*/
/* First check if the prefix command is a synonym.                     */
/*---------------------------------------------------------------------*/
/* check for prefix synonym, if so use the macro name not the synonym */
    get_valid_macro_file_name(find_prefix_synonym(t_area),temp_cmd,&errnum);

    if (errnum == 0)
       found_right = TRUE;
   }

/*---------------------------------------------------------------------*/
/* No valid macro file name found ?                                    */
/*                                                                     */
/* Scan 'temp_cmd' from left to right looking for an embedded macro    */
/* file name.                                                          */
/*---------------------------------------------------------------------*/
 if (!found_right)
   for ( ctr = 0; ctr <= strlen(orig_cmd) && !found_left; ctr++)
     {
      substr(&tmpstr[0], orig_cmd, ctr + 1, strlen(temp_cmd) - ctr);
      strcpy(t_area, tmpstr);
/*---------------------------------------------------------------------*/
/* First check if the prefix command is a synonym.                     */
/*---------------------------------------------------------------------*/
/* check for prefix synonym, if so use the macro name not the synonym */
      get_valid_macro_file_name(find_prefix_synonym(t_area),temp_cmd,&errnum);

      if (errnum == 0)
         found_left = TRUE;
     }

 if (found_right || found_left)
   {
    parm_length = strlen(orig_cmd) - strlen(t_area);

    if (found_right)
       substr(&tmpstr[0],orig_cmd,strlen(t_area) + 1,parm_length);
    else
       substr(&tmpstr[0],orig_cmd,1,parm_length);

    strcpy(parms, tmpstr);
    line_number = CURRENT_VIEW->ppc[ppc_idx].ppc_line_number;
    sprintf(pm_parms," PREFIX %d %s",line_number,parms);
    strcat(temp_cmd, pm_parms);     /* add on the parameter list */
    prefix_current_line = line_number;
    in_prefix_macro = TRUE;
    pmacro_rc = execute_macro(temp_cmd,TRUE);
    in_prefix_macro = FALSE;
   }
 else
    pmacro_rc = RC_NOT_COMMAND;

 return(pmacro_rc);
}
#endif
/***********************************************************************/
#ifdef PROTO
static char *substr(char *temp_string,char *string,int start,int length)
#else
static char *substr(temp_string, string, start, length)
char *temp_string, *string;
int start, length;
#endif
/***********************************************************************/
/* This function is like the REXX SUBSTR function, it returns a        */
/* pointer to a string containing the result.                          */
/* Note: The address of the beginning of a blank string is passed to be*/
/* modified by address: substr(&tmpstr[0],<string>,<start>,<length>);  */
/* ( In this case, the result is contained in 'tmpstr'  :-)            */
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register i, target_idx = 0;
/*--------------------------- processing ------------------------------*/
 for (i = start - 1; target_idx <= length - 1; i++)
    temp_string[target_idx++] = string[i];
 temp_string[target_idx] = '\0';   /*MH*/
 return(temp_string);
}
/***********************************************************************/
#ifdef PROTO
void add_prefix_command(LINE *curr,long line_number,bool block_command)
#else
void add_prefix_command(curr,line_number,block_command)
LINE *curr;
long line_number;
bool block_command;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
extern bool prefix_changed;
extern char *pre_rec;
extern unsigned short pre_rec_len;
/*--------------------------- local data ------------------------------*/
 register int i;
 int prefix_index;
 char temp_prefix_array[PREFIX_WIDTH+1];
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("prefix.c:  add_prefix_command");
#endif
 if (CURRENT_VIEW->prefix_command_index >= MAX_PENDING_PREFIX_COMMANDS-1)
   {
    display_error(1,(char *)"too many pending prefix commands");
#ifdef TRACE
    trace_return();
#endif
    return;
   }
 prefix_changed = FALSE;
/*---------------------------------------------------------------------*/
/* Add prefix command to pending prefix command array...               */
/*---------------------------------------------------------------------*/
 for (i=0;i<pre_rec_len;i++)
     temp_prefix_array[i] = pre_rec[i];
 temp_prefix_array[pre_rec_len] = '\0';
 strtrunc(temp_prefix_array);
/*---------------------------------------------------------------------*/
/* First check to see if the prefix command is blank. If so, then      */
/* remove the reference to the line number and set curr->pre = (-1).   */
/*---------------------------------------------------------------------*/
 if (blank_field(temp_prefix_array))
    clear_pending_prefix_command(curr->pre,curr);
 else
   {
/*---------------------------------------------------------------------*/
/* If the input line already points to an entry in the array, use the  */
/* existing entry in the array, otherwise add to the next entry.       */
/*---------------------------------------------------------------------*/
    prefix_index = (-1);
    for (i=0;i<CURRENT_VIEW->prefix_command_index;i++)
      {
       if (line_number == CURRENT_VIEW->ppc[i].ppc_line_number)
         prefix_index = i;
      }
    if (prefix_index == (-1))                   /* new entry required */
      {
       strcpy(CURRENT_VIEW->ppc[CURRENT_VIEW->prefix_command_index].ppc_command,temp_prefix_array);
       curr->pre = CURRENT_VIEW->prefix_command_index++;
      }
    else
      {
       strcpy(CURRENT_VIEW->ppc[prefix_index].ppc_command,temp_prefix_array);
       curr->pre = prefix_index;
      }
    CURRENT_VIEW->ppc[curr->pre].ppc_line_number = line_number;
    CURRENT_VIEW->ppc[curr->pre].ppc_block_command = block_command;
   }
/*---------------------------------------------------------------------*/
/* Clear the pending prefix command line.                              */
/*---------------------------------------------------------------------*/
 memset(pre_rec,' ',PREFIX_WIDTH);
 pre_rec_len = 0;
 pre_rec[PREFIX_WIDTH] = '\0';
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef PROTO
int add_prefix_synonym(char *synonym,char *macroname)
#else
int add_prefix_synonym(synonym,macroname)
char *synonym,*macroname;
#endif
/***********************************************************************/
/* Parameters:                                                         */
/*    synonym: synonym for prefix macro                                */
/*  macroname: name of REXX macro file                                 */
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int j;
 int rc;
 LINE *curr,*next;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("prefix.c:  add_prefix_synonym");
#endif
/*---------------------------------------------------------------------*/
/* First thing is to delete any definitions that may exist for the     */
/* supplied synonym.                                                   */
/*---------------------------------------------------------------------*/
 curr = first_prefix_synonym;
 while(curr != NULL)
   {
    if (strcmp(curr->name,synonym) == 0)
      {
       free(curr->name);
       free(curr->line);
       curr = lll_del(&first_prefix_synonym,&last_prefix_synonym,curr,DIRECTION_FORWARD);
      }
    else
       curr = curr->next;
   }
/*---------------------------------------------------------------------*/
/* Lastly append the synonym at the end of the linked list.            */
/*---------------------------------------------------------------------*/
 curr = lll_add(first_prefix_synonym,last_prefix_synonym,sizeof(LINE));
 if (curr == NULL)
   {
    display_error(30,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 curr->line = (char *)malloc((strlen(macroname)+1)*sizeof(char));
 if (curr->line == NULL)
   {
    display_error(30,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 strcpy(curr->line,macroname);
 curr->name = (char *)malloc((strlen(synonym)+1)*sizeof(char));
 if (curr->name == NULL)
   {
    display_error(30,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_OUT_OF_MEMORY);
   }
 strcpy(curr->name,synonym);
 last_prefix_synonym = curr;
 if (first_prefix_synonym == NULL)
    first_prefix_synonym = last_prefix_synonym;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man***************************************************************************
NAME
     find_prefix_synonym

SYNOPSIS
     char *find_prefix_synonym(synonym)
     char *synonym;

DESCRIPTION
     The find_prefix_synonym function finds a synonym for 'synonym'
     and returns that value. If no synonym exists, the 'synonym' is
     returned unchanged.

     This function is only available if REXX support is available.
     
RETURN VALUE
     Either the macroname associated with 'synonym' or 'synonym'.
*******************************************************************************/
#ifdef PROTO
char *find_prefix_synonym(char *synonym)
#else
char *find_prefix_synonym(synonym)
char *synonym;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 LINE *curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("prefix.c:  find_prefix_synonym");
#endif
 curr = first_prefix_synonym;
 while(curr != NULL)
   {
    if (strcmp(synonym,curr->name) == 0)
      {
#ifdef TRACE
       trace_return();
#endif
       return(curr->line);
      }
    curr = curr->next;
   }
#ifdef TRACE
 trace_return();
#endif
 return(synonym);
}
/*man***************************************************************************
NAME
     find_prefix_oldname

SYNOPSIS
     char *find_prefix_oldname(oldname)
     char *oldname;

DESCRIPTION
     The find_prefix_oldname function finds the synonym for
     'oldname' and returns that value. If no synonym exists, the 
     'oldname' is returned unchanged.

     This function is only available if REXX support is available.
     
RETURN VALUE
     Either the synonym associated with 'oldname' or 'oldname'.
*******************************************************************************/
#ifdef PROTO
char *find_prefix_oldname(char *oldname)
#else
char *find_prefix_oldname(oldname)
char *oldname;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 LINE *curr;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("prefix.c:  find_prefix_oldname");
#endif
 curr = first_prefix_synonym;
 while(curr != NULL)
   {
    if (strcmp(oldname,curr->line) == 0)
      {
#ifdef TRACE
       trace_return();
#endif
       return(curr->name);
      }
    curr = curr->next;
   }
#ifdef TRACE
 trace_return();
#endif
 return(oldname);
}
