/***********************************************************************/
/* COMMSET1.C - SET commands A-N                                       */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1993 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Header: C:\THE\RCS\commset1.c 1.4 1993/09/01 16:25:50 MH Interim MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"

/*#define DEBUG 1*/
/*man-start*********************************************************************



========================================================================
SET COMMAND REFERENCE
========================================================================
**man-end**********************************************************************/

/*-------------------------- external data ----------------------------*/
extern VIEW_DETAILS *vd_current;
extern char in_profile;    /* indicates if processing profile */
extern char mode_insert;        /* defines insert mode toggle */
extern char current_screen;
extern SCREEN_DETAILS screen[MAX_SCREENS];        /* screen structures */
extern WINDOW *foot,*error_window,*divider;
/*man-start*********************************************************************
COMMAND
     arbchar - set arbitrary character for change commands

SYNTAX
     [SET] ARBchar ON|OFF [char]

DESCRIPTION
     Set the character to use as an 'arbitrary character' when using
     change commands. When on the character specified no longer matches
     itself in locating a character to change but matches all characters.

COMPATIBILITY
     XEDIT: See below
     KEDIT: See below
     Only allows for single character matching, not multiple strings.

DEFAULT
     OFF ?

SEE ALSO
     Change,Schange and Locate

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Arbchar(char *params)
#else
int Arbchar(params)
char *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
/*--------------------------- local data ------------------------------*/
#define ARB_PARAMS  3
 char *word[ARB_PARAMS+1];
 unsigned short num_params;
 int rc = RC_INVALID_OPERAND;
 char arbsts=CURRENT_VIEW->arbchar_status;
 char arbchr=CURRENT_VIEW->arbchar_char;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Arbchar");
#endif
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,ARB_PARAMS,WORD_DELIMS,TEMP_PARAM);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* No parameters, error.                                               */
/*---------------------------------------------------------------------*/
    case 0: 
         display_error(3,(char *)""); 
         break;
/*---------------------------------------------------------------------*/
/* 1 or 2 parameters, validate them...                                 */
/*---------------------------------------------------------------------*/
    case 1: 
    case 2: 
         if (equal((char *)"on",word[0],2))
           {
            arbsts = ON;
            rc = RC_OK;
           }
         if (equal((char *)"off",word[0],3))
           {
            arbsts = OFF;
            rc = RC_OK;
           }
/*---------------------------------------------------------------------*/
/* If not a valid first parameter, display an error and exit.          */
/*---------------------------------------------------------------------*/
         if (rc != RC_OK)
           {
            display_error(1,word[0]);
            break;
           }
/*---------------------------------------------------------------------*/
/* For 1 parameter, don't check any more.                              */
/*---------------------------------------------------------------------*/
         if (num_params == 1)
            break;
         rc = RC_INVALID_OPERAND;
/*---------------------------------------------------------------------*/
/* For 2 parameters, check that a single character has been supplied...*/
/*---------------------------------------------------------------------*/
         if (strlen(word[1]) != 1)
           {
            display_error(1,word[1]);
            break;
           }
         arbchr = word[1][0];
         rc = RC_OK;
         break;
/*---------------------------------------------------------------------*/
/* Too many parameters...                                              */
/*---------------------------------------------------------------------*/
    default:
         display_error(2,(char *)""); 
         break;
   }
/*---------------------------------------------------------------------*/
/* If valid parameters, change the settings...                         */
/*---------------------------------------------------------------------*/
 if (rc == RC_OK)
   {
    CURRENT_VIEW->arbchar_char = arbchr;
    CURRENT_VIEW->arbchar_status = arbsts;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     autosave - set autosave period

SYNTAX
     [SET] AUtosave n|OFF

DESCRIPTION
     The AUTOSAVE command sets the interval between automatic saves
     of the file, or turns it off altogether.

COMPATIBILITY
     XEDIT: Does not support [mode] option.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Autosave(char *params)
#else
int Autosave(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern char AUSx;
/*--------------------------- local data ------------------------------*/
#define AUS_PARAMS  1
 char *word[AUS_PARAMS+1];
 register int i;
 unsigned short num_params;
 long num_pages,num_lines;
 unsigned short x,y;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Autosave");
#endif
 num_params = param_split(params,word,AUS_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params == 0)
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params != 1)
   {
    display_error(2,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (equal((char *)"off",word[0],3))
   {
    CURRENT_FILE->autosave = 0;
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
 if (!valid_positive_integer(word[0]))
   {
    display_error(4,(char *)word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 CURRENT_FILE->autosave = AUSx =  (char)atoi(word[0]);
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     backup - indicate if a backup copy of the file is to be kept

SYNTAX
     [SET] BACKup ON|OFF

DESCRIPTION
     The BACKUP command allows the user to determine if a backup copy
     of the original file is to be kept when the file being edited is
     saved or filed.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Does not support TEMP or KEEP options.

DEFAULT
     ON - a backup file is kept

SEE ALSO
     file,ffile,save,ssave

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Backup(char *params)
#else
int Backup(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool BACKUPx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Backup");
#endif
 if (equal((char *)"off",params,3))
    BACKUPx = FALSE;
 else
    if (equal((char *)"on",params,2))
       BACKUPx = TRUE;
    else
      {
       display_error(1,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
 CURRENT_FILE->backup = BACKUPx;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     beep - turn on or off the audible alarm when displaying errors

SYNTAX
     [SET] BEEP ON|OFF

DESCRIPTION
     The BEEP command allows the user to determine if an audible alarm
     is sounded when an error is displayed.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     OFF

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int BeepSound(char *params)
#else
int BeepSound(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool BEEPx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:BeepSound");
#endif
 if (equal((char *)"off",params,3))
    BEEPx = FALSE;
 else
    if (equal((char *)"on",params,2))
       BEEPx = TRUE;
    else
      {
       display_error(1,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     case - set case sensitivity parameters

SYNTAX
     [SET] CASE Mixed|Lower|Upper [Respect|Ignore] [Respect|Ignore]
                                  [Respect|Ignore]

DESCRIPTION
     The CASE command sets the editor's handling of the case of text.

     The first option (which is mandatory) controls how text is entered
     by the user. When LOWER or UPPER are in effect, the shift or caps
     lock keys have no effect on the text being entered. When MIXED is
     in effect, text is entered in the case set by the use of the shift
     and caps lock keys.

     The second option determines how the editor determines if a string
     target matches text in the file when the target is used in a LOCATE
     command.  With IGNORE in effect, a match is
     found irrespective of the case of the target or the found text.
     The following strings are treated as equivalent: the THE The ThE...
     With RESPECT in effect, the target and text must be the same case.
     Therefore a target of 'The' only matches text containing 'The', not
     'THE' or 'ThE' etc.

     The third option determines how the editor determines if a string
     target matches text in the file when the target is used in a CHANGE
     command.  With IGNORE in effect, a match is
     found irrespective of the case of the target or the found text.
     The following strings are treated as equivalent: the THE The ThE...
     With RESPECT in effect, the target and text must be the same case.
     Therefore a target of 'The' only matches text containing 'The', not
     'THE' or 'ThE' etc.

     The fourth option determines how the editor determines the sort 
     order of upper and lower case with the SORT command.
     With IGNORE in effect, upper and lower case letters are treated as
     equivalent.
     With RESPECT in effect, upper and lower case letters are treated as
     different values and uppercase characters will sort before lowercase
     characters.

COMPATIBILITY
     XEDIT: Adds support for case significance in CHANGE commands.
     KEDIT: Adds support for LOWER option.
     Both:  Adds support for case significance in SORT command.

DEFAULT
     MIXED IGNORE RESPECT RESPECT

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Case(char *params)
#else
int Case(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
/*--------------------------- local data ------------------------------*/
#define CAS_PARAMS  4
 char parm[CAS_PARAMS];
 char *word[CAS_PARAMS+1];
 register int i;
 unsigned short num_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Case");
#endif
 num_params = param_split(params,word,CAS_PARAMS,WORD_DELIMS,TEMP_PARAM);
/*---------------------------------------------------------------------*/
/* Validate the first parameter: must be Mixed, Upper or Lower         */
/*---------------------------------------------------------------------*/
 parm[0] = (char)UNDEFINED_OPERAND;
 if (equal((char *)"mixed",word[0],1))
    parm[0] = CASE_MIXED;
 if (equal((char *)"upper",word[0],1))
    parm[0] = CASE_UPPER;
 if (equal((char *)"lower",word[0],1))
    parm[0] = CASE_LOWER;
 if (parm[0] == (char)UNDEFINED_OPERAND)
    {
     display_error(1,(char *)word[0]);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Save the current values of each remaining case setting.             */
/*---------------------------------------------------------------------*/
 parm[1] = CURRENT_VIEW->case_locate;
 parm[2] = CURRENT_VIEW->case_change;
 parm[3] = CURRENT_VIEW->case_sort;
/*---------------------------------------------------------------------*/
/* Validate the remainder of the arguments.                            */
/* Each must be Respect or Ignore, if present.                         */
/*---------------------------------------------------------------------*/
 for (i=1;i<num_params;i++)
   {
    if (strcmp(word[1],"") != 0)
      {
       if (equal((char *)"respect",word[i],1))
          parm[i] = CASE_RESPECT;
       else
          if (equal((char *)"ignore",word[i],1))
             parm[i] = CASE_IGNORE;
          else
            {
             display_error(1,(char *)word[i]);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
      }
   }
/*---------------------------------------------------------------------*/
/* Set the new values of case settings for the view.                   */
/*---------------------------------------------------------------------*/
 CURRENT_VIEW->case_enter  = parm[0];
 CURRENT_VIEW->case_locate = parm[1];
 CURRENT_VIEW->case_change = parm[2];
 CURRENT_VIEW->case_sort   = parm[3];

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     clearscreen - indicate if the screen is to be cleared on exit

SYNTAX
     [SET] CLEARScreen ON|OFF

DESCRIPTION
     The CLEARSCREEN command allows the user to request that the screen be
     cleared on exit from THE.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     OFF - a clear screen does not occur.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Clearscreen(char *params)
#else
int Clearscreen(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 extern bool CLEARSCREENx;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Clearscreen");
#endif
 if (equal((char *)"off",params,3))
    CLEARSCREENx = FALSE;
 else
    if (equal((char *)"on",params,2))
       CLEARSCREENx = TRUE;
    else
      {
       display_error(1,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     clock - turn on or display of time on status line

SYNTAX
     [SET] CLOCK ON|OFF

DESCRIPTION
     The CLOCK command turns on or off the display of the time on the
     status line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     ON - the time is displayed.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Clock(char *params)
#else
int Clock(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 extern bool CLOCKx;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Clock");
#endif
 if (equal((char *)"off",params,3))
    CLOCKx = FALSE;
 else
    if (equal((char *)"on",params,2))
       CLOCKx = TRUE;
    else
      {
       display_error(1,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
 clear_footing();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     cmdarrows - sets the functionality of the arrows keys

SYNTAX
     [SET] CMDArrows Retrieve|Tab Scroll|Tab Scroll|Tab

DESCRIPTION
     The CMDARROWS command determines the action that occurs when the
     arrows keys are hit.

     CMDARROWS RETRIEVE SCROLL SCROLL (the default) will set the up and 
     down arrows to retrieve the last or next command entered on the command 
     line and to scroll the screen when at the top or bottom of the main
     text window. The left and right arrows will cause the main text window 
     to scroll if appropriate.

     CMDARROWS TAB TAB TAB will set the up and down arrows to move to the
     last or first line respectively of the main window and to return to 
     the command line when the top or bottom of the main text window is 
     reached. The left and right arrows will wrap around the window if on
     the command line or no prefix is on or to move between the prefix area.
     This behaviour is more like XEDIT.

DEFAULT
     RETRIEVE SCROLL SCROLL

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Cmdarrows(char *params)
#else
int Cmdarrows(params)
char *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
extern char CMDARROWSTABCMDx;
extern char CMDARROWSTABTXTx;
extern char CMDARROWSTABLRx;
/*--------------------------- local data ------------------------------*/
#define CMD_PARAMS  3
 char *word[CMD_PARAMS+1];
 char parm[CMD_PARAMS];
 int num_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Cmdarrows");
#endif
 parm[0] = (char)UNDEFINED_OPERAND;
 parm[1] = (char)UNDEFINED_OPERAND;
/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied.                    */
/* 2 Parameters are mandatory.                                         */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,CMD_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params < 3)
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 3)
   {
    display_error(2,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Determine values for first parameter; command line behaviour        */
/*---------------------------------------------------------------------*/
 if (equal((char *)"tab",word[0],1))
   {
    CMDARROWSTABCMDx = TRUE;
    parm[0] = TRUE;
   }
 if (equal((char *)"retrieve",word[0],1))
   {
    CMDARROWSTABCMDx = FALSE;
    parm[0] = TRUE;
   }
 if (parm[0] == (char)UNDEFINED_OPERAND)
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Determine values for second parameter; main text window behaviour   */
/*---------------------------------------------------------------------*/
 if (equal((char *)"tab",word[1],1))
   {
    CMDARROWSTABTXTx = TRUE;
    parm[1] = TRUE;
   }
 if (equal((char *)"scroll",word[1],1))
   {
    CMDARROWSTABTXTx = FALSE;
    parm[1] = TRUE;
   }
 if (parm[1] == (char)UNDEFINED_OPERAND)
   {
    display_error(1,word[1]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Determine values for third parameter; left/right arrow keys         */
/*---------------------------------------------------------------------*/
 if (equal((char *)"tab",word[2],1))
   {
    CMDARROWSTABLRx = TRUE;
    parm[2] = TRUE;
   }
 if (equal((char *)"scroll",word[2],1))
   {
    CMDARROWSTABLRx = FALSE;
    parm[2] = TRUE;
   }
 if (parm[2] == (char)UNDEFINED_OPERAND)
   {
    display_error(1,word[2]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     cmdline - sets the position of the command line.

SYNTAX
     [SET] CMDline Top|Bottom

DESCRIPTION
     The CMDLINE command sets the position of the command line, either
     at the top of the screen or at the bottom of the screen.

COMPATIBILITY
     XEDIT: Does not allow CMDLINE to be turned OFF.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Cmdline(char *params)
#else
int Cmdline(params)
char *params;
#endif
/***********************************************************************/
{
/*------------------------- external data -----------------------------*/
extern char CMD_LINEx;
/*--------------------------- local data ------------------------------*/
 char cmd_place='?';
 int off;
 unsigned short y,x;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Cmdline");
#endif
 if (equal((char *)"top",params,1))
   {
    cmd_place='T';
    off = 1;
   }
 if (equal((char *)"bottom",params,1))
   {
    cmd_place='B';
    off = (-1);
   }
 if (cmd_place=='?')
   {
    display_error(1,(char *)params);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If the setting supplied is the same as the current setting, just    */
/* return without doing anything.                                      */
/*---------------------------------------------------------------------*/
 if (cmd_place == CURRENT_VIEW->cmd_line)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Now we need to move the windows around.                             */
/*---------------------------------------------------------------------*/
 CURRENT_VIEW->cmd_line = cmd_place;
 if (cmd_place == 'T')
    CURRENT_VIEW->command_row = CURRENT_SCREEN.origin_y+1;
 else
    CURRENT_VIEW->command_row = CURRENT_SCREEN.origin_y+CURRENT_SCREEN.screen_rows-1;
/*---------------------------------------------------------------------*/
/* If we are in the profile, exit here.                                */
/*---------------------------------------------------------------------*/
 if (in_profile)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }

 getbegyx(CURRENT_WINDOW_MAIN,y,x);
 mvwin(CURRENT_WINDOW_MAIN,y+off,x);
 if (CURRENT_VIEW->prefix)
   {
    getbegyx(CURRENT_WINDOW_PREFIX,y,x);
    mvwin(CURRENT_WINDOW_PREFIX,y+off,x);
   }
 getbegyx(CURRENT_WINDOW_COMMAND,y,x);
 mvwin(CURRENT_WINDOW_COMMAND,CURRENT_VIEW->command_row,x);
 getbegyx(CURRENT_WINDOW_ARROW,y,x);
 mvwin(CURRENT_WINDOW_ARROW,CURRENT_VIEW->command_row,x);
 touchwin(CURRENT_WINDOW_ARROW);
 wnoutrefresh(CURRENT_WINDOW_ARROW);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     colour - set colours for display

SYNTAX
     [SET] COLOUR|COLOR area [modifier] [foreground background]

DESCRIPTION
     The COLOUR command changes the colours or display attributes of
     various display areas in THE.

     Valid values for 'area':
       filearea - area containing file lines
       curline  - the current line
       block    - marked block
       cblock   - current line if in marked block
       cmdline  - command line
       idline   - line containing file specific info
       msgline  - error messages
       arrow    - command line prompt
       prefix   - prefix area
       pending  - pending commands in prefix
       scale    - line showing scale line (N/A)
       tofeof   - *** Top of File *** and *** Bottom of File *** lines
       ctofeof  - as above if the same as current line
       tabline  - line showing tab positions (N/A)
       shadow   - hidden line marker lines (N/A)
       statarea - line showing status of editing session
       divider  - dividing line between vertical split screens (N/A)
       reserved - any reserved lines

     Valid values for 'foreground' and 'background':
       black,blue,green,cyan,red,magenta,yellow,white

     Valid values for 'modifier':
       normal,blink,bold,bright,high,reverse,underline

     It is an error to attempt to set a colour on a mono display.

COMPATIBILITY
     XEDIT: Functionally compatible. See below.
     KEDIT: Functionally compatible. See below.
     Does not implement all modifiers.

STATUS  
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Colour(char *params)
#else
int Colour(params)
char *params;
#endif

/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 struct areas
 {
  char *area;
  short area_min_len;
  short area_window;
 };
 typedef struct areas AREAS;
 static AREAS valid_areas[ATTR_MAX]=
 {
  {(char *)"filearea",1,WINDOW_MAIN},
  {(char *)"curline",2,WINDOW_MAIN},
  {(char *)"block",1,WINDOW_MAIN},
  {(char *)"cblock",2,WINDOW_MAIN},
  {(char *)"cmdline",1,WINDOW_COMMAND},
  {(char *)"idline",1,WINDOW_IDLINE},
  {(char *)"msgline",1,WINDOW_ERROR},
  {(char *)"arrow",1,WINDOW_ARROW},
  {(char *)"prefix",2,WINDOW_PREFIX},
  {(char *)"pending",1,WINDOW_PREFIX},
  {(char *)"scale",1,WINDOW_MAIN},
  {(char *)"tofeof",2,WINDOW_MAIN},
  {(char *)"ctofeof",2,WINDOW_MAIN},
  {(char *)"tabline",1,WINDOW_MAIN},
  {(char *)"shadow",2,WINDOW_MAIN},
  {(char *)"statarea",2,WINDOW_FOOTING},
  {(char *)"divider",1,WINDOW_DIVIDER},
  {(char *)"reserved",1,WINDOW_RESERVED}
 };
#define COL_PARAMS 2
 char *word[COL_PARAMS+1];
 char parm[COL_PARAMS];
 register int i;
 unsigned short num_params;
 int area;
 chtype fg,bg,mod;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Colour");
#endif
 num_params = param_split(params,word,COL_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params < 2 )
    {
     display_error(3,(char *)"");
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Check that the supplied area matches one of the values in the area  */
/* array and that the length is at least as long as the minimum.       */
/*---------------------------------------------------------------------*/
 parm[0] = FALSE;
 for (i=0;i<ATTR_MAX;i++)
    {
     if (equal(valid_areas[i].area,word[0],valid_areas[i].area_min_len))
       {
        parm[0] = TRUE;
        area = i;
        break;
       }
    }
 if (parm[0] == FALSE)
    {
     display_error(1,(char *)word[0]);
#ifdef TRACE
     trace_return();
#endif
     return(RC_INVALID_OPERAND);
    }
/*---------------------------------------------------------------------*/
/* Determine colours and modifiers.                                    */
/*---------------------------------------------------------------------*/

 if (parse_colours(word[1],&fg,&bg,&mod) != RC_OK)
   {
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* If we are still in the profile, save the colour setup for later...  */
/*---------------------------------------------------------------------*/

 if (in_profile)
   {
    colour[area] = (chtype)1;
    save_fg[area] = fg;
    save_bg[area] = bg;
    save_mod[area] = mod;
#ifdef TRACE
    trace_return();
#endif
    return(RC_OK);
   }
/*---------------------------------------------------------------------*/
/* Set the colours.                                                    */
/*---------------------------------------------------------------------*/

 set_colour(area,fg,bg,mod);

/*---------------------------------------------------------------------*/
/* Update the appropriate window with the new colour combination.      */
/*---------------------------------------------------------------------*/

 switch (valid_areas[area].area_window)
   {
    case WINDOW_MAIN:
                        if (area == ATTR_FILEAREA)
                           wattrset(CURRENT_WINDOW_MAIN,colour[area]);
                        redraw_window(CURRENT_WINDOW_MAIN);
                        show_page();
                        touchwin(CURRENT_WINDOW_MAIN);
                        wnoutrefresh(CURRENT_WINDOW_MAIN);
                        break;
    case WINDOW_PREFIX:
                        if (CURRENT_VIEW->prefix)
                          {
                           wattrset(CURRENT_WINDOW_PREFIX,colour[area]);
                           redraw_window(CURRENT_WINDOW_PREFIX);
                           touchwin(CURRENT_WINDOW_PREFIX);
                           wnoutrefresh(CURRENT_WINDOW_PREFIX);
                          }
                        break;
    case WINDOW_COMMAND:
                        wattrset(CURRENT_WINDOW_COMMAND,colour[area]);
                        redraw_window(CURRENT_WINDOW_COMMAND);
                        touchwin(CURRENT_WINDOW_COMMAND);
                        wnoutrefresh(CURRENT_WINDOW_COMMAND);
                        break;
    case WINDOW_ARROW:
                        wattrset(CURRENT_WINDOW_ARROW,colour[area]);
                        redraw_window(CURRENT_WINDOW_ARROW);
                        touchwin(CURRENT_WINDOW_ARROW);
                        wnoutrefresh(CURRENT_WINDOW_ARROW);
                        break;
    case WINDOW_IDLINE:
                        wattrset(CURRENT_WINDOW_IDLINE,colour[area]);
                        redraw_window(CURRENT_WINDOW_IDLINE);
                        touchwin(CURRENT_WINDOW_IDLINE);
                        wnoutrefresh(CURRENT_WINDOW_IDLINE);
                        break;
    case WINDOW_FOOTING:
                        wattrset(foot,colour[area]);
                        redraw_window(foot);
                        touchwin(foot);
                        wnoutrefresh(foot);
                        break;
    case WINDOW_ERROR:
                        wattrset(error_window,colour[area]);
                        break;
    case WINDOW_DIVIDER:
                        wattrset(divider,colour[area]);
                        break;
    case WINDOW_RESERVED:
                        wattrset(stdscr,colour[area]);
                        break;
    default:
                        break;
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     curline - set position of current line on screen

SYNTAX
     [SET] CURLine M[+n|-n] | [+|-]n

DESCRIPTION
     The CURLINE command sets the position of the current line to
     the physical screen line specified by supplied arguments.

     The two forms of parameters are:
     M[+n|-n] - this sets the current line to be relative to the
                middle of the screen. A positvie value adds to the
                middle line number, a negative subtracts from it.
                eg. M+3 on a 24 line screen will be line 15
                    M-5 on a 24 line screen will be line 7

     [+|-]n   - this sets the current line to be relative to the
                top of the screen (if positive or no sign) or
                relative to the bottom of the screen if negative.
                eg. +3 or 3 will set current line to line 3
                    -3 on a 24 line screen will be line 21

     If the resulting line is outside the bounds of the screen
     the position of the current line will become the middle line
     on the screen.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Curline(char *params)
#else
int Curline(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define CUR_PARAMS  1
 char *word[CUR_PARAMS+1];
 int num_params;
 int rc;
 int base,off;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Curline");
#endif
 num_params = param_split(params,word,CUR_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params < 1)
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 1)
   {
    display_error(2,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Parse the parameter...                                              */
/*---------------------------------------------------------------------*/
 if (*params == 'M'
 ||  *params == 'm')
   {
    base = CURLINE_MIDDLE;
    params++;
    if (blank_field(params))
       off = 0;
    else
      {
       if ((*params != '-'
       &&  *params != '+')
       || ((off = atoi(params)) == 0))
         {
          display_error(1,params);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
      }
   }
 else
   {
    if ((off = atoi(params)) == 0)
      {
       display_error(1,params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    base = (off > 0) ? CURLINE_TOP : CURLINE_BOTTOM;
   }
 CURRENT_VIEW->current_base = (char)base;
 CURRENT_VIEW->current_off = (char)off;
 if (!in_profile)
   {
    post_process_line(CURRENT_VIEW->focus_line);
    CURRENT_VIEW->current_row = calculate_current_row(CURRENT_VIEW->current_base,
                                                      CURRENT_VIEW->current_off,
                                                      CURRENT_SCREEN.rows);
    show_page();
   }
 
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     dirinclude - set the file mask for directory command

SYNTAX
     [SET] DIRInclude * 
     [SET] DIRInclude [Normal] [Readonly] [System] [Hidden] [Directory]

DESCRIPTION
     The DIRINCLUDE command sets the file mask for files that will be
     displayed on subsequent DIRECTORY commands.
     * will set the mask to all files, the other options will set the
     mask to include those options specified together with "normal"
     files eg.

      DIRINCLUDE R S

     will display readonly and system files together with "normal" files
     the next time the DIRECTORY command is issued.

     The effects of DIRINCLUDE are ignored in the Unix version.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

SEE ALSO
     directory

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Dirinclude(char *params)
#else
int Dirinclude(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 int rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Dirinclude");
#endif
 rc = set_dirtype(params);
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     display - specify which level of lines to display

SYNTAX
     [SET] DISPlay n [m|*]

DESCRIPTION
     The DISPLAY command sets the selection level for lines to be
     displayed on the screen.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

SEE ALSO
     SET SCOPE, SET SELECT, ALL

STATUS
     Not started.
**man-end**********************************************************************/
#ifdef PROTO
int Display(char *params)
#else
int Display(params)
char *params;
#endif
/***********************************************************************/
{
 extern char file_read; /* indicates if we have read the file */
 extern char DISPLAY_LOWx;
 extern char DISPLAY_HIGHx;
/*--------------------------- local data ------------------------------*/
 int rc=RC_OK;
#define DIS_PARAMS  2
 char *word[DIS_PARAMS+1];
 unsigned short num_params;
 long col1,col2;
 unsigned short x,y;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Display");
#endif
 display_error(0,(char *)"This function has not yet been implemented");

#ifdef TO_BE_IMPLEMENTED

/*---------------------------------------------------------------------*/
/* Validate the parameters that have been supplied. One only           */
/* parameter MUST be supplied. The first parameter MUST be a positive  */
/* integer. The second can be a positive integer or '*'. If no second  */
/* parameter is supplied, defaults to p1. The second parameter MUST be */
/* >= first parameter. '*' is regarded as the biggest number and is    */
/* literally 255.                                                      */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,DIS_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params < 1)
   {
    display_error(3,"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 2)
   {
    display_error(2,"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (!valid_positive_integer(word[0]))
   {
    display_error(4,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 col1 = atoi(word[0]);
 if (strcmp(word[1],"*") == 0)
    col2 = 255;
 else
    if (num_params == 1)      /* no second parameter, default to first */
       col2 = col1;
    else
       if (!valid_positive_integer(word[1]))
         {
          display_error(4,word[1]);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
       else
          col2 = atoi(word[1]);

 if (col2 > 255)
    col2 = 255;
 if (col1 > col2)
   {
    display_error(6,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (file_read)
   {
    CURRENT_VIEW->display_low = col1;
    CURRENT_VIEW->display_high   = col2;
    getyx(CURRENT_WINDOW,y,x);
    show_page();
    if (CURRENT_VIEW->current_window != WINDOW_COMMAND)
       wmove(CURRENT_WINDOW,y,x);
   }
 else
   {
    DISPLAY_LOWx = col1;
    DISPLAY_HIGHx = col2;
   }

#endif

#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     eolout - set end of line terminator

SYNTAX
     [SET] EOLout CRLF|LF

DESCRIPTION
     The EOLOUT command allows the user to specify the combination of
     characters that terminate a line. Lines of text in U*ix files are
     usually terminated with a LF, whereas in DOS they usually end with
     a CR and LF combination.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Eolout(char *params)
#else
int Eolout(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern char EOLx;
/*--------------------------- local data ------------------------------*/
#define EOLOUT_PARAMS  1
 char *word[EOLOUT_PARAMS+1];
 unsigned short num_params;
 char eolchar;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Eolout");
#endif
 num_params = param_split(params,word,EOLOUT_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (equal((char *)"lf",word[0],2))
    eolchar = EOLOUT_LF;
 else
    if (equal((char *)"crlf",word[0],4))
       eolchar = EOLOUT_CRLF;
    else
      {
       display_error(1,(char *)word[0]);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
 EOLx = CURRENT_FILE->eolout = eolchar;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     etmode - indicate if extended display mode is possible

SYNTAX
     [SET] ETMODE ON|OFF

DESCRIPTION
     The ETMODE command allows the user to specify if extended ASCII
     codes ( > 127) are to be displayed or should be displayed as
     the NONDISP character.

COMPATIBILITY
     XEDIT: Similar function but deals with Double-Byte characters
     KEDIT: N/A

DEFAULT
     ON  - for DOS and OS/2 platforms
     OFF - for other platforms

SEE ALSO
     [SET] NONDisp

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Etmode(char *params)
#else
int Etmode(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool extended_display_mode;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Etmode");
#endif
 if (equal((char *)"off",params,3))
    extended_display_mode = FALSE;
 else
    if (equal((char *)"on",params,2))
       extended_display_mode= TRUE;
    else
      {
       display_error(1,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
 if (!in_profile)
    show_page();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     hex - set how hexidecimal strings are treated in string operands

SYNTAX
     [SET] HEX ON|OFF

DESCRIPTION
     The HEX set command determines whether hexidecimal strings are
     treated as such in string operands.
     With HEX ON, any string operand of the form
        /x'31 32 33'/ or
        /d'49 50 51'/
     will be converted to /123/ before the command is executed.
     With HEX OFF, no conversion is done.

     This conversion should work wherever a string operand is used
     in any command.

COMPATIBILITY
     XEDIT: Adds support for decimal representation. See below.
     KEDIT: Compatible. See below.
     Spaces must seperate each character representation.

DEFAULT
     HEX OFF

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Hex(char *params)
#else
int Hex(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern char HEXx;
/*--------------------------- local data ------------------------------*/
#define HEX_PARAMS  1
 char parm[HEX_PARAMS];
 char *word[HEX_PARAMS+1];
 register int i;
 unsigned short num_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Hex");
#endif
 num_params = param_split(params,word,HEX_PARAMS,WORD_DELIMS,TEMP_PARAM);

 parm[0] = (char)UNDEFINED_OPERAND;
 if (equal((char *)"on",word[0],2))
    parm[0] = ON;
 if (equal((char *)"off",word[0],3))
    parm[0] = OFF;
 if (parm[0] == (char)UNDEFINED_OPERAND)
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 HEXx = CURRENT_VIEW->hex = parm[0];
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     hexdisplay - turn on or display of character under cursor

SYNTAX
     [SET] HEXDISPlay ON|OFF

DESCRIPTION
     The HEXDISPLAY command turns on or off the display of the character under
     the cursor on the status line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     ON - the current character is displayed.

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Hexdisplay(char *params)
#else
int Hexdisplay(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 extern bool HEXDISPLAYx;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Hexdisplay");
#endif
 if (equal((char *)"off",params,3))
    HEXDISPLAYx = FALSE;
 else
    if (equal((char *)"on",params,2))
       HEXDISPLAYx = TRUE;
    else
      {
       display_error(1,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
 clear_footing();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     impmacro - set implied macro command processing

SYNTAX
     [SET] IMPMACro ON|OFF

DESCRIPTION
     The IMPMACRO command is used to toggle implied macro processing
     from the command line. By turning this feature on you can then
     issue a macro command without the need to prefix the macro name
     with the MACRO command.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     ON

SEE ALSO
     Macro,Macropath

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Impmacro(char *params)
#else
int Impmacro(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool IMPMACROx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Impmacro");
#endif
 if (equal((char *)"off",params,3))
    IMPMACROx = FALSE;
 else
    if (equal((char *)"on",params,2))
       IMPMACROx = TRUE;
    else
      {
       display_error(1,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
 CURRENT_VIEW->imp_macro = IMPMACROx;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     impos - set implied operating system command processing

SYNTAX
     [SET] IMPOS ON|OFF
     [SET] IMPcmscp ON|OFF

DESCRIPTION
     The IMPMOS command is used to toggle implied operating system
     command processing from the command line. By turning this feature 
     on you can then issue an operating system command without the need 
     to prefix the operating system command with the OS command.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: N/A

DEFAULT
     ON

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Impos(char *params)
#else
int Impos(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool IMPOSx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Impos");
#endif
 if (equal((char *)"off",params,3))
    IMPOSx = FALSE;
 else
    if (equal((char *)"on",params,2))
       IMPOSx = TRUE;
    else
      {
       display_error(1,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
 CURRENT_VIEW->imp_os = IMPOSx;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     insertmode - put editor into insert mode

SYNTAX
     [SET] INSERTMode ON|OFF|TOGGLE

DESCRIPTION
     The INSERTMODE command toggles the insert mode within THE.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     TOGGLE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Insertmode(char *params)
#else
int Insertmode(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Insertmode");
#endif
 if (equal((char *)"off",params,3))
    mode_insert = NO;
 else
    if (equal((char *)"on",params,2))
       mode_insert = YES;
    else
       if (equal((char *)"toggle",params,6))
          mode_insert = (mode_insert) ? NO : YES;
       else
         {
          display_error(1,(char *)params);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
 if (!in_profile)
    draw_cursor(ON);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     linend - allow/disallow multiple commands on command line

SYNTAX
     [SET] LINENd ON|OFF [char]

DESCRIPTION
     The LINEND command allows or disallows the execution of multiple
     commands on the command line. When setting LINEND ON, a character
     is specified as the LINEND character which delimits each command.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     OFF #

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Linend(char *params)
#else
int Linend(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
#define LE_PARAMS  2
 char *word[LE_PARAMS+1];
 unsigned short num_params;
 bool le_status=CURRENT_VIEW->linend_status;
 char le_value=CURRENT_VIEW->linend_value;
 int rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Linend");
#endif
 num_params = param_split(params,word,LE_PARAMS,WORD_DELIMS,TEMP_PARAM);
 switch(num_params)
   {
    case 1:
    case 2:
           if (equal((char *)"off",word[0],3))
              le_status = FALSE;
           else
              if (equal((char *)"on",word[0],2))
                 le_status = TRUE;
              else
                {
                 display_error(1,word[0]);
                 rc = RC_INVALID_OPERAND;
                 break;
                }
           if (num_params == 1)
              break;
           if (strlen(word[1]) > 1)
             {
              display_error(1,word[1]);
              break;
             }
           le_value = word[1][0];
           break;
    case 0:
           display_error(3,"");
           rc = RC_INVALID_OPERAND;
           break;
    default:
           display_error(2,"");
           rc = RC_INVALID_OPERAND;
           break;
   }
 if (rc == RC_OK)
   {
    CURRENT_VIEW->linend_status = le_status;
    CURRENT_VIEW->linend_value = le_value;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/*man-start*********************************************************************
COMMAND
     macroext - set default macro extension value

SYNTAX
     [SET] MACROExt [ext]

DESCRIPTION
     The MACROEXT command sets the value of the file extension to be
     used for macro files. When a macro file name is specified on the
     command line, a fullstop '.' then this value will be appended.
     If no value is specified for ext, then THE assumes that the
     supplied macro file name is the fully specified name for a macro.

     The length of ext must be 10 characters or less.

     The macro extension is only appended to a file if that file does
     not include any path specifiers.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: N/A

DEFAULT
     the

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Macroext(char *params)
#else
int Macroext(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern char macro_suffix[12];
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Macroext");
#endif
/*---------------------------------------------------------------------*/
/* If no value is specified for ext, set the value of macro_suffix to  */
/* "", otherwise set it to the supplied value, prefixed with '.'       */
/*---------------------------------------------------------------------*/
 if (strlen(params) == 0)
    strcpy(macro_suffix,"");
 else
   {
    if (strlen(params) > 10)
      {
       display_error(7,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
    strcpy(macro_suffix,".");
    strcat(macro_suffix,params);
   }
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     macropath - set up default path for macro commands

SYNTAX
     [SET] MACROPath path[s]

DESCRIPTION
     The MACROPATH command sets up the search path from which macro
     command files are executed. Each directory is seperated by a
     colon (Unix) or semi-colon (DOS & OS/2).
     No check is done at this stage to validate the supplied path.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Incompatible.

DEFAULT
     Path specified by env variable THE_MACRO_PATH or directory
     specified in THE_HOME_DIR followed by the current directory
     if THE_MACRO_PATH not defined.

SEE ALSO
     Macro,Impmacro.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Macropath(char *params)
#else
int Macropath(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern char the_macro_path[MAX_FILE_NAME+1];
 extern char sp_path[MAX_FILE_NAME+1] ;
 extern char sp_fname[MAX_FILE_NAME+1] ;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Macropath");
#endif
/*---------------------------------------------------------------------*/
/* No checking is done on macro path supplied other than it contains a */
/* value. Path delimiters are translated if necessary.                 */
/*---------------------------------------------------------------------*/
 if (strlen(params) == 0)
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 strcpy(the_macro_path,params);
 (void *)strtrans(the_macro_path,OSLASH,ISLASH);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     margins - set left and right margins

SYNTAX
     [SET] MARgins left right [[+|-]indent]

DESCRIPTION
     The MARGINS command sets the left and right margins and the
     number of columns to indent a paragraph.
     These values are used with the WORDWRAP option.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Compatible.

DEFAULT
     1 72 0

SEE ALSO
     [SET] WORDWRAP.

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Margins(char *params)
#else
int Margins(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern char *temp_cmd;
/*--------------------------- local data ------------------------------*/
#define MAR_PARAMS  3
 char *word[MAR_PARAMS+1];
 int num_params;
 int rc;
 unsigned short left,right;
 short indent;
 bool offset,consistancy_error;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Margins");
#endif
/*---------------------------------------------------------------------*/
/* Two parameters are mandatory, the third is optional.                */
/*---------------------------------------------------------------------*/
 num_params = param_split(params,word,MAR_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params < 2)
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params > 3)
   {
    display_error(2,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Parse the parameters...                                             */
/*---------------------------------------------------------------------*/
 if ((left = atoi(word[0])) == 0)
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Right margin value can be *, set to maximum line length.            */
/*---------------------------------------------------------------------*/
 if (*(word[1]+1) == '*')
   {
    right = max_line_length;
   }
 else
   {
    if ((right = atoi(word[1])) == 0)
      {
       display_error(1,word[1]);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
   }
/*---------------------------------------------------------------------*/
/* Left margin must be less than right margin.                         */
/*---------------------------------------------------------------------*/
 if (right < left)
   {
    display_error(5,word[1]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* Obtain current values for indent, in case they aren't changed by    */
/* the current command. (ie. no third parameter)                       */
/*---------------------------------------------------------------------*/
 indent = CURRENT_VIEW->margin_indent;
 offset = CURRENT_VIEW->margin_indent_offset;
/*---------------------------------------------------------------------*/
/* Determine the type of offset for the indent value. If a sign is     */
/* specified, then the number supplied is relative to the left margin  */
/* otherwise it is an absolute column value.                           */
/*---------------------------------------------------------------------*/
 if (num_params == 3)
   {
    if (*(word[2]) == '-'
    ||  *(word[2]) == '+')
      {
       offset = TRUE;
       if ((indent = atoi(word[2])) == 0)
         {
          if (strcmp(word[2],"+0") != 0)
            {
             display_error(1,word[2]);
#ifdef TRACE
             trace_return();
#endif
             return(RC_INVALID_OPERAND);
            }
         }
      }
    else
      {
       offset = FALSE;
/*---------------------------------------------------------------------*/
/* Absolute indent cannot be negative.                                 */
/*---------------------------------------------------------------------*/
       if ((indent = atoi(word[2])) < 0)
         {
          display_error(1,word[2]);
#ifdef TRACE
          trace_return();
#endif
          return(RC_INVALID_OPERAND);
         }
      }
   }
/*---------------------------------------------------------------------*/
/* Once all values are determined, validate the relationship between   */
/* the margins and the indent values.                                  */
/* Rules:                                                              */
/*       o If indent is a negative offset, the resultant column value  */
/*         cannot be negative.                                         */
/*       o If indent is a positive offset, the resultant column value  */
/*         cannot be > max_line_length or right margin                 */
/*       o If indent is an absolute value, it cannot be > right margin */
/*---------------------------------------------------------------------*/
 consistancy_error = FALSE;
 if (offset
 && indent < 0
 && indent + left < 0)
    consistancy_error = TRUE;
 if (offset
 && indent > 0
 && indent + left > right)
    consistancy_error = TRUE;
 if (offset
 && indent > 0
 && indent + left > max_line_length)
    consistancy_error = TRUE;
 if (!offset
 && indent > right)
    consistancy_error = TRUE;
 if (consistancy_error)
   {
    if (offset)
       sprintf(temp_cmd,"%d %d %+d",left,right,indent);
    else
       sprintf(temp_cmd,"%d %d %d",left,right,indent);
    display_error(12,temp_cmd);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
/*---------------------------------------------------------------------*/
/* All OK, so save the values...                                       */
/*---------------------------------------------------------------------*/
 CURRENT_VIEW->margin_left = left;
 CURRENT_VIEW->margin_right = right;
 CURRENT_VIEW->margin_indent = indent;
 CURRENT_VIEW->margin_indent_offset = offset;
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     msgmode - set display of messages on or off

SYNTAX
     [SET] MSGMode ON|OFF

DESCRIPTION
     The MSGMODE set command determines whether error messages will be 
     displayed or suppressed.

COMPATIBILITY
     XEDIT: Does not support [Short|Long] options.
     KEDIT: Compatible

DEFAULT
     ON

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Msgmode(char *params)
#else
int Msgmode(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern char MSGMx;
/*--------------------------- local data ------------------------------*/
#define MSG_PARAMS  1
 char parm[MSG_PARAMS];
 char *word[MSG_PARAMS+1];
 unsigned short num_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Msgmode");
#endif
 num_params = param_split(params,word,MSG_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params > 1)
   {
    display_error(2,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params < 1)
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 parm[0] = (char)UNDEFINED_OPERAND;
 if (equal((char *)"on",word[0],2))
    parm[0] = TRUE;
 if (equal((char *)"off",word[0],3))
    parm[0] = FALSE;
 if (parm[0] == (char)UNDEFINED_OPERAND)
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (in_profile)
    MSGMx = parm[0];
 CURRENT_VIEW->message_mode = parm[0];
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     newlines - set behaviour of adding new lines

SYNTAX
     [SET] NEWLines Aligned|Left

DESCRIPTION
     The NEWLINES set command determines where the cursor displays after
     a new line is added to the file.
     With ALIGNED, the cursor will display in the column of the new line
     immediately underneath the first non-blank character in the line
     above.
     With LEFT, the cursor will display in the first column of the new line.

COMPATIBILITY
     XEDIT: N/A
     KEDIT: Same command, different functionality.

DEFAULT
     NEWLINES ALIGNED

STATUS
     Complete
**man-end**********************************************************************/
#ifdef PROTO
int Newlines(char *params)
#else
int Newlines(params)
char *params;
#endif
/***********************************************************************/
{
/*--------------------------- extern data -----------------------------*/
extern char NEWLx;
/*--------------------------- local data ------------------------------*/
#define NEW_PARAMS  1
 char parm[NEW_PARAMS];
 char *word[NEW_PARAMS+1];
 unsigned short num_params;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Newlines");
#endif
 num_params = param_split(params,word,NEW_PARAMS,WORD_DELIMS,TEMP_PARAM);
 if (num_params > 1)
   {
    display_error(2,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 if (num_params < 1)
   {
    display_error(3,(char *)"");
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }

 parm[0] = (char)UNDEFINED_OPERAND;
 if (equal((char *)"aligned",word[0],1))
    parm[0] = TRUE;
 if (equal((char *)"left",word[0],1))
    parm[0] = FALSE;
 if (parm[0] == (char)UNDEFINED_OPERAND)
   {
    display_error(1,word[0]);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 NEWLx = CURRENT_VIEW->newline_aligned = parm[0];
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     nodisp - set the character that displays for non-displaying characters

SYNTAX
     [SET] NONDisp char

DESCRIPTION
     The NONDISP command allows the user to change the character that 
     is displayed for non-displaying commands when [SET] ETMODE is OFF.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: N/A

DEFAULT
     #

SEE ALSO
     [SET] ETMODE

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Nondisp(char *params)
#else
int Nondisp(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool NONDISPx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Nondisp");
#endif
 if (strlen(params) != 1)
   {
    display_error(1,params);
#ifdef TRACE
    trace_return();
#endif
    return(RC_INVALID_OPERAND);
   }
 NONDISPx = *params;
 if (!in_profile)
    show_page();
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/*man-start*********************************************************************
COMMAND
     number - turn prefix numbers on or off

SYNTAX
     [SET] NUMber ON|OFF

DESCRIPTION
     The NUMBER command allows the user to toggle the display of numbers
     in the prefix area.

COMPATIBILITY
     XEDIT: Compatible.
     KEDIT: Compatible.

DEFAULT
     ON

SEE ALSO
     Prefix

STATUS
     Complete.
**man-end**********************************************************************/
#ifdef PROTO
int Number(char *params)
#else
int Number(params)
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool NUMBERx;
/*--------------------------- local data ------------------------------*/
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("commset1.c:Number");
#endif
 if (equal((char *)"off",params,3))
    NUMBERx = FALSE;
 else
    if (equal((char *)"on",params,2))
       NUMBERx = TRUE;
    else
      {
       display_error(1,(char *)params);
#ifdef TRACE
       trace_return();
#endif
       return(RC_INVALID_OPERAND);
      }
 CURRENT_VIEW->number = NUMBERx;
 if (!in_profile)
    show_page();

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
