/***********************************************************************/
/* QUERY.C -                                                           */
/* This file contains functions related to QUERY,STATUS and EXTRACT    */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1993 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                     email: M.Hessling@gu.edu.au
 * 36 David Road                     Phone: +61 7 849 7731
 * Holland Park                      Fax:   +61 7 875 5314
 * QLD 4121
 * Australia
 */

/*
$Header: C:\THE\RCS\query.c 1.4 1993/09/01 16:27:03 MH Interim MH $
*/

#include <stdio.h>

#include "the.h"
#include "proto.h"
#include "query.h"

#ifdef PROTO
static int extract_pending(int,char *);
static int extract_point(int,char *);
static int extract_prefix(int,char *);
#else
static int extract_pending();
static int extract_point();
static int extract_prefix();
#endif

/*man-start*********************************************************************


========================================================================
QUERY, EXTRACT and STATUS
========================================================================

     The following lists the valid variables that can be queried and
     also shows what values are returned. For both QUERY and STATUS,
     the values are concatenated together and displayed as a single
     line. For EXTRACT the REXX variables that are set are defined.
     The capitalised portion of the variable is the minimum abbreviation
     recognised.
     The bracketed text at the end of the description indicates from
     which commands a valid response will be supplied (Q-Query,E-Extract
     and S-Status).

ALT
     The number of alterations to the current file since the last SAVE
     or AUTOSAVE.
     (QES)

     alt.0           - 2
     alt.1           - Number of alterations since last SAVE or AUTOSAVE
     alt.2           - Number of alterations since last SAVE

ARBchar
     The status of ARBCHAR and the ARBCHAR character.
     (QES)

     arbchar.0       - 2
     arbchar.1       - ON|OFF
     arbchar.2       - ARBCHAR character

AUtosave
     The status of AUTOSAVE and/or the frequency setting.
     (QES)

     autosave.0      - 1
     autosave.1      - OFF|n

BACKup
     Indicates if a .bak file is kept after editing.
     (QES)

     backup.0        - 1
     backup.1        - ON|OFF

BEEP
     Indicates if the bell is sounded on display of error messages.
     (QES)

     beep.0          - 1
     beep.1          - ON|OFF

BLOCK
     Returns information about the marked block, if any.
     (E)

     block.0         - 6 if a marked block exists, or 1 for NONE
     block.1         - type of marked block (LINE|BOX|NONE)
     block.2         - line number of start of block
     block.3         - column number of start of block
     block.4         - line number of end of block
     block.5         - column number of end of block
     block.6         - file name containing marked block

CASE
     The settings related to the CASE of characters entered, searched
     for, changed and sorted.
     (QES)

     case.0          - 4
     case.1          - MIXED|UPPER|LOWER
     case.2          - RESPECT|IGNORE
     case.3          - RESPECT|IGNORE
     case.4          - RESPECT|IGNORE

CLEARScreen
     Indicates if the screen is cleared on exit from THE.
     (QES)

     clearscreen.0   - 1
     clearscreen.1   - ON|OFF

CLOCK
     Indicates if the time is displayed on the status line.
     (QES)

     clock.0         - 1
     clock.1         - ON|OFF

CMDArrows
     Returns the settings for arrow key behaviour.
     (QES)

     cmdarrows.0     - 3
     cmdarrows.1     - RETRIEVE|TAB
     cmdarrows.2     - SCROLL|TAB
     cmdarrows.3     - SCROLL|TAB

CMDline
     The settings for the command line.
     (QES)

     cmdline.0       - 3
     cmdline.1       - TOP|BOTTOM
     cmdline.2 *     - line number within window of command line
     cmdline.3 *     - contents of command line
               * these values are only returned with EXTRACT

CURline
     The value and position of the current line within the current file.
     (E)

     curline.0       - 3
     curline.1       - -1
     curline.2       - line number within window of current line
     curline.3       - contents of current line

DIRFILEID
     The value of the path and filename of the focus line in a DIR.DIR
     file.
     (E)

     dirfileid.0     - 2
     dirfileid.1     - full path of directory
     dirfileid.2     - file name at focus line

EOF
     Indicates if the current line is on the *** Bottom of file *** line.
     (QES)

     eof.0           - 1
     eof.2           - ON|OFF

EOLout
     Returns the value of the end-of-line character(s).
     (QES)

     eolout.0        - 1
     eolout.1        - LF|CRLF

ETMODE
     Indicates if extended display mode is set.
     (QES)

     etmode.0        - 1
     etmode.1        - ON|OFF

FNAME
     The file name of the current file.
     (E)

     fname.0         - 1
     fname.1         - File name.

FPATH
     The path name of the current file. This includes a trailing
     directory seperator.
     (E)

     fpath.0         - 1
     fpath.1         - File path.

FTYPE
     The extension of the current file. The characters following
     the trailing '.' character.
     (E)

     ftype.0         - 1
     ftype.1         - File extension.

GETENV variable
     The value of the supplied environment variable or ***invalid***
     if the variable does not exist.
     (E)

     getenv.0        - 1
     getenv.1        - value of variable

HEX
     Indicates if HEX is on or off.
     (QES)

     hex.0           - 1
     hex.1           - ON|OFF

HEXDISPlay
     Indicates if the current character is displayed on the status line.
     (QES)

     hexdisplay.0    - 1
     hexdisplay.1    - ON|OFF

IMPMACro
     Indicates if implied macro processing is on or off.
     (QES)

     impmacro.0      - 1
     impmacro.1      - ON|OFF

IMPOS
     Indicates if implied operating system command processing is on 
     or off.
     (QES)

     impos.0         - 1
     impos.1         - ON|OFF

INSERTmode
     Indicates if currently in insert mode or overstrike mode.
     (QES)

     insertmode.1    - 1
     insertmode.1    - ON|OFF

LASTmsg
     Return the text of the last error message generated.
     (E)

     lastmsg.0       - 1
     lastmsg.1       - text of last message.

LASTRC
     Return code from last command issued from command line.
     (QES)

     lastrc.0        - 1
     lastrc.1        - Last return code.

LENgth
     Length of the current line.
     (QES)

     length.0        - 1
     length.1        - Length of current line.

LIne
     Line number of current line in current file.
     (QES)

     line.0          - 1
     line.1          - Line number

LINENd
     Indicates if multiple commands allowed on command line
     and the delimiter.
     (QES)

     linend.0        - 2
     linend.1        - ON|OFF
     linend.2        - delimiter

MACROExt
     The current setting for a macro's file extension.
     (QES)

     macroext.0      - 1
     macroext.1      - Default file extension

MACROPath
     The path that THE looks for by default for macro files.
     (E)

     macropath.0     - 1
     macropath.1     - Path for macro files.

MARgins
     The settings for left and right margins and paragraph indent.
     (QES)

     margins.0       - 3
     margins.1       - left column
     margins.2       - right column
     margins.3       - indent value (column or offset from left margin)

MONITOR
     Indicates if the combination of monitor and the curses package
     supports colour. If the curses package supports colour, then
     monitor.3 is set to COLOR and monitor.2 can be COLOR or MONO
     depending on whether the monitor supports colour. If monitor.3
     is MONO then monitor.2 will also be set to MONO.
     (QES)

     monitor.0       - 2
     monitor.1       - COLOR|MONO
     monitor.2       - COLOR|MONO

MSGMode
     Indicates if MSGMODE setting is on or off.
     (QES)

     msgmode.0       - 1
     msgmode.1       - ON|OFF

NBFile
     Returns with the number of files currently in the ring.
     (QES)

     nbfile.0        - 1
     nbfile.1        - Number of files in ring

NEWLines
     Indicates if NEWLINES variable is set to LEFT or ALIGNED.
     (QES)

     newlines.0      - 1
     newlines.1      - ALIGNED|LEFT

NONDisp
     Returns the character that is displayed for extended characters
     that are not displayed.
     (QES)

     nondisp.0       - 1
     nondisp.1       - char

NUMber
     Indicates if line numbers are displayed in the prefix area.
     (QES)

     number.0        - 1
     number.1        - ON|OFF

PENDing [BLOCK] [OLDNAME] name|*
     Returns information about pening prefix commands.
     (E)

     pending.0       - 4
     pending.1       - line number in file
     pending.2       - newname - actual name entered in prefix area
     pending.3       - oldname - original name of macro after
                                 synonym resolution
     pending.4       - BLOCK or null

Point [*]
     Returns the name and line number of the focus line, or names and
     line numbers of all lines in a file if '*' is specified.
     (E)

     When 'EXTract /Point' is specified:

     point.0         - 0 or 1       (0 if focus line not named)
     point.1         - line number and name of line (if line is named)

     When 'EXTract /Point *' is specified:

     point.0         - number of named lines in the file
     point.1         - line number and name for first named line
     point.n         - line number and name for nth named line

PREfix [Synonym name|*]
     Indicates if prefix is displayed for the view and if displayed
     where is is displayed.
     (QES)

     prefix.0        - 1 or 2     (1 if prefix.1 is OFF, 2 otherwise)
     prefix.1        - ON|OFF
     prefix.2        - LEFT|RIGHT                 (if prefix.1 is ON)

     With [Synonym name] option, the name of the macrofile (oldname) 
     is returned that is associated with the synonym, name. If name 
     is not a synonym then name is returned.
     (E)

     prefix.0        - 1
     prefix.1        - oldname

     With [Synonym *] option, all prefix synonyms are returned.
     (E)

     prefix.0        - number of prefix synonyms
     prefix.1        - newname oldname
     prefix.n        - newname oldname

PRINTER
     Returns the value of the printer port or spooler.
     (QES)

     printer.0       - 1
     printer.1       - port or spooler name

REPROFile
     Indicates if the profile file is to be re-executed for each
     subsequent file to be edited.
     (QES)

     reprofile.0     - 1
     reprofile.1     - ON|OFF

REXXOUTput
     Indicates if REXX output is captured to a file or not and the
     line number limit of lines to be displayed.
     (QES)

     rexxoutput.0    - 2
     rexxoutput.1    - FILE|DISPLAY
     rexxoutput.2    - line number limit

SIze
     Returns the number of lines in the current file.
     (QES)

     size.0          - 1
     size.1          - Lines in current file.

STAY
     Indicates if STAY is on or off.
     (QES)

     stay.0          - 1
     stay.1          - ON|OFF

TABKey
     Returns settings about behaviour of TAB key.
     tabkey.1 is behaviour while in insert mode
     tabkey.2 is behaviour while not in insert mode
     (QES)

     tabkey.0        - 2
     tabkey.1        - TAB|CHARACTER
     tabkey.2        - TAB|CHARACTER

TABS
     Returns settings about tab columns.
     (QES)

     tabs.0          - 2
     tabs.1          - INCR
     tabs.2          - size of tab

TABSIn
     Indicates if TABSIN processing is on or off and the size of the
     tabs setting.
     (QES)

     tabsin.0        - 2
     tabsin.1        - ON|OFF
     tabsin.2        - size of tabs

TABSOut
     Indicates if TABSOUT processing is on or off and the size of the
     tabs setting.
     (QES)

     tabsout.0       - 2
     tabsout.1       - ON|OFF
     tabsout.2       - size of tabs

TERMinal
     Identifies the terminal type currently being used.
     (QES)

     terminal.0      - 1
     terminal.1      - DOS|OS2|$TERM value under Unix

TOF
     Indicates if the current line is on the *** Top of file *** line.
     (QES)

     tof.0           - 1
     tof.2           - ON|OFF

Verify
     Returns verify column settings.
     (QES)

     verify.0        - 1
     verify.1        - Column pair of verify start and end columns.

VERSION
     Returns information about name of application (THE) and version
     information.
     (QES)

     version.0       - 4
     version.1       - THE
     version.3       - version string eg. 1.5
     version.3       - platform version (DOS,OS2,UNIX)
     version.4       - version status information eg. beta

Width
     Returns maximum line width setting.
     (QES)

     width.0         - 1
     width.1         - Maximum line width value.

WORDWrap
     Indicates if WORDWARP is on or off.
     (QES)

     wordwrap.0      - 1
     wordwrap.1      - ON|OFF

Zone
     Returns zone column settings.
     (QES)

     zone.0          - 2
     zone.1          - Zone start column
     zone.2          - Zone end column



========================================================================
IMPLIED EXTRACT, BOOLEAN and OTHER FUNCTIONS
========================================================================

IMPLIED EXTRACT

     The above REXX variables set above by the EXTRACT command may also
     be obtained by a REXX macro as an implied EXTRACT. Each variable
     above that may be set by an explicit EXTRACT command may also be
     used as an external function to REXX to obtain the same information.
     eg. The REXX commands:

         'EXTRACT /SIZE/CURLINE/'
         Say size.1 curline.1

     may be substituted with:

         Say size.1() curline.1()

BOOLEAN FUNCTIONS

     THE also provides other information to the REXX interpreter via
     boolean functions. These functions return either '1' or '0'
     depending on the information queried.

  after()
     Returns '1' if the cursor is currently after the last non-blank
     character on the line, or if the line is blank.

  blank()
     Returns '1' if the line the cursor is on is completely blank.

  block()
     Returns '1' if the marked block is within the current view.

  before()
     Returns '1' if the cursor is currently before the first non-blank
     character on the line, or if the line is blank.

  bottomedge()
     Returns '1' if the cursor is on the bottom edge of the filearea
     or prefix area.

  command()
     Returns '1' if the cursor is on the command line.

  current()
     Returns '1' if the cursor is on the current line.

  dir()
     Returns '1' if the current file is the special DIR.DIR file.

  end()
     Returns '1' if the cursor is on the last non-blank character on
     the line.

  eof()
     Returns '1' if the cursor is on the '*** Bottom of file ***' line
     and the cursor is not on the command line.

  first()
     Returns '1' if the cursor is in column 1 of the current window.

  focuseof()
     Returns '1' if the focus line is the '*** Bottom of file ***' line
     whether the cursor is on it or not.

  focustof()
     Returns '1' if the focus line is the '*** Top of file ***' line
     whether the cursor is on it or not.

  inblock()
     Returns '1' if the cursor is in the marked block.

  initial()
     Returns '1' if the function is called from the profile file.

  inprefix()
     Returns '1' if the cursor is located in the prefix area.

  leftedge()
     Returns '1' if the cursor is on the left edge of the file area.

  modifiable()
     Returns '1' if the cursor is located in an area that can be changed.
     ie. not on '*** Top of File ***' or '*** Bottom of File ***' lines.

  rightedge()
     Returns '1' if the cursor is on the right edge of the file area.

  spacechar()
     Returns '1' if the cursor is on a space character.

  tof()
     Returns '1' if the cursor is on the '*** Top of file ***' line
     and the cursor is not on the command line.

  topedge()
     Returns '1' if the cursor is on the top edge of the file area.

  verone()
     Returns '1' if the column 1 of the file is being displayed in
     column 1.

OTHER FUNCTIONS

   The following functions provide functions to simplify THE macros
   written in REXX.

  valid_target(target)
     Returns 0 if the supplied 'target' is invalid.
     Returns 'ALL' if the supplied 'target' is 'ALL'.
     If a valid target, returns the number of lines from the focus
     line to the 'target', either positive if the target is toward the
     end of file or negative if the target is toward the top of file.

**man-end**********************************************************************/

struct query_item
 {
  char query;                                /* valid query response ? */
  char *name;                                          /* name of item */
  int min_len;                       /* minimum length of abbreviation */
  int item_number;                           /* unique number for item */
  int number_values;                      /* number of values returned */
 };
typedef struct query_item QUERY_ITEM;

/***********************************************************************/
/* Keep the following items in alphabetic order of name.               */
/***********************************************************************/
 static QUERY_ITEM item[] =
  {
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"alt",3,ITEM_ALT,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"arbchar",3,ITEM_ARBCHAR,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"autosave",2,ITEM_AUTOSAVE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"backup",4,ITEM_BACKUP,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"beep",4,ITEM_BEEP,1},
#if !defined(NOREXX)
   {                         QUERY_EXTRACT             ,"block",5,ITEM_BLOCK,0},
#else
   {QUERY_NONE                                         ,"block",5,ITEM_BLOCK,0},
#endif
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"case",4,ITEM_CASE,4},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"clearscreen",6,ITEM_CLEARSCREEN,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"clock",5,ITEM_CLOCK,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"cmdarrows",4,ITEM_CMDARROWS,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"cmdline",3,ITEM_CMDLINE,2},
   {                         QUERY_EXTRACT|QUERY_MODIFY,"curline",3,ITEM_CURLINE,3},
#if !defined(NOREXX)
   {                         QUERY_EXTRACT             ,"dirfileid",9,ITEM_DIRFILEID,2},
#else
   {QUERY_NONE                                         ,"dirfileid",9,ITEM_DIRFILEID,2},
#endif
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"eof",3,ITEM_EOF,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"eolout",3,ITEM_EOLOUT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"etmode",6,ITEM_ETMODE,1},
#if !defined(NOREXX)
   {                         QUERY_EXTRACT             ,"fname",5,ITEM_FNAME,1},
   {                         QUERY_EXTRACT             ,"fpath",5,ITEM_FPATH,1},
   {                         QUERY_EXTRACT             ,"ftype",5,ITEM_FTYPE,1},
   {                         QUERY_EXTRACT             ,"getenv",6,ITEM_GETENV,1},
#else
   {QUERY_NONE                                         ,"fname",5,ITEM_FNAME,1},
   {QUERY_NONE                                         ,"fpath",5,ITEM_FPATH,1},
   {QUERY_NONE                                         ,"ftype",5,ITEM_FTYPE,1},
   {QUERY_NONE                                         ,"getenv",6,ITEM_GETENV,1},
#endif
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"hex",3,ITEM_HEX,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"hexdisplay",7,ITEM_HEXDISPLAY,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"impmacro",6,ITEM_IMPMACRO,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"impos",5,ITEM_IMPOS,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"insertmode",6,ITEM_INSERTMODE,1},
#if !defined(NOREXX)
   {                         QUERY_EXTRACT             ,"lastmsg",4,ITEM_LASTMSG,1},
#else
   {QUERY_NONE                                         ,"lastmsg",4,ITEM_LASTMSG,1},
#endif
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"lastrc",6,ITEM_LASTRC,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"length",3,ITEM_LENGTH,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"line",2,ITEM_LINE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"linend",5,ITEM_LINEND,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"macroext",6,ITEM_MACROEXT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"macropath",6,ITEM_MACROPATH,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"margins",3,ITEM_MARGINS,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"monitor",7,ITEM_MONITOR,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"msgmode",4,ITEM_MSGMODE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"nbfile",3,ITEM_NBFILE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"newlines",4,ITEM_NEWLINES,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"nondisp",4,ITEM_NONDISP,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"number",3,ITEM_NUMBER,1},
#if !defined(NOREXX)
   {                         QUERY_EXTRACT             ,"pending",4,ITEM_PENDING,4},
#else
   {QUERY_NONE                                         ,"pending",4,ITEM_PENDING,4},
#endif
#if !defined(NOREXX)
   {                         QUERY_EXTRACT             ,"point",1,ITEM_POINT,0},
#else
   {QUERY_NONE                                         ,"point",1,ITEM_POINT,0},
#endif
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"prefix",3,ITEM_PREFIX,0},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"printer",7,ITEM_PRINTER,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"reprofile",6,ITEM_REPROFILE,1},
#if !defined(NOREXX)
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"rexxoutput",7,ITEM_REXXOUTPUT,2},
#else
   {QUERY_NONE                                         ,"rexxoutput",7,ITEM_REXXOUTPUT,2},
#endif
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"size",2,ITEM_SIZE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"stay",4,ITEM_STAY,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"tabkey",4,ITEM_TABKEY,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"tabs",4,ITEM_TABS,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"tabsin",5,ITEM_TABSIN,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"tabsout",5,ITEM_TABSOUT,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"terminal",4,ITEM_TERMINAL,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"tof",3,ITEM_TOF,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"verify",1,ITEM_VERIFY,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"version",7,ITEM_VERSION,4},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,"width",1,ITEM_WIDTH,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"wordwrap",5,ITEM_WORDWRAP,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,"zone",1,ITEM_ZONE,2},
   {0                                                   ,NULL,0,0,0},
  };

VALUE item_values[MAX_VARIABLES_RETURNED];
/*-------------------------- external data ----------------------------*/
extern VIEW_DETAILS *vd_current,*vd_first,*vd_mark;
/***********************************************************************/
#ifdef PROTO
int find_item(char *item_name,char query_type)
#else
int find_item(item_name,query_type)
char *item_name;
char query_type;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register int i;
 int itemno = (-1);
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c   :find_item");
#endif
/*---------------------------------------------------------------------*/
/* First find the item to be evaluated...                              */
/*---------------------------------------------------------------------*/
 for (i=0;item[i].name != NULL;i++)
    {
     if (equal(item[i].name,item_name,item[i].min_len)
     && (item[i].query & query_type))
       {
        itemno = item[i].item_number;
        break;
       }
    }
#ifdef TRACE
 trace_return();
#endif
 return(itemno);
}
/***********************************************************************/
#ifdef PROTO
int show_status(void)
#else
int show_status()
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
extern WINDOW *foot;
/*--------------------------- local data ------------------------------*/
 register int i,j;
 int lineno=0,colno=0;
 int number_variables;
 bool left_col = TRUE;
 int item_width,column,column_width,col[3];
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c   :show_status");
#endif
/*---------------------------------------------------------------------*/
/* For each item that is displayable, display it...                    */
/*---------------------------------------------------------------------*/
 column = 0;
 column_width = COLS / 3;
 col[0] = 0;
 col[1] = column_width+1;
 col[2] = col[1]+column_width+1;
 wclear(stdscr);
 for (i=0;item[i].name != NULL;i++)
    {
/*---------------------------------------------------------------------*/
/* Get only those settings that are queryable...                       */
/*---------------------------------------------------------------------*/
     if (item[i].query & QUERY_STATUS)
       {
        number_variables = get_item_values(item[i].item_number,"",QUERY_STATUS,0L,NULL,0L);
        item_width = 0;
/*---------------------------------------------------------------------*/
/* Obtain the total length of the setting values...                    */
/*---------------------------------------------------------------------*/
        for (j=0;j<number_variables+1;j++)
           item_width += item_values[j].len+1;
        item_width--;     /* reduce by 1 for last value's blank at end */
/*---------------------------------------------------------------------*/
/* If the length of the variables is > the screen width, go to next    */
/* line.                                                               */
/*---------------------------------------------------------------------*/
        if (item_width+col[column] > COLS)
          {
           column = colno = 0;
           lineno++;
          }
        else
           colno = col[column];
/*---------------------------------------------------------------------*/
/* Display the variables. For the first value, display in BOLD.        */
/*---------------------------------------------------------------------*/
        for (j=0;j<number_variables+1;j++)
          {
           if (j == 0)
              attrset(A_BOLD);
           mvaddstr(lineno,colno,item_values[j].value);
           attrset(A_NORMAL);
           colno += item_values[j].len+1;
          }
/*---------------------------------------------------------------------*/
/* Now have to determine where to display the next set of variables... */
/* If the just-displayed values go past the last column, or we are     */
/* already in the last column, go to the next line...                  */
/*---------------------------------------------------------------------*/
        if (colno >= col[2] || column == 2)
          {
           lineno++;
           column = 0;
          }
        else
          {
/*---------------------------------------------------------------------*/
/* ...else if the the just-displayed values go past the 2nd column...  */
/* If the just-displayed values go past the last column, go on to the  */
/* next line...                                                        */
/*---------------------------------------------------------------------*/
           if (colno >= col[1])
              column = 2;
           else
/*---------------------------------------------------------------------*/
/* ...else go to the next column.                                      */
/*---------------------------------------------------------------------*/
              column++;
          }
       }
    }
 mvaddstr(LINES-2,0,HIT_ANY_KEY);
 refresh();
 (void)my_getch(stdscr);
 wclear(stdscr);
 refresh();

 restore_THE();

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
int save_status(char *filename)
#else
int save_status(filename)
char *filename;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern char sp_path[MAX_FILE_NAME+1] ;
 extern char sp_fname[MAX_FILE_NAME+1] ;
 extern char *the_version;
/*--------------------------- local data ------------------------------*/
 register int i,j;
 int number_variables,rc;
 FILE *fp;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c   :save_status");
#endif
/*---------------------------------------------------------------------*/
/* Get the fully qualified filename from the supplied filename.        */
/*---------------------------------------------------------------------*/
 strtrans(filename,OSLASH,ISLASH);
 if ((rc = splitpath(filename)) != RC_OK)
   {
    display_error(10,filename);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* splitpath() has set up sp_path  and sp_fname. Append the filename.  */
/*---------------------------------------------------------------------*/
 strcat(sp_path,sp_fname);
/*---------------------------------------------------------------------*/
/* If the file already exists, display an error.                       */
/*---------------------------------------------------------------------*/
  if (file_exists(sp_path))
   {
    display_error(8,filename);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 fp = fopen(sp_path,"w");
/*---------------------------------------------------------------------*/
/* For each item that is modifiable, get its value...                  */
/*---------------------------------------------------------------------*/
 fprintf(fp,"/* This file generated by STATUS command of THE %s */\n",the_version);
 for (i=0;item[i].name != NULL;i++)
    {
/*---------------------------------------------------------------------*/
/* Get only those settings that are modifiable...                      */
/*---------------------------------------------------------------------*/
     if (item[i].query & QUERY_MODIFY)
       {
        number_variables = get_item_values(item[i].item_number,"",QUERY_MODIFY,0L,NULL,0L);
        fputs("'set",fp);
/*---------------------------------------------------------------------*/
/* Write the variables to the file...                                  */
/*---------------------------------------------------------------------*/
        for (j=0;j<number_variables+1;j++)
          {
           fputc(' ',fp);
           fputs(item_values[j].value,fp);
          }
        fprintf(fp,"'\n");
       }
    }
 fflush(fp);
 fclose(fp);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef PROTO
int get_item_values(int itemno,char *itemargs,char query_type,unsigned long argc,char *arg,unsigned long arglen)
#else
int get_item_values(itemno,itemargs,query_type,argc,arg,arglen)
int itemno;
char *itemargs;
char query_type;
unsigned long argc;
char *arg;
unsigned long arglen;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
extern char term_name[20];
extern char dir_path[MAX_FILE_NAME+1];
extern char *temp_cmd;
extern unsigned short file_start;
extern char *the_version;
extern short max_line_length;
extern char the_macro_path[MAX_FILE_NAME+1];
extern char macro_suffix[12];
extern char CLEARSCREENx;
extern char CMDARROWSTABCMDx;
extern char CMDARROWSTABTXTx;
extern char CMDARROWSTABLRx;
extern char REPROFILEx;
extern bool CAPREXXOUTx;
extern long CAPREXXMAXx;
extern char TABI_ONx;
extern char TABI_Nx;
extern char TABO_ONx;
extern char TABO_Nx;
extern bool BEEPx;
extern bool CLOCKx;
extern bool HEXDISPLAYx;
extern char number_of_files;
extern char mode_insert;
extern char tabkey_insert;
extern char tabkey_overwrite;
extern int lastrc;
extern char last_message[160];
extern char in_profile;
extern unsigned short rec_len;
extern unsigned short cmd_rec_len;
extern unsigned short pre_rec_len;
extern char current_screen;
extern SCREEN_DETAILS screen[MAX_SCREENS];
extern bool colour_support;
extern bool extended_display_mode;
extern bool NONDISPx;
extern long prefix_current_line;
extern bool in_prefix_macro;
extern char *cmd_rec;
extern char *rec;

#if defined(UNIX) || defined(OS2)
extern char *spooler_name;
#endif
/*--------------------------- local data ------------------------------*/
 static char num0[3];
 static char num1[7];
 static char num2[7];
 static char num3[15];
 static char num4[15];
 static LINE *curr;
 int y,x,rc,i;
 int number_variables = item[itemno].number_values;
 long true_line,num_lines;
 bool bool_flag;
 char cursor_char;
 char *tmpbuf;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   get_item_values");
#endif
/*---------------------------------------------------------------------*/
/* Now that we have a valid itme, set up its values...                 */
/*---------------------------------------------------------------------*/
 switch(itemno)
   {
#if !defined(NOREXX)
    case ITEM_AFTER_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         bool_flag = FALSE;
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_MAIN:
                 if ((x+CURRENT_VIEW->verify_col-1) >= rec_len)
                    bool_flag = TRUE;
                 break;
            case WINDOW_COMMAND:
                 if (x >= cmd_rec_len)
                    bool_flag = TRUE;
                 break;
            case WINDOW_PREFIX:
                 if (x >= pre_rec_len)
                    bool_flag = TRUE;
                 break;
           }
         if (bool_flag)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_ALT:
         sprintf(num1,"%d",CURRENT_FILE->autosave_alt);
         sprintf(num2,"%d",CURRENT_FILE->save_alt);
         item_values[1].value = num1;
         item_values[2].value = num2;
         item_values[1].len = strlen(num1);
         item_values[2].len = strlen(num2);
         break;
    case ITEM_ARBCHAR:
         if (CURRENT_VIEW->arbchar_status)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
         else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         num1[0] = CURRENT_VIEW->arbchar_char;
         num1[1] ='\0';
         item_values[2].value = num1;
         item_values[2].len = 1;
         break;
    case ITEM_AUTOSAVE:
         sprintf(num1,"%d",CURRENT_FILE->autosave);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         if (CURRENT_FILE->autosave == 0)
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_BACKUP:
         if (CURRENT_FILE->backup)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
#if !defined(NOREXX)
    case ITEM_BEFORE_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         bool_flag = FALSE;
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_MAIN:
                 if ((x+CURRENT_VIEW->verify_col-1) < memne(rec,' ',rec_len))
                    bool_flag = TRUE;
                 break;
            case WINDOW_COMMAND:
                 if (x < memne(cmd_rec,' ',cmd_rec_len))
                    bool_flag = TRUE;
                 break;
            case WINDOW_PREFIX:/* cursor can't go before 1st non-blank */
                 break;
           }
         if (bool_flag)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_BEEP:
         if (BEEPx)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
#if !defined(NOREXX)
    case ITEM_BLANK_FUNCTION:
         bool_flag = FALSE;
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_MAIN:
                 if (rec_len == 0)
                    bool_flag = TRUE;
                 break;
            case WINDOW_PREFIX:
                 if (pre_rec_len == 0)
                    bool_flag = TRUE;
                 break;
            case WINDOW_COMMAND:
                 if (cmd_rec_len == 0)
                    bool_flag = TRUE;
                 break;
           }
         if (bool_flag)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_BLOCK:
         if (MARK_VIEW == NULL)
           {
            item_values[1].value = "NONE";
            item_values[1].len = 4;
            number_variables = 1;
           }
          else
           {
            number_variables = 6;
            if (MARK_VIEW->mark_type == M_LINE)
              {
               item_values[1].value = "LINE";
               item_values[1].len = 4;
              }
            else
              {
               item_values[1].value = "BOX";
               item_values[1].len = 3;
              }
            sprintf(num1,"%d",MARK_VIEW->mark_start_line);
            item_values[2].value = num1;
            item_values[2].len = strlen(num1);
            sprintf(num2,"%d",MARK_VIEW->mark_start_col);
            item_values[3].value = num2;
            item_values[3].len = strlen(num2);
            sprintf(num3,"%d",MARK_VIEW->mark_end_line);
            item_values[4].value = num3;
            item_values[4].len = strlen(num3);
            sprintf(num4,"%d",MARK_VIEW->mark_end_col);
            item_values[5].value = num4;
            item_values[5].len = strlen(num4);
            strcpy(temp_cmd,MARK_FILE->fpath);
            strcat(temp_cmd,MARK_FILE->fname);
            item_values[6].value = (char *)temp_cmd;
            item_values[6].len = strlen(temp_cmd);
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_BLOCK_FUNCTION:
         if (CURRENT_VIEW == MARK_VIEW)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_BOTTOMEDGE_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_MAIN
         &&  y == CURRENT_SCREEN.rows-1)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_CASE:
         switch(CURRENT_VIEW->case_enter)
           {
            case CASE_MIXED:
                 item_values[1].value = "MIXED";
                 break;
            case CASE_UPPER:
                 item_values[1].value = "UPPER";
                 break;
            case CASE_LOWER:
                 item_values[1].value = "LOWER";
                 break;
            default:
                 break;
           }
         item_values[1].len = 5;
         switch(CURRENT_VIEW->case_locate)
           {
            case CASE_IGNORE:
                 item_values[2].value = "IGNORE";
                 item_values[2].len = 6;
                 break;
            case CASE_RESPECT:
                 item_values[2].value = "RESPECT";
                 item_values[2].len = 7;
                 break;
            default:
                 break;
           }
         switch(CURRENT_VIEW->case_change)
           {
            case CASE_IGNORE:
                 item_values[3].value = "IGNORE";
                 item_values[3].len = 6;
                 break;
            case CASE_RESPECT:
                 item_values[3].value = "RESPECT";
                 item_values[3].len = 7;
                 break;
            default:
                 break;
           }
         switch(CURRENT_VIEW->case_sort)
           {
            case CASE_IGNORE:
                 item_values[4].value = "IGNORE";
                 item_values[4].len = 6;
                 break;
            case CASE_RESPECT:
                 item_values[4].value = "RESPECT";
                 item_values[4].len = 7;
                 break;
            default:
                 break;
           }
         break;
    case ITEM_CLEARSCREEN:
         if (CLEARSCREENx)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_CLOCK:
         if (CLOCKx)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
#if !defined(NOREXX)
    case ITEM_COMMAND_FUNCTION:
         if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_CMDARROWS:
         if (CMDARROWSTABCMDx)
           {
            item_values[1].value = "TAB";
            item_values[1].len = 3;
           }
          else
           {
            item_values[1].value = "RETRIEVE";
            item_values[1].len = 8;
           }
         if (CMDARROWSTABTXTx)
           {
            item_values[2].value = "TAB";
            item_values[2].len = 3;
           }
          else
           {
            item_values[2].value = "SCROLL";
            item_values[2].len = 6;
           }
         if (CMDARROWSTABLRx)
           {
            item_values[3].value = "TAB";
            item_values[3].len = 3;
           }
          else
           {
            item_values[3].value = "SCROLL";
            item_values[3].len = 6;
           }
         break;
    case ITEM_CMDLINE:
         if (CURRENT_VIEW->cmd_line == 'B')
           {
            item_values[1].value = "BOTTOM";
            item_values[1].len = 6;
           }
          else
           {
            item_values[1].value = "TOP";
            item_values[1].len = 3;
           }
         if (query_type == QUERY_EXTRACT)
           {
            getbegyx(CURRENT_WINDOW_COMMAND,y,x);
            sprintf(num1,"%d",y);
            item_values[2].value = num1;
            item_values[2].len = strlen(num1);
            item_values[3].value = cmd_rec;
            item_values[3].len = cmd_rec_len;
            number_variables = 3;
           }
         else
            number_variables = 1;
         break;
    case ITEM_CURLINE:
         if (query_type == QUERY_MODIFY)
           {
            sprintf(num1,"%d",CURRENT_VIEW->current_row+1);
            item_values[1].value = num1;
            item_values[1].len = strlen(num1);
            number_variables = 1;
           }
         else
           {
            item_values[1].value = "-1";
            item_values[1].len = 2;
            sprintf(num1,"%d",CURRENT_VIEW->current_row+1);
            item_values[2].value = num1;
            item_values[2].len = strlen(num1);
#ifdef USE_VOID
            curr = (LINE *)ll_find((void *)CURRENT_FILE->first_line,get_true_line());
#else
            curr = lll_find(CURRENT_FILE->first_line,get_true_line());
#endif
            item_values[3].value = (char *)curr->line;
            item_values[3].len = curr->length;
           }
         break;
#if !defined(NOREXX)
    case ITEM_CURRENT_FUNCTION:
         if (CURRENT_VIEW->current_window != WINDOW_COMMAND
         &&  CURRENT_VIEW->focus_line == CURRENT_VIEW->current_line)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_DIR_FUNCTION:
         if (CURRENT_FILE->pseudo_file == PSEUDO_DIR)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_DIRFILEID:
         true_line = (-1L);
         if (CURRENT_FILE->pseudo_file == PSEUDO_DIR)
           {
            if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
               if (CURRENT_TOF || CURRENT_BOF)
                  true_line = (-1L);
               else
                  true_line = CURRENT_VIEW->current_line;
            else
               if (FOCUS_TOF || FOCUS_BOF)
                  true_line = (-1L);
               else
                  true_line = CURRENT_VIEW->focus_line;
           }
         else
            true_line = (-1L);
         if (true_line == (-1L))
           {
            item_values[1].value = "";
            item_values[1].len = 0;
            item_values[2].value = "";
            item_values[2].len = 0;
           }
         else
           {
#ifdef USE_VOID
            curr = (LINE *)ll_find((void *)CURRENT_FILE->first_line,true_line);
#else
            curr = lll_find(CURRENT_FILE->first_line,true_line);
#endif
            item_values[1].value = (char *)dir_path;
            item_values[1].len = strlen(dir_path);
            if (curr->length < file_start)
              {
               item_values[2].value = "";
               item_values[2].len = 0;
              }
            else
              {
               item_values[2].value = (char *)curr->line+file_start;
               item_values[2].len = strlen(curr->line+file_start);
              }
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_END_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_MAIN
         &&  (x+CURRENT_VIEW->verify_col) == rec_len)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_EOF:
         if (CURRENT_BOF)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
#if !defined(NOREXX)
    case ITEM_EOF_FUNCTION:
         if (FOCUS_BOF && CURRENT_VIEW->current_window != WINDOW_COMMAND)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_EOLOUT:
         if (CURRENT_FILE->eolout == EOLOUT_LF)
           {
            item_values[1].value = "LF";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "CRLF";
            item_values[1].len = 4;
           }
         break;
    case ITEM_ETMODE:
         if (extended_display_mode)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
#if !defined(NOREXX)
    case ITEM_FIRST_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (x == 0)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_FOCUSEOF_FUNCTION:
         if (FOCUS_BOF)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_FOCUSTOF_FUNCTION:
         if (FOCUS_TOF)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_FNAME:
         item_values[1].value = (char *)CURRENT_FILE->fname;
         item_values[1].len = strlen(CURRENT_FILE->fname);
         break;
#endif
#if !defined(NOREXX)
    case ITEM_FPATH:
         item_values[1].value = (char *)CURRENT_FILE->fpath;
         item_values[1].len = strlen(CURRENT_FILE->fpath);
         break;
#endif
#if !defined(NOREXX)
    case ITEM_FTYPE:
         x = strzreveq(CURRENT_FILE->fname,'.');
         if (x == (-1))
           {
            item_values[1].value = "";
            item_values[1].len = 0;
           }
         else
           {
            item_values[1].value = (char *)CURRENT_FILE->fname+x+1;
            item_values[1].len = strlen(CURRENT_FILE->fname+x+1);
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_GETENV:
         if (query_type == QUERY_FUNCTION)
            tmpbuf = getenv(arg);
         else
            tmpbuf = getenv(itemargs);
         if (tmpbuf == NULL)
            item_values[1].value = "***invalid***";
         else
            item_values[1].value = tmpbuf;
         item_values[1].len = strlen(item_values[1].value);
         break;
#endif
    case ITEM_HEX:
         if (CURRENT_VIEW->hex)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_HEXDISPLAY:
         if (HEXDISPLAYx)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_IMPMACRO:
         if (CURRENT_VIEW->imp_macro)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_IMPOS:
         if (CURRENT_VIEW->imp_os)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
#if !defined(NOREXX)
    case ITEM_INBLOCK_FUNCTION:
         bool_flag = FALSE;
         if (CURRENT_VIEW == MARK_VIEW
         &&  CURRENT_VIEW->current_window == WINDOW_MAIN)
           {
            getyx(CURRENT_WINDOW_MAIN,y,x);
            switch(MARK_VIEW->mark_type)
              {
               case M_LINE:
                    if ((CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line)
                    &&  (CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line))
                      bool_flag = TRUE;
                    break;
               case M_BOX:
                    if ((CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line)
                    &&  (CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line)
                    &&  (x + CURRENT_VIEW->verify_start >= MARK_VIEW->mark_start_col)
                    &&  (x + CURRENT_VIEW->verify_start <= MARK_VIEW->mark_end_col))
                      bool_flag = TRUE;
                    break;
               default:
                    break;
              }
           }
         if (bool_flag)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_INITIAL_FUNCTION:
         if (in_profile)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_INPREFIX_FUNCTION:
         if (CURRENT_VIEW->current_window == WINDOW_PREFIX)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_INSERTMODE:
         if (mode_insert)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
#if !defined(NOREXX)
    case ITEM_LASTMSG:
         item_values[1].value = (char *)last_message;
         item_values[1].len = strlen(last_message);
         break;
#endif
    case ITEM_LASTRC:
         sprintf(num1,"%d",lastrc);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
#if !defined(NOREXX)
    case ITEM_LEFTEDGE_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_MAIN
         &&  x == 0)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_LENGTH:
#ifdef USE_VOID
         curr = (LINE *)ll_find((void *)CURRENT_FILE->first_line,CURRENT_VIEW->current_line);
#else
         curr = lll_find(CURRENT_FILE->first_line,CURRENT_VIEW->current_line);
#endif
         sprintf(num1,"%d",curr->length);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
    case ITEM_LINE:
         sprintf(num1,"%d",get_true_line());
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
    case ITEM_LINEND:
         if (CURRENT_VIEW->linend_status)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         num1[0] = CURRENT_VIEW->linend_value;
         num1[1] = '\0';
         item_values[2].value = num1;
         item_values[2].len = 1;
         break;
    case ITEM_MACROEXT:
         if (strlen(macro_suffix) == 0)
            item_values[1].value = (char *)macro_suffix;
         else
            item_values[1].value = (char *)macro_suffix+1;
         item_values[1].len = strlen(macro_suffix);
         break;
    case ITEM_MACROPATH:
         item_values[1].value = (char *)the_macro_path;
         item_values[1].len = strlen(the_macro_path);
         break;
    case ITEM_MARGINS:
         sprintf(num1,"%d",CURRENT_VIEW->margin_left);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         sprintf(num2,"%d",CURRENT_VIEW->margin_right);
         item_values[2].value = num2;
         item_values[2].len = strlen(num2);
         if (CURRENT_VIEW->margin_indent_offset)
            sprintf(num3,"%+d",CURRENT_VIEW->margin_indent);
         else
            sprintf(num3,"%d",CURRENT_VIEW->margin_indent);
         item_values[3].value = num3;
         item_values[3].len = strlen(num3);
         break;
#if !defined(NOREXX)
    case ITEM_MODIFIABLE_FUNCTION:
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_MAIN:
                 if (FOCUS_TOF || FOCUS_BOF)
                    bool_flag = FALSE;
                 else
                    bool_flag = TRUE;
                 break;
            default:
                 bool_flag = TRUE;
                 break;
           }
         if (bool_flag)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_MONITOR:
#ifdef A_COLOR
         if (colour_support)
           {
            item_values[1].value = "COLOR";
            item_values[1].len = 5;
           }
          else
           {
            item_values[1].value = "MONO";
            item_values[1].len = 4;
           }
          item_values[2].value = "COLOR";
          item_values[2].len = 5;
          break;
#else
          item_values[1].value = "MONO";
          item_values[1].len = 4;
          item_values[2].value = "MONO";
          item_values[2].len = 4;
         break;
#endif
    case ITEM_MSGMODE:
         if (CURRENT_VIEW->message_mode)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_NBFILE:
         sprintf(num1,"%d",number_of_files);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
    case ITEM_NEWLINES:
         if (CURRENT_VIEW->newline_aligned)
           {
            item_values[1].value = "ALIGNED";
            item_values[1].len = 7;
           }
          else
           {
            item_values[1].value = "LEFT";
            item_values[1].len = 4;
           }
         break;
    case ITEM_NONDISP:
         num1[0] = NONDISPx;
         num1[1] ='\0';
         item_values[1].value = num1;
         item_values[1].len = 1;
         break;
    case ITEM_NUMBER:
         if (CURRENT_VIEW->number)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
         else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
#if !defined(NOREXX)
    case ITEM_PENDING:
         number_variables = extract_pending(itemno,itemargs);
         break;
#endif
#if !defined(NOREXX)
    case ITEM_POINT:
         number_variables = extract_point(itemno,itemargs);
         break;
#endif
    case ITEM_PREFIX:
#if !defined(NOREXX)
         if (strcmp(itemargs,"") != 0)
           {
            number_variables = extract_prefix(itemno,itemargs);
            break;
           }
#else
         if (strcmp(itemargs,"") != 0)
           {
            display_error(2,(char *)itemargs);
            number_variables = EXTRACT_ARG_ERROR;
            break;
           }
#endif
         switch(CURRENT_VIEW->prefix)
           {
            case PREFIX_LEFT:
                 item_values[1].value = "ON";
                 item_values[1].len = 2;
                 item_values[2].value = "LEFT";
                 item_values[2].len = 4;
                 number_variables = 2;
                 break;
            case PREFIX_RIGHT:
                 item_values[1].value = "ON";
                 item_values[1].len = 2;
                 item_values[2].value = "RIGHT";
                 item_values[2].len = 5;
                 number_variables = 2;
                 break;
            default:
                 item_values[1].value = "OFF";
                 item_values[1].len = 3;
                 item_values[2].value = "";  /* this set to empty deliberately */
                 item_values[2].len = 0;
                 number_variables = 1;
                 break;
           }
         break;
    case ITEM_PRINTER:
#if defined(UNIX) || defined(OS2)
         item_values[1].value = (char *)spooler_name;
         item_values[1].len = strlen(spooler_name);
#else
         item_values[1].value = "LPT1";
         item_values[1].len = 4;
#endif
         break;
    case ITEM_REPROFILE:
         if (REPROFILEx)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
#if !defined(NOREXX)
    case ITEM_REXXOUTPUT:
         if (CAPREXXOUTx)
           {
            item_values[1].value = "FILE";
            item_values[1].len = 4;
           }
          else
           {
            item_values[1].value = "DISPLAY";
            item_values[1].len = 7;
           }
         sprintf(num1,"%d",CAPREXXMAXx);
         item_values[2].value = num1;
         item_values[2].len = strlen(num1);
         break;
#endif
#if !defined(NOREXX)
    case ITEM_RIGHTEDGE_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_MAIN
         &&  x == getmaxx(CURRENT_WINDOW)-1)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_SIZE:
         sprintf(num1,"%d",CURRENT_FILE->number_lines);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
#if !defined(NOREXX)
    case ITEM_SPACECHAR_FUNCTION:
         cursor_char = (char)winch(CURRENT_WINDOW) & A_CHARTEXT;
         if (cursor_char == ' ')
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_STAY:
         if (CURRENT_VIEW->stay)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_TABKEY:
         if (tabkey_insert == 'T')
           {
            item_values[1].value = "TAB";
            item_values[1].len = 3;
           }
          else
           {
            item_values[1].value = "CHARACTER";
            item_values[1].len = 9;
           }
         if (tabkey_overwrite == 'T')
           {
            item_values[2].value = "TAB";
            item_values[2].len = 3;
           }
          else
           {
            item_values[2].value = "CHARACTER";
            item_values[2].len = 9;
           }
         break;
    case ITEM_TABS:
         sprintf(num1,"%d",CURRENT_VIEW->tabs);
         item_values[1].value = "INCR";
         item_values[1].len = 4;
         item_values[2].value = num1;
         item_values[2].len = strlen(num1);
         break;
    case ITEM_TABSIN:
         if (TABI_ONx)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         sprintf(num1,"%d",TABI_Nx);
         item_values[2].value = num1;
         item_values[2].len = strlen(num1);
         break;
    case ITEM_TABSOUT:
         if (TABO_ONx)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         sprintf(num1,"%d",TABO_Nx);
         item_values[2].value = num1;
         item_values[2].len = strlen(num1);
         break;
    case ITEM_TERMINAL:
         item_values[1].value = term_name;
         item_values[1].len = strlen(term_name);
         break;
    case ITEM_TOF:
         if (CURRENT_TOF)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
#if !defined(NOREXX)
    case ITEM_TOF_FUNCTION:
         if (FOCUS_TOF && CURRENT_VIEW->current_window != WINDOW_COMMAND)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_TOPEDGE_FUNCTION:
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_MAIN
         &&  y == 0)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
#if !defined(NOREXX)
    case ITEM_VALID_TARGET_FUNCTION:
         if (argc != 1)   /* incorrect no of arguments - invalid */
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
            break;
           }
         if (memcmpi("ALL",arg,arglen) == 0)
           {
            item_values[1].value = "ALL";
            item_values[1].len = 3;
            break;
           }
         tmpbuf = (char *)malloc(sizeof(char)*(arglen+1));
         if (tmpbuf == (char *)NULL)
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
            break;
           }
         memcpy(tmpbuf,arg,arglen);
         *(tmpbuf+arglen) = '\0';
         if (in_prefix_macro)
            true_line = prefix_current_line;
         else
            true_line = get_true_line();
         num_lines = valid_target(tmpbuf,true_line);
         free(tmpbuf);
         if (num_lines == TARGET_NOT_FOUND
         ||  num_lines == TARGET_ERROR)
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
            break;
           }
         sprintf(num3,"%d",num_lines);
         item_values[1].value = num3;
         item_values[1].len = strlen(num3);
         break;
#endif
    case ITEM_VERIFY:
         sprintf(num3,"%d %d",CURRENT_VIEW->verify_start,CURRENT_VIEW->verify_end);
         item_values[1].value = num3;
         item_values[1].len = strlen(num3);
         break;
#if !defined(NOREXX)
    case ITEM_VERONE_FUNCTION:
         if (CURRENT_VIEW->verify_col == 1)
           {
            item_values[1].value = "1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = "0";
            item_values[1].len = 1;
           }
         break;
#endif
    case ITEM_VERSION:
         item_values[1].value = "THE";
         item_values[1].len = 3;
         item_values[2].value = (char *)the_version;
         item_values[2].len = strlen(the_version);
#if defined(DOS)
         item_values[3].value = "DOS";
#elif defined(OS2)
         item_values[3].value = "OS2";
#elif defined(UNIX)
         item_values[3].value = "UNIX";
#else
         item_values[3].value = "???";
#endif
         item_values[3].len = strlen(item_values[3].value);
         item_values[4].value = "01-Dec-93";
         item_values[4].len = strlen(item_values[4].value);
         break;
    case ITEM_WIDTH:
         sprintf(num1,"%d",max_line_length);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         break;
    case ITEM_WORDWRAP:
         if (CURRENT_VIEW->wordwrap)
           {
            item_values[1].value = "ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = "OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_ZONE:
         sprintf(num1,"%d",CURRENT_VIEW->zone_start);
         item_values[1].value = num1;
         item_values[1].len = strlen(num1);
         sprintf(num2,"%d",CURRENT_VIEW->zone_end);
         item_values[2].value = num2;
         item_values[2].len = strlen(num2);
         break;
    default:
         break;
   }
/*---------------------------------------------------------------------*/
/* If an "error" condition exists, do not set any values.              */
/*---------------------------------------------------------------------*/
 if (number_variables >= 0)
   {
    switch(query_type)
      {
       case QUERY_EXTRACT:
       case QUERY_FUNCTION:
            sprintf(num0,"%d",number_variables);
            item_values[0].value = num0;
            item_values[0].len = strlen(num0);
            break;
       case QUERY_STATUS:
       case QUERY_QUERY:
       case QUERY_MODIFY:
            item_values[0].value = item[itemno].name;
            item_values[0].len = strlen(item[itemno].name);
            break;
       default:
            break;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
#if !defined(NOREXX)
/***********************************************************************/
#ifdef PROTO
static int extract_pending(int itemno,char *params)
#else
static int extract_pending(itemno,params)
int itemno;
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register int i;
#define PEN_PARAMS  4
 char *word[PEN_PARAMS+1];
 unsigned short num_params;
 int number_variables = item[itemno].number_values;
 bool find_block=FALSE;
 bool find_oldname=FALSE;
 bool valid_args = FALSE;
 int pending_idx;
 char *name;
 static char num1[7];
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_pending");
#endif
 num_params = param_split(params,word,PEN_PARAMS,WORD_DELIMS,TEMP_PARAM);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* No parameters, error.                                               */
/*---------------------------------------------------------------------*/
    case 0:
           display_error(3,(char *)"");
           number_variables = EXTRACT_ARG_ERROR;
           break;
/*---------------------------------------------------------------------*/
/* 1  parameter, only name.                                            */
/*---------------------------------------------------------------------*/
    case 1:
           if (equal((char *)"block",word[0],5)
           ||  equal((char *)"oldname",word[0],7))
             {
              display_error(3,(char *)"");
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           name = word[0];
           break;
/*---------------------------------------------------------------------*/
/* 2  parameters, BLOCK and name or OLDNAME and name.                  */
/*---------------------------------------------------------------------*/
    case 2:
           if (equal((char *)"block",word[0],5)
           &&  equal((char *)"oldname",word[1],7))
             {
              display_error(3,(char *)"");
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           if (equal((char *)"block",word[1],5)
           &&  equal((char *)"oldname",word[0],7))
             {
              display_error(3,(char *)"");
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           if (equal((char *)"block",word[0],5))
             {
              name = word[1];
              find_block = TRUE;
              break;
             }
           if (equal((char *)"oldname",word[0],7))
             {
              name = word[1];
              find_oldname = TRUE;
              break;
             }
           display_error(1,word[0]);
           number_variables = EXTRACT_ARG_ERROR;
           break;
/*---------------------------------------------------------------------*/
/* 3  parameters, BLOCK and OLDNAME and name.                          */
/*---------------------------------------------------------------------*/
    case 3:
           if (equal((char *)"block",word[0],5)
           &&  equal((char *)"oldname",word[1],7))
              valid_args = TRUE;
           if (equal((char *)"block",word[1],5)
           &&  equal((char *)"oldname",word[0],7))
              valid_args = TRUE;
           if (!valid_args)
             {
              display_error(1,word[0]);
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           find_block = find_oldname = TRUE;
           name = word[2];
           break;
/*---------------------------------------------------------------------*/
/* Too many parameters, error.                                         */
/*---------------------------------------------------------------------*/
    default:
           display_error(2,(char *)"");
           number_variables = EXTRACT_ARG_ERROR;
           break;
   }
/*---------------------------------------------------------------------*/
/* If the validation of parameters is successful...                    */
/*---------------------------------------------------------------------*/
 if (number_variables >= 0)
   {
/*---------------------------------------------------------------------*/
/* No pending prefix commands, return 0.                               */
/*---------------------------------------------------------------------*/
    if (CURRENT_VIEW->prefix_command_index == 0)
       number_variables = 0;
    else
/*---------------------------------------------------------------------*/
/* Have to search each pending prefix command...                       */
/*---------------------------------------------------------------------*/
      {
       pending_idx = (-1);
/*---------------------------------------------------------------------*/
/* If we are to look for OLDNAME, find a synonym for it if one exists..*/
/*---------------------------------------------------------------------*/
       if (find_oldname)
          name = find_prefix_oldname(name);
/*---------------------------------------------------------------------*/
/* For each pending prefix command...                                  */
/*---------------------------------------------------------------------*/
       for (i=0;i<CURRENT_VIEW->prefix_command_index;i++)
         {
/*---------------------------------------------------------------------*/
/* If we want to match on any name...                                  */
/*---------------------------------------------------------------------*/
          if (strcmp(name,"*") == 0)
            {
/*---------------------------------------------------------------------*/
/* Are we matching on any BLOCK command...                             */
/*---------------------------------------------------------------------*/
             if (find_block)
               {
                if (CURRENT_VIEW->ppc[i].ppc_block_command)
                  {
/*---------------------------------------------------------------------*/
/* We have found the first BLOCK command with any name.                */
/*---------------------------------------------------------------------*/
                   pending_idx = i;
                   break;
                  }
                else
/*---------------------------------------------------------------------*/
/* Go back and look for another...                                     */
/*---------------------------------------------------------------------*/
                   continue;
               }
             else
               {
/*---------------------------------------------------------------------*/
/* We have found the first command with any name.                      */
/*---------------------------------------------------------------------*/
                pending_idx = i;
                break;
               }
            }
/*---------------------------------------------------------------------*/
/* We want to find a specific command...                               */
/*---------------------------------------------------------------------*/
          if (strcmp(CURRENT_VIEW->ppc[i].ppc_command,name) == 0)
            {
/*---------------------------------------------------------------------*/
/* Are we looking for a specific BLOCK command...                      */
/*---------------------------------------------------------------------*/
             if (find_block)
               {
                if (CURRENT_VIEW->ppc[i].ppc_block_command)
                  {
/*---------------------------------------------------------------------*/
/* We have found the first specific BLOCK command.                     */
/*---------------------------------------------------------------------*/
                   pending_idx = i;
                   break;
                  }
               }
             else
               {
/*---------------------------------------------------------------------*/
/* We have found the first specific command.                           */
/*---------------------------------------------------------------------*/
                pending_idx = i;
                break;
               }
            }
         }
/*---------------------------------------------------------------------*/
/* Did we find a matching pending prefix command ?                     */
/*---------------------------------------------------------------------*/
       if (pending_idx == (-1))
          number_variables = 0;
       else
         {
/*---------------------------------------------------------------------*/
/* Yes we did. Set all of the REXX variables to the correct values...  */
/*---------------------------------------------------------------------*/
          sprintf(num1,"%d",CURRENT_VIEW->ppc[pending_idx].ppc_line_number);
          item_values[1].value = num1;
          item_values[1].len = strlen(num1);
          item_values[2].value = CURRENT_VIEW->ppc[pending_idx].ppc_command;
          item_values[2].len = strlen(item_values[2].value);
          item_values[3].value = find_prefix_synonym(CURRENT_VIEW->ppc[pending_idx].ppc_command);
          item_values[3].len = strlen(item_values[2].value);
          if (CURRENT_VIEW->ppc[pending_idx].ppc_block_command)
             item_values[4].value = "BLOCK";
          else
             item_values[4].value = "";
          item_values[4].len = strlen(item_values[4].value);
         }
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
/***********************************************************************/
#ifdef PROTO
static int extract_point(int itemno,char *params)
#else
static int extract_point(itemno,params)
int itemno;
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 register int i;
 int number_variables = item[itemno].number_values;
 static char num4[15];
 LINE *curr;
 long true_line;
 int rc;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_point");
#endif
 if (strcmp(params,"") == 0) /* get name for focus line only */
   {
    true_line = get_true_line();
    curr = lll_find(CURRENT_FILE->first_line,true_line);
    if (curr->name == NULL)  /* line not named */
       number_variables = 0;
    else
      {
       sprintf(num4,"%d %s",true_line,curr->name);
       item_values[1].value = num4;
       item_values[1].len = strlen(num4);
       number_variables = 1;
      }
   }
 else
   {
    if (strcmp(params,"*") != 0)     /* argument must be "*" */
      {
       display_error(1,(char *)params);
       number_variables = EXTRACT_ARG_ERROR;
      }
    else
      {
       curr = CURRENT_FILE->first_line;
       for(true_line=0,i=0;curr != NULL;true_line++)
         {
          if (curr->name != NULL)  /* line is named */
            {
             sprintf(num4,"%d %s",true_line,curr->name);
             rc = set_rexx_variable(item[itemno].name,num4,++i);
             if (rc == RC_SYSTEM_ERROR)
               {
                display_error(54,(char *)"");
#ifdef TRACE
                trace_return();
#endif
                return(EXTRACT_ARG_ERROR);
               }
            }
          curr = curr->next;
         }
       sprintf(num4,"%d",i);
       rc = set_rexx_variable(item[itemno].name,num4,0);
       if (rc == RC_SYSTEM_ERROR)
         {
          display_error(54,(char *)"");
          number_variables = EXTRACT_ARG_ERROR;
         }
       else
          number_variables = EXTRACT_VARIABLES_SET;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
/***********************************************************************/
#ifdef PROTO
static int extract_prefix(int itemno,char *params)
#else
static int extract_prefix(itemno,params)
int itemno;
char *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern LINE *first_prefix_synonym;
/*--------------------------- local data ------------------------------*/
 register int i;
#define PRE_PARAMS  3
 char *word[PRE_PARAMS+1];
 unsigned short num_params;
 int number_variables = item[itemno].number_values;
 static char num4[15];
 LINE *curr;
 int rc;
 char *tmpbuf;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_prefix");
#endif
 num_params = param_split(params,word,PRE_PARAMS,WORD_DELIMS,TEMP_PARAM);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* None or 1 parameters, error.                                        */
/*---------------------------------------------------------------------*/
    case 0:
    case 1:
           display_error(3,(char *)"");
           number_variables = EXTRACT_ARG_ERROR;
           break;
/*---------------------------------------------------------------------*/
/* 2  parameters, Synonym and name.                                    */
/*---------------------------------------------------------------------*/
    case 2:
           if (!equal((char *)"synonym",word[0],1))
             {
              display_error(13,(char *)"");
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           break;
/*---------------------------------------------------------------------*/
/* Too many parameters, error.                                         */
/*---------------------------------------------------------------------*/
    default:
           display_error(2,(char *)"");
           number_variables = EXTRACT_ARG_ERROR;
           break;
   }
/*---------------------------------------------------------------------*/
/* If the validation of parameters is successful...                    */
/*---------------------------------------------------------------------*/
 if (number_variables >= 0)
   {
    if (strcmp(word[1],"*") == 0)
/*---------------------------------------------------------------------*/
/* Get oldname for all synonyms...                                     */
/*---------------------------------------------------------------------*/
      {
       curr = first_prefix_synonym;
       i = 0;
       while(curr != NULL)
         {
          tmpbuf = (char *)malloc(sizeof(char)*(strlen(curr->name)+strlen(curr->line)+1));
          if (tmpbuf == (char *)NULL)
            {
             display_error(30,(char *)"");
#ifdef TRACE
             trace_return();
#endif
             return(EXTRACT_ARG_ERROR);
            }
          strcpy(tmpbuf,curr->name);
          strcat(tmpbuf," ");
          strcat(tmpbuf,curr->line);
          rc = set_rexx_variable(item[itemno].name,tmpbuf,++i);
          free(tmpbuf);
          if (rc == RC_SYSTEM_ERROR)
            {
             display_error(54,(char *)"");
#ifdef TRACE
             trace_return();
#endif
             return(EXTRACT_ARG_ERROR);
            }
          curr = curr->next;
         }
       sprintf(num4,"%d",i);
       rc = set_rexx_variable(item[itemno].name,num4,0);
       if (rc == RC_SYSTEM_ERROR)
         {
          display_error(54,(char *)"");
          number_variables = EXTRACT_ARG_ERROR;
         }
       else
          number_variables = EXTRACT_VARIABLES_SET;
      }
    else
/*---------------------------------------------------------------------*/
/* Get oldname for named synonym...                                    */
/*---------------------------------------------------------------------*/
      {
       item_values[1].value = find_prefix_synonym(word[1]);
       item_values[1].len = strlen(item_values[1].value);
       number_variables = 1;
      }
 }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
#endif
