=head1 NAME

Data::ID::Exim - generate Exim message IDs

=head1 SYNOPSIS

	use Data::ID::Exim qw(exim_mid);

	$mid = exim_mid;

=head1 DESCRIPTION

This module exports one function, C<exim_mid>, which generates Exim
message IDs, as if the Perl program were an Exim process.

=cut

package Data::ID::Exim;

use warnings;
use strict;

use Exporter;
use Time::HiRes qw(gettimeofday);

our $VERSION = "0.000";

our @ISA = qw(Exporter);

our @EXPORT_OK = qw(exim_mid);

=head1 FUNCTIONS

=over

=item exim_mid

Generates an Exim message ID.  (This ID may, of course, be used to label
things other than mail messages, but Exim refers to them as message IDs.)
The ID is based on the time and process ID, such that it is guaranteed
to be unique among IDs generated by this algorithm on this host.
This function is completely interoperable with Exim, in the sense that
it uses exactly the same algorithm so that the uniqueness guarantee
applies between IDs generated by this function and by Exim itself.

The format of the message ID is three groups of base 62 digits, separated
by hyphens.  The first group, of six digits, gives the integral number of
seconds since the epoch.  The second group, also of six digits, gives the
process ID.  The third group, of two digits, gives the fractional part
of the number of seconds since the epoch, in units of 1/2000 of a second
(500us).  The function does not return until the clock has advanced far
enough that another call would generate a different ID.

The strange structure of the ID comes from compatibility with earlier
versions of Exim, in which the last two digits were a sequence number.

=item exim_mid(HOST_NUMBER)

Exim has limited support for making message IDs unique among a group
of hosts.  Each host is assigned a number in the range 0 to 16 inclusive.
The last two digits of the message IDs give the host number multiplied by
200 plus the fractional part of the number of seconds since the epoch in
units of 1/200 of a second (5ms).  This makes message IDs unique across
the group of hosts, at the expense of generation rate.

To generate this style of ID, pass the host number to C<exim_mid>.
The host number must be configured by some out-of-band mechanism.

=cut

sub base62_digit($) {
	my($n) = @_;
	if($n < 10) {
		chr(ord("0") + $n);
	} elsif($n < 10+26) {
		chr(ord("A") + $n - 10);
	} else {
		chr(ord("a") + $n - (10+26));
	}

}

sub base62($$) {
	my($ndigits, $value) = @_;
	my $str = "";
	while($ndigits--) {
		$str = base62_digit($value % 62) . $str;
		$value /= 62;
	}
	$str;
}

sub make_fraction($$) {
	my($host_number, $usec) = @_;
	defined($host_number) ?
		200*$host_number + int($usec / 5000) :
		int($usec / 500);
}

sub exim_mid(;$) {
	my($host_number) = @_;
	my($sec, $usec) = gettimeofday;
	my $frac = make_fraction($host_number, $usec);
	my($new_sec, $new_usec, $new_frac);
	do {
		($new_sec, $new_usec) = gettimeofday;
		$new_frac = make_fraction($host_number, $new_usec);
	} while($new_sec == $sec && $new_frac == $frac);
	base62(6, $sec)."-".base62(6, $$)."-".base62(2, $frac);
}

=back

=head1 BUGS

Can theoretically generate duplicate message IDs during a leap second.
Exim suffers the same problem.

=head1 SEE ALSO

L<Data::ID::Maildir>,
L<UUID>,
L<Win32::Genguid>

=head1 AUTHOR

Andrew Main (Zefram) <zefram@fysh.org>

=head1 COPYRIGHT

Copyright (C) 2004 Andrew Main (Zefram) <zefram@fysh.org>

This module is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut

1;
