package InfluxDB::Client::Response;
# ABSTRACT: response class of the InfluxDB::Client

use strict;
use warnings;
use v5.10;

use Carp;
use JSON;
use HTTP::Status ();

use InfluxDB::Client::Response::Status;
use InfluxDB::Client::Response::Meta;

our $VERSION = '0.1';

sub new {
    my ( $class, %args ) = @_;
    
    my $self = bless {
        _status  => $args{status} || InfluxDB::Client::Response::Status::OK,
        _error   => undef,
        _content => {},
        # we create an empty meta object. this could be filled later
        # we don't have enough usecases where to fill this in the constructor to handle it here
        _meta    => InfluxDB::Client::Response::Meta->new,
    } => $class;
    
    if ( $args{http_response} ) {
        $self->parse_http_response($args{http_response});
    } elsif ( $args{content} ) {
        $self->content($args{content});
    }
    
    return $self;
}


sub status {
    my ( $self, $status ) = @_;
    if ( $status ) {
        # TODO: type checking
        $self->{_status} = $status;
    }
    return $self->{_status};
}

sub http_code {
    return $_[0]->{_http_code};
}

sub content {
    my ( $self, $content ) = @_;
    if ( $content ) {
        if ( ref $content ) {
            # if we got an reference, we just store it
            $self->{_content} = $content;
        } else {
            # otherwise we parse it as a JSON string
            my $json = JSON->new();
            eval { $self->{_content} = $json->decode($content); };

            # if JSON parsing failed, we set the status to INTFAIL
            if ($@) {
                $self->{_error} = $@;
                $self->{_status} = InfluxDB::Client::Response::Status::INTFAIL;
            } else {
                # we check if we have an error in our result
                if(exists $self->{_content}{error} && defined $self->{_content}{error}) {
                    $self->{_error} = $self->{_content}{error};
                    # the status code should already be set to error, but we make sure it really is
                    $self->{_status} = InfluxDB::Client::Response::Status::SRVFAIL
                        if ( $self->{_status} == InfluxDB::Client::Response::Status::OK );
                }
                
                # we strip the "results" from the parsed data
                $self->{_content} = $self->{_content}{results} if exists $self->{_content}{results};
            }
        }
    }
    return $self->{_content};
}

sub error {
    my ( $self, $error ) = @_;
    $self->{_error} = $error if $error;
    return $self->{_error};
}

sub meta {
    return $_[0]->{_meta};
}

sub parse_http_response  {
    my ( $self, $response ) = @_;
    
    croak 'invalid response given'
        unless( $response && ref $response && ref $response eq 'HTTP::Response' );
    # set the status according to the response code of the http_response
    if ( $response->code == HTTP::Status::HTTP_UNAUTHORIZED ||  $response->code == HTTP::Status::HTTP_FORBIDDEN ) {
        $self->{_status} = InfluxDB::Client::Response::Status::AUTH;
    } elsif ( $response->code >= 400 ) {
        $self->{_status} = InfluxDB::Client::Response::Status::SRVFAIL;
    } else {
        $self->{_status} = InfluxDB::Client::Response::Status::OK;
    }
    
    # parse our content
    # we leave it up to the calling function to normalize the response data if needed
    $self->content($response->content()) if $response->content();
    
    # set the http_code to the value from the response
    $self->{_http_code} = $response->code;
    
    # we read some meta values and store them in the Meta Object
    $self->meta->influxdb_version($response->header('X-Influxdb-Version'));
    $self->meta->request_id($response->header('Request-Id'));
    $self->meta->server($response->header('Client-Peer'));
    
    # return the status, so the caller could check if the parsing worked
    return $self->status;
}


1;

__END__

=pod

=encoding UTF-8

=head1 NAME

InfluxDB::Client::Response - response class of the InfluxDB::Client

=head1 VERSION

version 0.1

=head1 METHODS

=head2 new(%args)

creates a new C<InfluxDB::Client::Response> instance.

The arguments are:

=over 4

=item *

C<http_response> the L<HTTP::Response> object returned from a request to the InfluxDB

=item *

C<status> the L<HTTP::Response> object returned from a request to the InfluxDB

=item *

C<content> the L<HTTP::Response> object returned from a request to the InfluxDB

=back

If C<http_response> is given, all other arguments are ignored.

C<status> will default to L<InfluxDB::Client::Response::Status::OK>.

If C<status> is L<InfluxDB::Client::Response::Status::OK> and C<content> is a string, C<content> will be parsed as JSON.

=head2 status()

B<returns:> the L<InfluxDB::Client::Response::Status> value of the response

=head2 status($status)

Sets the status to C<$status>

B<returns:> the new L<InfluxDB::Client::Response::Status> value of the response

=head2 http_code()

B<returns:> the original code of the L<HTTP::Response> if this object was
created or updated with a valid L<HTTP::Response>

=head2 content()

B<returns:> the content of the response

=head2 content($content)

Sets the content to C<$content>

If C<$content> is a string, it will be parsed with L<JSON>.
If any parsing errors occour, the result will be undef,
the status will be set to L<InfluxDB::Client::Response::Status::INTFAIL>
and C<error()> will return the full error message of L<JSON>

If C<$content> is a reference, it will just stored as the new content without further processing

B<returns:> the new content of the response

=head2 error()

B<returns:> the error string of the object

=head2 error($error)

Sets the error string to C<$status>

B<returns:> the new error string of the object

=head2 meta()

B<returns:> the L<InfluxDB::Client::Response::Meta> Object of the response

=head2 parse_http_response($response)

Parses the given C<$response>, and updates the object

C<$response> must be a L<HTTP::Response> object

B<returns:> the new L<InfluxDB::Client::Response::Status> value of the response

=head1 AUTHOR

Thomas Berger <thomas.berger@1und1.de>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2017 by 1&1 Telecommunication SE.

This is free software, licensed under:

  The Apache License, Version 2.0, January 2004

=cut
