package InfluxDB::Client::DataSet;
# ABSTRACT: simple container to hold and prepare data for C<InfluxDB::Client->write()>


use strict;
use warnings;

use Carp;
use Time::HiRes;
use InfluxDB::LineProtocol;

our $VERSION = '0.1';

sub new {
    my ($class, %args );
    # check if we have only one argument
    if ( @_ == 2 ) {
        my $measurement;
        ( $class, $measurement ) = @_;
        $args{measurement} = $measurement;
    } else {
        # else we assume we got an hash
        ( $class, %args ) = @_;
    }

    # check the precision
    $args{precision} //= 'ns';
    
    # set our timestamp function
    my $get_ts;
    {
        ## no critic qw(TestingAndDebugging::ProhibitNoStrict)
        no strict 'refs';
        my $ts_func = 'InfluxDB::LineProtocol::ts_' . $args{precision};
        *{"$class\::get_ts"} = \&$ts_func;
    }

    # set the timestamp
    $args{timestamp} //= $class->get_ts();
    
    # validate the measurement parameter
    croak 'no measurement given'
        unless defined  $args{measurement};
    croak 'measurement must not be a reference'
        if ( ref $args{measurement} );
    
    # validate the values parameters
    if (exists $args{values} && defined $args{values}) {
        croak 'values must be a hash reference'
            unless (ref $args{values} && ref $args{values} eq 'HASH');
    }
    
    # validate the tags parameters
    if (exists $args{tags} && defined $args{tags}) {
        croak 'values must be a hash reference'
            unless (ref $args{tags} && ref $args{tags} eq 'HASH');
    }
    
    # construct $self
    my $self = bless {
        _measurement => $args{measurement},
        _values      => $args{values} // {},
        _tags        => $args{tags} // {},
        _precision   => $args{precision},
        _ts          => $args{timestamp}
    } => $class;
    return $self;
}


sub measurement {
    my ( $self, $m ) = @_;
    if ( $m ) {
        croak 'measurement must not be a reference'
            if ( ref $m );
        $self->{_measurement} = $m;
    }
    return $self->{_measurement};
}


sub value {
    $_[0]->_set_tag_or_values('values',@_);
}

sub tag {
    $_[0]->_set_tag_or_values('tags',@_);
}

sub precision {
    return $_[0]->{_precision};
}

sub ts {
    my ( $self, $ts ) = @_;
    $self->{_ts} = $ts if $ts;
    return $self->{_ts};
}

sub set_ts {
    my ( $self, $ts ) = @_;
    $self->ts($ts // $self->get_ts);
}



sub as_string {
    my ( $self ) = @_;
    return InfluxDB::LineProtocol::data2line($self->{_measurement}, $self->{_values}, $self->{_tags}, $self->{_ts});
}

# helper for the tag() and value() functions
sub _set_tag_or_values {
    my $self = shift;
    my $elem = shift;
    # check if we have a single paramter
    if ( @_ == 1 ) {
        my ($p) = @_;
        # if the paramter is a hasreference, we override our internal storage
        if (ref $p ) {
            croak "either a $elem key or a hash reference wanted"
                unless (ref $p eq 'HASH');
            # copy the hash reference
            $self->{'_'.$elem} = { %$p };
        } else {
            # we just fetch the element and return it
            return $self->{'_'.$elem}{$p} if ( exists $self->{'_'.$elem}{$p} );
            # return undef if the key does not exist
            return;
        }
    } else {
        my ( %pairs ) = @_;
        foreach my $k ( keys %pairs ) {
            $self->{'_'.$elem}{$k} = $pairs{$k};
        }
    }
    # we return a copy of our internal storage
    return { %{$self->{'_'.$elem}} }
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

InfluxDB::Client::DataSet - simple container to hold and prepare data for C<InfluxDB::Client->write()>

=head1 VERSION

version 0.1

=head1 SYNOPSIS

    use InfluxDB::Client;
    use InfluxDB::Client::DataSet;

    my $dataset1 = InfluxDB::Client::DataSet->new('measurement');
    $dataset1->value(cost => 42, funfactor => 1000);
    $dataset1->value(available => 3);
    $dataset1->tag(tag => 'wanted');
    
    # does the same as above, but in one call
    my $dataset2 = InfluxDB::Client::DataSet->new(
        measurement => 'measurement',
        values => {
            cost        => 42,
            funfactor   => 1000,
            available   => 3
        },
        tags => {
            tag => 'wanted'
        }
    );

=head1 METHODS

=head2 new($measurement)

creates a new, empty L<InfluxDB::Client::DataSet> for the given measurement.

The current timestamp is saved on creation

=head2 new(%args)

creates a new L<InfluxDB::Client::DataSet>

The arguments are:

=over 4

=item *

C<measurement> name of the measurement the data should be written to

=item *

C<values> a hash reference of the values to write

=item *

C<tags> a hash reference of the tags to write

=item *

C<precision> defines the precision of the given/generated timestamp value, may be one of C<ns>,C<u>,C<ms>,C<s>,C<m>,C<h>. Defaults to C<ns>

=item *

C<timestamp> an integer representing the epoch in the given precision

=back

=head2 measurement()

B<returns:> the measurement

=head2 measurement($measurement)

Sets the measurement to C<$measurement>

B<returns:> the new measurement

=head2 value()

B<returns:> a hash reference of values

=head2 value($key)

B<returns:> the value with the given C<$key>

=head2 value(\%values)

override the current values with the given hash reference

B<returns:> a hash reference of values

=head2 value(%values)

sets the given values

B<returns:> a hash reference of values

=head2 tag()

B<returns:> a hash reference of tags

=head2 tag($key)

B<returns:> the tag with the given C<$key>

=head2 tag(\%tags)

override the current tags with the given hash reference

B<returns:> a hash reference of tags

=head2 tag(%tags)

sets the given tags

B<returns:> a hash reference of tags

=head2 precision()

B<returns:> a precision of the dataset

=head2 ts()

B<returns:> the timestamp of the DataSet

=head2 ts($timestamp)

sets the timestamp to C<$timestamp>

B<returns:> the new timestamp of the DataSet

=head2 set_ts($timestamp)

sets the timestamp to C<$timestamp>. If the C<$timestamp> is empty,
the equivalent of I<now> is used. 

using this method is the same as C<$ds-E<gt>ts($timestamp // $ds-E<gt>get_ts);>

B<returns:> the new timestamp of the DataSet

=head2 get_ts()

B<returns:> the current timestamp in the precision given on construction if the object

I<this method is a coderef to the according function in InfluxDB::LineProtocol and generated within the constructor>

=head2 as_string()

B<returns:> the dataset in the InfluxDB line protocol format

=head1 AUTHOR

Thomas Berger <thomas.berger@1und1.de>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2017 by 1&1 Telecommunication SE.

This is free software, licensed under:

  The Apache License, Version 2.0, January 2004

=cut
