package Spp::Optimizer;

=head1 NAME

Spp::Optimizer - Optimizer of match data

=head1 VERSION

Version 0.01

=cut

our $VERSION = '0.02';


=head1 SYNOPSIS

Optimizer the match data structure to AST of Spp

    use Spp::Optimizer qw(opt_atom);

    my $ast = opt_atom(match_rule($str, $rule));

=head1 EXPORT

opt_atom

=cut

use Exporter;
our @ISA = qw(Exporter);
our @EXPORT_OK = qw(opt_atom);

use 5.020;
use Carp qw(croak);
use experimental qw(switch autoderef);

use Spp::Tools;

#######################
# Optimizer Ast
#######################

sub opt_exprs {
  my $atoms = shift;
  my $atoms_opt = [];
  for my $atom (values $atoms) {
    push $atoms_opt, opt_atom($atom);
  }
  return add_exprs($atoms_opt);
}

sub opt_atom {
  my $atom = shift;
  return ['true'] if $atom eq '{}';
  my ($name, $value) = @{$atom};
  # see $name; exit;
  if (is_perl_array($name)) {
    my $opt_atoms = opt_atoms($atom);
    return add_exprs($opt_atoms);
  }
  given ($name) {
    when ('sym')     { return opt_sym($value)    }
    when ('char')    { return opt_char($value)   }
    when ('Str')     { return opt_str($value)    }
    when ('Keyword') { return opt_str($value)    }
    when ('String')  { return opt_string($value) }
    when ('List')    { return opt_list($value)   }
    when ('Array')   { return opt_array($value)  }
    when ('Sarray')  { return opt_sarray($value) }
    when ('Hash')    { return opt_hash($value)   }
    when ('Pair')    { return opt_atoms($value)  }
    when ('Rule')    { return opt_rule($value)   }
    when ('Group')   { return opt_group($value)  }
    when ('Strs')    { return opt_strs($value)   }
    when ('Chclass') { return opt_chclass($value)}
    when ('Action')  { return opt_atom($value)   }
    when ('Alias')   { return opt_alias($value)  }
    when ('chars')   { return $value             }
    when ('int')     { return $value + 0         }
    when ('any')     { return ['any']            }
    when ('dot')     { return ['dot']            }
    when ('ctoken')  { return ['ctoken', $value] }
    when ('rtoken')  { return ['rtoken', $value] }
    when ('gtoken')  { return ['gtoken', $value] }
    when ('assert')  { return ['assert', $value] }
    when ('cclass')  { return ['cclass', $value] }
    # if String only have blank interlatation
    when ('inter')   { return ''                 }
    see $atom;
    default { error("Unknown type to optimizer: $name") }
  }
}

sub opt_sym {
  my $name = shift;
  given ($name) {
    when ('nil')  { return ['nil'] }
    when ('true')  { return ['true'] }
    when ('false') { return ['false'] }
    default { return ['sym', $name] }
  }
}

sub opt_str_char {
  my $value = shift;
  my $char = tail($value);
  given ($char) {
    when ('n') { return "\n" }
    when ('r') { return "\r" }
    when ('t') { return "\t" }
    default { return $char }
  }
}

sub opt_char {
  my $char = shift;
  return opt_str_char($char);
}

sub opt_sarray {
  my $ast = shift;
  my ($strs, $str) = ([], '');
  for my $node (values $ast) {
    my ($name, $value) = @{ $node };
    given ($name) {
      when ('cstr') { $str = $str . $value }
      when ('char') { $str = $str . opt_str_char($value) }
      when ('blank') {
        push $strs, $str if len($str) > 0;
        $str = '';
      }
      default { error("Unknown Aarray node name: $name") }
    }
  }
  push $strs, $str if len($str) > 0;
  return ['array', $strs];
}

sub opt_str {
  my $ast = shift;
  return '' if $ast eq "''";
  my $chars = [];
  for my $node (values $ast) {
    my ($name, $value) = @{$node};
    if ($name eq 'char') {
      push $chars, opt_str_char($value);
    } else {
      push $chars, $value;
    }
  }
  return perl_join($chars);
}

# my $ast = [["dstr","str"]];
# see opt_string($ast);
sub opt_string {
  my $ast = shift;
  return '' if $ast eq '""';
  my $nodes = [];
  for my $node (values $ast) {
    my ($name, $value) = @{$node};
    given ($name) {
      when ('char') {
        my $char_str = opt_str_char($value);
        push $nodes, $char_str;
      }
      when ('dstr') {
        push $nodes, $value;
      }
      default {
        push $nodes, opt_atom($node);
      }
    }
  }
  my $str_nodes = combin_str_nodes($nodes);
  if (len($str_nodes) == 1 and is_str($nodes->[0]) ) {
    return first($str_nodes);
   }
  return ['string', $str_nodes];
}

sub combin_str_nodes {
  my $nodes = shift;
  my $str_nodes = [];
  my ($str, $str_mode) = ('', 0);
  for my $node (values $nodes) {
    if (is_str($node)) {
      $str = $str . tail($node);
      $str_mode = 1;
    }
    else {
      if ($str_mode == 1) {
        push $str_nodes, $str;
      }
      push $str_nodes, $node;
      $str = '';
      $str_mode = 0;
    }
  }
  if ($str_mode == 1) {
    push $str_nodes, $str;
  }
  return $str_nodes;
}

sub opt_hash {
  my $hash = shift;
  # hash would saved as array to contain keys value
  return ['hash', []] if $hash eq '{}';
  my $hash_opt = [];
  for my $pair (values $hash) {
    if (first($pair) eq 'Pair') {
      my $pair_value = tail($pair);
      my $opt_pair = opt_atoms($pair_value);
      push $hash_opt, $opt_pair;
    } else {
      error("Hash key name not is <Pair>");
    }
  }
  return ['hash', $hash_opt];
}

sub opt_atoms {
  my $atoms = shift;
  my $atoms_opt = [];
  for my $atom (values $atoms) {
    push $atoms_opt, opt_atom($atom);
  }
  return $atoms_opt;
}

sub opt_list {
  my $atoms = shift;
  return ['list', []] if $atoms eq '()';
  my $atoms_opt = opt_atoms($atoms);
  return ['list', $atoms_opt];
}

sub opt_array {
  my $atoms = shift;
  return ['array', []] if $atoms eq '[]';
  my $atoms_opt = opt_atoms($atoms);
  return ['array', $atoms_opt];
}

sub opt_rule {
  my $atoms = shift;
  return ['true'] if $atoms eq ':{}';
  my $rule_value = opt_token($atoms);
  return ['rule', $rule_value];
}

sub opt_group {
  my $ast = shift;
  # blank group would return true :{ () }
  return ['true'] if $ast eq '()';
  my $atoms = opt_token_atoms($ast);
  if (in(first($atoms), ['lbranch', 'branch'])) {
    return $atoms;
  }
  return ['group', $atoms];
}

sub opt_token {
  my $args = shift;
  my $atoms = opt_token_atoms($args);
  if (in(first($atoms), ['lbranch', 'branch'])) {
    return $atoms;
  }
  return first($atoms) if len($atoms) == 1;
  return ['token', $atoms];
}

#my $data = [["assert","^"],["Group",[["rtoken","ws"],["branch","||"],["ctoken","atom"]]],["rept","+"],["assert",'$']];
#my $opt_data = opt_token_atoms($data);
#see $opt_data;
#exit;
sub opt_token_atoms {
  my $atoms = shift;
  # see $atoms;
  $atoms = delete_branch($atoms);
  # see $atoms;
  my $lb_flag = ['lbranch','|'];
  if (in($lb_flag, $atoms)) {
    my $branch = gather_branch($atoms, $lb_flag);
    # see $branch;
    return opt_lbranch($branch);
  }
  my $b_flag = ['branch', '||'];
  if (in($b_flag, $atoms)) {
    # say 'atoms include b-flag';
    my $branch = gather_branch($atoms, $b_flag);
    my $opt_branch = opt_branch($branch);
    # see $opt_branch;
    return $opt_branch;
  }
  my ($index, $opt_atoms) = (0, []);
  while ($index < len($atoms)) {
    my $look = next_atom_is_look($atoms, $index);
    my $rept = next_atom_is_rept($atoms, $index);
    if ($look) {
      # see $look;
      push $opt_atoms, $look;
      $index += 3;
    } elsif ($rept) {
      push $opt_atoms, $rept;
      $index += 2;
    } else {
      my $atom = $atoms->[$index];
      push $opt_atoms, opt_atom($atom);
      $index += 1;
    }
  }
  return $opt_atoms;
}

sub delete_branch {
  my $ast = shift;
  my $ast_len = len($ast);
  my ($index, $flag_pass) = (0, 0);
  my $atoms = [];
  while ($index < $ast_len) {
    my $atom = $ast->[$index];
    $index++;
    if (in(first($atom), ['lbranch', 'branch'])) {
      if ($flag_pass == 1 and $index < $ast_len) {
        $flag_pass = 0;
        push $atoms, $atom;
      }
    }
    else {
      $flag_pass = 1;
      push $atoms, $atom;
    }
  }
  return $atoms;
}

sub gather_branch {
  # see [@_];
  my ($atoms, $branch_atom) = @_;
  my $branch = [];
  my $flag = 0;
  my $opt_atoms = [];
  for my $atom (values $atoms) {
    # see $atom;
    if (is_same($atom, $branch_atom)) {
      push $opt_atoms, $branch;
      $branch = [];
      $flag = 0;
    } else {
      push $branch, $atom;
      $flag = 1;
    }
  }
  push $opt_atoms, $branch if $flag == 1;
  # see $opt_atoms;
  return $opt_atoms;
}

sub opt_lbranch {
   my $ast   = shift;
   my $atoms = [];
   for my $branch (values $ast) {
     push $atoms, opt_token($branch);
   }
   return ['lbranch', $atoms];
}

sub opt_branch {
   my $ast     = shift;
   my $atoms   = [];
   for my $branch (values $ast) {
     push $atoms, opt_token($branch);
   }
   return ['branch', $atoms];
}

sub next_atom_is_rept {
   my ( $atoms, $atom_pos ) = @_;
   if ( $atom_pos < len($atoms) - 1 ) {
      my $next_atom = $atoms->[ $atom_pos + 1 ];
      if ( first($next_atom) eq 'rept' ) {
        my $atom = @{$atoms}[$atom_pos];
        my $atom_opt = opt_atom($atom);
        my $rept_opt = opt_rept($next_atom->[1]);
        return ['rept', [$atom_opt, $rept_opt]];
      }
   }
   return 0;
}

sub next_atom_is_look {
   my ( $atoms, $index ) = @_;
   return 0 if $index >= len($atoms) - 1;
   my $next_atom = $atoms->[$index+1];
   return 0 if first($next_atom) ne 'look';
   my $first_atom = $atoms->[$index];
   my $look_atom = $atoms->[ $index + 2 ];
   my $atom_opt = opt_atom($first_atom);
   my $rept_opt = opt_rept($next_atom->[1]);
   my $look_opt = opt_atom($look_atom);
   return ['look', [$atom_opt, $rept_opt, $look_opt]];
}

sub opt_alias {
  my $ast = shift;
  my $alias_name = $ast->[0][1];
  my $alias_atom = opt_atom($ast->[1]);
  return ['alias', [$alias_name, $alias_atom]];
}

sub opt_strs {
  my $ast = shift;
  my $str_list = [];
  foreach my $node (values $ast) {
    if ( is_str($node) ) {
      push $str_list, $node;
    }
  }
  return ['strs', $str_list];
}

sub opt_rept {
   my $str = shift;
   given ($str) {
     when ('?')  { return [ 0,  1, $str ] }
     when ('*')  { return [ 0, -1, $str ] }
     when ('+')  { return [ 1, -1, $str ] }
     when ('*?') { return [ 0, -1, $str ] }
     when ('+?') { return [ 1, -1, $str ] }
     default { error("Unkown Rept value: $str") }
   }
}

sub opt_chclass {
  my $ast = shift;
  my $atoms = [];
  for my $node (values $ast) {
    push $atoms, opt_chclass_node($node);
  }
  return ['chclass', $atoms];
}

sub opt_chclass_node {
  my $node = shift;
  my ($name, $value) = @{$node};
  given ($name) {
    when ('flip')   { return ['flip'] }
    when ('cchar')  { return $value }
    when ('char')   { return $value }
    when ('cclass') { return ['cclass', $value] }
    when ('range')  { return ['range', $value] }
    default { error("Unknown char class node: $name") }
  }
}

=head1 AUTHOR

Michael Song, C<< <10435916 at qq.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-spp at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Spp>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Spp::Optimizer

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Spp>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Spp>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Spp>

=item * Search CPAN

L<http://search.cpan.org/dist/Spp/>

=back

=head1 ACKNOWLEDGEMENTS

=head1 LICENSE AND COPYRIGHT

Copyright 2015 Michael Song.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=cut

1; # End of Spp::Optimizer
