# ************************************************************************* 
# Copyright (c) 2014, SUSE LLC
# 
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
# 
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
# 
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
# 
# 3. Neither the name of SUSE LLC nor the names of its contributors may be
# used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
# ************************************************************************* 
#
# tests to see if parser works for various commands

#!perl
use 5.012;
use strict;
use warnings FATAL => 'all';

use App::Dochazka::CLI::Parser qw( parse );
use App::Dochazka::CLI::CommandMap qw( $semantic_tree );
use Data::Dumper;
use Test::More;

my ( $cmd, $r, $r_should_be );

$cmd = "GET employee nick worker";
$r = parse( $cmd );
is_deeply( $r, {
    'th' => {
              'GET' => 'GET',
              '_TERM' => 'worker',
              '_REST' => '',
              'EMPLOYEE' => 'employee',
              'NICK' => 'nick'
            },
    'ts' => [
              'GET',
              'EMPLOYEE',
              'NICK',
              '_TERM'
            ],
    'nc' => 'GET EMPLOYEE NICK _TERM'
} );

$cmd = "ACTIVITY";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'ACTIVITY'
            ],
    'th' => {
              'ACTIVITY' => 'ACTIVITY',
              '_REST' => ''
            },
    'nc' => 'ACTIVITY'
} );

$cmd = "ACTIVITY ALL";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'ACTIVITY',
              'ALL'
            ],
    'th' => {
              'ACTIVITY' => 'ACTIVITY',
              'ALL' => 'ALL',
              '_REST' => ''
            },
    'nc' => 'ACTIVITY ALL'
} );

$cmd = "ACTIVITY ALL DISABLED";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'ACTIVITY',
              'ALL',
              'DISABLED'
            ],
    'th' => {
              'ACTIVITY' => 'ACTIVITY',
              'ALL' => 'ALL',
              'DISABLED' => 'DISABLED',
              '_REST' => ''
            },
    'nc' => 'ACTIVITY ALL DISABLED'
} );

$cmd = "EMPLOYEE";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'EMPLOYEE',
            ],
    'th' => {
              'EMPLOYEE' => 'EMPLOYEE',
              '_REST' => ''
            },
    'nc' => 'EMPLOYEE'
} );

$cmd = "EMPLOYEE PROFILE";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'EMPLOYEE',
              'PROFILE'
            ],
    'th' => {
              'EMPLOYEE' => 'EMPLOYEE',
              'PROFILE' => 'PROFILE',
              '_REST' => ''
            },
    'nc' => 'EMPLOYEE PROFILE'
} );

$cmd = "EMPLOYEE SHOW";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'EMPLOYEE',
              'SHOW'
            ],
    'th' => {
              'EMPLOYEE' => 'EMPLOYEE',
              'SHOW' => 'SHOW',
              '_REST' => ''
            },
    'nc' => 'EMPLOYEE SHOW'
} );

$cmd = "EMPLOYEE=worker";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'EMPLOYEE_SPEC',
            ],
    'th' => {
              'EMPLOYEE_SPEC' => 'EMPLOYEE=worker',
              '_REST' => ''
            },
    'nc' => 'EMPLOYEE_SPEC'
} );

$cmd = "EMPLOYEE=worker PROFILE";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'EMPLOYEE_SPEC',
              'PROFILE'
            ],
    'th' => {
              'EMPLOYEE_SPEC' => 'EMPLOYEE=worker',
              'PROFILE' => 'PROFILE',
              '_REST' => ''
            },
    'nc' => 'EMPLOYEE_SPEC PROFILE'
} );

$cmd = "EMPLOYEE=worker SHOW";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'EMPLOYEE_SPEC',
              'SHOW'
            ],
    'th' => {
              'EMPLOYEE_SPEC' => 'EMPLOYEE=worker',
              'SHOW' => 'SHOW',
              '_REST' => ''
            },
    'nc' => 'EMPLOYEE_SPEC SHOW'
} );

$cmd = "EMPLOYEE SET SEC_ID foobar";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'EMPLOYEE',
              'SET',
              'SEC_ID',
              '_TERM'
            ],
    'th' => {
              'SEC_ID' => 'SEC_ID',
              'SET' => 'SET',
              'EMPLOYEE' => 'EMPLOYEE',
              '_TERM' => 'foobar',
              '_REST' => ''
            },
    'nc' => 'EMPLOYEE SET SEC_ID _TERM'
} );

$cmd = "EMPLOYEE SET FULLNAME Johannes Runner";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'EMPLOYEE',
              'SET',
              'FULLNAME'
            ],
    'th' => {
              'EMPLOYEE' => 'EMPLOYEE',
              'SET' => 'SET',
              'FULLNAME' => 'FULLNAME',
              '_REST' => 'Johannes Runner'
            },
    'nc' => 'EMPLOYEE SET FULLNAME'
} );

$cmd = "EMPLOYEE=orc63 SET SEC_ID foobar";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'EMPLOYEE_SPEC',
              'SET',
              'SEC_ID',
              '_TERM'
            ],
    'th' => {
              'SEC_ID' => 'SEC_ID',
              'SET' => 'SET',
              'EMPLOYEE_SPEC' => 'EMPLOYEE=orc63',
              '_TERM' => 'foobar',
              '_REST' => ''
            },
    'nc' => 'EMPLOYEE_SPEC SET SEC_ID _TERM'
} );

$cmd = "EMPLOYEE=orc63 SET FULLNAME Just Another Orc in the Rye";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'EMPLOYEE_SPEC',
              'SET',
              'FULLNAME'
            ],
    'th' => {
              'EMPLOYEE_SPEC' => 'EMPLOYEE=orc63',
              'SET' => 'SET',
              'FULLNAME' => 'FULLNAME',
              '_REST' => 'Just Another Orc in the Rye'
            },
    'nc' => 'EMPLOYEE_SPEC SET FULLNAME'
} );

$cmd = "SCHEDULE DUMP";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'SCHEDULE',
              'DUMP',
            ],
    'th' => {
              'SCHEDULE' => 'SCHEDULE',
              'DUMP' => 'DUMP',
              '_REST' => '',
            },
    'nc' => "SCHEDULE DUMP"
} );

$cmd = "SCHEDULE MEMORY";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'SCHEDULE',
              'MEMORY',
            ],
    'th' => {
              'SCHEDULE' => 'SCHEDULE',
              'MEMORY' => 'MEMORY',
              '_REST' => '',
            },
    'nc' => "SCHEDULE MEMORY"
} );

$cmd = "SCHEDULE NEW";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'SCHEDULE',
              'NEW',
            ],
    'th' => {
              'SCHEDULE' => 'SCHEDULE',
              'NEW' => 'NEW',
              '_REST' => '',
            },
    'nc' => "SCHEDULE NEW"
} );

$cmd = "SCHEDULE MON 33:33 TUE 23:00";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'SCHEDULE',
              '_DOW',
              '_TIME',
              '_DOW1',
              '_TIME1',
            ],
    'th' => {
              'SCHEDULE' => 'SCHEDULE',
              '_DOW' => 'MON',
              '_TIME' => '33:33',
              '_DOW1' => 'TUE',
              '_TIME1' => '23:00',
              '_REST' => '',
            },
    'nc' => "SCHEDULE _DOW _TIME _DOW1 _TIME1"
} );

$cmd = "SCHEDULE MON 33:33 - TUE 23:00";
$r = parse( $cmd );
is_deeply( $r, {
    'ts' => [
              'SCHEDULE',
              '_DOW',
              '_TIME',
              '_HYPHEN',
              '_DOW1',
              '_TIME1',
            ],
    'th' => {
              'SCHEDULE' => 'SCHEDULE',
              '_DOW' => 'MON',
              '_TIME' => '33:33',
              '_DOW1' => 'TUE',
              '_HYPHEN' => '-',
              '_TIME1' => '23:00',
              '_REST' => '',
            },
    'nc' => "SCHEDULE _DOW _TIME _HYPHEN _DOW1 _TIME1"
} );

$cmd = "SCHEDULE MON 33:33-23:00";
$r_should_be = {
    'ts' => [
              'SCHEDULE',
              '_DOW',
              '_TIMERANGE',
            ],
    'th' => {
              'SCHEDULE' => 'SCHEDULE',
              '_DOW' => 'MON',
              '_TIMERANGE' => '33:33-23:00',
              '_REST' => '',
            },
    'nc' => "SCHEDULE _DOW _TIMERANGE"
};
$r = parse( $cmd );
is_deeply( $r, $r_should_be );

$cmd = "SCHEDULE ALL 33:33-23:00";
$r_should_be = {
    'ts' => [
              'SCHEDULE',
              'ALL',
              '_TIMERANGE',
            ],
    'th' => {
              'SCHEDULE' => 'SCHEDULE',
              'ALL' => 'ALL',
              '_TIMERANGE' => '33:33-23:00',
              '_REST' => '',
            },
    'nc' => "SCHEDULE ALL _TIMERANGE"
};
$r = parse( $cmd );
is_deeply( $r, $r_should_be );

done_testing;
