# ************************************************************************* 
# Copyright (c) 2014, SUSE LLC
# 
# All rights reserved.
# 
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are met:
# 
# 1. Redistributions of source code must retain the above copyright notice,
# this list of conditions and the following disclaimer.
# 
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
# 
# 3. Neither the name of SUSE LLC nor the names of its contributors may be
# used to endorse or promote products derived from this software without
# specific prior written permission.
# 
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.
# ************************************************************************* 
#
# top-level CLI module
#
package App::Dochazka::CLI;

use 5.006;
use strict;
use warnings FATAL => 'all';

use App::CELL qw( $CELL );
use Data::Dumper;
use Exporter 'import';
use Test::More;



=head1 NAME

App::Dochazka::CLI - Dochazka command line client



=head1 VERSION

Version 0.150

=cut

our $VERSION = '0.150';



=head1 SYNOPSIS

Dochazka command line client.

    bash$ dochazka-cli
    Dochazka(2014-08-08) demo> 



=head1 DESCRIPTION

L<App::Dochazka::CLI> is the Command Line Interface (CLI) component of the
Dochazka Attendance & Time Tracking system. 

It implements a number of commands, which can be divided
into three categories:

=over

=item Commands for generating HTTP requests

The CLI enables the user to generate HTTP requests to an L<App::Dochazka::REST>
server and view the server's responses. Commands in this category have a direct
correlation with the REST server resources, always starting with one of the
four basic HTTP methods supported by the REST server: C<GET>, C<PUT>, C<POST>,
C<DELETE>. Commands in this category are useful mainly for Dochazka developers,
but may also be used by administrators and normal users -- e.g., for
troubleshooting and ad hoc testing. Effort has been made to ensure that the CLI
supports all the resources. See L<"Commands for generating HTTP requests">. 

=item Commands for Dochazka administrators

Dochazka administrators have access to all of Dochazka's resources and can
call them directly by generating appropriate HTTP requests. This can be a bit
tedious, however, so "shortcut" commands have been implemented for the more
commonly used administrative procedures. See L<"Commands for Dochazka
administrators">.

=item Commands for Dochazka users

Finally there are the day-to-day commands used by Dochazka users (employees
of the organization) to enter attendance data, view their profile, generate
reports, etc. See L<"Commands for Dochazka users">.

=back


=head1 COMMANDS

=head2 Commands for generating HTTP requests

Commands in this category start with the HTTP method and continue 
with the resource (except it is written with spaces instead of forward
slashes) and concludes with the request entity, if any.

All CLI commands must be written on a single line of input, and
commands in this category are no exception.

Examples:

=over

=item DELETE INTERVAL IID 24

The equivalent of C<< DELETE interval/iid/24 >>

=item POST DOCU TEXT "employee/eid/:eid"

The equivalent of C<< POST docu/text >> with the request entity shown.

=item POST EMPLOYEE EID { "eid" : 15, "sec_id" : 340 }

Update the secondary ID of the employee with EID 15.

=back


=head2 Commands for Dochazka administrators

Dochazka administrators have a frequent need to be able to do things like
change an employee's privilege level or schedule, remove locks, etc. They
also need to be able to view attendance data entered by employees. For these
purposes they need convenient ways to specify the employee they are interested
in.

=head2 Commands for Dochazka supervisors

Supervisors have the ability to view attendance data and other records
belonging to their reports (e.g. employees in their team or department).
These are a subset of the administrative commands.


=head2 Commands for Dochazka administrators

These commands are a subset of the commands introduced in the previous
section - the main difference is that ordinary users can only view and
act on their own attendance records, whereas administrators can do so
on records of any employee.

=head1 PACKAGE VARIABLES AND EXPORTS

=over

=item * C<< $current_date >>

The current date, as determined by C<bin/dochazka-cli>

=item * C<< $current_emp >>

The L<App::Dochazka::Model::Employee> object of the current employee.

=item * C<< $current_priv >>

The privlevel of the current employee.

=item * C<< $debug_mode >>

Tells parser whether to display debug messages

=back

=cut

our ( $current_date, $current_emp, $current_priv, $debug_mode, $message );
our @EXPORT_OK = qw( 
    $current_date 
    $current_emp 
    $current_priv 
    $debug_mode
);

1;


