package Debug::Smart;

use warnings;
use strict;
our $VERSION = '0.002';

use 5.008;
use Carp;
use Path::Class;
use IO::File;
use base qw(Exporter);

our @EXPORT = qw(LOG YAML DUMP CLOSE);
our @EXPORT_OK = qw(TRACE);
my $fh_hashref;
my $arg_hashref;

END {
    _make_temp(_get_fh(), 'fh');
    _make_temp(_get_arg(), 'arg');
    $fh_hashref = undef;
    $arg_hashref = undef;;
    CLOSE();
}

sub import {
    my $package = shift;
    my ($caller_package, $caller_name, $line) = caller(0);
    my $TRUE = 1;
    my $FALSE = 0;
    my $file = file($caller_name);
    my $arg;
    $arg->{-path} = $file->dir;
    $arg->{-name} = "$caller_package.debug_log";
    $arg->{-timestamp} = $FALSE;

    my @symbols = ();
    push @_, @EXPORT;
    while (@_) {
        my $key = shift;
        if ($key =~ /^[-]/) {
            if ($key =~ /-path|-p/) {
                $arg->{$key} = shift;
            }
            elsif ($key =~ /-name|-n/) {
                $arg->{$key} = shift;
            }
            elsif ($key =~ /-timestamp|-ts/) {
                $arg->{$key} = $TRUE;
            }
            elsif ($key =~ /-append/) {
                $arg->{$key} = $TRUE;
            }
            elsif ($key =~ /-trace/) {
                $arg->{$key} = shift;
                push @symbols, 'TRACE';
                _tracefilter($arg);
            }
        }
        else {
            push @symbols, $key;
        }
    }
    _open($caller_package, $arg);
    $arg_hashref->{$caller_package} = $arg;
    _make_temp(undef, 'fh');
    _make_temp(undef, 'arg');
    Exporter::export($package, $caller_package, @symbols);
}

sub _tracefilter {
    my $arg = shift;
    require Filter::Util::Call;
    my $done = 0;
    Filter::Util::Call::filter_add(
        sub {
            return 0 if $done;
            my ($data, $end) = ('', '');
            while (my $status = Filter::Util::Call::filter_read()) {
                return $status if $status < 0;
                if (/^__(?:END|DATA)__\r?$/) {
                    $end = $_;
                    last;
                }
                $data .= $_;
                $_ = '';
            }
            $_ = $data;
            my $target = $arg->{-trace};
            if (ref $target eq 'ARRAY') {
                foreach my $val (@{$target}) {
                    my $name = "'$val'";
                    my $escape = '\\' . $val;
                    s{([^;]*$escape[^;]*;)}{$1TRACE $name => $val;}gm;
                }
            }
            elsif (not ref $target) {
                my $name = "'$target'";
                my $escape = '\\' . $target;
                s{([^;]*$escape[^;]*;)}{$1TRACE $name => $target;}gm;
            }
            else {
                croak 'you must use SCALAR or ARRAY REF : ' . ref $target;
            }
            $done = 1;
        }
    );
}

sub _open {
    my ($caller_package, $arg) = @_;
    my $mode = $arg->{-append} ? 'a' : 'w';
    my $fh = IO::File->new("$arg->{-path}/$arg->{-name}", $mode) or
        croak "IO::File can't open the file : "
        . $arg->{-path} . " name : " . $arg->{-name};

    $fh_hashref->{$caller_package} = $fh;
}


sub LOG {
    my $value = shift;
    my $arg = _temp_arg() || _get_arg();
    _make_temp(undef, 'arg');
    $value = '[' . localtime(time) . ']' . $value if $arg->{-timestamp};
    my $fh = _temp_fh() || _get_fh() ;
    _make_temp(undef, 'fh');
    print $fh "$value\n" or croak "Can't print value.";
    $fh->flush;
    return $value;
}

sub DUMP {
    my $message = shift;
    eval "require Data::Dumper";
    croak "Data::Dumper is not installed" if $@;

    my $fh = _temp_fh() || _get_fh();
    _make_temp($fh, 'fh');
    my $arg = _temp_arg() || _get_arg();
    _make_temp($arg, 'arg');
    LOG("[$message #DUMP]");
    print $fh Data::Dumper::Dumper(@_) or croak "Can't print value.";
    $fh->flush;
    return wantarray ? @_ : $_[0];
}

sub YAML {
    my $message = shift;
    eval "require YAML";
    croak "YAML is not installed." if $@;

    my $fh = _temp_fh() || _get_fh();
    _make_temp($fh, 'fh');
    my $arg = _temp_arg() || _get_arg();
    _make_temp($arg, 'arg');
    LOG("[$message #YAML]");
    print $fh YAML::Dump(@_) or croak "Can't print value.";
    $fh->flush;
    return wantarray ? @_ : $_[0];
}

sub TRACE {
    my ($name, $value) = @_;
    my ($caller_package, $caller_name, $line) = caller(0);
    my $message = "TRACE name:$name line:$line";
    my $type = ref $value;
    _make_temp(_get_fh(), 'fh');
    _make_temp(_get_arg(), 'arg');
    if ($type eq 'SCALAR' || $type eq '') {
        LOG("[$message] $value");
    }
    elsif ($type eq 'ARRAY' || $type eq 'HASH' || $type eq 'REF') {
        DUMP($message, $value);
    }
    elsif ($type eq 'CODE' || $type eq 'GLOB') {
        croak "can't trace CODE or GLOB type.";
    }
    else {
        DUMP($message, $value);
    }
}

sub CLOSE {
    my $fh = _temp_fh() || _get_fh();
    _make_temp(undef, 'fh');
    if ($fh) {
        $fh->close;
    }
    my $arg = _temp_arg() || _get_arg();
    _make_temp(undef, 'arg');
    if ($arg && -z "$arg->{-path}/$arg->{-name}") {
        unlink "$arg->{-path}/$arg->{-name}";
    }
}

sub _get_fh {
    my $caller_package = caller(1);
    return $fh_hashref->{$caller_package};
}

sub _get_arg {
    my $caller_package = caller(1);
    return $arg_hashref->{$caller_package};
}

sub _make_temp {
    my ($value, $type) = @_;
    no strict 'refs';
    no warnings;
    *{"Debug::Smart::_temp_$type"} = sub {
        return $value; 
    };
}


=head1 NAME

Debug::Smart - Debug messages for smart logging to the file 

=head1 VERSION

version 0.002

=cut

=head1 SYNOPSIS

    use Debug::Smart -timestamp;
    
    LOG("write a message");
    DUMP("dump the data structures", $arg);
    YAML("dump the data structures back into yaml", $arg)

=head1 DESCRIPTION

B<Debug::Smart> provides debug methods that is easy to use.

This module automatically creates and opens the file for logging.
It is created to location of the file that used this module.
And the name of the file is the namespace + I<.debug_log> with using this module.
It exports a function that you can put just about anywhere in your Perl code to
make it logging.

To change the location or filename, you can use the options.
Please refer to B<OPTIONS> for more information on.

    package Example;

    use Debug::Smart;
    #file name "Example.debug_log"


    package Example;
    
    use Debug::Smart -name => 'mydebug.log';
    #file name "mydebug.log"

B<WARNING:>
This module automatically determines the output location and the filename when
you don't use some options.
You should carefully use it, otherwise the file of same name is overwrited.
And this module uses a source filter.  If you don't like that, don't use this.

=head1 TRACE

You can trace the variables if you use I<-trace> option.
This option specifies the variable's name that is type of B<SCALAR> or B<ARRAY_REF>.
B<TRACE> function is automatically added by source code filter(B<Fillter::Util::Call>) and
outputs the specified variable's value that each time appeared in the source code.

    # you shuld use sigle quote
    use Debug::Smart -trace => '$var';

    my $var = 1;
    $var = 2;
    $var = 10;

done.

    my $var = 1;TRACE $var;
    $var = 2;TRACE $var;
    $var = 10; TRACE $var;

=head1 EXPORT

=over

=item LOG

To write variable to the file.

=item DUMP

To write the variable structures to the file with Data::Dumper::Dumper.

=item YAML 

To write the variable structures to the file with YAML::Dump.

=item TRACE

This function traces valiables.
(TRACE is not export if you don't use I<-trace> option)

=item CLOSE

This function closes the filehandle when you purposefully want to close it.
Normally, this module closes the filehandle when the program ends.

=back

=head1 OPTIONS

    use Debug::Smart -path => '/path/to/';

I<-path> option specify output location of the log file. 

    use Debug::Smart -name => 'filename';

I<-filename> option specify the filename.

    use Debug::Smart -timestamp;

I<-timestamp> option add timestamp to the head of logging message.

    use Debug::Smart -append

I<-append> option is append mode. Writing at end-of-file.
Default is write mode. It will be overwritten.

    use Debug::Smart -trace => '$foo';
or
    use Debug::Smart -trace => ['$foo', '$bar'];

I<-trace> option traces the variable of specified the name. 
You should write the single quoted variable's name.

=head1 SEE ALSO

Filter::Util::Call

=head1 AUTHOR

Kazuhiro Shibuya, C<< <k42uh1r0@gmail.com> >>

=head1 BUGS

Please report any bugs or feature requests to
C<bug-debug-simple@rt.cpan.org>, or through the web interface at
L<http://rt.cpan.org>.  I will be notified, and then you'll automatically be
notified of progress on your bug as I make changes.

=head1 COPYRIGHT & LICENSE

Copyright 2007 Kazuhiro Shibuya, All Rights Reserved.

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut

1; # End of Debug::Smart
