# Copyright (c) Philippe Verdret, 1995-1998
# A very simple template processor 

use strict
require 5.004;

=head1 NAME

Parse::Template - Template Processor (0.3)

=head1 SYNOPSIS

  use Parse::Template;

  my %template = 
    (
     'TOP' =>  q!before <<$template->eval('DATA')>> after!,
     'DATA' => q!Insert data: ! .
               q!1. List: <<"@list$N">>! .
               q!2. Hash: <<"$hash{'key_value'}$N">>! .
               q!3. Sub: <<&SUB()$N>>!
    );
 
  my $tmplt = new Parse::Template (\%template);
  open FH, "< foo";

  $tmplt->env('var' => '(value!)');
  $tmplt->env('list' => [1, 2, 10], 
                'N' => "\n",
              'tmplt' => \$tmplt, 
              'FH' => \*FH,
              'SUB' => sub { "->content generated by a sub<-" },
              'hash' => { 'key_value' => q!It's an hash value! });
  print $tmplt->eval('TOP'), "\n";

=head1 DESCRIPTION

Parse::Template permits evaluating Perl expressions placed within a
template.  These expressions must be surrounded by C<&lt;&lt;> and
C<&gt;&gt;>, and they are evaluated within an environment specific to
each instance of the C<Parse::Template> class.

C<Parse::Template> creates a class specific to each instance.  This
class inherits from the class C<Parse::Template> and contains the
environment that is specific to the template.

The C<env()> method permits constructing the required evaluation
environment.  Each entry must be specified using a key consisting of
the name of the symbol to be created, associated with a reference
whose type is that of entry to be created (for example, a reference to
an array to create an array).  A scalar variable is defined by
declaring the name of the variable, associated with its value.  A
scalar variable containing a reference is defined by writing C<'var'
=&gt; \$variable>, where C<$variable> is a lexical variable that contains
the reference.

This package was initially created to serve as a code generator
for the C<Parse::Lex> class.  You will find examples of its use
in the classes C<Parse::Lex> et C<Parse::CLex>.

=head1 METHODS

=over 4

=item new HASH

Constructor for the class.

=item env HASH

=item env SYMBOL

Permits defining the environment that is specific to a template.
C<env(SYMBOL)> returns the reference associated with the symbol, or
C<undef> if the symbol is not defined.  The reference that is returned
is of the type indicated by the character (C<&, $, %, @, *>) that
prefixes the symbol.

Example:

  $tmplt->env('LIST' => [1, 2, 3])}   Defines a list

  @{$tmplt->env('*LIST')}             Returns the list

  @{$tmplt->env('@LIST')}             Ditto


=item getPart PART_LABEL

Returns the designated part of the template.

=item ppregexp REGEXP

Preprocesses a regular expression so that it can be inserted into a
template where the regular expression delimiter is either the
character "/" or the character "!".

=item setPart PART_LABEL => TEXT

A template is defined by a hash (associative array).  C<setPart()>
permits defining a new entry within this hash.

=item undef

Permits destroying a template instance (see L<BUGS>).

=back

=head1 NOTES CONCERNING THE CURRENT VERSION

This is an experimental module.  Send me your comments.

=head1 BUGS

An instance is not destroyed when the variable within which it
is placed is itself destroyed.  This is because this instance
is also referenced within its class.  Therefore, don't use this
class to create a large number of instances, or if possible,
use the C<undef()> method.

=head1 AUTHOR

Philippe Verdret 

=head1 COPYRIGHT

Copyright (c) 1995-1998 Philippe Verdret. All rights reserved.
This module is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=cut

use strict;
package Parse::Template;
$Parse::Template::VERSION = '0.3';

use constant DEBUG => 0;	

my $sym = 'sym00';
sub genSymbol { $sym++ }	# generate: sym00, sym01, sym02, etc.

sub new {
  my $receiver = shift;
  my $class = genSymbol();
  my $self; 
  if (@_) {
    $self = bless {@_}, $class;
  } else {
    $self = bless {}, $class;
  }
  no strict;
  @{"${class}::ISA"} = ref $receiver || $receiver;
  $self->initialize();	
  $self;
}
sub initialize {
  my $self = shift;
  my $class = ref $self;
  no strict;
  ${"${class}::self"} = '';	# suppress a warning!
  ${"${class}::self"} = $self;
  $self;
}
#sub DESTROY {			
#  my $self = shift;		
#  my $class = ref $self;
#  undef %{"${class}::"};
#}
sub undef {
  my $self = shift;
  my $class = ref $self;
  unless (@_) {
    undef %{"${class}::"};
  } else {}
}
use constant TRACE_ENV => 0;
sub env {
  my $self = shift;
  my $class = ref $self;
  my $symbol = shift;
  no strict;
  if (@_) {
    while (@_) {
      my $value = shift;
      print STDERR "${class}::$symbol\t$value\n" if TRACE_ENV;
      if (ref $value) {
	*{"${class}::$symbol"} = $value;
      } else {			# scalar value
      	*{"${class}::$symbol"} = \$value;
      }
      $symbol = shift if @_;
    }
  } elsif (defined *{"${class}::$symbol"}) { # borrowed from Exporter.pm
    return \&{"${class}::$symbol"} unless $symbol =~ s/^(\W)//;
    my $type = $1;
    return 
      $type eq '&' ? \&{"${class}::$symbol"} :
	$type eq '$' ? \${"${class}::$symbol"} :
	    $type eq '@' ? \@{"${class}::$symbol"} :
	    $type eq '%' ? \%{"${class}::$symbol"} :
	    $type eq '*' ?  *{"${class}::$symbol"} :
	    do { require Carp; Carp::croak("Can't find symbol: $type$symbol") };
  } else {
    undef;
  }
}
# Purpose:  validate the regexp and replace "!" or "/" by "\!" or "\/"
# Arguments: a regexp
# Returns: the preprocessed regexp
sub ppregexp {
  #  my $self = $_[0]; # useless
  my $regexp = $_[1];
  eval { '' =~ /$regexp/ };
  if ($@) {			
    die "$@";			
  }
  $regexp =~ s{
    ((?:\G|[^\\])(?:\\{2,2})*)	# Context before
    ([/!\"])			# Delimiters used
  }{$1\\$2}xg;
  $regexp;
}
sub getPart {		
  my $self = shift;
  my $part = shift;
  $self->{$part};
}
sub setPart {		
  my $self = shift;
  my $part = shift;
  $self->{$part} = shift;
}

sub eval {
  my $self = shift;
  my $class = ref $self;
  my $part = shift;
  my $code = $self->{$part};
  unless (defined $code) {
    die "'$part' template part not defined";
  }
  local $^W = 0 if $^W;
  $code =~ s{<<(.*?)>>}{	# the magic substitution
    if (DEBUG) {
      print STDERR "expression to eval {package $class; $1}\n";
    } 
    qq!package $class; $1!;
  }eegsx;
  die "$@" if $@;
  $code;
}
1;
__END__
