package Archive::Libarchive::API;

use strict;
use warnings;

# ABSTRACT: Comprehensive API documentation for Archive::Libarchive
our $VERSION = '0.03'; # VERSION

# NOTE: This document is autogenerated from the template
# maint/tt/Doc.pm.tt
# Please see maint/README.md for details on updating.

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Archive::Libarchive::API - Comprehensive API documentation for Archive::Libarchive

=head1 VERSION

version 0.03

=head1 SYNOPSIS

 use 5.020;
 use Archive::Libarchive qw( :const );
 
 my $r = Archive::Libarchive::ArchiveRead->new;
 $r->support_filter_all;
 $r->support_format_all;
 $r->open_filename("archive.tar", 10240) == ARCHIVE_OK
   or die $r->error_string;
 
 my $e = Archive::Libarchive::Entry->new;
 say $e->pathname while $r->next_header($e) == ARCHIVE_OK;

=head1 DESCRIPTION

This document covers comprehensive list of methods and constants provided by
L<Archive::Libarchive> classes.  The top level document L<Archive::Libarchive>
is a better place to start, along with the main documentation pages for each
class, as while not comprehensive, do cover in more detail the methods and
features that you are most likely to use in practice, and they include examples.
This document is intended primarily for reference.

This document includes method signatures for each method provided by the API,
but because of the large number of methods, the exact function of all methods
is not provided.  You can usually get the idea from the method names, or
consult the C<libarchive> documentation.  A familiarity with the C<libarchive>
API is therefore helpful.

Because C<libarchive> contains class name prefixes for its methods to provide
a pseudo object oriented interface in C, and because this Perl interface is
properly object oriented, the names of all methods have been changed.  (Mostly
by removing the class prefix, although, some have been modified beyond that
because simply removing the class prefix makes some method names confusing).
To help you find the equivalent C<libarchive> documentation, the original
C<libarchive> function names are included in the method usage.

Some methods are marked as C<(optional)>.  These methods were added to
C<libarchive> after the current minimum supported by L<Archive::Libarchive>,
and therefore might not be available depending on the version of C<libarchive>
you have installed.  If you need your code to work on both old code and to
take advantage of these methods on new versions you can use the C<can> method
to see if these methods are available.

If you need one or more of these methods and your system C<libarchive> is too
old, then you can force a share install of L<Alien::Libarchive3>.

 env ALIEN_INSTALL_TYPE=share cpanm --reinstall Alien::Libarchive3

=head1 Archive::Libarchive

=over 4

=item Main documentation and examples: L<Archive::Libarchive>

=back

=head2 archive_bzlib_version

 # archive_bzlib_version
 my $string = archive_bzlib_version();

The C<bzlib> version that C<libarchive> was built with.  This will return C<undef> if the library was
not found at build time.

=head2 archive_liblz4_version

 # archive_liblz4_version
 my $string = archive_liblz4_version();

The C<liblz4> version that C<libarchive> was built with.  This will return C<undef> if the library was
not found at build time.

=head2 archive_liblzma_version

 # archive_liblzma_version
 my $string = archive_liblzma_version();

The C<liblzma> version that C<libarchive> was built with.  This will return C<undef> if the library was
not found at build time.

=head2 archive_libzstd_version

 # archive_libzstd_version (optional)
 my $string = archive_libzstd_version();

The C<zstd> version that C<libarchive> was built with.  This will return C<undef> if the library was
not found at build time.

=head2 archive_version_details

 # archive_version_details
 my $string = archive_version_details();

Detailed textual name/version of the library and its dependencies. This has the form:

=over 4

=item C<libarchive x.y.z zlib/a.b.c liblzma/d.e.f ... etc ...>

=back

the list of libraries described here will vary depending on how libarchive was compiled.

=head2 archive_version_number

 # archive_version_number
 my $int = archive_version_number();

The C<libarchive> version expressed as an integer.  This will be the major, minor and patch
levels each using up to three digits, so 3.5.1 will be C<3005001>.

=head2 archive_version_string

 # archive_version_string
 my $string = archive_version_string();

The C<libarchive> version as a string.

=head2 archive_zlib_version

 # archive_zlib_version
 my $string = archive_zlib_version();

The C<zlib> version that C<libarchive> was built with.  This will return C<undef> if the library was
not found at build time.

=head2 versions

 my %versions = Archive::Libarchive->versions();

This returns a hash of C<libarchive> and L<Archive::Libarchive> versions and dependency versions.  This
may be useful in a test report diagnostic.

=head1 Archive::Libarchive::Archive

=over 4

=item Main documentation and examples: L<Archive::Libarchive::Archive>

=back

=head2 clear_error

 # archive_clear_error
 $ar->clear_error;

Clear the error for the corresponding archive instance.

=head2 copy_error

 # archive_copy_error
 $ar->copy_error($ar);

=head2 entry

 # archive_entry_new2
 my $e = $ar->entry;

This method creates a new L<Archive::Libarchive::Entry> instance, like when
you create an instance with that class' L<new|Archive::Libarchive::Entry/new>
method, except this form will pull character-set conversion information from
the specified archive instance.

=head2 errno

 # archive_errno
 my $int = $ar->errno;

Returns the system C<errno> code for the archive instance.  For non-system level
errors, this will not have a sensible value.

=head2 error_string

 # archive_error_string
 my $string = $ar->error_string;

Returns a human readable diagnostic of error for the corresponding archive instance.

=head2 file_count

 # archive_file_count
 my $int = $ar->file_count;

=head2 filter_bytes

 # archive_filter_bytes
 my $sint64 = $ar->filter_bytes($int);

=head2 filter_code

 # archive_filter_code
 my $code = $ar->filter_code($num);

This will return the filter code at position C<$num>.  For the total
number of positions see the
L<filter_count method|Archive::Libarchive::API/filter_count>.

The constant prefix for this method is C<ARCHIVE_FILTER_>.  This will
return a dualvar where the string is the lowercase name without the
prefix and the integer is the constant value.  For the full list see
L<Archive::Libarchive::API/CONSTANTS>.

=head2 filter_count

 # archive_filter_count
 my $int = $ar->filter_count;

=head2 filter_name

 # archive_filter_name
 my $string = $ar->filter_name($int);

=head2 format

 # archive_format
 my $code = $ar->format;

This will return the format code at position C<$num>.

The constant prefix for this method is C<ARCHIVE_FORMAT_>.  This will
return a dualvar where the string is the lowercase name without the
prefix and the integer is the constant value.  For the full list see
L<Archive::Libarchive::API/CONSTANTS>.

=head2 format_name

 # archive_format_name
 my $string = $ar->format_name;

=head2 seek_data

 # archive_seek_data
 my $sint64_1 = $ar->seek_data($sint64_2, $int);

=head2 set_error

 # archive_set_error
 $ar->set_error($errno, $string);

This will set the C<errno> code and human readable diagnostic for the archive
instance.  Not all errors have a corresponding C<errno> code, so you can
set that to zero (C<0>) in that case.

=head1 Archive::Libarchive::ArchiveRead

=over 4

=item Main documentation and examples: L<Archive::Libarchive::ArchiveRead>

=item Parent class: L<Archive::Libarchive::Archive>

=back

=head2 add_passphrase

 # archive_read_add_passphrase
 my $int = $r->add_passphrase($string);

=head2 append_filter

 # archive_read_append_filter
 my $int = $r->append_filter($code);

Append filter to manually specify the order in which filters will be
applied.  This will accept either a string representation of the filter
code, or the constant.  The constant prefix is C<ARCHIVE_FILTER_>.  So
for a gzipped file this would be either C<'gzip'> or C<ARCHIVE_FILTER_GZIP>.
For the full list see L<Archive::Libarchive::API/CONSTANTS>.

=head2 append_filter_program

 # archive_read_append_filter_program
 my $int = $r->append_filter_program($string);

=head2 append_filter_program_signature

 # archive_read_append_filter_program_signature
 my $int = $r->append_filter_program_signature($string, $opaque, $size_t);

=head2 close

 # archive_read_close
 my $int = $r->close;

=head2 extract

 # archive_read_extract
 my $int1 = $r->extract($e, $int2);

=head2 extract2

 # archive_read_extract2
 my $int = $r->extract2($e, $ar);

=head2 extract_set_skip_file

 # archive_read_extract_set_skip_file
 $r->extract_set_skip_file($sint64_1, $sint64_2);

=head2 format_capabilities

 # archive_read_format_capabilities
 my $int = $r->format_capabilities;

=head2 has_encrypted_entries

 # archive_read_has_encrypted_entries
 my $int = $r->has_encrypted_entries;

=head2 header_position

 # archive_read_header_position
 my $sint64 = $r->header_position;

=head2 new

 # archive_read_new
 my $r = Archive::Libarchive::ArchiveRead->new;

Create a new archive read object.

=head2 next_header

 # archive_read_next_header
 my $code = $r->next_header($e);

Returns the next L<Archive::Libarchive::Entry> object.

=head2 open

 # archive_read_open1
 # archive_read_set_callback_data
 # archive_read_set_close_callback
 # archive_read_set_open_callback
 # archive_read_set_read_callback
 # archive_read_set_seek_callback
 # archive_read_set_skip_callback
 $r->open(%callbacks);

This is a basic open method, which relies on callbacks for its implementation.  The
only callback that is required is the C<read> callback.  The C<open> and C<close>
callbacks are made available mostly for the benefit of the caller.  The C<skip>
and C<seek> callbacks are used if available for some formats like C<zip> to improve
performance.  All callbacks should return a L<normal status code|Archive::Libarchive/CONSTANTS>,
which is C<ARCHIVE_OK> on success.

Unlike the C<libarchive> C-API, this interface doesn't provide a facility for
passing in "client" data.  In Perl this is implemented using a closure, which should
allow you to pass in arbitrary variables via proper scoping.

=over 4

=item open

 $r->open(open => sub ($r) {
   ...
 });

Called immediately when the archive is "opened";

=item read

 $r->open(read => sub ($r, $ref) {
   $$ref = ...;
   ...
   return $size.
 });

Called when new data is required.  What is passed in is a scalar reference.  You should
set this scalar to the next block of data.  On success you should return the size of
the data in bytes, and on failure return a L<normal status code|Archive::Libarchive/CONSTANTS>.

=item seek

 $r->open(seek => sub ($r, $offset, $whence) {
   ...
 });

Called to seek to the new location.  The C<$offset> and C<$whence> arguments work exactly
like the C<libc> C<fseek> function.

=item skip

 $r->open(skip => sub ($r, $request) {
   ...
 });

Called to skip the next C<$request> bytes.  Should return the actual number of bytes skipped
on success (which can be less than or equal to C<$request>), and on failure return a
L<normal status code|Archive::Libarchive/CONSTANTS>.

=item close

 $r->open(close => sub ($r) {
   ...
 });

This is called when the archive instance is closed.

=back

=head2 open_FILE

 $r->open_FILE($file_pointer);

This takes either a L<FFI::C::File>, or an opaque pointer to a libc file pointer.

=head2 open_fd

 # archive_read_open_fd
 my $int1 = $r->open_fd($int2, $size_t);

=head2 open_file

 # archive_read_open_file
 my $int = $r->open_file($string, $size_t);

=head2 open_filename

 # archive_read_open_filename
 my $int = $r->open_filename($string, $size_t);

=head2 open_filename_w

 # archive_read_open_filename_w
 my $int = $r->open_filename_w($wstring, $size_t);

=head2 open_filenames

 # archive_read_open_filenames
 my $int = $r->open_filenames(\@filenames, $size_t);

Open a multi-file archive (typically for RAR format).   The C<$size_t> argument is
the block size.

=head2 open_memory

 # archive_write_open_memory
 my $code = $r->open_memory(\$buffer);

Open's the in-memory archive.

=head2 open_perlfile

 $r->open_perlfile(*FILE);

This takes a perl file handle and reads the archive from there.

=head2 read_data

 # archive_read_data
 my $size_or_code = $r->read_data(\$buffer, $size);
 my $size_or_code = $r->read_data(\$buffer);

Read in data from the content section of the archive entry.  The output is written into
C<$buffer>.  Up to C<$size> bytes will be read.  This will return the number of bytes
read on success, zero (C<0>) on EOF and a L<normal status code|Archive::Libarchive/CONSTANTS>
on error.

=head2 read_data_skip

 # archive_read_data_skip
 my $int = $r->read_data_skip;

=head2 set_filter_option

 # archive_read_set_filter_option
 my $int = $r->set_filter_option($string1, $string2, $string3);

=head2 set_format

 # archive_read_set_format
 my $int = $r->set_format($code);

Set the format manually.  This will accept either a string representation
of the format, or the constant.  The constant prefix is C<ARCHIVE_FORMAT_>.
So for a tar file this would be either C<'tar'> or C<ARCHIVE_FORMAT_TAR>.

=head2 set_format_option

 # archive_read_set_format_option
 my $int = $r->set_format_option($string1, $string2, $string3);

=head2 set_option

 # archive_read_set_option
 my $int = $r->set_option($string1, $string2, $string3);

=head2 set_options

 # archive_read_set_options
 my $int = $r->set_options($string);

=head2 set_passphrase_callback

 # archive_read_set_passphrase_callback
 my $int = $r->set_passphrase_callback(sub ($r) {
   ...
   return $passphrase;
 });

Set a callback that will be called when a passphrase is required, for example with a .zip
file with encrypted entries.

=head2 support_compression_all

 # archive_read_support_compression_all
 my $int = $r->support_compression_all;

=head2 support_compression_bzip2

 # archive_read_support_compression_bzip2
 my $int = $r->support_compression_bzip2;

=head2 support_compression_compress

 # archive_read_support_compression_compress
 my $int = $r->support_compression_compress;

=head2 support_compression_gzip

 # archive_read_support_compression_gzip
 my $int = $r->support_compression_gzip;

=head2 support_compression_lzip

 # archive_read_support_compression_lzip
 my $int = $r->support_compression_lzip;

=head2 support_compression_lzma

 # archive_read_support_compression_lzma
 my $int = $r->support_compression_lzma;

=head2 support_compression_none

 # archive_read_support_compression_none
 my $int = $r->support_compression_none;

=head2 support_compression_program

 # archive_read_support_compression_program
 my $int = $r->support_compression_program($string);

=head2 support_compression_program_signature

 # archive_read_support_compression_program_signature
 my $int = $r->support_compression_program_signature($string, $opaque, $size_t);

=head2 support_compression_rpm

 # archive_read_support_compression_rpm
 my $int = $r->support_compression_rpm;

=head2 support_compression_uu

 # archive_read_support_compression_uu
 my $int = $r->support_compression_uu;

=head2 support_compression_xz

 # archive_read_support_compression_xz
 my $int = $r->support_compression_xz;

=head2 support_filter_all

 # archive_read_support_filter_all
 my $int = $r->support_filter_all;

=head2 support_filter_by_code

 # archive_read_support_filter_by_code (optional)
 my $int1 = $r->support_filter_by_code($int2);

=head2 support_filter_bzip2

 # archive_read_support_filter_bzip2
 my $int = $r->support_filter_bzip2;

=head2 support_filter_compress

 # archive_read_support_filter_compress
 my $int = $r->support_filter_compress;

=head2 support_filter_grzip

 # archive_read_support_filter_grzip
 my $int = $r->support_filter_grzip;

=head2 support_filter_gzip

 # archive_read_support_filter_gzip
 my $int = $r->support_filter_gzip;

=head2 support_filter_lrzip

 # archive_read_support_filter_lrzip
 my $int = $r->support_filter_lrzip;

=head2 support_filter_lz4

 # archive_read_support_filter_lz4
 my $int = $r->support_filter_lz4;

=head2 support_filter_lzip

 # archive_read_support_filter_lzip
 my $int = $r->support_filter_lzip;

=head2 support_filter_lzma

 # archive_read_support_filter_lzma
 my $int = $r->support_filter_lzma;

=head2 support_filter_lzop

 # archive_read_support_filter_lzop
 my $int = $r->support_filter_lzop;

=head2 support_filter_none

 # archive_read_support_filter_none
 my $int = $r->support_filter_none;

=head2 support_filter_program

 # archive_read_support_filter_program
 my $int = $r->support_filter_program($string);

=head2 support_filter_program_signature

 # archive_read_support_filter_program_signature
 my $int = $r->support_filter_program_signature($string, $opaque, $size_t);

=head2 support_filter_rpm

 # archive_read_support_filter_rpm
 my $int = $r->support_filter_rpm;

=head2 support_filter_uu

 # archive_read_support_filter_uu
 my $int = $r->support_filter_uu;

=head2 support_filter_xz

 # archive_read_support_filter_xz
 my $int = $r->support_filter_xz;

=head2 support_filter_zstd

 # archive_read_support_filter_zstd (optional)
 my $int = $r->support_filter_zstd;

=head2 support_format_7zip

 # archive_read_support_format_7zip
 my $int = $r->support_format_7zip;

=head2 support_format_all

 # archive_read_support_format_all
 my $int = $r->support_format_all;

=head2 support_format_ar

 # archive_read_support_format_ar
 my $int = $r->support_format_ar;

=head2 support_format_by_code

 # archive_read_support_format_by_code
 my $int1 = $r->support_format_by_code($int2);

=head2 support_format_cab

 # archive_read_support_format_cab
 my $int = $r->support_format_cab;

=head2 support_format_cpio

 # archive_read_support_format_cpio
 my $int = $r->support_format_cpio;

=head2 support_format_empty

 # archive_read_support_format_empty
 my $int = $r->support_format_empty;

=head2 support_format_gnutar

 # archive_read_support_format_gnutar
 my $int = $r->support_format_gnutar;

=head2 support_format_iso9660

 # archive_read_support_format_iso9660
 my $int = $r->support_format_iso9660;

=head2 support_format_lha

 # archive_read_support_format_lha
 my $int = $r->support_format_lha;

=head2 support_format_mtree

 # archive_read_support_format_mtree
 my $int = $r->support_format_mtree;

=head2 support_format_rar

 # archive_read_support_format_rar
 my $int = $r->support_format_rar;

=head2 support_format_rar5

 # archive_read_support_format_rar5 (optional)
 my $int = $r->support_format_rar5;

=head2 support_format_raw

 # archive_read_support_format_raw
 my $int = $r->support_format_raw;

=head2 support_format_tar

 # archive_read_support_format_tar
 my $int = $r->support_format_tar;

=head2 support_format_warc

 # archive_read_support_format_warc
 my $int = $r->support_format_warc;

=head2 support_format_xar

 # archive_read_support_format_xar
 my $int = $r->support_format_xar;

=head2 support_format_zip

 # archive_read_support_format_zip
 my $int = $r->support_format_zip;

=head2 support_format_zip_seekable

 # archive_read_support_format_zip_seekable
 my $int = $r->support_format_zip_seekable;

=head2 support_format_zip_streamable

 # archive_read_support_format_zip_streamable
 my $int = $r->support_format_zip_streamable;

=head1 Archive::Libarchive::ArchiveWrite

=over 4

=item Main documentation and examples: L<Archive::Libarchive::ArchiveWrite>

=item Parent class: L<Archive::Libarchive::Archive>

=back

=head2 add_filter

 # archive_write_add_filter
 my $int = $w->add_filter($code);

Add filter to be applied when writing the archive.
This will accept either a string representation of the filter
code, or the constant.  The constant prefix is C<ARCHIVE_FILTER_>.  So
for a gzipped file this would be either C<'gzip'> or C<ARCHIVE_FILTER_GZIP>.
For the full list see L<Archive::Libarchive::API/CONSTANTS>.

=head2 add_filter_b64encode

 # archive_write_add_filter_b64encode
 my $int = $w->add_filter_b64encode;

=head2 add_filter_by_name

 # archive_write_add_filter_by_name
 my $int = $w->add_filter_by_name($string);

=head2 add_filter_bzip2

 # archive_write_add_filter_bzip2
 my $int = $w->add_filter_bzip2;

=head2 add_filter_compress

 # archive_write_add_filter_compress
 my $int = $w->add_filter_compress;

=head2 add_filter_grzip

 # archive_write_add_filter_grzip
 my $int = $w->add_filter_grzip;

=head2 add_filter_gzip

 # archive_write_add_filter_gzip
 my $int = $w->add_filter_gzip;

=head2 add_filter_lrzip

 # archive_write_add_filter_lrzip
 my $int = $w->add_filter_lrzip;

=head2 add_filter_lz4

 # archive_write_add_filter_lz4
 my $int = $w->add_filter_lz4;

=head2 add_filter_lzip

 # archive_write_add_filter_lzip
 my $int = $w->add_filter_lzip;

=head2 add_filter_lzma

 # archive_write_add_filter_lzma
 my $int = $w->add_filter_lzma;

=head2 add_filter_lzop

 # archive_write_add_filter_lzop
 my $int = $w->add_filter_lzop;

=head2 add_filter_none

 # archive_write_add_filter_none
 my $int = $w->add_filter_none;

=head2 add_filter_program

 # archive_write_add_filter_program
 my $int = $w->add_filter_program($string);

=head2 add_filter_uuencode

 # archive_write_add_filter_uuencode
 my $int = $w->add_filter_uuencode;

=head2 add_filter_xz

 # archive_write_add_filter_xz
 my $int = $w->add_filter_xz;

=head2 add_filter_zstd

 # archive_write_add_filter_zstd (optional)
 my $int = $w->add_filter_zstd;

=head2 close

 # archive_write_close
 my $int = $w->close;

=head2 data

 # archive_write_data
 my $size_or_code = $w->write_data(\$buffer);

Write the entry content data to the archive.  This takes a reference to the buffer.
Returns the number of bytes written on success, and a L<normal status code|Archive::Libarchive/CONSTANTS>
on error.

=head2 fail

 # archive_write_fail
 my $int = $w->fail;

=head2 finish_entry

 # archive_write_finish_entry
 my $int = $w->finish_entry;

=head2 get_bytes_in_last_block

 # archive_write_get_bytes_in_last_block
 my $int = $w->get_bytes_in_last_block;

=head2 get_bytes_per_block

 # archive_write_get_bytes_per_block
 my $int = $w->get_bytes_per_block;

=head2 new

 # archive_write_new
 my $w = Archive::Libarchive::ArchiveWrite->new;

Create a new archive write object.

=head2 open

 # archive_write_open
 $w->open(%callbacks);

This is a basic open method, which relies on callbacks for its implementation.  The
only callback that is required is the C<write> callback.  The C<open> and C<close>
callbacks are made available mostly for the benefit of the caller.  All callbacks
should return a L<normal status code|Archive::Libarchive/CONSTANTS>, which is
C<ARCHIVE_OK> on success.

Unlike the C<libarchive> C-API, this interface doesn't provide a facility for
passing in "client" data.  In Perl this is implemented using a closure, which should
allow you to pass in arbitrary variables via proper scoping.

=over 4

=item open

 $w->open(open => sub ($w) {
   ...
 });

Called immediately when the archive is "opened";

=item write

 $w->open(write => sub ($w, $ref) {
   ... = $$ref;
   return $size;
 });

This callback is called when data needs to be written to the archive.  It is passed in
as a reference to a scalar that contains the raw data.  On success you should return the actual size of
the data written in bytes, and on failure return a L<normal status code|Archive::Libarchive/CONSTANTS>.

=item close

 $w->open(open => sub ($w) {
   ...
 });

This is called when the archive instance is closed.

=back

=head2 open_FILE

 # archive_write_open_FILE
 $w->open_FILE($file_pointer);

This takes either a L<FFI::C::File>, or an opaque pointer to a libc file pointer.

=head2 open_fd

 # archive_write_open_fd
 my $int1 = $w->open_fd($int2);

=head2 open_filename

 # archive_write_open_filename
 my $int = $w->open_filename($string);

=head2 open_filename_w

 # archive_write_open_filename_w
 my $int = $w->open_filename_w($wstring);

=head2 open_memory

 # archive_write_open_memory
 $w->open_memory(\$buffer);

This takes a reference to scalar and stores the archive in memory there.

=head2 open_perlfile

 $w->open_perlfile(*FILE);

This takes a perl file handle and stores the archive there.

=head2 set_bytes_in_last_block

 # archive_write_set_bytes_in_last_block
 my $int1 = $w->set_bytes_in_last_block($int2);

=head2 set_bytes_per_block

 # archive_write_set_bytes_per_block
 my $int1 = $w->set_bytes_per_block($int2);

=head2 set_filter_option

 # archive_write_set_filter_option
 my $int = $w->set_filter_option($string1, $string2, $string3);

=head2 set_format

 # archive_write_set_format
 my $int = $w->set_format($code);

Set the output format.  This will accept either a string representation
of the format, or the constant.  The constant prefix is C<ARCHIVE_FORMAT_>.
So for a tar file this would be either C<'tar'> or C<ARCHIVE_FORMAT_TAR>.

=head2 set_format_7zip

 # archive_write_set_format_7zip
 my $int = $w->set_format_7zip;

=head2 set_format_ar_bsd

 # archive_write_set_format_ar_bsd
 my $int = $w->set_format_ar_bsd;

=head2 set_format_ar_svr4

 # archive_write_set_format_ar_svr4
 my $int = $w->set_format_ar_svr4;

=head2 set_format_by_name

 # archive_write_set_format_by_name
 my $int = $w->set_format_by_name($string);

=head2 set_format_cpio

 # archive_write_set_format_cpio
 my $int = $w->set_format_cpio;

=head2 set_format_cpio_newc

 # archive_write_set_format_cpio_newc
 my $int = $w->set_format_cpio_newc;

=head2 set_format_filter_by_ext

 # archive_write_set_format_filter_by_ext
 my $int = $w->set_format_filter_by_ext($string);

=head2 set_format_filter_by_ext_def

 # archive_write_set_format_filter_by_ext_def
 my $int = $w->set_format_filter_by_ext_def($string1, $string2);

=head2 set_format_gnutar

 # archive_write_set_format_gnutar
 my $int = $w->set_format_gnutar;

=head2 set_format_iso9660

 # archive_write_set_format_iso9660
 my $int = $w->set_format_iso9660;

=head2 set_format_mtree

 # archive_write_set_format_mtree
 my $int = $w->set_format_mtree;

=head2 set_format_mtree_classic

 # archive_write_set_format_mtree_classic
 my $int = $w->set_format_mtree_classic;

=head2 set_format_option

 # archive_write_set_format_option
 my $int = $w->set_format_option($string1, $string2, $string3);

=head2 set_format_pax

 # archive_write_set_format_pax
 my $int = $w->set_format_pax;

=head2 set_format_pax_restricted

 # archive_write_set_format_pax_restricted
 my $int = $w->set_format_pax_restricted;

=head2 set_format_raw

 # archive_write_set_format_raw
 my $int = $w->set_format_raw;

=head2 set_format_shar

 # archive_write_set_format_shar
 my $int = $w->set_format_shar;

=head2 set_format_shar_dump

 # archive_write_set_format_shar_dump
 my $int = $w->set_format_shar_dump;

=head2 set_format_ustar

 # archive_write_set_format_ustar
 my $int = $w->set_format_ustar;

=head2 set_format_v7tar

 # archive_write_set_format_v7tar
 my $int = $w->set_format_v7tar;

=head2 set_format_warc

 # archive_write_set_format_warc
 my $int = $w->set_format_warc;

=head2 set_format_xar

 # archive_write_set_format_xar
 my $int = $w->set_format_xar;

=head2 set_format_zip

 # archive_write_set_format_zip
 my $int = $w->set_format_zip;

=head2 set_option

 # archive_write_set_option
 my $int = $w->set_option($string1, $string2, $string3);

=head2 set_options

 # archive_write_set_options
 my $int = $w->set_options($string);

=head2 set_passphrase

 # archive_write_set_passphrase
 my $int = $w->set_passphrase($string);

=head2 set_passphrase_callback

 # archive_write_set_passphrase_callback
 my $int = $w->set_passphrase_callback(sub ($w) {
   ...
   return $passphrase;
 });

Set a callback that will be called when a passphrase is required, for example with a .zip
file with encrypted entries.

=head2 set_skip_file

 # archive_write_set_skip_file
 my $int = $w->set_skip_file($sint64_1, $sint64_2);

=head2 write_header

 # archive_write_header
 my $int = $w->write_header($e);

=head2 zip_set_compression_deflate

 # archive_write_zip_set_compression_deflate
 my $int = $w->zip_set_compression_deflate;

=head2 zip_set_compression_store

 # archive_write_zip_set_compression_store
 my $int = $w->zip_set_compression_store;

=head1 Archive::Libarchive::DiskRead

=over 4

=item Main documentation and examples: L<Archive::Libarchive::DiskRead>

=item Parent class: L<Archive::Libarchive::ArchiveRead>

=back

=head2 disk_can_descend

 # archive_read_disk_can_descend
 my $int = $dr->disk_can_descend;

=head2 disk_current_filesystem

 # archive_read_disk_current_filesystem
 my $int = $dr->disk_current_filesystem;

=head2 disk_current_filesystem_is_remote

 # archive_read_disk_current_filesystem_is_remote
 my $int = $dr->disk_current_filesystem_is_remote;

=head2 disk_current_filesystem_is_synthetic

 # archive_read_disk_current_filesystem_is_synthetic
 my $int = $dr->disk_current_filesystem_is_synthetic;

=head2 disk_descend

 # archive_read_disk_descend
 my $int = $dr->disk_descend;

=head2 disk_gname

 # archive_read_disk_gname
 my $string = $dr->disk_gname($sint64);

=head2 disk_open

 # archive_read_disk_open
 my $int = $dr->disk_open($string);

=head2 disk_open_w

 # archive_read_disk_open_w
 my $int = $dr->disk_open_w($wstring);

=head2 disk_set_atime_restored

 # archive_read_disk_set_atime_restored
 my $int = $dr->disk_set_atime_restored;

=head2 disk_set_behavior

 # archive_read_disk_set_behavior
 my $int1 = $dr->disk_set_behavior($int2);

=head2 disk_set_standard_lookup

 # archive_read_disk_set_standard_lookup
 my $int = $dr->disk_set_standard_lookup;

=head2 disk_set_symlink_hybrid

 # archive_read_disk_set_symlink_hybrid
 my $int = $dr->disk_set_symlink_hybrid;

=head2 disk_set_symlink_logical

 # archive_read_disk_set_symlink_logical
 my $int = $dr->disk_set_symlink_logical;

=head2 disk_set_symlink_physical

 # archive_read_disk_set_symlink_physical
 my $int = $dr->disk_set_symlink_physical;

=head2 disk_uname

 # archive_read_disk_uname
 my $string = $dr->disk_uname($sint64);

=head2 new

 my $dr = Archive::Libarchive::DiskRead->new;

Create a new disk read object.

=head1 Archive::Libarchive::DiskWrite

=over 4

=item Main documentation and examples: L<Archive::Libarchive::DiskWrite>

=item Parent class: L<Archive::Libarchive::ArchiveWrite>

=back

=head2 disk_gid

 # archive_write_disk_gid
 my $sint64_1 = $dw->disk_gid($string, $sint64_2);

=head2 disk_set_options

 # archive_write_disk_set_options
 my $int1 = $dw->disk_set_options($int2);

=head2 disk_set_skip_file

 # archive_write_disk_set_skip_file
 my $int = $dw->disk_set_skip_file($sint64_1, $sint64_2);

=head2 disk_set_standard_lookup

 # archive_write_disk_set_standard_lookup
 my $int = $dw->disk_set_standard_lookup;

=head2 disk_uid

 # archive_write_disk_uid
 my $sint64_1 = $dw->disk_uid($string, $sint64_2);

=head2 new

 # archive_write_disk_new
 my $dw = Archive::Libarchive::DiskWrite->new;

Create a new disk write object.

=head1 Archive::Libarchive::Entry

=over 4

=item Main documentation and examples: L<Archive::Libarchive::Entry>

=back

=head2 acl_add_entry

 # archive_entry_acl_add_entry
 my $int1 = $e->acl_add_entry($int2, $int3, $int4, $int5, $string);

=head2 acl_add_entry_w

 # archive_entry_acl_add_entry_w
 my $int1 = $e->acl_add_entry_w($int2, $int3, $int4, $int5, $wstring);

=head2 acl_clear

 # archive_entry_acl_clear
 $e->acl_clear;

=head2 acl_count

 # archive_entry_acl_count
 my $int1 = $e->acl_count($int2);

=head2 acl_from_text

 # archive_entry_acl_from_text (optional)
 my $int1 = $e->acl_from_text($string, $int2);

=head2 acl_from_text_w

 # archive_entry_acl_from_text_w (optional)
 my $int1 = $e->acl_from_text_w($wstring, $int2);

=head2 acl_next

 # archive_entry_acl_next
 my $int1 = $e->acl_next($int2, $int*1, $int*2, $int*3, $int*4, \$string);

=head2 acl_reset

 # archive_entry_acl_reset
 my $int1 = $e->acl_reset($int2);

=head2 acl_text

 # archive_entry_acl_text
 my $string = $e->acl_text($int);

=head2 acl_text_w

 # archive_entry_acl_text_w
 my $wstring = $e->acl_text_w($int);

=head2 acl_to_text

 # archive_entry_acl_to_text (optional)
 my $string = $e->acl_to_text(\$ssize_t, $int);

=head2 acl_to_text_w

 # archive_entry_acl_to_text_w (optional)
 my $wstring = $e->acl_to_text_w(\$ssize_t, $int);

=head2 acl_types

 # archive_entry_acl_types (optional)
 my $int = $e->acl_types;

=head2 atime

 # archive_entry_atime
 my $time_t = $e->atime;

=head2 atime_is_set

 # archive_entry_atime_is_set
 my $int = $e->atime_is_set;

=head2 atime_nsec

 # archive_entry_atime_nsec
 my $long = $e->atime_nsec;

=head2 birthtime

 # archive_entry_birthtime
 my $time_t = $e->birthtime;

=head2 birthtime_is_set

 # archive_entry_birthtime_is_set
 my $int = $e->birthtime_is_set;

=head2 birthtime_nsec

 # archive_entry_birthtime_nsec
 my $long = $e->birthtime_nsec;

=head2 clear

 # archive_entry_clear
 $e->clear;

=head2 clone

 # archive_entry_clone
 my $e2 = $e->clone;

Clone the entry instance.

=head2 copy_fflags_text

 # archive_entry_copy_fflags_text
 my $string1 = $e->copy_fflags_text($string2);

=head2 copy_fflags_text_w

 # archive_entry_copy_fflags_text_w
 my $wstring1 = $e->copy_fflags_text_w($wstring2);

=head2 copy_gname

 # archive_entry_copy_gname
 $e->copy_gname($string);

=head2 copy_gname_w

 # archive_entry_copy_gname_w
 $e->copy_gname_w($wstring);

=head2 copy_hardlink

 # archive_entry_copy_hardlink
 $e->copy_hardlink($string);

=head2 copy_hardlink_w

 # archive_entry_copy_hardlink_w
 $e->copy_hardlink_w($wstring);

=head2 copy_link

 # archive_entry_copy_link
 $e->copy_link($string);

=head2 copy_link_w

 # archive_entry_copy_link_w
 $e->copy_link_w($wstring);

=head2 copy_mac_metadata

 # archive_entry_copy_mac_metadata
 $e->copy_mac_metadata($meta);

Sets the mac metadata to C<$meta>.

=head2 copy_pathname

 # archive_entry_copy_pathname
 $e->copy_pathname($string);

=head2 copy_pathname_w

 # archive_entry_copy_pathname_w
 $e->copy_pathname_w($wstring);

=head2 copy_sourcepath

 # archive_entry_copy_sourcepath
 $e->copy_sourcepath($string);

=head2 copy_sourcepath_w

 # archive_entry_copy_sourcepath_w
 $e->copy_sourcepath_w($wstring);

=head2 copy_stat

 # archive_entry_copy_stat
 $e->copy_stat($stat);

Copies the values from a L<FFI::C::Stat> instance.

Not currently implemented on Windows.

=head2 copy_symlink

 # archive_entry_copy_symlink
 $e->copy_symlink($string);

=head2 copy_symlink_w

 # archive_entry_copy_symlink_w
 $e->copy_symlink_w($wstring);

=head2 copy_uname

 # archive_entry_copy_uname
 $e->copy_uname($string);

=head2 copy_uname_w

 # archive_entry_copy_uname_w
 $e->copy_uname_w($wstring);

=head2 ctime

 # archive_entry_ctime
 my $time_t = $e->ctime;

=head2 ctime_is_set

 # archive_entry_ctime_is_set
 my $int = $e->ctime_is_set;

=head2 ctime_nsec

 # archive_entry_ctime_nsec
 my $long = $e->ctime_nsec;

=head2 dev

 # archive_entry_dev
 my $dev_t = $e->dev;

=head2 dev_is_set

 # archive_entry_dev_is_set
 my $int = $e->dev_is_set;

=head2 devmajor

 # archive_entry_devmajor
 my $dev_t = $e->devmajor;

=head2 devminor

 # archive_entry_devminor
 my $dev_t = $e->devminor;

=head2 digest

 # archive_entry_digest
 my $string = $e->digest($type);

This is used to query the raw hex digest for the given entry. The type of digest is
provided as an argument.  The type may be passed in as either a string or an integer
constant.  The constant prefix is C<ARCHIVE_ENTRY_DIGEST_>.  So for an MD5 digest
you could pass in either C<'md5'> or C<ARCHIVE_ENTRY_DIGEST_MD5>.

=head2 fflags

 # archive_entry_fflags
 $e->fflags($ulong*1, $ulong*2);

=head2 fflags_text

 # archive_entry_fflags_text
 my $string = $e->fflags_text;

=head2 filetype

 # archive_entry_filetype
 my $code = $e->filetype;

This returns the type of file for the entry.  This will be a dualvar where the string
is one of C<mt>, C<reg>, C<lnx>, C<sock>, C<chr>, C<blk>, C<dir> or C<ifo>, and
integer values will match the corresponding C<AE_IF> prefixed constant.  See
L<Archive::Libarchive::API/CONSTANTS> for the full list.

=head2 gid

 # archive_entry_gid
 my $sint64 = $e->gid;

=head2 gname

 # archive_entry_gname
 my $string = $e->gname;

=head2 gname_utf8

 # archive_entry_gname_utf8
 my $string_utf8 = $e->gname_utf8;

=head2 hardlink

 # archive_entry_hardlink
 my $string = $e->hardlink;

=head2 hardlink_utf8

 # archive_entry_hardlink_utf8
 my $string_utf8 = $e->hardlink_utf8;

=head2 ino

 # archive_entry_ino
 my $sint64 = $e->ino;

=head2 ino64

 # archive_entry_ino64
 my $sint64 = $e->ino64;

=head2 ino_is_set

 # archive_entry_ino_is_set
 my $int = $e->ino_is_set;

=head2 is_data_encrypted

 # archive_entry_is_data_encrypted
 my $int = $e->is_data_encrypted;

=head2 is_encrypted

 # archive_entry_is_encrypted
 my $int = $e->is_encrypted;

=head2 is_metadata_encrypted

 # archive_entry_is_metadata_encrypted
 my $int = $e->is_metadata_encrypted;

=head2 mac_metadata

 # archive_entry_mac_metadata
 my $meta = $e->mac_metadata;

Get the mac metadata from the entry.

=head2 mode

 # archive_entry_mode
 my $mode_t = $e->mode;

=head2 mtime

 # archive_entry_mtime
 my $time_t = $e->mtime;

=head2 mtime_is_set

 # archive_entry_mtime_is_set
 my $int = $e->mtime_is_set;

=head2 mtime_nsec

 # archive_entry_mtime_nsec
 my $long = $e->mtime_nsec;

=head2 new

 my $e = Archive::Libarchive::Entry->new;

Create a new Entry object.

=head2 nlink

 # archive_entry_nlink
 my $uint = $e->nlink;

=head2 pathname

 # archive_entry_pathname
 my $string = $e->pathname;

=head2 pathname_utf8

 # archive_entry_pathname_utf8
 my $string_utf8 = $e->pathname_utf8;

=head2 perm

 # archive_entry_perm
 my $mode_t = $e->perm;

=head2 rdev

 # archive_entry_rdev
 my $dev_t = $e->rdev;

=head2 rdevmajor

 # archive_entry_rdevmajor
 my $dev_t = $e->rdevmajor;

=head2 rdevminor

 # archive_entry_rdevminor
 my $dev_t = $e->rdevminor;

=head2 set_atime

 # archive_entry_set_atime
 $e->set_atime($time_t, $long);

=head2 set_birthtime

 # archive_entry_set_birthtime
 $e->set_birthtime($time_t, $long);

=head2 set_ctime

 # archive_entry_set_ctime
 $e->set_ctime($time_t, $long);

=head2 set_dev

 # archive_entry_set_dev
 $e->set_dev($dev_t);

=head2 set_devmajor

 # archive_entry_set_devmajor
 $e->set_devmajor($dev_t);

=head2 set_devminor

 # archive_entry_set_devminor
 $e->set_devminor($dev_t);

=head2 set_fflags

 # archive_entry_set_fflags
 $e->set_fflags($ulong1, $ulong2);

=head2 set_filetype

 # archive_entry_set_filetype
 $e->set_filetype($code);

This sets the type of the file for the entry.  This will accept either a string value
which is one of C<mt>, C<reg>, C<lnx>, C<sock>, C<chr>, C<blk>, C<dir> or C<ifo>,
or an integer constant value with the C<AE_IF> prefix.  See
L<Archive::Libarchive::API/CONSTANTS> for the full list.

=head2 set_gid

 # archive_entry_set_gid
 $e->set_gid($sint64);

=head2 set_gname

 # archive_entry_set_gname
 $e->set_gname($string);

=head2 set_gname_utf8

 # archive_entry_set_gname_utf8
 $e->set_gname_utf8($string);

=head2 set_hardlink

 # archive_entry_set_hardlink
 $e->set_hardlink($string);

=head2 set_hardlink_utf8

 # archive_entry_set_hardlink_utf8
 $e->set_hardlink_utf8($string);

=head2 set_ino

 # archive_entry_set_ino
 $e->set_ino($sint64);

=head2 set_ino64

 # archive_entry_set_ino64
 $e->set_ino64($sint64);

=head2 set_is_data_encrypted

 # archive_entry_set_is_data_encrypted
 $e->set_is_data_encrypted($char);

=head2 set_is_metadata_encrypted

 # archive_entry_set_is_metadata_encrypted
 $e->set_is_metadata_encrypted($char);

=head2 set_link

 # archive_entry_set_link
 $e->set_link($string);

=head2 set_link_utf8

 # archive_entry_set_link_utf8
 $e->set_link_utf8($string);

=head2 set_mode

 # archive_entry_set_mode
 $e->set_mode($mode_t);

=head2 set_mtime

 # archive_entry_set_mtime
 $e->set_mtime($time_t, $long);

=head2 set_nlink

 # archive_entry_set_nlink
 $e->set_nlink($uint);

=head2 set_pathname

 # archive_entry_set_pathname
 $e->set_pathname($string);

=head2 set_pathname_utf8

 # archive_entry_set_pathname_utf8
 $e->set_pathname_utf8($string);

=head2 set_perm

 # archive_entry_set_perm
 $e->set_perm($mode_t);

=head2 set_rdev

 # archive_entry_set_rdev
 $e->set_rdev($dev_t);

=head2 set_rdevmajor

 # archive_entry_set_rdevmajor
 $e->set_rdevmajor($dev_t);

=head2 set_rdevminor

 # archive_entry_set_rdevminor
 $e->set_rdevminor($dev_t);

=head2 set_size

 # archive_entry_set_size
 $e->set_size($sint64);

=head2 set_symlink

 # archive_entry_set_symlink
 $e->set_symlink($string);

=head2 set_symlink_type

 # archive_entry_set_symlink_type (optional)
 $e->set_symlink_type($int);

=head2 set_symlink_utf8

 # archive_entry_set_symlink_utf8
 $e->set_symlink_utf8($string);

=head2 set_uid

 # archive_entry_set_uid
 $e->set_uid($sint64);

=head2 set_uname

 # archive_entry_set_uname
 $e->set_uname($string);

=head2 set_uname_utf8

 # archive_entry_set_uname_utf8
 $e->set_uname_utf8($string);

=head2 size

 # archive_entry_size
 my $sint64 = $e->size;

=head2 size_is_set

 # archive_entry_size_is_set
 my $int = $e->size_is_set;

=head2 sourcepath

 # archive_entry_sourcepath
 my $string = $e->sourcepath;

=head2 sourcepath_w

 # archive_entry_sourcepath_w
 my $wstring = $e->sourcepath_w;

=head2 sparse_add_entry

 # archive_entry_sparse_add_entry
 $e->sparse_add_entry($sint64_1, $sint64_2);

=head2 sparse_clear

 # archive_entry_sparse_clear
 $e->sparse_clear;

=head2 sparse_count

 # archive_entry_sparse_count
 my $int = $e->sparse_count;

=head2 sparse_next

 # archive_entry_sparse_next
 my $int = $e->sparse_next($sint64*1, $sint64*2);

=head2 sparse_reset

 # archive_entry_sparse_reset
 my $int = $e->sparse_reset;

=head2 stat

 # archive_entry_stat
 my $stat = $e->stat;

Returns a L<FFI::C::Stat> instance filled out from the entry metadata.

Not currently implemented on Windows.

=head2 strmode

 # archive_entry_strmode
 my $string = $e->strmode;

=head2 symlink

 # archive_entry_symlink
 my $string = $e->symlink;

=head2 symlink_type

 # archive_entry_symlink_type (optional)
 my $int = $e->symlink_type;

=head2 symlink_utf8

 # archive_entry_symlink_utf8
 my $string_utf8 = $e->symlink_utf8;

=head2 uid

 # archive_entry_uid
 my $sint64 = $e->uid;

=head2 uname

 # archive_entry_uname
 my $string = $e->uname;

=head2 uname_utf8

 # archive_entry_uname_utf8
 my $string_utf8 = $e->uname_utf8;

=head2 unset_atime

 # archive_entry_unset_atime
 $e->unset_atime;

=head2 unset_birthtime

 # archive_entry_unset_birthtime
 $e->unset_birthtime;

=head2 unset_ctime

 # archive_entry_unset_ctime
 $e->unset_ctime;

=head2 unset_mtime

 # archive_entry_unset_mtime
 $e->unset_mtime;

=head2 unset_size

 # archive_entry_unset_size
 $e->unset_size;

=head2 update_gname_utf8

 # archive_entry_update_gname_utf8
 my $int = $e->update_gname_utf8($string);

=head2 update_hardlink_utf8

 # archive_entry_update_hardlink_utf8
 my $int = $e->update_hardlink_utf8($string);

=head2 update_link_utf8

 # archive_entry_update_link_utf8
 my $int = $e->update_link_utf8($string);

=head2 update_pathname_utf8

 # archive_entry_update_pathname_utf8
 my $int = $e->update_pathname_utf8($string);

=head2 update_symlink_utf8

 # archive_entry_update_symlink_utf8
 my $int = $e->update_symlink_utf8($string);

=head2 update_uname_utf8

 # archive_entry_update_uname_utf8
 my $int = $e->update_uname_utf8($string);

=head2 xattr_add_entry

 # archive_entry_xattr_add_entry
 my $int = $e->xattr_add_entry($name, $value);

Adds an xattr name/value pair.

=head2 xattr_clear

 # archive_entry_xattr_clear
 $e->xattr_clear;

=head2 xattr_count

 # archive_entry_xattr_count
 my $int = $e->xattr_count;

=head2 xattr_next

 # archive_entry_xattr_next
 my $int = $e->xattr_next(\$name, $value);

Fetches the next xattr name/value pair.

=head2 xattr_reset

 # archive_entry_xattr_reset
 my $int = $e->xattr_reset;

=head1 Archive::Libarchive::EntryLinkResolver

=over 4

=item Main documentation and examples: L<Archive::Libarchive::EntryLinkResolver>

=back

=head2 new

 my $r = Archive::Libarchive::EntryLinkResolver->new;

Create a new entry link resolver object.

=head2 set_strategy

 # archive_entry_linkresolver_set_strategy
 $lr->set_strategy($int);

=head1 Archive::Libarchive::Match

=over 4

=item Main documentation and examples: L<Archive::Libarchive::Match>

=item Parent class: L<Archive::Libarchive::Archive>

=back

=head2 exclude_entry

 # archive_match_exclude_entry
 my $int1 = $m->exclude_entry($int2, $e);

=head2 exclude_pattern

 # archive_match_exclude_pattern
 my $int = $m->exclude_pattern($string);

=head2 exclude_pattern_from_file

 # archive_match_exclude_pattern_from_file
 my $int1 = $m->exclude_pattern_from_file($string, $int2);

=head2 exclude_pattern_from_file_w

 # archive_match_exclude_pattern_from_file_w
 my $int1 = $m->exclude_pattern_from_file_w($wstring, $int2);

=head2 exclude_pattern_w

 # archive_match_exclude_pattern_w
 my $int = $m->exclude_pattern_w($wstring);

=head2 excluded

 # archive_match_excluded
 my $int = $m->excluded($e);

=head2 include_date

 # archive_match_include_date
 my $int1 = $m->include_date($int2, $string);

=head2 include_date_w

 # archive_match_include_date_w
 my $int1 = $m->include_date_w($int2, $wstring);

=head2 include_file_time

 # archive_match_include_file_time
 my $int1 = $m->include_file_time($int2, $string);

=head2 include_file_time_w

 # archive_match_include_file_time_w
 my $int1 = $m->include_file_time_w($int2, $wstring);

=head2 include_gid

 # archive_match_include_gid
 my $int = $m->include_gid($sint64);

=head2 include_gname

 # archive_match_include_gname
 my $int = $m->include_gname($string);

=head2 include_gname_w

 # archive_match_include_gname_w
 my $int = $m->include_gname_w($wstring);

=head2 include_pattern

 # archive_match_include_pattern
 my $int = $m->include_pattern($string);

=head2 include_pattern_from_file

 # archive_match_include_pattern_from_file
 my $int1 = $m->include_pattern_from_file($string, $int2);

=head2 include_pattern_from_file_w

 # archive_match_include_pattern_from_file_w
 my $int1 = $m->include_pattern_from_file_w($wstring, $int2);

=head2 include_pattern_w

 # archive_match_include_pattern_w
 my $int = $m->include_pattern_w($wstring);

=head2 include_time

 # archive_match_include_time
 my $int1 = $m->include_time($int2, $time_t, $long);

=head2 include_uid

 # archive_match_include_uid
 my $int = $m->include_uid($sint64);

=head2 include_uname

 # archive_match_include_uname
 my $int = $m->include_uname($string);

=head2 include_uname_w

 # archive_match_include_uname_w
 my $int = $m->include_uname_w($wstring);

=head2 new

 my $r = Archive::Libarchive::Match->new;

Create a new archive match object.

=head2 owner_excluded

 # archive_match_owner_excluded
 my $int = $m->owner_excluded($e);

=head2 path_excluded

 # archive_match_path_excluded
 my $int = $m->path_excluded($e);

=head2 path_unmatched_inclusions

 # archive_match_path_unmatched_inclusions
 my $int = $m->path_unmatched_inclusions;

=head2 path_unmatched_inclusions_next

 # archive_match_path_unmatched_inclusions_next
 my $int = $m->path_unmatched_inclusions_next(\$string);

=head2 set_inclusion_recursion

 # archive_match_set_inclusion_recursion (optional)
 my $int1 = $m->set_inclusion_recursion($int2);

=head2 time_excluded

 # archive_match_time_excluded
 my $int = $m->time_excluded($e);

=head1 CONSTANTS

The status code constants are documented in the main documentation page
L<Archive::Libarchive>.

These constants can be imported individually from L<Archive::Libarchive> using
the normal L<Exporter> interface, or all of them can be imported with the C<:const>
export tag or along with functions with the C<:all> tag.

For the rest please consult the C<libarchive>
documentation.

=over 4

=item ARCHIVE_ENTRY_ACL_ADD_FILE

=item ARCHIVE_ENTRY_ACL_ADD_SUBDIRECTORY

=item ARCHIVE_ENTRY_ACL_APPEND_DATA

=item ARCHIVE_ENTRY_ACL_DELETE

=item ARCHIVE_ENTRY_ACL_DELETE_CHILD

=item ARCHIVE_ENTRY_ACL_ENTRY_DIRECTORY_INHERIT

=item ARCHIVE_ENTRY_ACL_ENTRY_FAILED_ACCESS

=item ARCHIVE_ENTRY_ACL_ENTRY_FILE_INHERIT

=item ARCHIVE_ENTRY_ACL_ENTRY_INHERITED

=item ARCHIVE_ENTRY_ACL_ENTRY_INHERIT_ONLY

=item ARCHIVE_ENTRY_ACL_ENTRY_NO_PROPAGATE_INHERIT

=item ARCHIVE_ENTRY_ACL_ENTRY_SUCCESSFUL_ACCESS

=item ARCHIVE_ENTRY_ACL_EVERYONE

=item ARCHIVE_ENTRY_ACL_EXECUTE

=item ARCHIVE_ENTRY_ACL_GROUP

=item ARCHIVE_ENTRY_ACL_GROUP_OBJ

=item ARCHIVE_ENTRY_ACL_INHERITANCE_NFS4

=item ARCHIVE_ENTRY_ACL_LIST_DIRECTORY

=item ARCHIVE_ENTRY_ACL_MASK

=item ARCHIVE_ENTRY_ACL_OTHER

=item ARCHIVE_ENTRY_ACL_PERMS_NFS4

=item ARCHIVE_ENTRY_ACL_PERMS_POSIX1E

=item ARCHIVE_ENTRY_ACL_READ

=item ARCHIVE_ENTRY_ACL_READ_ACL

=item ARCHIVE_ENTRY_ACL_READ_ATTRIBUTES

=item ARCHIVE_ENTRY_ACL_READ_DATA

=item ARCHIVE_ENTRY_ACL_READ_NAMED_ATTRS

=item ARCHIVE_ENTRY_ACL_STYLE_COMPACT

=item ARCHIVE_ENTRY_ACL_STYLE_EXTRA_ID

=item ARCHIVE_ENTRY_ACL_STYLE_MARK_DEFAULT

=item ARCHIVE_ENTRY_ACL_STYLE_SEPARATOR_COMMA

=item ARCHIVE_ENTRY_ACL_STYLE_SOLARIS

=item ARCHIVE_ENTRY_ACL_SYNCHRONIZE

=item ARCHIVE_ENTRY_ACL_TYPE_ACCESS

=item ARCHIVE_ENTRY_ACL_TYPE_ALARM

=item ARCHIVE_ENTRY_ACL_TYPE_ALLOW

=item ARCHIVE_ENTRY_ACL_TYPE_AUDIT

=item ARCHIVE_ENTRY_ACL_TYPE_DEFAULT

=item ARCHIVE_ENTRY_ACL_TYPE_DENY

=item ARCHIVE_ENTRY_ACL_TYPE_NFS4

=item ARCHIVE_ENTRY_ACL_TYPE_POSIX1E

=item ARCHIVE_ENTRY_ACL_USER

=item ARCHIVE_ENTRY_ACL_USER_OBJ

=item ARCHIVE_ENTRY_ACL_WRITE

=item ARCHIVE_ENTRY_ACL_WRITE_ACL

=item ARCHIVE_ENTRY_ACL_WRITE_ATTRIBUTES

=item ARCHIVE_ENTRY_ACL_WRITE_DATA

=item ARCHIVE_ENTRY_ACL_WRITE_NAMED_ATTRS

=item ARCHIVE_ENTRY_ACL_WRITE_OWNER

=item ARCHIVE_ENTRY_DIGEST_MD5

=item ARCHIVE_ENTRY_DIGEST_RMD160

=item ARCHIVE_ENTRY_DIGEST_SHA1

=item ARCHIVE_ENTRY_DIGEST_SHA256

=item ARCHIVE_ENTRY_DIGEST_SHA384

=item ARCHIVE_ENTRY_DIGEST_SHA512

=item ARCHIVE_EOF

=item ARCHIVE_EXTRACT_ACL

=item ARCHIVE_EXTRACT_CLEAR_NOCHANGE_FFLAGS

=item ARCHIVE_EXTRACT_FFLAGS

=item ARCHIVE_EXTRACT_HFS_COMPRESSION_FORCED

=item ARCHIVE_EXTRACT_MAC_METADATA

=item ARCHIVE_EXTRACT_NO_AUTODIR

=item ARCHIVE_EXTRACT_NO_HFS_COMPRESSION

=item ARCHIVE_EXTRACT_NO_OVERWRITE

=item ARCHIVE_EXTRACT_NO_OVERWRITE_NEWER

=item ARCHIVE_EXTRACT_OWNER

=item ARCHIVE_EXTRACT_PERM

=item ARCHIVE_EXTRACT_SAFE_WRITES

=item ARCHIVE_EXTRACT_SECURE_NOABSOLUTEPATHS

=item ARCHIVE_EXTRACT_SECURE_NODOTDOT

=item ARCHIVE_EXTRACT_SECURE_SYMLINKS

=item ARCHIVE_EXTRACT_SPARSE

=item ARCHIVE_EXTRACT_TIME

=item ARCHIVE_EXTRACT_UNLINK

=item ARCHIVE_EXTRACT_XATTR

=item ARCHIVE_FAILED

=item ARCHIVE_FATAL

=item ARCHIVE_FILTER_BZIP2

=item ARCHIVE_FILTER_COMPRESS

=item ARCHIVE_FILTER_GRZIP

=item ARCHIVE_FILTER_GZIP

=item ARCHIVE_FILTER_LRZIP

=item ARCHIVE_FILTER_LZ4

=item ARCHIVE_FILTER_LZIP

=item ARCHIVE_FILTER_LZMA

=item ARCHIVE_FILTER_LZOP

=item ARCHIVE_FILTER_NONE

=item ARCHIVE_FILTER_PROGRAM

=item ARCHIVE_FILTER_RPM

=item ARCHIVE_FILTER_UU

=item ARCHIVE_FILTER_XZ

=item ARCHIVE_FILTER_ZSTD

=item ARCHIVE_FORMAT_7ZIP

=item ARCHIVE_FORMAT_AR

=item ARCHIVE_FORMAT_AR_BSD

=item ARCHIVE_FORMAT_AR_GNU

=item ARCHIVE_FORMAT_BASE_MASK

=item ARCHIVE_FORMAT_CAB

=item ARCHIVE_FORMAT_CPIO

=item ARCHIVE_FORMAT_CPIO_AFIO_LARGE

=item ARCHIVE_FORMAT_CPIO_BIN_BE

=item ARCHIVE_FORMAT_CPIO_BIN_LE

=item ARCHIVE_FORMAT_CPIO_POSIX

=item ARCHIVE_FORMAT_CPIO_SVR4_CRC

=item ARCHIVE_FORMAT_CPIO_SVR4_NOCRC

=item ARCHIVE_FORMAT_EMPTY

=item ARCHIVE_FORMAT_ISO9660

=item ARCHIVE_FORMAT_ISO9660_ROCKRIDGE

=item ARCHIVE_FORMAT_LHA

=item ARCHIVE_FORMAT_MTREE

=item ARCHIVE_FORMAT_RAR

=item ARCHIVE_FORMAT_RAR_V5

=item ARCHIVE_FORMAT_RAW

=item ARCHIVE_FORMAT_SHAR

=item ARCHIVE_FORMAT_SHAR_BASE

=item ARCHIVE_FORMAT_SHAR_DUMP

=item ARCHIVE_FORMAT_TAR

=item ARCHIVE_FORMAT_TAR_GNUTAR

=item ARCHIVE_FORMAT_TAR_PAX_INTERCHANGE

=item ARCHIVE_FORMAT_TAR_PAX_RESTRICTED

=item ARCHIVE_FORMAT_TAR_USTAR

=item ARCHIVE_FORMAT_WARC

=item ARCHIVE_FORMAT_XAR

=item ARCHIVE_FORMAT_ZIP

=item ARCHIVE_MATCH_CTIME

=item ARCHIVE_MATCH_EQUAL

=item ARCHIVE_MATCH_MTIME

=item ARCHIVE_MATCH_NEWER

=item ARCHIVE_MATCH_OLDER

=item ARCHIVE_OK

=item ARCHIVE_READDISK_HONOR_NODUMP

=item ARCHIVE_READDISK_MAC_COPYFILE

=item ARCHIVE_READDISK_NO_ACL

=item ARCHIVE_READDISK_NO_FFLAGS

=item ARCHIVE_READDISK_NO_TRAVERSE_MOUNTS

=item ARCHIVE_READDISK_NO_XATTR

=item ARCHIVE_READDISK_RESTORE_ATIME

=item ARCHIVE_READ_FORMAT_CAPS_ENCRYPT_DATA

=item ARCHIVE_READ_FORMAT_CAPS_ENCRYPT_METADATA

=item ARCHIVE_READ_FORMAT_CAPS_NONE

=item ARCHIVE_READ_FORMAT_ENCRYPTION_DONT_KNOW

=item ARCHIVE_READ_FORMAT_ENCRYPTION_UNSUPPORTED

=item ARCHIVE_RETRY

=item ARCHIVE_WARN

=back

=head1 NOT IMPLEMENTED

The following methods are not currently implemented.  The reason for this
is usually for one of 1) the method doesn't make sense in the context of
Perl 2) the methods have been renamed, and this is the old name kept for
compatibility in C<libarchive> 3) the methods have deprecated and will
be removed in a future version of C<libarchive> 4) we haven't gotten around
to writing bindings for them.

=over 4

=item archive_compression

=item archive_compression_name

=item archive_entry_acl

=item archive_entry_gname_w

=item archive_entry_hardlink_w

=item archive_entry_linkify

=item archive_entry_partial_links

=item archive_entry_pathname_w

=item archive_entry_symlink_w

=item archive_entry_uname_w

=item archive_free

=item archive_position_compressed

=item archive_position_uncompressed

=item archive_read_add_callback_data

=item archive_read_append_callback_data

=item archive_read_finish

=item archive_read_next_header

=item archive_read_open

=item archive_read_open2

=item archive_read_open_memory2

=item archive_read_prepend_callback_data

=item archive_read_set_callback_data2

=item archive_utility_string_sort

=item archive_write_finish

=item archive_write_open_file

=item archive_write_open_memory

=item archive_write_set_compression_bzip2

=item archive_write_set_compression_compress

=item archive_write_set_compression_gzip

=item archive_write_set_compression_lzip

=item archive_write_set_compression_lzma

=item archive_write_set_compression_none

=item archive_write_set_compression_program

=item archive_write_set_compression_xz

=back

=head1 SEE ALSO

=over 4

=item L<Archive::Libarchive>

This is the main top-level module for using C<libarchive> from
Perl.  It is the best place to start reading the documentation.
It pulls in the other classes and C<libarchive> constants so
that you only need one C<use> statement to effectively use
C<libarchive>.

=item L<Archive::Libarchive::Archive>

The base class of all archive classes.  This includes some common error
reporting functionality among other things.

=item L<Archive::Libarchive::ArchiveRead>

This class is used for reading from archives.

=item L<Archive::Libarchive::ArchiveWrite>

This class is for creating new archives.

=item L<Archive::Libarchive::DiskRead>

This class is for reading L<Archive::Libarchive::Entry> objects from disk
so that they can be written to L<Archive::Libarchive::ArchiveWrite> objects.

=item L<Archive::Libarchive::DiskWrite>

This class is for writing L<Archive::Libarchive::Entry> objects to disk
that have been written from L<Archive::Libarchive::ArchiveRead> objects.

=item L<Archive::Libarchive::Entry>

This class represents a file in an archive, or on disk.

=item L<Archive::Libarchive::EntryLinkResolver>

This class exposes the C<libarchive> link resolver API.

=item L<Archive::Libarchive::Match>

This class exposes the C<libarchive> match API.

=item L<Alien::Libarchive3>

If a suitable system C<libarchive> can't be found, then this
L<Alien> will be installed to provide it.

=item L<libarchive.org|http://libarchive.org/>

The C<libarchive> project home page.

=item L<https://github.com/libarchive/libarchive/wiki>

The C<libarchive> project wiki.

=item L<https://github.com/libarchive/libarchive/wiki/ManualPages>

Some of the C<libarchive> man pages are listed here.

=back

=head1 AUTHOR

Graham Ollis <plicease@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2021 by Graham Ollis.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
