# @(#)Ident: TracingStacks.pm 2013-05-09 14:38 pjf ;

package Unexpected::TraitFor::TracingStacks;

use namespace::autoclean;
use version; our $VERSION = qv( sprintf '0.1.%d', q$Rev: 11 $ =~ /\d+/gmx );

use Moose::Role;
use MooseX::Types   -declare => [ q(Tracer) ];
use MooseX::Types::LoadableClass qw(LoadableClass);
use MooseX::Types::Moose         qw(HashRef Object);
use Scalar::Util                 qw(weaken);

requires qw(BUILD);

subtype Tracer, as Object,
   where   { $_->can( q(frames) ) },
   message { blessed $_ ? 'Object '.(blessed $_).' is missing a frames method'
                        : "Scalar ${_} is not on object reference" };

# Object attributes (public)
has 'trace'       => is => 'ro', isa => Tracer,
   builder        => '_build_trace', handles => [ qw(frames) ],
   init_arg       => undef, lazy => 1;

has 'trace_args'  => is => 'ro', isa => HashRef,
   builder        => '_build_trace_args', lazy => 1;

has 'trace_class' => is => 'ro', isa => LoadableClass, coerce => 1,
   default        => sub { q(Devel::StackTrace) };

# Construction
before 'BUILD' => sub {
   my $self = shift; $self->trace; return;
};

# Public methods
sub filtered_frames {
   return grep { $_->subroutine !~ m{ :: __ANON__ \z }mx } $_[ 0 ]->frames;
}

sub stacktrace {
   my ($self, $skip) = @_; my (@lines, %seen, $subr);

   for my $frame (reverse $self->filtered_frames) {
      my $package = $frame->package; my $l_no;

      unless ($l_no = $seen{ $package } and $l_no == $frame->line) {
         push @lines, join q( ), $subr || $package, 'line', $frame->line;
         $seen{ $package } = $frame->line;
      }

      $subr = $frame->subroutine;
   }

   defined $skip or $skip = 0; pop @lines while ($skip--);

   return wantarray ? reverse @lines : (join "\n", reverse @lines)."\n";
}

sub trace_frame_filter { # Lifted from StackTrace::Auto
   my $self = shift; my $found_mark = 0; weaken( $self );

   return sub {
      my ($raw)    = @_;
      my  $subr    = $raw->{caller}->[ 3 ];
     (my  $package = $subr) =~ s{ :: \w+ \z }{}mx;

      if    ($found_mark == 3) { return 1 }
      elsif ($found_mark == 2) {
         $subr =~ m{ :: new \z }mx and $self->isa( $package ) and return 0;
         $found_mark++; return 1;
      }
      elsif ($found_mark == 1) {
         $subr =~ m{ :: new \z }mx and $self->isa( $package ) and $found_mark++;
         return 0;
      }

      $subr =~ m{ :: _build_trace \z }mx and $found_mark++;
      return 0;
   }
}

# Private methods
sub _build_trace {
   return $_[ 0 ]->trace_class->new( %{ $_[ 0 ]->trace_args } );
}

sub _build_trace_args {
   return { no_refs          => 1,
            respect_overload => 0,
            max_arg_length   => 0,
            frame_filter     => $_[ 0 ]->trace_frame_filter, };
}

1;

__END__

=pod

=encoding utf8

=head1 Name

Unexpected::TraitFor::TracingStacks - Provides a minimalist stacktrace

=head1 Synopsis

   use Moose;

   with 'Unexpected::TraitFor::TracingStacks';

=head1 Version

This documents version v0.1.$Rev: 11 $ of
L<Unexpected::TraitFor::TracingStacks>

=head1 Description

Provides a minimalist stacktrace

=head1 Configuration and Environment

Modifies C<BUILD> in the consuming class. Forces the instantiation of
the C<trace> attribute

Defines the following attributes;

=over 3

=item C<trace>

An instance of the C<trace_class>

=item C<trace_args>

A hash ref of arguments passed the C<trace_class> constructor when the
C<trace> attribute is instantiated

=item C<trace_class>

A loadable class which defaults to L<Devel::StackTrace>

=back

=head1 Subroutines/Methods

=head2 filtered_frames

   @frames = $self->filtered_frames;

Currently frames with subroutine names matching C<__ANON__> are
filtered out

=head2 stacktrace

   $lines = $self->stacktrace( $num_lines_to_skip );

Returns a minimalist stack trace. Defaults to skipping zero frames
from the stack

=head2 trace_frame_filter

Lifted from L<StackTrace::Auto> this method filters out frames from the
raw stacktrace that are not of interest. It is very clever

=head1 Diagnostics

None

=head1 Dependencies

=over 3

=item L<namespace::autoclean>

=item L<Moose::Role>

=item L<MooseX::Types>

=item L<MooseX::Types::LoadableClass>

=item L<MooseX::Types::Moose>

=item L<Scalar::Util>

=back

=head1 Incompatibilities

There are no known incompatibilities in this module

=head1 Bugs and Limitations

There are no known bugs in this module.
Please report problems to the address below.
Patches are welcome

=head1 Acknowledgements

Larry Wall - For the Perl programming language

=head1 Author

Peter Flanigan, C<< <pjfl@cpan.org> >>

=head1 License and Copyright

Copyright (c) 2013 Peter Flanigan. All rights reserved

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself. See L<perlartistic>

This program is distributed in the hope that it will be useful,
but WITHOUT WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE

=cut

# Local Variables:
# mode: perl
# tab-width: 3
# End:
