package Bigtop::Docs::Syntax;

=head1 NAME

Bigtop::Docs::Syntax - A (mostly) complete Bigtop syntax description

=head1 Intro

This document is a fleshing out of Bigtop::Docs::Keywords.  Note
that you might find Bigtop::Docs::Cookbook more useful depending on
the complexity of your questions and how you like to use docs.  This
is like a text book, the cookbook teaches exclusively by example.
See also the examples directory in the distribution and Bigtop::Docs::TentTut
(tentmaker shows docs in a browser, its docs are taken directly from the
backends, so it is more up to date than any other doc).

=head1 Anatomy of a Bigtop File

Bigtop files are designed to be easy to write and to read (by you and the
computer).  There is a basic structure based on brace delimited blocks,
but it's not complicated like a programming language.  It's a descriptive
language.  There is no flow of control.  That means that you can
put things in whatever order you like, except that config comes first
and some recipients of the generated files may care about order.  For
instance, your command line SQL tool probably expects to see the
definition for a sequence before it sees a table which uses it.  These
orderings are usually fairly intuitive, especially when that other program
complains about missing definitions, etc.

The following skeleton is the smallest legal bigtop file which describes
almost nothing (and will do nothing if you feed it to bigtop):

    config {}
    app App::Name {}

=head2 config Section

At the top level there are two sections in a Bigtop file.  The order is
enforced (though I can't really defend that).  First comes config.
It lists things that make the output specific.  In it, there are statements
and backend blocks.  Each statement can take exactly one value.
If there are any characters that Perl wouldn't like in an identifier,
enclose the value in backquotes (the quote usually found under tilde, not
the ones on the same key with double quotes).

The available config statements are:

=over 4

=item engine

This is the web engine you plan to use.  Example:

    engine MP13;

This notation is stolen from Catalyst and is also used in Gantry.
Your actual choices are dependent on which framework you use and
what the backend for that framework is expecting.  Gantry supports these:

    MP13 - mod_perl 1.3.x
    MP20 - mod_perl 2.x
    CGI  - cgi/fast cgi

=item template_engine

The templating system you want to use.  Example:

    template_engine TT;

Where TT could be replaced with one of these.

    Default
    Mason
    Your::Soon::To::Be::Replaced::We::Hope::Home::Rolled::Templater

The Default will probably provide raw text/html output.  Which of these
you can choose, and what they do, is determined by your framework.

[ A fact which is still strange to me: Bigtop has no knowledge of the
templates your application will use (though some of its backends might). ]

=item base_dir

Deprecated.

Optional, defaults to the current directory.  Ignored with a warning except
in create mode.

A path to where the app will be built.  Enclose paths in backquotes.  Example:

    base_dir `/home/phil/play`;

=item app_dir

Deprecated.

Optional, defaults to the h2xs style directory name for your app.  Ignored
with a warning except in create mode.

A path relative to base_dir where Build.PL, Changes, README, etc. will live.
For example, if your app block looks like this:

    app MyNS::Apps::Name

the default will be MyNS-Apps-Name under the base_dir.  If you wanted
extra nesting you could say:

    app_dir `MyNS/Apps/Name`;

then bigtop would make those intervening directories for you.

=back

In addition to these hard coded keys, there are backend blocks which
depend on what Bigtop:: modules you have installed.  These typically
generate files on the disk, which could be Perl modules, httpd.confs, etc.

The system is simplistic and best described by an example.  Suppose you list

    SomeType  SomeBackend {}

in the config section.  Bigtop will assume the following (yes, I know how
dangerous assumptions can be):

=over 4

=item *

You have a package called Bigtop::Backend::SomeType::SomeBackend, defined in
the usual way, which is installed on your system.  (i.e. There is a file
called SomeBackend.pm which defines the package
Bigtop::Backend::SomeType::SomeBackend, which lives in
the path Bigtop/Backend/SomeType/SomeBackend.pm, relative to an
@INC member.)

=item *

That package has a method called gen_SomeType which does whatever backends
of this type should do (it might make .pm files, .conf files, .sql files
etc.).  This allows bigtop to call gen_SomeType when the user types:

    bigtop file.bigtop SomeType

=back

The distribution comes with a number of these Bigtop:: backends.
[ See L<Bigtop Backends> below for a list. ]  If you write a useful one,
please send it in so we can include it in a future release.

As a user, you can mark a file as not to be generated by saying:

    SomeType SomeBackend { no_gen 1; }

This serves as a reminder that you generated SomeType in the past, but
have messed with the result (even though you were probably warned not to).
But, more importantly, it allows SomeBackend to register the keywords it
handles, so their absence doesn't generate parse errors.

Most backends also honor the template statement.  If you say:

    SomeType SomeModule { template my_template.tt; }

Bigtop::Backend::SomeType::SomeModule will use C<my_template.tt> instead of
the one hard coded inside it.  Note that the template must be directly
usable by Template Toolit.  Further, it needs to define the same blocks
as the backend's normal template.  It's usually best to start by copying
the template from the here doc in the backend into a file, then modify
it.

That is all there is to the config section.  A typical one looks
like this:

    config {
        # engine          MP13;
        engine          MP20;
        template_engine TT;
        Init            Std             { no_gen 1; }
        SQL             Postgres        {}
        Control         Gantry          {}
        Model           GantryDBIxClass {}
    }

Note that one engine statement is commented out.  A comment is any line
whose first non-whitespace character is a pound sign.

After the L<app Section>, there is a list of all the current backends,
what keywords they understand, and what they do with them (if they
do anything special).

=head2 app Section

The app section is far more interesting than the config section.  This is
where you describe the data in your application and how you want the
user to view and manipulate it.  This is the heart of the Bigtop file.
Theoretically, you can completely recast the app (say by exchanging
Postgres for mysql, Catalyst for Gantry, mod_perl 1.3 for mod_perl 2.0,
etc.) without changing the app section.  Whether you can do that in
practice depends on whether there are backends for the system you want
to move to (if you are interested in writing backends, see
Bigtop::Docs::Modules for some pointers).  Reality also impinges on
this ideal.  Some backends understand different keywords, because
they work with different concepts.

In the app section, there are two key components:

=over 4

=item Blocks

Most blocks look like this:

    block_type Name {
        ... statements or blocks
    }

Controller and method blocks are exceptions.  These have a type property
(the type is required for methods but is optional for controllers):

    method name is stub { ... }

The legal values of this property are specified in the
Bigtop::Backend::Control:: backend you choose.  See it's docs for a
list of the types and what they mean, but the trivial type, C<stub>,
is always supported.

=item Statements

Some examples of statements:

    label          Description;
    label          `First Name`;
    header_options Add => `$site->location . '/add`;
    cols           first_name, last_name, age;

A statement is a legal keyword followed by a value and terminated with a
semi-colon.  Legal values are numbers, valid Perl idents (including colons),
anything you enclose in backward quotes (see 'A Note About Quotes in Strings'
if you need to embed quotes in strings), and comma separated lists of
the above.

Note that we sometimes use => instead of comma, they mean substantially
different things.  Only use => when told to.  Always use the => when told to.

(Footnote on fat commas:  In the grammar, comma is the separator for items
in a list, while the fat comma (=>) forms a pair which becomes a single
element in the list.  If your backend is expecting two items and it
gets one pair instead, it will be sorely confused.  Likewise, if your backend
is expecting a pair, but you give it two items separated by comma, it will be
just as confused.  This is different from the way Perl treats the fat
comma.  Remember: in Bigtop a fat comma makes a pair a comma never does.)

Which keywords are legal depends on which backends you are using.
Whether a particular keyword's value will work is up to the backend and to
whatever program ends up receiving what the backend produced.

A Note About Quotes in Strings:

You can put whatever you like into backquoted strings (as long as you don't
like backquotes).  But if you put quotes in such strings, you may need to
escape them in some way.  For example, if the string is a label for
html presentation, your backend will likely take your:

    label `Customer's Name`;

and make it something like:

    some_display_function('Customer's Name');

This will not work, since the generated Perl now has an extra single quote.
Usually you avoid this by protecting the quote like this:

    label `Customer\'s Name';

This results in the correct final output:

    some_display_function('Customer\'s Name');

You must read the backend documentation to find out what will happen
to the value you provide, but the above is typical.

[ Note for those who read tests as examples:  In the tests you will
see quotes protected by two backslashes.  This is because the tests
use here documents to build bigtop input.  The first backslash is
eaten by the heredoc processor.  Since you will supply input from 
file (at least most of the time), you should not need two backslashes. ]

=back

The general structure of the app section is:

    app App::Name { }

In fact, that is a valid specification, but it doesn't do much (you could
generate Init for such an app, but that would do slightly less than h2xs).

Inside the braces there are statements and blocks.

While the app level keywords are all defined by various backends,
here are some that are commonly available:

=over 4

=item authors

(registered by Bigtop::Backend::Control and Bigtop::Backend::Init::Std)

A comma separated list of authors (use backquotes around these to
protect spaces, etc.).  These authors will be put into the README
file and into the AUTHORS section of the stub modules' POD.

    authors `Phil Crow` => `philcrow2000@yahoo.com, `My Friend`;

Each author item may be either a name or name => email pair.

If you omit the authors statement, bigtop will use the same method as
h2xs.  This means the author name will be taken from the gcos field
of the password file for the current user (or the moral equivalent on
non-unix systems).

=item contact_us

(registered by Bigtop::Backend::Control and Bigtop::Backend::Init::Std)

Describe how to contact or join project developers:

     contact_us `Visit us at www.useourproject.org, or join our mailing
 list by sending subscribe to list-sub@useourproject.org`;

Note that all manner of whitespace is legal inside the quotes, the
value will be used literally.

=item copyright_holder

(registered by Bigtop::Backend::Control and Bigtop::Backend::Init::Std)

By default the copyright holder listed in README and POD is the first
author.  If you want someone else, list them here.  For example, the app
might belong to your company:

    copyright_holder `The World Company`;

=item license_text

(registered by Bigtop::Backend::Control and Bigtop::Backend::Init::Std)

By default the license in README and POD is what h2xs produced for perl 5.8.0.
Use this, if you want something different (e.g. due to company policy):

    license_text `All rights reserved.`;

Include the complete text here.  Feel free to include new lines in the
backquotes.

=item location

(registered by Bigtop::Backend::HttpdConf, Bigtop::Backend::CGI,
and Bigtop::Backend::SiteLook::GantryDefault)

The root location in your apache conf or cgi dispatching script.  Defaults
to /.

=item uses

(registered by Bigtop::Backend::Control, understood by
Bigtop::Backend::Control::Gantry)

A list of modules which will be used in the base modules of the application.

=item literal

(part of the grammar)

Literal means literal.  Enclose the value in backquotes and it will appear
in the result unmodified.  (Exception: one new line will be added if the
literal does not already have trailing whitespace.)  This means
you are primarily responsible for white space management.

At the app level, there are several valid literal types:

=over 4

=item HttpdConf

(registered by Bigtop::Backend::HttpdConf)

Places arbitrary directives between location blocks in httpd.conf:

    literal HttpdConf `Include /some/other/file.conf`;

You can certainly use this to add locations for which you don't have
controllers defined in your Bigtop file.  This opens the option of continuing
to regenerate even after you have extra locations or other directives.

Literal HttpdConf directives appear in the generated output
with controller locations in the order these appear in the Bigtop file.

=item Location

(registered by Bigtop::Backend::HttpdConf)

Places Apache directives directly into the root location of the app:

    literal Location `    require valid-user`;

=item PerlTop

(registered by Bigtop::Backend::HttpdConf and Bigtop::Backend::CGI)

Places perl code immediately under the #!/usr/bin/perl line in either the
<Perl> block in the generated httpd.conf or in the CGI/FastCGI dispatching
script (and app.server if you had the CGI backend build one).  May be
repeated.  Output appears in the same order as the statements in the Bigtop
file.  Example:

    literal PerlTop `    use lib '/home/myuser/perl/lib';`;

=item PerlBlock

(handled by Bigtop::Backend::HttpdConf)

Places perl code into the <Perl> block in the generated httpd.conf after
the use statement for the base module of the app.  Output appears in
the same area as the use statements for controller blocks, in
the same order as the statements and controller blocks appear in the
bigtop file.  Example:

    controller SubName {
        #...
    }
    literal PerlBlock `    use Some::Module;`;
    controller OtherName {
        #...
    }

yeilds:

    <Perl>
        #!/usr/bin/perl

        use BaseModule;
        use SubName;
        use Some::Module;
        use OtherName;
    </Perl>

You may prefer to have a single literal PerlBlock at the bottom of the
bigtop file, below all of the controllers.

=item SQL

(registered by Bigtop::Backend::SQL)

Places SQL statements directly into schema.* generated by
Bigtop::Backend::SQL::*
modules.  Literal SQL statements are mixed with sequences and tables in
the generated file with all of them appearing in the output in the order
they appear in the input.  There is no way to have literal statements
apply to specific SQL backends.  If you use a literal SQL statement,
its contents will go into every schema.* file.

=item Conf

(registered by Bigtop::Backend::Conf)

Places config variable/value pairs into the config file at the top level.

=back

There are additional literal statements at other levels.

=back

Currently there are five legal block types at the app level: sequence,
config, table, join_table, and controller.  These are described in detail
directly below.  

=over 4

=item sequence

(understood by Bigtop::Backend::SQL::*, Bigtop::Backend::Model::*,
and some Bigtop::Backend::Control::* modules)

If you are going to generate default id's from sequences, use this to
declare the sequence.  Give the sequence a name and a block:

    sequence address_seq {}

Currently the block must be empty (but it might eventually allow you to
specify things like starting value, increment, etc.).

=item config

(understood by Bigtop::Backend::Conf::*, Bigtop::Backend::HttpdConf::*,
and Bigtop::Backend::CGI::*)

[ For historical reasons set_vars can be used as a synonym for config.
That use is deprecated. ]

This allows you to dump config parameters into your Gantry::Conf instance
config file, PerlSetVar statements into the root location for your app in
your httpd.conf, or the equivalent in your cgi dispatching script or stand
alone server.  Simply list the name of a variable and its value in each
statement (remember to use backquotes):

    config {
        dbconn `dbi:Pg:dbname=yourdb` => no_accessor;
        dbuser `some_user`            => no_accessor;

        smtp_host `smtp.example.com`;
    }

If you tag the statement with => no_accessor, the controller backend
will skip making an accessor for the variable (thus assuming that your
framework is handling it).  It will also omit fishing the parameter from
the config info into the site object in the controller's generated init
method.

Note that all the backends which understand config blocks allow a gen_root
statement in their backend block in the bigtop config section.  It will
manufacture a root parameter with a relative path to the generated html
templates (the path is 'html', which will be relative to the build directory).

=item table

(understood by Bigtop::Backend::SQL::*, Bigtop::Backend::Model::*,
and Bigtop::Backend::Control::*)

The general structure of a table is:

    table name { }

Inside you can list either statements or field blocks.
There are three legal statements at present:

=over 4

=item data last_name => Crow, first_name => Phil;

(registered by Bigtop::Backend::SQL)

This corresponds directly to an INSERT INTO statement for the given data.
You may use as many data statements as you like.  Obviously, all fields
you name must be defined for the table and your values must be of the
proper SQL type.  Note that it is usually a bad idea to specify a value
for the primary key, in a data statement, when the table has
auto-incrementing behavior.

=item foreign_display `%last_name, %first_name`;

(registered by Bigtop::Backend::Model)

This becomes the user's view of rows from this table when other tables point
to them via a foreign_key.  Always use backquotes.  The formatting is
simplistic.  Any text, which looks like an ident, abutted after a percent
must be a field in the table whose value will go in that spot.  Literal
percents are allowed at the end of the string or with trailing whitespace
or punctuation (except underscore, '_').  Example:

    foreign_display `name: %last, %first, score: %score%`;

=item model_base_class

(registered by Bigtop::Backend::Model)

Each Model backend has a default base class for its models.  This overrides
that default.

    model_base_class Gantry::Utils::AuthCDBI;

To specify the default, use the same statment, but put it in the model's
backend block in the bigtop config section.

=item not_for

(registered by Bigtop::Backend::Parser, understood by Bigtop::Backend::Model::*
and Bigtop::Backend::SQL::*)

Tells one or more backends to skip this table.  Choices:

    not_for SQL;
    not_for Model;
    not_for Model, SQL;

For example, you might use this to obtain a model for a table that shouldn't
appear in your in schema.* (perhaps the table lives in an external
database).

=item sequence sequence_name;

(registered by Bigtop::Backend::SQL understood by Bigtop::Backend::SQL::*
and Bigtop::Backend::Model::*);

This means that the table might use the named sequence, but how it uses
it is up to the field blocks (see below).  You can only name one
sequence per table.

=back

There is only one legal block type:

    field name { }

The field block can contain only statements.  The legal keywords
typically include:

=over 4

=item is

(registered and required by Bigtop::Backend::SQL)

Allows a comma separated list of SQL descriptors for this field.  Choose from:

=over 4

=item type (must be first)

This should be a valid SQL type (or something your backend will convert
into one) like int4, varchar, etc.  Note that int4 is properly converted
to something the database understands for all available backends.

=item primary_key (can only be used by one field in each table)

(handled by Bigtop::Backend::SQL::* and Bigtop::Backend::Model::*)

This column is the primary key.  Multi-column primary keys are not supported.

=item auto (also known as assign_by_sequence)

(handled by Bigtop::Backend::SQL::* required by Bigtop::Backend::Model::Gantry)

This column's default value will be the next value of the sequence for
this table, if there is one and your database understands sequences.
Otherwise, this will indicate that column should be marked auto-increment,
so the database will auto-generate a sequence of value for the primary key.

=back

Example field blocks (but see below for more statements):

    field id   { is int4, primary_key, auto; }
    field name { is varchar; }

=item label

(registered by Bigtop::Backend::Control::Gantry)

What the user will see as a description of the field in main_listing
tables and on input/edit forms.

    label `First Name`;

=item date_select_text

(registered by Bigtop::Backend::Control::Gantry)

For fields of type date, this provides a date popup (if your backend supports
it).  The text is used as the label for the href that triggers the popup.

    date_select_text `Launch Calendar Popup`;

See 'How can I let my users pick dates easily?' in Gantry::Docs::FAQ
for details.

=item html_form_constraint

(registered by Bigtop::Backend::Control::Gantry)

A regex or code which will generate one immediately.  If supplied, any
value the user supplies will have to match this.

    html_form_constraint `qr[^\d{5}(-\d{4})?]`;

These are passed to Data::FormValidator, see its docs for full details.

=item html_form_display_size

(registered by Bigtop::Backend::Control::Gantry)

For fields of html_form_type text, this controls how wide the field appears.

    html_form_display_size 8;

=item html_form_optional

(registered by Bigtop::Backend::Control::Gantry)

By default, fields are required (except for the primary key, which is
assumed to be generated by the database using a sequence).  This means
that users must supply a value for them (which may or may not be carefully
validated).  If the field is not required, set this to a true value.

    html_form_optional 1;

=item html_form_options

(registered by Bigtop::Backend::Control::Gantry)

The options for fields of html_form_type select (except fields with refers_to
statements).  Provide a comma separated list of option pairs like this:

    field times_per_day {
        is                int4;
        label             `Notification Frequency`;
        html_form_type    select;
        html_form_options
            Daily              => 1,
            `Every other hour` => 12,
            Hourly             => 24;
    }

In each pair, the key is what they user will see, the value is what the
form's POST processor will see.  Note the careful use of backquotes for
options with spaces in them.  If your option has any characters that Perl
wouldn't like in an ident name, use backquotes.

Omit this if your select field is really a foreign key (has a refers_to
statement).  The user will be allowed to pick from all available rows
in the foreign key table, seeing each row through its foreign_display.

=item html_form_type

(registered by Bigtop::Backend::Control::Gantry)

What input type the user will use to enter/edit values of this field.

    html_form_type textarea;

Choose from text, textarea, or select.  If you choose select, and your
field does not have a refers_to statement, you need to include the
html_form_options statement.

=item html_form_rows and html_form_cols

For fields of type textarea, controls height and width of the text area's box.

=item non_essential

(registered by Bigtop::Backend::Model)

Tells the Model backend that this field should not be retrieved until someone
accesses it.  Not all models pay attention to this suggestion.

=item refers_to

(registered by Bigtop::Backend::SQL, understood by
Bigtop::Backend::Control::Gantry,  Bigtop::Backend::Model::Gantry,
Bigtop::Backend::Model::GantryDBIxClass, and
Bigtop::Backend::Model::GantryCDBI)

Indicates that this column is a foreign key holding the single integer
primary key of another table.  The other table must exist.

Use an html_form_type of select with refers_to fields.  Bigtop and Gantry
will do the rest.  When the user needs to set a refers_to field, she
will be presented with a pull down list.  The items in the list will
be from the foreign table and will show the foreign_display from each
of its rows.

Model backends will tell their respective ORMs that this field is a foreign
key, so they can make the proper relationship definitions (e.g. by calling
belongs_to, has_a, etc.).

=back

=item join_table

=over 4

=item joins (required)

(registered by Bigtop::Backend::SQL and Bigtop::Backend::Model, understood by
Bigtop::Backend::SQL::SQLite, Bigtop::Backend::SQL::Postgres,
Bigtop::Backend::SQL::MySQL, and Bigtop::Backend::Model::GantryDBIxClass)

A pair of tables which have a many-to-many relationship.  The order
of the tables is not important.  Example:

    join_table tshirt_color {
        joins tshirt => color;
    }

The name of the join table is arbitrary, but command line tools make it
by joining the first table's name to the second table's name with an
underscore.

The has_many relationships will be named for the table, by appending an
s to the table name.  If that won't work for your table, see the names
statement directly below.

=item names (optional)

(registered by Bigtop::Backend::SQL and Bigtop::Backend::Model, understood by
Bigtop::Backend::SQL::SQLite, Bigtop::Backend::SQL::Postgres,
Bigtop::Backend::SQL::MySQL, and Bigtop::Backend::Model::GantryDBIxClass)

Names default to table names with a trailing s appended so color is called
colors.  If you want the has_many relationships to have custom names do this:

    join_table fox_sock {
        joins fox   => sock;
        names foxes => socks;
    }

Note that if you are providing one name, you must provide the other one.
The names should be in the same order as they are in the joins statement.

=back

=item controller

    controller Control::Module [ is type ] { ... }

Controllers have several statments:

=over 4

=item controls_table

(registered by Bigtop::Backend::Control)

Says that this controller manages a given table.  Example:

    controls_table transaction;

=item literal Location

(registered by Bigtop::Backend::HttpdConf)

Puts literal text into the apache location block for this controller.  Example:

    literal Location `    require group tech`;

Then, C<require group tech> will appear as the last statement in
the location block for the controller.

=item literal GantryLocation

(registered by Bigtop::Backend::Conf)

Puts literal variable/value pairs into the Config::General output
in the GantryLocation for this controller.

=item location

(registered by Bigtop::Backend::HttpdConf, Bigtop::Backend::CGI, and
Bigtop::Backend::SiteLook::GantryDefault)

The absolute location where users will point their browsers for this
controller.  (This is still not a full URL.  It will be in the same
host or virtual host as the other controllers.)

=item page_link_label

(registered by Bigtop::Backend::SiteLook::GantryDefault)

Indicates that this controller should be included in default site
navigation.  The value is the link text.

=item rel_location

(registered by Bigtop::Backend::HttpdConf, Bigtop::Backend::CGI, and
Bigtop::Backend::SiteLook::GantryDefault)

The location, relative to the app level location, where users will point
their browsers for this controller.  (For Gantry apps, you must specify
either rel_location or location for each controller.)

=item text_description

(registered by Bigtop::Backend::Control)

Used to fill in various blanks in user messages relating to items being
added, edited, and deleted by automated CRUD schemes.

=item uses

(registered by Bigtop::Backend::Control)

Takes a list of modules which will be used at the top of the generated
module.  Do not include the data model package here, controls_table
is responsible for using it.  If your controller has a type, you should
also leave out the module corresponding to that type.  For example,
if you use the Control Gantry backend and you say

    controller Transaction is CRUD {
        #...
    }

you shouldn't put Gantry::Plugins::CRUD in your uses statement.  Doing
so will result in duplicate use statements.

By default, modules mentioned in uses statements will appear with import
lists which explicitly import all items in their @EXPORT.
But, you may supply the modules to use as a pair, where the key is the name
of the module and the value is a literal string which becomes the
import list for the module:

    uses Gantry::Utils::AutoCRUD => `qw( do_add do_edit )`;

This generates:

    use Gantry::Utils::AutoCRUD qw( do_add do_edit );

To turn off all explicit importation:

    uses Gantry::Utils::AutotCRUD => ``;

which will generate:

    use Gantry::Utils::AutoCRUD;

To decline imports:

    uses Gantry::Utils::AutoCRUD => `qw()`;

=back

There are three controller types:

=over 4

=item AutoCRUD

(understood by Bigtop::Backend::Control::Gantry)

The Gantry Control backend puts Gantry::Plugins::AutoCRUD into your
use list (so don't put it there yourself, or it will appear twice in
the output).

=item CRUD

(understood by Bigtop::Backend::Control::Gantry)

Puts Gantry::Plugins::CRUD in your use list (so don't put it there
yourself).  Also generates a mass of code stubs to get you started
with semi-automated CRUD.  See "How do I use Gantry's CRUD?" in
Gantry::Docs::FAQ for details.

=item stub

(understood by Bigtop::Backend::Control::*)

This has no effect and is provided solely for symmetry.

This is the default, so:

    controller MySpecialOne is stub {
    }

is exactly equivalent to

    controller MySpecialOne {
    }

=back

There are two blocks for controllers:

=over 4

=item config

(part of the grammar understood by Bigtop::Backend::Conf::General,
Bigtop::Backend::Control::Gantry and Bigtop::Backend::HttpdConf::Gantry)

[ For historical reasons set_vars can be used as a synonym for config.
That use is deprecated. ]

This block works like the app level config block, but dumps its
variables into the location for the controller.

=back

=item method

    method name is type { ... }

What goes in the block is controlled by the type (and how the backend
treats it).  Everything in the block must be a statement.

Most of the method statements depend on the method's type.  There is one
they all share: extra_args.  It allows you to augment the parameter list
the method expects.  Example:

    extra_args `$id`, `@some_list`;

So the value for this statement is a comma separated list.  Each argument
must be separately backquoted.  They must include their sigils.  These
values will be used as is.  (Note that typically you can only have
one array in this list and it must be last.  The first array in the
list will take all the remaining items from @_.)

The method types are:

=over 4

=item main_listing

(understood by Bigtop::Backend::Control::Gantry)

You should probably call this method do_main, since Gantry calls that by
default.  It makes an html table of rows from the controlled table.  Its
statements are:

=over 4

=item cols

A comma separated list of the fields from the controlled table which should
be included in the html table.

=item col_labels (optional)

By default the columns are labeled with the value of the field's label
statement in the table block.  If no label is given, the field's name
is used as the default (and the bigtop user is warned).  To pick your
own labels, use this statement.

Each item in the list can be plain text or a pair.  If it is a pair,
the key is link text and the value is Perl code which will generate
an href link.  Example:

    col_labels `Full Name`,
               Date => `$site->location() . '/order_by_date'`;

Presumably you have supplied a do_order_by_date method in the controller
to handle reordering the html table by date.

col_labels is used from left to right.  If it runs out of entries, the
remaining labels revert to default behavior.

=item header_options

Gantry's results.tt main listing template has a spot for table level
links at the far right of the heading row.  To populate it, use
this statement.  Use either strings or text => link pairs.  Example:

    header_options Add, Reports => `$self->location() . '/table_reports'`;

For simple strings the link will take users to:

    ( my $label = $your_label ) =~ s/ /_/g;
    $self->location() . '/' . lc( $label );

So, if the above location was /loc/subloc, Add will link to

    /loc/subloc/add

=item html_template

By default main_listings use Gantry's results.tt.  Use this to
override that template with one of yours.  Put it in a location where your
template processor can find it.

=item row_options

These options are like header options, but they appear at the right side
of each row in the main listing.  Typical example:

    row_options Edit, Delete;

These link to

    ( my $label = $your_label ) =~ s/ /_/g;
    $self->location() . '/' . lc( $label ) . "/$id";

So, if the above location was /loc/subloc, Edit will link to

    /loc/subloc/edit/$id

To change the link location, use a pair (as shown above for header_options):

    row_options Edit => `$self->location() . "/exotic_edit/$id";

=item title

The browser window title to display while do_main's page is displayed.

=back

=item AutoCRUD_form

This method must be called form, since Gantry's AutoCRUD scheme will call
it by name (well, you could call it _form, but that's deprecated).
The statements for this type are:

=over 4

=item fields

A comma separated list of the names of the fields from the controlled
table that should appear on the form.  They will appear in the order given.
Example:

    fields name, address, city, state, zip, phone;

=item all_fields_but

Negated form of the fields statement.  Here you list the fields you don't
want.  They appear in the order they are specified in the table.  Example:

    all_fields_but id;

=item extra_keys

Gantry's AutoCRUD scheme uses form.tt to display the form.  That
template expects a hash describing the form's layout, validation
requirements, etc.  You can add keys to the hash with this statement.
Provide as many comma separated pairs as you like.  The keys and
their values will be used as is in the hash for form.tt.  Example:

    javascript => `$self->calendar_month_js( 'my_form_name' )`;

This is one step in the process of allowing easy date selection
described in 'How can I let my users pick dates easily?' in
Gantry::Docs::FAQ.

Use any keys you want, but know that your template is your audience.
If it doesn't know about the key, it will be ignored.

=item form_name

Provides a name attribute to the html form element (yes, I know this
is not in compliance with the current xhtml).  You need to use this
statement if you are using Gantry's popup calendar scheme for user
date input.

=back

=item CRUD_form

If you use this method type, you might want to mark the controller's
type as CRUD:

    controller MyModule is CRUD {
        #...
    }

If you do mark the controller's type as CRUD, name this
something_form.  That is, end the name with _form.  The backend
strips that and uses what's left as the name of its
generated Gantry::Plugins::CRUD object.

See 'How do I use Gantry's CRUD?' in Bigtop::Docs::Cookbook for
details about what the backend makes in this case.

=back

=back

=head1 Bigtop Backends

This section lists the available backends.  Bigtop::Docs::Cookbook has more
details on what each one makes and where it puts its output.

=over 4

=item CGI Gantry

By default, this makes a single file app.cgi in the build directory which
you must copy to your cgi-bin directory.  If you want FastCGI instead,
set fast_cgi:

    CGI Gantry { fast_cgi 1; }

This backend can also make a stand alone server for testing your app.
The server will be based on HTTP::Server::Simple.  To get a server, set
with_server:

    CGI Gantry { with_server 1; }

The default server binds to port 8080, to change that, add server_port:

    CGI Gantry { with_server 1;
                 server_port 8192; }

If you want to use Gantry::Conf, you should use the instance statement:

    CGI Gantry { instance name; }

This will put the single hash key GantryConfInstance into the CGI engine
object.  You should also use the Conf General backend to make the config
file for Gantry::Conf to use.  If you don't use /etc/gantry.conf for
your master conf file, add the conffile statement:

    CGI Gantry { instance name; conffile `/path/to/your/master.conf`; }

=item Conf General

Puts all app and controller level config statements into a Config::General
formatted file (Appname.conf in the docs subdirectory of the build directory).
Use the gen_root statement to have it generate a root parameter at the base
location level:

    Conf General { gen_root 1; }

=item Control Gantry

You can control whether the use Gantry statement in the base module of your
app has the Engine and TemplateEngine import values.  By default, you don't
get them.  If you want them in the controller, say this:

    Control Gantry { full_use 1; }

In this case, you should probably put a C<full_use 0;> statement somewhere
else (like in HttpdConf Gantry's block, see below).

Setting full_use to 1 makes it harder to port the app from one engine
or template engine to another, which is why it isn't the default.

If you use DBIx::Class as your ORM (which we recommend), make sure
to include C<dbix 1;>.  In normal use, DBIx::Class performs queries
via a call to resultset on the schema for your whole database.  That
requires slightly different coding than directly querying the models.
Hence this flag.  It defaults to false, but that is subject change
since dbic is now our preferred ORM.

=item HttpdConf Gantry

You may turn off PerlSetVar generation for app and controller level config
block statements by setting skip_config to a true value:

    HttpdConf Gantry { skip_config 1; }

If you do not want the Engine and TemplateEngine import values in your use
App::Name statement in the Perl block of the generated httpd.conf, say
something like this:

    HttpdConf Gantry { full_use 0; }

By default the use statement will have a full import list.

=item Init Std

=item Model GantryDBIxClass

If you use this backend, you should set the dbix statement in the Control
Gantry backend to a true value.

=item Model Gantry

=item Model GantryCDBI

=item SiteLook Gantry

Allows you to include a gantry_wrapper statement in its config block:

    SiteLook Gantry {
        gantry_wrapper `/path/to/gantry/root/sample_wrapper.tt`;
    }

The path needs to point to a wrapper with the structure of the
sample_wrapper.tt that ships with Gantry.  By default, the backend
just uses the one that came with Gantry.  So, you don't usually need
this statement.

=item SQL SQLite

=item SQL Postgres

=item SQL MySQL

=back

=head1 AUTHOR

Phil Crow <philcrow2000@yahoo.com>

=cut
