package Acme::CPANModules::TextTable;

our $AUTHORITY = 'cpan:PERLANCAR'; # AUTHORITY
our $DATE = '2021-07-26'; # DATE
our $DIST = 'Acme-CPANModules-TextTable'; # DIST
our $VERSION = '0.010'; # VERSION

use 5.010001;
use strict;
use warnings;
use utf8;

sub _make_table {
    my ($cols, $rows, $celltext) = @_;
    my $res = [];
    push @$res, [];
    for (0..$cols-1) { $res->[0][$_] = "col" . ($_+1) }
    for my $row (1..$rows) {
        push @$res, [ map { $celltext // "row$row.$_" } 1..$cols ];
    }
    $res;
}

our $LIST = {
    summary => 'Modules that generate text tables',
    entry_features => {
        wide_char => {summary => 'Whether the use of wide characters (e.g. Kanji) in cells does not cause the table to be misaligned'},
        color_data =>  {summary => 'Whether module supports ANSI colors (i.e. text with ANSI color codes can still be aligned properly)'},
        multiline_data => {summary => 'Whether module supports aligning data cells that contain newlines'},

        box_char => {summary => 'Whether module can utilize box-drawing characters'},
        custom_border => {summary => 'Whether module allows customizing border in some way'},

        align_row => {summary => "Whether module supports aligning text horizontally in a row (left/right/middle)"},
        align_column => {summary => "Whether module supports aligning text horizontally in a column (left/right/middle)"},
        align_cell => {summary => "Whether module supports aligning text horizontally in individual cells (left/right/middle)"},

        valign_row => {summary => "Whether module supports aligning text vertically in a row (top/bottom/middle)"},
        valign_column => {summary => "Whether module supports aligning text vertically in a column (top/bottom/middle)"},
        valign_cell => {summary => "Whether module supports aligning text vertically in individual cells (top/bottom/middle)"},

        rowspan => {summary => "Whether module supports row spans"},
        colspan => {summary => "Whether module supports column spans"},

        custom_color => {summary => 'Whether the module produces colored table and supports customizing color in some way'},
        color_theme => {summary => 'Whether the module supports color theme/scheme'},

        speed => {summary => "Rendering speed", schema=>'str*'},

        column_width => {summary => 'Whether module allows setting the width of columns'},
        per_column_width => {summary => 'Whether module allows setting column width on a per-column basis'},
        row_height => {summary => 'Whether module allows setting the height of rows'},
        per_row_height => {summary => 'Whether module allows setting row height on a per-row basis'},

        pad => {summary => 'Whether module allows customizing cell horizontal padding'},
        vpad => {summary => 'Whether module allows customizing cell vertical padding'},
    },
    entries => [
        {
            module => 'Text::Table::Any',
            description => <<'_',

This is a frontend for many text table modules as backends. The interface is
dead simple, following <pm:Text::Table::Tiny>. The main drawback is that it
currently does not allow passing (some, any) options to each backend.

_
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Any::table(rows=>$table, header_row=>1);
            },
            features => {
                align_cell     => {value=>undef, summary=>"Depends on backend"},
                align_column   => {value=>undef, summary=>"Depends on backend"},
                align_row      => {value=>undef, summary=>"Depends on backend"},
                box_char       => {value=>undef, summary=>"Depends on backend"},
                color_data     => {value=>undef, summary=>"Depends on backend"},
                color_theme    => {value=>undef, summary=>"Depends on backend"},
                colspan        => {value=>undef, summary=>"Depends on backend"},
                custom_border  => {value=>undef, summary=>"Depends on backend"},
                custom_color   => {value=>undef, summary=>"Depends on backend"},
                multiline_data => {value=>undef, summary=>"Depends on backend"},
                rowspan        => {value=>undef, summary=>"Depends on backend"},
                speed          => {value=>undef, summary=>"Depends on backend"},
                valign_cell    => {value=>undef, summary=>"Depends on backend"},
                valign_column  => {value=>undef, summary=>"Depends on backend"},
                valign_row     => {value=>undef, summary=>"Depends on backend"},
                wide_char_data => {value=>undef, summary=>"Depends on backend"},
            },
        },

        {
            module => 'Text::UnicodeBox::Table',
            description => <<'_',

The main feature of this module is the various border style it provides drawn
using Unicode box-drawing characters. It allows per-row style. The rendering
speed is particularly slow compared to other modules.

_
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::UnicodeBox::Table->new;
                $t->add_header(@{ $table->[0] });
                $t->add_row(@{ $table->[$_] }) for 1..$#{$table};
                $t->render;
            },
            features => {
                align_cell => 0,
                align_column => 1,
                box_char => 0,
                color_data => 1,
                color_theme => 0,
                colspan => 0,
                custom_border => 1,
                custom_color => 0,
                multiline_data => 0,
                rowspan => 0,
                wide_char_data => 1,
                speed => "slow",
            },
        },

        {
            module => 'Text::Table::Manifold',
            description => <<'_',

Two main features of this module is per-column aligning and wide character
support. This module, aside from doing its rendering, can also be told to pass
rendering to HTML, CSV, or other text table module like
<pm:Text::UnicodeBox::Table>); so in this way it is similar to
<pm:Text::Table::Any>.

_
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::Table::Manifold->new;
                $t->headers($table->[0]);
                $t->data([ @{$table}[1 .. $#{$table}] ]);
                join("\n", @{$t->render(padding => 1)}) . "\n";
            },
            features => {
                align_cell => 0,
                align_column => 1,
                box_char => undef, # ?
                color_data => 1,
                color_theme => 0,
                colspan => 0,
                custom_border => {value=>0, summary=>"But this module can pass rendering to other module like Text::UnicodeBox::Table"},
                custom_color => 0,
                multiline_data => 0,
                rowspan => 0,
                wide_char_data => 1,
            },
        },

        {
            module => 'Text::ANSITable',
            description => <<'_',

This 2013 project was my take in creating a text table module that can handle
color, multiline text, wide characters. I also threw in various formatting
options, e.g. per-column/row/cell align/valign/pad/vpad, conditional formatting,
and so on. I even added a couple of features I never used: hiding rows and
specifying columns to display which can be in different order from the original
specified columns or can contain the same original columns multiple times. I
think this module offers the most formatting options on CPAN.

In early 2021, I needed colspan/rowspan and I implemented this in a new module:
<pm:Text::Table::Span> (later renamed to <pm:Text::Table::More>). I plan to add
this feature too to Text::ANSITable, but in the meantime I'm also adding more
formatting options which I need to Text::Table::More.

_
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::ANSITable->new(
                    use_utf8 => 0,
                    use_box_chars => 0,
                    use_color =>  0,
                    columns => $table->[0],
                    border_style => 'ASCII::SingleLine',
                );
                $t->add_row($table->[$_]) for 1..@$table-1;
                $t->draw;
            },
            features => {
                align_cell => 1,
                align_column => 1,
                align_row => 1,
                box_char => 1,
                color_data =>  1,
                color_theme => 1,
                colspan => 0,
                column_width => 1,
                custom_border => 1,
                custom_color => 1,
                multiline_data => 1,
                pad => 1,
                per_column_width => 1,
                per_row_height => 1,
                row_height => 1,
                rowspan => 0,
                speed => "slow",
                valign_cell => 1,
                valign_column => 1,
                valign_row => 1,
                vpad => 1,
                wide_char_data => 1,
            },
        },

        {
            module => 'Text::ASCIITable',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::ASCIITable->new();
                $t->setCols(@{ $table->[0] });
                $t->addRow(@{ $table->[$_] }) for 1..@$table-1;
                "$t";
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => 0,
                multiline_data => 1,
            },
        },
        {
            module => 'Text::FormatTable',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::FormatTable->new(join('|', ('l') x @{ $table->[0] }));
                $t->head(@{ $table->[0] });
                $t->row(@{ $table->[$_] }) for 1..@$table-1;
                $t->render;
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => 0,
                multiline_data => 1,
            },
        },
        {
            module => 'Text::MarkdownTable',
            bench_code => sub {
                my ($table) = @_;
                my $out = "";
                my $t = Text::MarkdownTable->new(file => \$out);
                my $fields = $table->[0];
                foreach (1..@$table-1) {
                    my $row = $table->[$_];
                    $t->add( {
                        map { $fields->[$_] => $row->[$_] } 0..@$fields-1
                    });
                }
                $t->done;
                $out;
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => 0,
                multiline_data => {value=>0, summary=>'Newlines stripped'},
            },
        },
        {
            module => 'Text::Table',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::Table->new(@{ $table->[0] });
                $t->load(@{ $table }[1..@$table-1]);
                $t;
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => {value=>undef, summary=>'Does not draw borders'},
                multiline_data => 1,
            },
        },
        {
            module => 'Text::Table::Tiny',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Tiny::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  1,
                box_char => 1,
                multiline_data => 0,
            },
        },
        {
            module => 'Text::Table::TinyBorderStyle',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyBorderStyle::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => 1,
                multiline_data => 0,
            },
        },
        {
            module => 'Text::Table::More',
            description => <<'_',

A module I wrote in early 2021. Main distinguishing feature is support for
rowspan/clospan. I plan to add more features to this module on an as-needed
basic. This module is now preferred to <pm:Text::ANSITable>, although currently
it does not nearly as many formatting options as Text::ANSITable.

_
            bench_code => sub {
                my ($table) = @_;
                Text::Table::More::generate_table(rows=>$table, header_row=>1);
            },
            features => {
                align_cell => 1,
                align_column => 1,
                align_row => 1,
                box_char => 1,
                color_data =>  1,
                color_theme => 0,
                colspan => 1,
                custom_border => 1,
                custom_color => 0,
                multiline_data => 1,
                rowspan => 1,
                speed => "slow",
                valign_cell => 1,
                valign_column => 1,
                valign_row => 1,
                wide_char_data => 1,
                column_width => 0, # todo
                per_column_width => 0, # todo
                row_height => 0, # todo
                per_row_height => 0, # todo
                pad => 0, # todo
                vpad => 0, # todo
            },
        },
        {
            module => 'Text::Table::Sprintf',
            description => <<'_',

A performant (see benchmark result) and lightweight (a page of code, no use of
modules at all), but with minimal extra features.

_
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Sprintf::table(rows=>$table, header_row=>1);
            },
            features => {
                box_char => 0,
                color_data =>  0,
                multiline_data => 0,
                speed => "fast",
                wide_char_data => 0,
            },
        },
        {
            module => 'Text::Table::TinyColor',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyColor::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 0,
                color_data =>  1,
                box_char => 0,
                multiline_data => 0,
            },
        },
        {
            module => 'Text::Table::TinyColorWide',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyColorWide::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  1,
                box_char => 0,
                multiline_data => 0,
            },
        },
        {
            module => 'Text::Table::TinyWide',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::TinyWide::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  0,
                box_char => 0,
            },
        },
        {
            module => 'Text::Table::Org',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::Org::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 0,
                color_data =>  0,
                box_char => 0,
                multiline_data => 0,
            },
        },
        {
            module => 'Text::Table::CSV',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::CSV::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  0,
                box_char => {value=>undef, summary=>"Irrelevant"},
                multiline_data => {value=>1, summary=>"But make sure your CSV parser can handle multiline cell"},
            },
        },
        {
            module => 'Text::Table::HTML',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::HTML::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  {value=>0, summary=>'Not converted to HTML color elements'},
                box_char => 0,
                multiline_data => 1,
            },
        },
        {
            module => 'Text::Table::HTML::DataTables',
            bench_code => sub {
                my ($table) = @_;
                Text::Table::HTML::DataTables::table(rows=>$table, header_row=>1);
            },
            features => {
                wide_char_data => 1,
                color_data =>  {value=>0, summary=>'Not converted to HTML color elements'},
                box_char => 0,
                multiline_data => 1,
            },
        },
        {
            module => 'Text::TabularDisplay',
            bench_code => sub {
                my ($table) = @_;
                my $t = Text::TabularDisplay->new(@{ $table->[0] });
                $t->add(@{ $table->[$_] }) for 1..@$table-1;
                $t->render; # doesn't add newline
            },
            features => {
                wide_char_data => 1,
                color_data =>  0,
                box_char => {value=>undef, summary=>"Irrelevant"},
                multiline_data => 1,
            },
        },
    ],

    bench_datasets => [
        {name=>'tiny (1x1)'          , argv => [_make_table( 1, 1)],},
        {name=>'small (3x5)'         , argv => [_make_table( 3, 5)],},
        {name=>'wide (30x5)'         , argv => [_make_table(30, 5)],},
        {name=>'long (3x300)'        , argv => [_make_table( 3, 300)],},
        {name=>'large (30x300)'      , argv => [_make_table(30, 300)],},
        {name=>'multiline data (2x1)', argv => [ [["col1", "col2"], ["foobar\nbaz\nqux\nquux","corge"]] ], include_by_default=>0 },
        {name=>'wide char data (1x2)', argv => [ [["col1"], ["no wide character"], ["宽字"]] ], include_by_default=>0 },
        {name=>'color data (1x2)'    , argv => [ [["col1"], ["no color"], ["\e[31m\e[1mwith\e[0m \e[32m\e[1mcolor\e[0m"]] ], include_by_default=>0 },
    ],

};

1;
# ABSTRACT: Modules that generate text tables

__END__

=pod

=encoding UTF-8

=head1 NAME

Acme::CPANModules::TextTable - Modules that generate text tables

=head1 VERSION

This document describes version 0.010 of Acme::CPANModules::TextTable (from Perl distribution Acme-CPANModules-TextTable), released on 2021-07-26.

=head1 SYNOPSIS

To run benchmark with default option:

 % bencher --cpanmodules-module TextTable

To run module startup overhead benchmark:

 % bencher --module-startup --cpanmodules-module TextTable

For more options (dump scenario, list/include/exclude/add participants, list/include/exclude/add datasets, etc), see L<bencher> or run C<bencher --help>.

=head1 ACME::CPANMODULES ENTRIES

=over

=item * L<Text::Table::Any> - Generate text table using one of several backends

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

This is a frontend for many text table modules as backends. The interface is
dead simple, following L<Text::Table::Tiny>. The main drawback is that it
currently does not allow passing (some, any) options to each backend.


=item * L<Text::UnicodeBox::Table> - High level interface providing easy table drawing

Author: L<EWATERS|https://metacpan.org/author/EWATERS>

The main feature of this module is the various border style it provides drawn
using Unicode box-drawing characters. It allows per-row style. The rendering
speed is particularly slow compared to other modules.


=item * L<Text::Table::Manifold> - Render tables in manifold formats

Author: L<RSAVAGE|https://metacpan.org/author/RSAVAGE>

Two main features of this module is per-column aligning and wide character
support. This module, aside from doing its rendering, can also be told to pass
rendering to HTML, CSV, or other text table module like
L<Text::UnicodeBox::Table>); so in this way it is similar to
L<Text::Table::Any>.


=item * L<Text::ANSITable> - Create nice formatted tables using extended ASCII and ANSI colors

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

This 2013 project was my take in creating a text table module that can handle
color, multiline text, wide characters. I also threw in various formatting
options, e.g. per-column/row/cell align/valign/pad/vpad, conditional formatting,
and so on. I even added a couple of features I never used: hiding rows and
specifying columns to display which can be in different order from the original
specified columns or can contain the same original columns multiple times. I
think this module offers the most formatting options on CPAN.

In early 2021, I needed colspan/rowspan and I implemented this in a new module:
L<Text::Table::Span> (later renamed to L<Text::Table::More>). I plan to add
this feature too to Text::ANSITable, but in the meantime I'm also adding more
formatting options which I need to Text::Table::More.


=item * L<Text::ASCIITable> - Create a nice formatted table using ASCII characters.

Author: L<LUNATIC|https://metacpan.org/author/LUNATIC>

=item * L<Text::FormatTable> - Format text tables

Author: L<TREY|https://metacpan.org/author/TREY>

=item * L<Text::MarkdownTable> - Write Markdown syntax tables from data

Author: L<VOJ|https://metacpan.org/author/VOJ>

=item * L<Text::Table> - Organize Data in Tables

Author: L<SHLOMIF|https://metacpan.org/author/SHLOMIF>

=item * L<Text::Table::Tiny> - generate simple text tables from 2D arrays

Author: L<NEILB|https://metacpan.org/author/NEILB>

=item * L<Text::Table::TinyBorderStyle> - Text::Table::Tiny + support for border styles

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item * L<Text::Table::More> - Generate text table with simple interface and many options

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

A module I wrote in early 2021. Main distinguishing feature is support for
rowspan/clospan. I plan to add more features to this module on an as-needed
basic. This module is now preferred to L<Text::ANSITable>, although currently
it does not nearly as many formatting options as Text::ANSITable.


=item * L<Text::Table::Sprintf> - Generate simple text tables from 2D arrays using sprintf()

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

A performant (see benchmark result) and lightweight (a page of code, no use of
modules at all), but with minimal extra features.


=item * L<Text::Table::TinyColor> - Text::Table::Tiny + support for colored text

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item * L<Text::Table::TinyColorWide> - Text::Table::Tiny + support for colored text & wide character

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item * L<Text::Table::TinyWide> - Text::Table::Tiny + support for wide character

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item * L<Text::Table::Org> - Generate Org tables

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item * L<Text::Table::CSV> - Generate CSV

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item * L<Text::Table::HTML> - Generate HTML table

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item * L<Text::Table::HTML::DataTables> - Generate HTML table with jQuery and DataTables plugin

Author: L<PERLANCAR|https://metacpan.org/author/PERLANCAR>

=item * L<Text::TabularDisplay> - Display text in formatted table output

Author: L<DARREN|https://metacpan.org/author/DARREN>

=back

=head1 ACME::CPANMODULES FEATURE COMPARISON MATRIX

 +-------------------------------+----------------+------------------+---------------+--------------+----------------+-----------------+-------------+-------------------+------------------+---------------------+--------------+------------+------------------+--------------------+-----------------+----------------+-------------------+----------+-----------------------+---------------------+-----------------+-----------+
 | module                        | align_cell *1) | align_column *2) | align_row *3) | box_char *4) | color_data *5) | color_theme *6) | colspan *7) | custom_border *8) | custom_color *9) | multiline_data *10) | rowspan *11) | speed *12) | valign_cell *13) | valign_column *14) | valign_row *15) | wide_char_data | column_width *16) | pad *17) | per_column_width *18) | per_row_height *19) | row_height *20) | vpad *21) |
 +-------------------------------+----------------+------------------+---------------+--------------+----------------+-----------------+-------------+-------------------+------------------+---------------------+--------------+------------+------------------+--------------------+-----------------+----------------+-------------------+----------+-----------------------+---------------------+-----------------+-----------+
 | Text::Table::Any              | N/A *22)       | N/A *22)         | N/A *22)      | N/A *22)     | N/A *22)       | N/A *22)        | N/A *22)    | N/A *22)          | N/A *22)         | N/A *22)            | N/A *22)     | N/A *22)   | N/A *22)         | N/A *22)           | N/A *22)        | N/A *22)       | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::UnicodeBox::Table       | no             | yes              | N/A           | no           | yes            | no              | no          | yes               | no               | no                  | no           | slow       | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::Manifold         | no             | yes              | N/A           | N/A          | yes            | no              | no          | no *23)           | no               | no                  | no           | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::ANSITable               | yes            | yes              | yes           | yes          | yes            | yes             | no          | yes               | yes              | yes                 | no           | slow       | yes              | yes                | yes             | yes            | yes               | yes      | yes                   | yes                 | yes             | yes       |
 | Text::ASCIITable              | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::FormatTable             | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::MarkdownTable           | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | no *24)             | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table                   | N/A            | N/A              | N/A           | N/A *25)     | no             | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::Tiny             | N/A            | N/A              | N/A           | yes          | yes            | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::TinyBorderStyle  | N/A            | N/A              | N/A           | yes          | no             | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::More             | yes            | yes              | yes           | yes          | yes            | no              | yes         | yes               | no               | yes                 | yes          | slow       | yes              | yes                | yes             | yes            | no                | no       | no                    | no                  | no              | no        |
 | Text::Table::Sprintf          | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | fast       | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::TinyColor        | N/A            | N/A              | N/A           | no           | yes            | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::TinyColorWide    | N/A            | N/A              | N/A           | no           | yes            | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::TinyWide         | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | N/A                 | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::Org              | N/A            | N/A              | N/A           | no           | no             | N/A             | N/A         | N/A               | N/A              | no                  | N/A          | N/A        | N/A              | N/A                | N/A             | no             | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::CSV              | N/A            | N/A              | N/A           | N/A *26)     | no             | N/A             | N/A         | N/A               | N/A              | yes *27)            | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::HTML             | N/A            | N/A              | N/A           | no           | no *28)        | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::Table::HTML::DataTables | N/A            | N/A              | N/A           | no           | no *28)        | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 | Text::TabularDisplay          | N/A            | N/A              | N/A           | N/A *26)     | no             | N/A             | N/A         | N/A               | N/A              | yes                 | N/A          | N/A        | N/A              | N/A                | N/A             | yes            | N/A               | N/A      | N/A                   | N/A                 | N/A             | N/A       |
 +-------------------------------+----------------+------------------+---------------+--------------+----------------+-----------------+-------------+-------------------+------------------+---------------------+--------------+------------+------------------+--------------------+-----------------+----------------+-------------------+----------+-----------------------+---------------------+-----------------+-----------+
 


Notes:

=over

=item 1. align_cell: Whether module supports aligning text horizontally in individual cells (left/right/middle)

=item 2. align_column: Whether module supports aligning text horizontally in a column (left/right/middle)

=item 3. align_row: Whether module supports aligning text horizontally in a row (left/right/middle)

=item 4. box_char: Whether module can utilize box-drawing characters

=item 5. color_data: Whether module supports ANSI colors (i.e. text with ANSI color codes can still be aligned properly)

=item 6. color_theme: Whether the module supports color theme/scheme

=item 7. colspan: Whether module supports column spans

=item 8. custom_border: Whether module allows customizing border in some way

=item 9. custom_color: Whether the module produces colored table and supports customizing color in some way

=item 10. multiline_data: Whether module supports aligning data cells that contain newlines

=item 11. rowspan: Whether module supports row spans

=item 12. speed: Rendering speed

=item 13. valign_cell: Whether module supports aligning text vertically in individual cells (top/bottom/middle)

=item 14. valign_column: Whether module supports aligning text vertically in a column (top/bottom/middle)

=item 15. valign_row: Whether module supports aligning text vertically in a row (top/bottom/middle)

=item 16. column_width: Whether module allows setting the width of columns

=item 17. pad: Whether module allows customizing cell horizontal padding

=item 18. per_column_width: Whether module allows setting column width on a per-column basis

=item 19. per_row_height: Whether module allows setting row height on a per-row basis

=item 20. row_height: Whether module allows setting the height of rows

=item 21. vpad: Whether module allows customizing cell vertical padding

=item 22. Depends on backend

=item 23. But this module can pass rendering to other module like Text::UnicodeBox::Table

=item 24. Newlines stripped

=item 25. Does not draw borders

=item 26. Irrelevant

=item 27. But make sure your CSV parser can handle multiline cell

=item 28. Not converted to HTML color elements

=back

=head1 BENCHMARKED MODULES

Version numbers shown below are the versions used when running the sample benchmark.

L<Text::Table::Any> 0.104

L<Text::UnicodeBox::Table>

L<Text::Table::Manifold> 1.03

L<Text::ANSITable> 0.602

L<Text::ASCIITable> 0.22

L<Text::FormatTable> 1.03

L<Text::MarkdownTable> 0.3.1

L<Text::Table> 1.133

L<Text::Table::Tiny> 1.02

L<Text::Table::TinyBorderStyle> 0.004

L<Text::Table::More> 0.014

L<Text::Table::Sprintf> 0.003

L<Text::Table::TinyColor> 0.002

L<Text::Table::TinyColorWide> 0.001

L<Text::Table::TinyWide> 0.001

L<Text::Table::Org> 0.02

L<Text::Table::CSV> 0.023

L<Text::Table::HTML> 0.004

L<Text::Table::HTML::DataTables> 0.007

L<Text::TabularDisplay> 1.38

=head1 BENCHMARK PARTICIPANTS

=over

=item * Text::Table::Any (perl_code)

L<Text::Table::Any>



=item * Text::UnicodeBox::Table (perl_code)

L<Text::UnicodeBox::Table>



=item * Text::Table::Manifold (perl_code)

L<Text::Table::Manifold>



=item * Text::ANSITable (perl_code)

L<Text::ANSITable>



=item * Text::ASCIITable (perl_code)

L<Text::ASCIITable>



=item * Text::FormatTable (perl_code)

L<Text::FormatTable>



=item * Text::MarkdownTable (perl_code)

L<Text::MarkdownTable>



=item * Text::Table (perl_code)

L<Text::Table>



=item * Text::Table::Tiny (perl_code)

L<Text::Table::Tiny>



=item * Text::Table::TinyBorderStyle (perl_code)

L<Text::Table::TinyBorderStyle>



=item * Text::Table::More (perl_code)

L<Text::Table::More>



=item * Text::Table::Sprintf (perl_code)

L<Text::Table::Sprintf>



=item * Text::Table::TinyColor (perl_code)

L<Text::Table::TinyColor>



=item * Text::Table::TinyColorWide (perl_code)

L<Text::Table::TinyColorWide>



=item * Text::Table::TinyWide (perl_code)

L<Text::Table::TinyWide>



=item * Text::Table::Org (perl_code)

L<Text::Table::Org>



=item * Text::Table::CSV (perl_code)

L<Text::Table::CSV>



=item * Text::Table::HTML (perl_code)

L<Text::Table::HTML>



=item * Text::Table::HTML::DataTables (perl_code)

L<Text::Table::HTML::DataTables>



=item * Text::TabularDisplay (perl_code)

L<Text::TabularDisplay>



=back

=head1 BENCHMARK DATASETS

=over

=item * tiny (1x1)

=item * small (3x5)

=item * wide (30x5)

=item * long (3x300)

=item * large (30x300)

=item * multiline data (2x1) (not included by default)

=item * wide char data (1x2) (not included by default)

=item * color data (1x2) (not included by default)

=back

=head1 SAMPLE BENCHMARK RESULTS

Run on: perl: I<< v5.30.2 >>, CPU: I<< Intel(R) Core(TM) i7-4770 CPU @ 3.40GHz (4 cores) >>, OS: I<< GNU/Linux LinuxMint version 19 >>, OS kernel: I<< Linux version 5.3.0-68-generic >>.

Benchmark with default options (C<< bencher --cpanmodules-module TextTable >>):

 #table1#
 {dataset=>"large (30x300)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+
 | Text::UnicodeBox::Table       |      1.4  |    700    |                 0.00% |             38200.92% |   0.00083 |      21 |
 | Text::ANSITable               |      3.24 |    309    |               127.30% |             16750.07% |   0.00027 |      20 |
 | Text::Table::More             |      6.65 |    150    |               366.41% |              8111.86% | 8.4e-05   |      20 |
 | Text::ASCIITable              |     15.4  |     65.1  |               977.22% |              3455.52% | 3.8e-05   |      21 |
 | Text::Table::TinyColorWide    |     21.4  |     46.7  |              1401.77% |              2450.38% | 3.5e-05   |      20 |
 | Text::FormatTable             |     22    |     45    |              1470.55% |              2338.69% | 8.2e-05   |      23 |
 | Text::Table::TinyWide         |     30.4  |     32.8  |              2035.98% |              1693.13% | 8.7e-06   |      20 |
 | Text::Table::Manifold         |     49    |     20    |              3369.28% |              1004.00% | 2.5e-05   |      20 |
 | Text::Table::Tiny             |     53.8  |     18.6  |              3672.14% |               915.36% |   1e-05   |      20 |
 | Text::TabularDisplay          |     58.9  |     17    |              4032.78% |               826.76% | 1.3e-05   |      22 |
 | Text::Table::TinyColor        |     78.6  |     12.7  |              5416.32% |               594.32% | 3.1e-06   |      20 |
 | Text::Table::HTML             |     79    |     13    |              5451.73% |               589.89% | 1.4e-05   |      20 |
 | Text::MarkdownTable           |    110    |      9    |              7680.32% |               392.28% | 3.6e-05   |      20 |
 | Text::Table                   |    140    |      7    |              9876.97% |               283.89% | 1.3e-05   |      20 |
 | Text::Table::HTML::DataTables |    160    |      6.1  |             11457.63% |               231.39% | 6.2e-06   |      20 |
 | Text::Table::CSV              |    282    |      3.55 |             19677.91% |                93.66% | 3.3e-06   |      21 |
 | Text::Table::Org              |    298    |      3.36 |             20807.96% |                83.19% | 2.9e-06   |      20 |
 | Text::Table::TinyBorderStyle  |    330    |      3.03 |             23023.92% |                65.63% | 9.1e-07   |      20 |
 | Text::Table::Any              |    520    |      1.9  |             36530.61% |                 4.56% | 4.3e-06   |      20 |
 | Text::Table::Sprintf          |    550    |      1.8  |             38200.92% |                 0.00% | 2.7e-06   |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+-----------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                   Rate  Text::UnicodeBox::Table  Text::ANSITable  Text::Table::More  Text::ASCIITable  Text::Table::TinyColorWide  Text::FormatTable  Text::Table::TinyWide  Text::Table::Manifold  Text::Table::Tiny  Text::TabularDisplay  Text::Table::HTML  Text::Table::TinyColor  Text::MarkdownTable  Text::Table  Text::Table::HTML::DataTables  Text::Table::CSV  Text::Table::Org  Text::Table::TinyBorderStyle  Text::Table::Any  Text::Table::Sprintf 
  Text::UnicodeBox::Table         1.4/s                       --             -55%               -78%              -90%                        -93%               -93%                   -95%                   -97%               -97%                  -97%               -98%                    -98%                 -98%         -99%                           -99%              -99%              -99%                          -99%              -99%                  -99% 
  Text::ANSITable                3.24/s                     126%               --               -51%              -78%                        -84%               -85%                   -89%                   -93%               -93%                  -94%               -95%                    -95%                 -97%         -97%                           -98%              -98%              -98%                          -99%              -99%                  -99% 
  Text::Table::More              6.65/s                     366%             106%                 --              -56%                        -68%               -70%                   -78%                   -86%               -87%                  -88%               -91%                    -91%                 -94%         -95%                           -95%              -97%              -97%                          -97%              -98%                  -98% 
  Text::ASCIITable               15.4/s                     975%             374%               130%                --                        -28%               -30%                   -49%                   -69%               -71%                  -73%               -80%                    -80%                 -86%         -89%                           -90%              -94%              -94%                          -95%              -97%                  -97% 
  Text::Table::TinyColorWide     21.4/s                    1398%             561%               221%               39%                          --                -3%                   -29%                   -57%               -60%                  -63%               -72%                    -72%                 -80%         -85%                           -86%              -92%              -92%                          -93%              -95%                  -96% 
  Text::FormatTable                22/s                    1455%             586%               233%               44%                          3%                 --                   -27%                   -55%               -58%                  -62%               -71%                    -71%                 -80%         -84%                           -86%              -92%              -92%                          -93%              -95%                  -96% 
  Text::Table::TinyWide          30.4/s                    2034%             842%               357%               98%                         42%                37%                     --                   -39%               -43%                  -48%               -60%                    -61%                 -72%         -78%                           -81%              -89%              -89%                          -90%              -94%                  -94% 
  Text::Table::Manifold            49/s                    3400%            1445%               650%              225%                        133%               125%                    63%                     --                -6%                  -15%               -35%                    -36%                 -55%         -65%                           -69%              -82%              -83%                          -84%              -90%                  -91% 
  Text::Table::Tiny              53.8/s                    3663%            1561%               706%              249%                        151%               141%                    76%                     7%                 --                   -8%               -30%                    -31%                 -51%         -62%                           -67%              -80%              -81%                          -83%              -89%                  -90% 
  Text::TabularDisplay           58.9/s                    4017%            1717%               782%              282%                        174%               164%                    92%                    17%                 9%                    --               -23%                    -25%                 -47%         -58%                           -64%              -79%              -80%                          -82%              -88%                  -89% 
  Text::Table::HTML                79/s                    5284%            2276%              1053%              400%                        259%               246%                   152%                    53%                43%                   30%                 --                     -2%                 -30%         -46%                           -53%              -72%              -74%                          -76%              -85%                  -86% 
  Text::Table::TinyColor         78.6/s                    5411%            2333%              1081%              412%                        267%               254%                   158%                    57%                46%                   33%                 2%                      --                 -29%         -44%                           -51%              -72%              -73%                          -76%              -85%                  -85% 
  Text::MarkdownTable             110/s                    7677%            3333%              1566%              623%                        418%               400%                   264%                   122%               106%                   88%                44%                     41%                   --         -22%                           -32%              -60%              -62%                          -66%              -78%                  -80% 
  Text::Table                     140/s                    9900%            4314%              2042%              829%                        567%               542%                   368%                   185%               165%                  142%                85%                     81%                  28%           --                           -12%              -49%              -52%                          -56%              -72%                  -74% 
  Text::Table::HTML::DataTables   160/s                   11375%            4965%              2359%              967%                        665%               637%                   437%                   227%               204%                  178%               113%                    108%                  47%          14%                             --              -41%              -44%                          -50%              -68%                  -70% 
  Text::Table::CSV                282/s                   19618%            8604%              4125%             1733%                       1215%              1167%                   823%                   463%               423%                  378%               266%                    257%                 153%          97%                            71%                --               -5%                          -14%              -46%                  -49% 
  Text::Table::Org                298/s                   20733%            9096%              4364%             1837%                       1289%              1239%                   876%                   495%               453%                  405%               286%                    277%                 167%         108%                            81%                5%                --                           -9%              -43%                  -46% 
  Text::Table::TinyBorderStyle    330/s                   23002%           10098%              4850%             2048%                       1441%              1385%                   982%                   560%               513%                  461%               329%                    319%                 197%         131%                           101%               17%               10%                            --              -37%                  -40% 
  Text::Table::Any                520/s                   36742%           16163%              7794%             3326%                       2357%              2268%                  1626%                   952%               878%                  794%               584%                    568%                 373%         268%                           221%               86%               76%                           59%                --                   -5% 
  Text::Table::Sprintf            550/s                   38788%           17066%              8233%             3516%                       2494%              2400%                  1722%                  1011%               933%                  844%               622%                    605%                 400%         288%                           238%               97%               86%                           68%                5%                    -- 
 
 Legends:
   Text::ANSITable: participant=Text::ANSITable
   Text::ASCIITable: participant=Text::ASCIITable
   Text::FormatTable: participant=Text::FormatTable
   Text::MarkdownTable: participant=Text::MarkdownTable
   Text::Table: participant=Text::Table
   Text::Table::Any: participant=Text::Table::Any
   Text::Table::CSV: participant=Text::Table::CSV
   Text::Table::HTML: participant=Text::Table::HTML
   Text::Table::HTML::DataTables: participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: participant=Text::Table::Manifold
   Text::Table::More: participant=Text::Table::More
   Text::Table::Org: participant=Text::Table::Org
   Text::Table::Sprintf: participant=Text::Table::Sprintf
   Text::Table::Tiny: participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: participant=Text::Table::TinyWide
   Text::TabularDisplay: participant=Text::TabularDisplay
   Text::UnicodeBox::Table: participant=Text::UnicodeBox::Table

 #table2#
 {dataset=>"long (3x300)"}
 +-------------------------------+-----------+------------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ms)  | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+------------+-----------------------+-----------------------+---------+---------+
 | Text::UnicodeBox::Table       |     11.2  | 89.3       |                 0.00% |             36003.77% | 7.8e-05 |      20 |
 | Text::ANSITable               |     29.5  | 33.8       |               163.76% |             13587.92% |   2e-05 |      20 |
 | Text::Table::More             |     61.7  | 16.2       |               451.03% |              6452.06% | 4.7e-06 |      20 |
 | Text::ASCIITable              |    160    |  6.1       |              1359.99% |              2372.89% | 6.9e-06 |      21 |
 | Text::FormatTable             |    199    |  5.02      |              1677.22% |              1931.48% | 3.8e-06 |      20 |
 | Text::Table::TinyColorWide    |    211    |  4.74      |              1782.01% |              1818.36% |   2e-06 |      20 |
 | Text::Table::TinyWide         |    294    |  3.4       |              2527.50% |              1274.07% | 2.9e-06 |      20 |
 | Text::TabularDisplay          |    430    |  2.3       |              3717.65% |               845.71% | 2.5e-06 |      22 |
 | Text::Table::Manifold         |    440    |  2.3       |              3856.28% |               812.57% | 3.1e-06 |      20 |
 | Text::Table::Tiny             |    470    |  2.1       |              4093.89% |               760.87% | 8.1e-06 |      21 |
 | Text::MarkdownTable           |    511    |  1.96      |              4459.96% |               691.76% | 1.8e-06 |      21 |
 | Text::Table                   |    640    |  1.6       |              5624.88% |               530.65% | 2.5e-06 |      20 |
 | Text::Table::TinyColor        |    730    |  1.4       |              6424.12% |               453.39% | 4.1e-06 |      21 |
 | Text::Table::HTML             |    736    |  1.36      |              6467.97% |               449.69% | 9.1e-07 |      20 |
 | Text::Table::HTML::DataTables |   1280    |  0.783     |             11306.74% |               216.51% | 4.3e-07 |      20 |
 | Text::Table::Org              |   2160    |  0.463     |             19174.58% |                87.31% | 2.1e-07 |      20 |
 | Text::Table::CSV              |   2170    |  0.461     |             19271.31% |                86.38% | 2.1e-07 |      20 |
 | Text::Table::TinyBorderStyle  |   2210    |  0.452     |             19662.37% |                82.69% | 2.7e-07 |      20 |
 | Text::Table::Any              |   3600    |  0.28      |             32228.61% |                11.68% | 9.1e-07 |      20 |
 | Text::Table::Sprintf          |   4044.06 |  0.2472763 |             36003.77% |                 0.00% | 1.8e-11 |      21 |
 +-------------------------------+-----------+------------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                      Rate  Text::UnicodeBox::Table  Text::ANSITable  Text::Table::More  Text::ASCIITable  Text::FormatTable  Text::Table::TinyColorWide  Text::Table::TinyWide  Text::TabularDisplay  Text::Table::Manifold  Text::Table::Tiny  Text::MarkdownTable  Text::Table  Text::Table::TinyColor  Text::Table::HTML  Text::Table::HTML::DataTables  Text::Table::Org  Text::Table::CSV  Text::Table::TinyBorderStyle  Text::Table::Any  Text::Table::Sprintf 
  Text::UnicodeBox::Table           11.2/s                       --             -62%               -81%              -93%               -94%                        -94%                   -96%                  -97%                   -97%               -97%                 -97%         -98%                    -98%               -98%                           -99%              -99%              -99%                          -99%              -99%                  -99% 
  Text::ANSITable                   29.5/s                     164%               --               -52%              -81%               -85%                        -85%                   -89%                  -93%                   -93%               -93%                 -94%         -95%                    -95%               -95%                           -97%              -98%              -98%                          -98%              -99%                  -99% 
  Text::Table::More                 61.7/s                     451%             108%                 --              -62%               -69%                        -70%                   -79%                  -85%                   -85%               -87%                 -87%         -90%                    -91%               -91%                           -95%              -97%              -97%                          -97%              -98%                  -98% 
  Text::ASCIITable                   160/s                    1363%             454%               165%                --               -17%                        -22%                   -44%                  -62%                   -62%               -65%                 -67%         -73%                    -77%               -77%                           -87%              -92%              -92%                          -92%              -95%                  -95% 
  Text::FormatTable                  199/s                    1678%             573%               222%               21%                 --                         -5%                   -32%                  -54%                   -54%               -58%                 -60%         -68%                    -72%               -72%                           -84%              -90%              -90%                          -90%              -94%                  -95% 
  Text::Table::TinyColorWide         211/s                    1783%             613%               241%               28%                 5%                          --                   -28%                  -51%                   -51%               -55%                 -58%         -66%                    -70%               -71%                           -83%              -90%              -90%                          -90%              -94%                  -94% 
  Text::Table::TinyWide              294/s                    2526%             894%               376%               79%                47%                         39%                     --                  -32%                   -32%               -38%                 -42%         -52%                    -58%               -60%                           -76%              -86%              -86%                          -86%              -91%                  -92% 
  Text::TabularDisplay               430/s                    3782%            1369%               604%              165%               118%                        106%                    47%                    --                     0%                -8%                 -14%         -30%                    -39%               -40%                           -65%              -79%              -79%                          -80%              -87%                  -89% 
  Text::Table::Manifold              440/s                    3782%            1369%               604%              165%               118%                        106%                    47%                    0%                     --                -8%                 -14%         -30%                    -39%               -40%                           -65%              -79%              -79%                          -80%              -87%                  -89% 
  Text::Table::Tiny                  470/s                    4152%            1509%               671%              190%               139%                        125%                    61%                    9%                     9%                 --                  -6%         -23%                    -33%               -35%                           -62%              -77%              -78%                          -78%              -86%                  -88% 
  Text::MarkdownTable                511/s                    4456%            1624%               726%              211%               156%                        141%                    73%                   17%                    17%                 7%                   --         -18%                    -28%               -30%                           -60%              -76%              -76%                          -76%              -85%                  -87% 
  Text::Table                        640/s                    5481%            2012%               912%              281%               213%                        196%                   112%                   43%                    43%                31%                  22%           --                    -12%               -15%                           -51%              -71%              -71%                          -71%              -82%                  -84% 
  Text::Table::TinyColor             730/s                    6278%            2314%              1057%              335%               258%                        238%                   142%                   64%                    64%                50%                  40%          14%                      --                -2%                           -44%              -66%              -67%                          -67%              -80%                  -82% 
  Text::Table::HTML                  736/s                    6466%            2385%              1091%              348%               269%                        248%                   149%                   69%                    69%                54%                  44%          17%                      2%                 --                           -42%              -65%              -66%                          -66%              -79%                  -81% 
  Text::Table::HTML::DataTables     1280/s                   11304%            4216%              1968%              679%               541%                        505%                   334%                  193%                   193%               168%                 150%         104%                     78%                73%                             --              -40%              -41%                          -42%              -64%                  -68% 
  Text::Table::Org                  2160/s                   19187%            7200%              3398%             1217%               984%                        923%                   634%                  396%                   396%               353%                 323%         245%                    202%               193%                            69%                --                0%                           -2%              -39%                  -46% 
  Text::Table::CSV                  2170/s                   19270%            7231%              3414%             1223%               988%                        928%                   637%                  398%                   398%               355%                 325%         247%                    203%               195%                            69%                0%                --                           -1%              -39%                  -46% 
  Text::Table::TinyBorderStyle      2210/s                   19656%            7377%              3484%             1249%              1010%                        948%                   652%                  408%                   408%               364%                 333%         253%                    209%               200%                            73%                2%                1%                            --              -38%                  -45% 
  Text::Table::Any                  3600/s                   31792%           11971%              5685%             2078%              1692%                       1592%                  1114%                  721%                   721%               650%                 599%         471%                    399%               385%                           179%               65%               64%                           61%                --                  -11% 
  Text::Table::Sprintf           4044.06/s                   36013%           13568%              6451%             2366%              1930%                       1816%                  1274%                  830%                   830%               749%                 692%         547%                    466%               449%                           216%               87%               86%                           82%               13%                    -- 
 
 Legends:
   Text::ANSITable: participant=Text::ANSITable
   Text::ASCIITable: participant=Text::ASCIITable
   Text::FormatTable: participant=Text::FormatTable
   Text::MarkdownTable: participant=Text::MarkdownTable
   Text::Table: participant=Text::Table
   Text::Table::Any: participant=Text::Table::Any
   Text::Table::CSV: participant=Text::Table::CSV
   Text::Table::HTML: participant=Text::Table::HTML
   Text::Table::HTML::DataTables: participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: participant=Text::Table::Manifold
   Text::Table::More: participant=Text::Table::More
   Text::Table::Org: participant=Text::Table::Org
   Text::Table::Sprintf: participant=Text::Table::Sprintf
   Text::Table::Tiny: participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: participant=Text::Table::TinyWide
   Text::TabularDisplay: participant=Text::TabularDisplay
   Text::UnicodeBox::Table: participant=Text::UnicodeBox::Table

 #table3#
 {dataset=>"small (3x5)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Text::UnicodeBox::Table       |     300   | 3         |                 0.00% |             41834.16% | 3.6e-05 |      20 |
 | Text::ANSITable               |    1110   | 0.901     |               282.97% |             10849.80% | 4.8e-07 |      20 |
 | Text::Table::More             |    2470   | 0.404     |               753.13% |              4815.31% | 2.1e-07 |      20 |
 | Text::ASCIITable              |    6100   | 0.16      |              1991.67% |              1904.82% | 1.9e-07 |      26 |
 | Text::FormatTable             |    8540   | 0.117     |              2847.04% |              1322.93% |   5e-08 |      23 |
 | Text::Table::Manifold         |    9330   | 0.107     |              3119.29% |              1202.59% | 5.3e-08 |      20 |
 | Text::Table                   |    9500   | 0.11      |              3162.28% |              1185.43% | 2.3e-07 |      26 |
 | Text::Table::TinyColorWide    |    9600   | 0.1       |              3199.07% |              1171.09% |   2e-07 |      23 |
 | Text::Table::TinyWide         |   13486.3 | 0.0741495 |              4552.56% |               801.31% | 1.8e-11 |      24 |
 | Text::Table::TinyBorderStyle  |   14600   | 0.0686    |              4927.82% |               734.04% | 2.7e-08 |      20 |
 | Text::MarkdownTable           |   15000   | 0.068     |              4982.13% |               725.13% | 3.5e-07 |      20 |
 | Text::Table::HTML::DataTables |   18602.6 | 0.0537559 |              6317.62% |               553.42% | 3.9e-11 |      20 |
 | Text::Table::Tiny             |   19000   | 0.053     |              6359.89% |               549.15% |   8e-08 |      20 |
 | Text::TabularDisplay          |   18900   | 0.0529    |              6419.63% |               543.20% | 2.7e-08 |      20 |
 | Text::Table::TinyColor        |   29400   | 0.034     |             10055.45% |               312.92% | 1.3e-08 |      21 |
 | Text::Table::HTML             |   38200   | 0.0262    |             13077.39% |               218.23% | 1.2e-08 |      25 |
 | Text::Table::Org              |   67200   | 0.0149    |             23080.98% |                80.90% | 5.1e-09 |      34 |
 | Text::Table::CSV              |   90000   | 0.01      |             30838.08% |                35.54% | 1.2e-07 |      20 |
 | Text::Table::Any              |  108000   | 0.00925   |             37208.05% |                12.40% | 2.8e-09 |      28 |
 | Text::Table::Sprintf          |  122000   | 0.00823   |             41834.16% |                 0.00% | 2.7e-09 |      30 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                      Rate  Text::UnicodeBox::Table  Text::ANSITable  Text::Table::More  Text::ASCIITable  Text::FormatTable  Text::Table  Text::Table::Manifold  Text::Table::TinyColorWide  Text::Table::TinyWide  Text::Table::TinyBorderStyle  Text::MarkdownTable  Text::Table::HTML::DataTables  Text::Table::Tiny  Text::TabularDisplay  Text::Table::TinyColor  Text::Table::HTML  Text::Table::Org  Text::Table::CSV  Text::Table::Any  Text::Table::Sprintf 
  Text::UnicodeBox::Table            300/s                       --             -69%               -86%              -94%               -96%         -96%                   -96%                        -96%                   -97%                          -97%                 -97%                           -98%               -98%                  -98%                    -98%               -99%              -99%              -99%              -99%                  -99% 
  Text::ANSITable                   1110/s                     232%               --               -55%              -82%               -87%         -87%                   -88%                        -88%                   -91%                          -92%                 -92%                           -94%               -94%                  -94%                    -96%               -97%              -98%              -98%              -98%                  -99% 
  Text::Table::More                 2470/s                     642%             123%                 --              -60%               -71%         -72%                   -73%                        -75%                   -81%                          -83%                 -83%                           -86%               -86%                  -86%                    -91%               -93%              -96%              -97%              -97%                  -97% 
  Text::ASCIITable                  6100/s                    1775%             463%               152%                --               -26%         -31%                   -33%                        -37%                   -53%                          -57%                 -57%                           -66%               -66%                  -66%                    -78%               -83%              -90%              -93%              -94%                  -94% 
  Text::FormatTable                 8540/s                    2464%             670%               245%               36%                 --          -5%                    -8%                        -14%                   -36%                          -41%                 -41%                           -54%               -54%                  -54%                    -70%               -77%              -87%              -91%              -92%                  -92% 
  Text::Table                       9500/s                    2627%             719%               267%               45%                 6%           --                    -2%                         -9%                   -32%                          -37%                 -38%                           -51%               -51%                  -51%                    -69%               -76%              -86%              -90%              -91%                  -92% 
  Text::Table::Manifold             9330/s                    2703%             742%               277%               49%                 9%           2%                     --                         -6%                   -30%                          -35%                 -36%                           -49%               -50%                  -50%                    -68%               -75%              -86%              -90%              -91%                  -92% 
  Text::Table::TinyColorWide        9600/s                    2900%             801%               304%               59%                16%           9%                     6%                          --                   -25%                          -31%                 -31%                           -46%               -47%                  -47%                    -65%               -73%              -85%              -90%              -90%                  -91% 
  Text::Table::TinyWide          13486.3/s                    3945%            1115%               444%              115%                57%          48%                    44%                         34%                     --                           -7%                  -8%                           -27%               -28%                  -28%                    -54%               -64%              -79%              -86%              -87%                  -88% 
  Text::Table::TinyBorderStyle     14600/s                    4273%            1213%               488%              133%                70%          60%                    55%                         45%                     8%                            --                   0%                           -21%               -22%                  -22%                    -50%               -61%              -78%              -85%              -86%                  -88% 
  Text::MarkdownTable              15000/s                    4311%            1225%               494%              135%                72%          61%                    57%                         47%                     9%                            0%                   --                           -20%               -22%                  -22%                    -50%               -61%              -78%              -85%              -86%                  -87% 
  Text::Table::HTML::DataTables  18602.6/s                    5480%            1576%               651%              197%               117%         104%                    99%                         86%                    37%                           27%                  26%                             --                -1%                   -1%                    -36%               -51%              -72%              -81%              -82%                  -84% 
  Text::Table::Tiny                19000/s                    5560%            1600%               662%              201%               120%         107%                   101%                         88%                    39%                           29%                  28%                             1%                 --                    0%                    -35%               -50%              -71%              -81%              -82%                  -84% 
  Text::TabularDisplay             18900/s                    5571%            1603%               663%              202%               121%         107%                   102%                         89%                    40%                           29%                  28%                             1%                 0%                    --                    -35%               -50%              -71%              -81%              -82%                  -84% 
  Text::Table::TinyColor           29400/s                    8723%            2550%              1088%              370%               244%         223%                   214%                        194%                   118%                          101%                 100%                            58%                55%                   55%                      --               -22%              -56%              -70%              -72%                  -75% 
  Text::Table::HTML                38200/s                   11350%            3338%              1441%              510%               346%         319%                   308%                        281%                   183%                          161%                 159%                           105%               102%                  101%                     29%                 --              -43%              -61%              -64%                  -68% 
  Text::Table::Org                 67200/s                   20034%            5946%              2611%              973%               685%         638%                   618%                        571%                   397%                          360%                 356%                           260%               255%                  255%                    128%                75%                --              -32%              -37%                  -44% 
  Text::Table::CSV                 90000/s                   29900%            8910%              3940%             1500%              1070%        1000%                   969%                        900%                   641%                          586%                 580%                           437%               430%                  429%                    240%               162%               49%                --               -7%                  -17% 
  Text::Table::Any                108000/s                   32332%            9640%              4267%             1629%              1164%        1089%                  1056%                        981%                   701%                          641%                 635%                           481%               472%                  471%                    267%               183%               61%                8%                --                  -11% 
  Text::Table::Sprintf            122000/s                   36352%           10847%              4808%             1844%              1321%        1236%                  1200%                       1115%                   800%                          733%                 726%                           553%               543%                  542%                    313%               218%               81%               21%               12%                    -- 
 
 Legends:
   Text::ANSITable: participant=Text::ANSITable
   Text::ASCIITable: participant=Text::ASCIITable
   Text::FormatTable: participant=Text::FormatTable
   Text::MarkdownTable: participant=Text::MarkdownTable
   Text::Table: participant=Text::Table
   Text::Table::Any: participant=Text::Table::Any
   Text::Table::CSV: participant=Text::Table::CSV
   Text::Table::HTML: participant=Text::Table::HTML
   Text::Table::HTML::DataTables: participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: participant=Text::Table::Manifold
   Text::Table::More: participant=Text::Table::More
   Text::Table::Org: participant=Text::Table::Org
   Text::Table::Sprintf: participant=Text::Table::Sprintf
   Text::Table::Tiny: participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: participant=Text::Table::TinyWide
   Text::TabularDisplay: participant=Text::TabularDisplay
   Text::UnicodeBox::Table: participant=Text::UnicodeBox::Table

 #table4#
 {dataset=>"tiny (1x1)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Text::UnicodeBox::Table       |       700 | 2         |                 0.00% |             53151.65% | 5.8e-05 |      20 |
 | Text::ANSITable               |      3950 | 0.253     |               494.32% |              8860.11% |   2e-07 |      22 |
 | Text::Table::More             |      9400 | 0.11      |              1320.53% |              3648.71% | 2.7e-07 |      20 |
 | Text::Table::Manifold         |     16000 | 0.064     |              2263.64% |              2152.95% |   1e-07 |      21 |
 | Text::Table::TinyBorderStyle  |     19000 | 0.053     |              2740.13% |              1774.97% | 1.1e-07 |      29 |
 | Text::ASCIITable              |     22000 | 0.046     |              3153.23% |              1536.89% | 6.2e-08 |      23 |
 | Text::Table::HTML::DataTables |     25000 | 0.04      |              3625.05% |              1329.55% | 6.2e-08 |      23 |
 | Text::Table                   |     25000 | 0.039     |              3722.96% |              1292.94% | 4.9e-08 |      24 |
 | Text::MarkdownTable           |     29000 | 0.034     |              4286.43% |              1114.01% | 1.7e-07 |      20 |
 | Text::FormatTable             |     42400 | 0.0236    |              6287.80% |               733.65% | 1.8e-08 |      24 |
 | Text::Table::TinyColorWide    |     57000 | 0.017     |              8530.95% |               516.98% | 2.7e-08 |      20 |
 | Text::Table::Tiny             |     68300 | 0.0147    |             10171.32% |               418.45% | 6.7e-09 |      20 |
 | Text::Table::TinyWide         |     74900 | 0.0134    |             11166.37% |               372.66% | 6.7e-09 |      20 |
 | Text::TabularDisplay          |     77000 | 0.013     |             11542.97% |               357.37% | 2.5e-08 |      23 |
 | Text::Table::TinyColor        |    114550 | 0.00873   |             17137.30% |               208.93% | 3.6e-11 |      21 |
 | Text::Table::HTML             |    181000 | 0.00552   |             27158.21% |                95.36% | 1.6e-09 |      21 |
 | Text::Table::Org              |    182000 | 0.0055    |             27269.66% |                94.56% |   5e-09 |      20 |
 | Text::Table::Any              |    258580 | 0.0038673 |             38811.12% |                36.85% | 3.6e-11 |      20 |
 | Text::Table::Sprintf          |    352000 | 0.00284   |             52919.27% |                 0.44% | 8.3e-10 |      20 |
 | Text::Table::CSV              |    354000 | 0.00283   |             53151.65% |                 0.00% | 2.5e-09 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                     Rate  Text::UnicodeBox::Table  Text::ANSITable  Text::Table::More  Text::Table::Manifold  Text::Table::TinyBorderStyle  Text::ASCIITable  Text::Table::HTML::DataTables  Text::Table  Text::MarkdownTable  Text::FormatTable  Text::Table::TinyColorWide  Text::Table::Tiny  Text::Table::TinyWide  Text::TabularDisplay  Text::Table::TinyColor  Text::Table::HTML  Text::Table::Org  Text::Table::Any  Text::Table::Sprintf  Text::Table::CSV 
  Text::UnicodeBox::Table           700/s                       --             -87%               -94%                   -96%                          -97%              -97%                           -98%         -98%                 -98%               -98%                        -99%               -99%                   -99%                  -99%                    -99%               -99%              -99%              -99%                  -99%              -99% 
  Text::ANSITable                  3950/s                     690%               --               -56%                   -74%                          -79%              -81%                           -84%         -84%                 -86%               -90%                        -93%               -94%                   -94%                  -94%                    -96%               -97%              -97%              -98%                  -98%              -98% 
  Text::Table::More                9400/s                    1718%             129%                 --                   -41%                          -51%              -58%                           -63%         -64%                 -69%               -78%                        -84%               -86%                   -87%                  -88%                    -92%               -94%              -95%              -96%                  -97%              -97% 
  Text::Table::Manifold           16000/s                    3025%             295%                71%                     --                          -17%              -28%                           -37%         -39%                 -46%               -63%                        -73%               -77%                   -79%                  -79%                    -86%               -91%              -91%              -93%                  -95%              -95% 
  Text::Table::TinyBorderStyle    19000/s                    3673%             377%               107%                    20%                            --              -13%                           -24%         -26%                 -35%               -55%                        -67%               -72%                   -74%                  -75%                    -83%               -89%              -89%              -92%                  -94%              -94% 
  Text::ASCIITable                22000/s                    4247%             450%               139%                    39%                           15%                --                           -13%         -15%                 -26%               -48%                        -63%               -68%                   -70%                  -71%                    -81%               -88%              -88%              -91%                  -93%              -93% 
  Text::Table::HTML::DataTables   25000/s                    4900%             532%               175%                    60%                           32%               14%                             --          -2%                 -14%               -41%                        -57%               -63%                   -66%                  -67%                    -78%               -86%              -86%              -90%                  -92%              -92% 
  Text::Table                     25000/s                    5028%             548%               182%                    64%                           35%               17%                             2%           --                 -12%               -39%                        -56%               -62%                   -65%                  -66%                    -77%               -85%              -85%              -90%                  -92%              -92% 
  Text::MarkdownTable             29000/s                    5782%             644%               223%                    88%                           55%               35%                            17%          14%                   --               -30%                        -50%               -56%                   -60%                  -61%                    -74%               -83%              -83%              -88%                  -91%              -91% 
  Text::FormatTable               42400/s                    8374%             972%               366%                   171%                          124%               94%                            69%          65%                  44%                 --                        -27%               -37%                   -43%                  -44%                    -63%               -76%              -76%              -83%                  -87%              -88% 
  Text::Table::TinyColorWide      57000/s                   11664%            1388%               547%                   276%                          211%              170%                           135%         129%                 100%                38%                          --               -13%                   -21%                  -23%                    -48%               -67%              -67%              -77%                  -83%              -83% 
  Text::Table::Tiny               68300/s                   13505%            1621%               648%                   335%                          260%              212%                           172%         165%                 131%                60%                         15%                 --                    -8%                  -11%                    -40%               -62%              -62%              -73%                  -80%              -80% 
  Text::Table::TinyWide           74900/s                   14825%            1788%               720%                   377%                          295%              243%                           198%         191%                 153%                76%                         26%                 9%                     --                   -2%                    -34%               -58%              -58%              -71%                  -78%              -78% 
  Text::TabularDisplay            77000/s                   15284%            1846%               746%                   392%                          307%              253%                           207%         200%                 161%                81%                         30%                13%                     3%                    --                    -32%               -57%              -57%              -70%                  -78%              -78% 
  Text::Table::TinyColor         114550/s                   22809%            2798%              1160%                   633%                          507%              426%                           358%         346%                 289%               170%                         94%                68%                    53%                   48%                      --               -36%              -36%              -55%                  -67%              -67% 
  Text::Table::HTML              181000/s                   36131%            4483%              1892%                  1059%                          860%              733%                           624%         606%                 515%               327%                        207%               166%                   142%                  135%                     58%                 --                0%              -29%                  -48%              -48% 
  Text::Table::Org               182000/s                   36263%            4500%              1900%                  1063%                          863%              736%                           627%         609%                 518%               329%                        209%               167%                   143%                  136%                     58%                 0%                --              -29%                  -48%              -48% 
  Text::Table::Any               258580/s                   51615%            6442%              2744%                  1554%                         1270%             1089%                           934%         908%                 779%               510%                        339%               280%                   246%                  236%                    125%                42%               42%                --                  -26%              -26% 
  Text::Table::Sprintf           352000/s                   70322%            8808%              3773%                  2153%                         1766%             1519%                          1308%        1273%                1097%               730%                        498%               417%                   371%                  357%                    207%                94%               93%               36%                    --                0% 
  Text::Table::CSV               354000/s                   70571%            8839%              3786%                  2161%                         1772%             1525%                          1313%        1278%                1101%               733%                        500%               419%                   373%                  359%                    208%                95%               94%               36%                    0%                -- 
 
 Legends:
   Text::ANSITable: participant=Text::ANSITable
   Text::ASCIITable: participant=Text::ASCIITable
   Text::FormatTable: participant=Text::FormatTable
   Text::MarkdownTable: participant=Text::MarkdownTable
   Text::Table: participant=Text::Table
   Text::Table::Any: participant=Text::Table::Any
   Text::Table::CSV: participant=Text::Table::CSV
   Text::Table::HTML: participant=Text::Table::HTML
   Text::Table::HTML::DataTables: participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: participant=Text::Table::Manifold
   Text::Table::More: participant=Text::Table::More
   Text::Table::Org: participant=Text::Table::Org
   Text::Table::Sprintf: participant=Text::Table::Sprintf
   Text::Table::Tiny: participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: participant=Text::Table::TinyWide
   Text::TabularDisplay: participant=Text::TabularDisplay
   Text::UnicodeBox::Table: participant=Text::UnicodeBox::Table

 #table5#
 {dataset=>"wide (30x5)"}
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | participant                   | rate (/s) | time (ms) | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors | samples |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+
 | Text::UnicodeBox::Table       |        53 |   19      |                 0.00% |             36286.32% |   5e-05 |      20 |
 | Text::ANSITable               |       142 |    7.06   |               169.86% |             13383.28% | 3.3e-06 |      21 |
 | Text::Table::More             |       310 |    3.2    |               491.85% |              6047.87% | 1.1e-05 |      20 |
 | Text::ASCIITable              |       650 |    1.54   |              1138.03% |              2839.04% | 1.1e-06 |      20 |
 | Text::FormatTable             |       872 |    1.15   |              1560.07% |              2091.86% | 9.1e-07 |      20 |
 | Text::Table::TinyColorWide    |      1070 |    0.933  |              1940.08% |              1683.57% | 4.8e-07 |      20 |
 | Text::Table                   |      1400 |    0.71   |              2584.33% |              1255.51% | 3.4e-06 |      22 |
 | Text::Table::TinyWide         |      1500 |    0.669  |              2747.03% |              1178.05% | 2.1e-07 |      20 |
 | Text::Table::Manifold         |      2050 |    0.487  |              3807.64% |               831.16% | 4.3e-07 |      20 |
 | Text::Table::Tiny             |      2500 |    0.4    |              4659.89% |               664.44% | 2.1e-07 |      20 |
 | Text::TabularDisplay          |      2800 |    0.36   |              5154.45% |               592.49% | 6.9e-07 |      20 |
 | Text::Table::TinyColor        |      3700 |    0.27   |              6938.91% |               416.93% | 1.6e-07 |      21 |
 | Text::MarkdownTable           |      4000 |    0.25   |              7603.74% |               372.32% | 2.4e-06 |      21 |
 | Text::Table::TinyBorderStyle  |      4050 |    0.247  |              7606.42% |               372.16% | 2.1e-07 |      20 |
 | Text::Table::HTML             |      4730 |    0.212  |              8899.63% |               304.31% |   2e-07 |      22 |
 | Text::Table::HTML::DataTables |      6200 |    0.16   |             11689.03% |               208.65% | 1.8e-07 |      29 |
 | Text::Table::Org              |     11200 |    0.0894 |             21200.84% |                70.82% | 2.7e-08 |      20 |
 | Text::Table::CSV              |     14400 |    0.0693 |             27377.76% |                32.42% | 2.7e-08 |      20 |
 | Text::Table::Any              |     18000 |    0.055  |             34594.43% |                 4.88% |   8e-08 |      20 |
 | Text::Table::Sprintf          |     19000 |    0.052  |             36286.32% |                 0.00% | 1.1e-07 |      20 |
 +-------------------------------+-----------+-----------+-----------------------+-----------------------+---------+---------+

Formatted as L<Benchmark.pm|Benchmark> result:

                                    Rate  Text::UnicodeBox::Table  Text::ANSITable  Text::Table::More  Text::ASCIITable  Text::FormatTable  Text::Table::TinyColorWide  Text::Table  Text::Table::TinyWide  Text::Table::Manifold  Text::Table::Tiny  Text::TabularDisplay  Text::Table::TinyColor  Text::MarkdownTable  Text::Table::TinyBorderStyle  Text::Table::HTML  Text::Table::HTML::DataTables  Text::Table::Org  Text::Table::CSV  Text::Table::Any  Text::Table::Sprintf 
  Text::UnicodeBox::Table           53/s                       --             -62%               -83%              -91%               -93%                        -95%         -96%                   -96%                   -97%               -97%                  -98%                    -98%                 -98%                          -98%               -98%                           -99%              -99%              -99%              -99%                  -99% 
  Text::ANSITable                  142/s                     169%               --               -54%              -78%               -83%                        -86%         -89%                   -90%                   -93%               -94%                  -94%                    -96%                 -96%                          -96%               -96%                           -97%              -98%              -99%              -99%                  -99% 
  Text::Table::More                310/s                     493%             120%                 --              -51%               -64%                        -70%         -77%                   -79%                   -84%               -87%                  -88%                    -91%                 -92%                          -92%               -93%                           -95%              -97%              -97%              -98%                  -98% 
  Text::ASCIITable                 650/s                    1133%             358%               107%                --               -25%                        -39%         -53%                   -56%                   -68%               -74%                  -76%                    -82%                 -83%                          -83%               -86%                           -89%              -94%              -95%              -96%                  -96% 
  Text::FormatTable                872/s                    1552%             513%               178%               33%                 --                        -18%         -38%                   -41%                   -57%               -65%                  -68%                    -76%                 -78%                          -78%               -81%                           -86%              -92%              -93%              -95%                  -95% 
  Text::Table::TinyColorWide      1070/s                    1936%             656%               242%               65%                23%                          --         -23%                   -28%                   -47%               -57%                  -61%                    -71%                 -73%                          -73%               -77%                           -82%              -90%              -92%              -94%                  -94% 
  Text::Table                     1400/s                    2576%             894%               350%              116%                61%                         31%           --                    -5%                   -31%               -43%                  -49%                    -61%                 -64%                          -65%               -70%                           -77%              -87%              -90%              -92%                  -92% 
  Text::Table::TinyWide           1500/s                    2740%             955%               378%              130%                71%                         39%           6%                     --                   -27%               -40%                  -46%                    -59%                 -62%                          -63%               -68%                           -76%              -86%              -89%              -91%                  -92% 
  Text::Table::Manifold           2050/s                    3801%            1349%               557%              216%               136%                         91%          45%                    37%                     --               -17%                  -26%                    -44%                 -48%                          -49%               -56%                           -67%              -81%              -85%              -88%                  -89% 
  Text::Table::Tiny               2500/s                    4650%            1664%               700%              285%               187%                        133%          77%                    67%                    21%                 --                  -10%                    -32%                 -37%                          -38%               -47%                           -60%              -77%              -82%              -86%                  -87% 
  Text::TabularDisplay            2800/s                    5177%            1861%               788%              327%               219%                        159%          97%                    85%                    35%                11%                    --                    -24%                 -30%                          -31%               -41%                           -55%              -75%              -80%              -84%                  -85% 
  Text::Table::TinyColor          3700/s                    6937%            2514%              1085%              470%               325%                        245%         162%                   147%                    80%                48%                   33%                      --                  -7%                           -8%               -21%                           -40%              -66%              -74%              -79%                  -80% 
  Text::MarkdownTable             4000/s                    7500%            2724%              1180%              516%               359%                        273%         184%                   167%                    94%                60%                   43%                      8%                   --                           -1%               -15%                           -36%              -64%              -72%              -78%                  -79% 
  Text::Table::TinyBorderStyle    4050/s                    7592%            2758%              1195%              523%               365%                        277%         187%                   170%                    97%                61%                   45%                      9%                   1%                            --               -14%                           -35%              -63%              -71%              -77%                  -78% 
  Text::Table::HTML               4730/s                    8862%            3230%              1409%              626%               442%                        340%         234%                   215%                   129%                88%                   69%                     27%                  17%                           16%                 --                           -24%              -57%              -67%              -74%                  -75% 
  Text::Table::HTML::DataTables   6200/s                   11775%            4312%              1900%              862%               618%                        483%         343%                   318%                   204%               150%                  125%                     68%                  56%                           54%                32%                             --              -44%              -56%              -65%                  -67% 
  Text::Table::Org               11200/s                   21152%            7797%              3479%             1622%              1186%                        943%         694%                   648%                   444%               347%                  302%                    202%                 179%                          176%               137%                            78%                --              -22%              -38%                  -41% 
  Text::Table::CSV               14400/s                   27317%           10087%              4517%             2122%              1559%                       1246%         924%                   865%                   602%               477%                  419%                    289%                 260%                          256%               205%                           130%               29%                --              -20%                  -24% 
  Text::Table::Any               18000/s                   34445%           12736%              5718%             2700%              1990%                       1596%        1190%                  1116%                   785%               627%                  554%                    390%                 354%                          349%               285%                           190%               62%               26%                --                   -5% 
  Text::Table::Sprintf           19000/s                   36438%           13476%              6053%             2861%              2111%                       1694%        1265%                  1186%                   836%               669%                  592%                    419%                 380%                          375%               307%                           207%               71%               33%                5%                    -- 
 
 Legends:
   Text::ANSITable: participant=Text::ANSITable
   Text::ASCIITable: participant=Text::ASCIITable
   Text::FormatTable: participant=Text::FormatTable
   Text::MarkdownTable: participant=Text::MarkdownTable
   Text::Table: participant=Text::Table
   Text::Table::Any: participant=Text::Table::Any
   Text::Table::CSV: participant=Text::Table::CSV
   Text::Table::HTML: participant=Text::Table::HTML
   Text::Table::HTML::DataTables: participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: participant=Text::Table::Manifold
   Text::Table::More: participant=Text::Table::More
   Text::Table::Org: participant=Text::Table::Org
   Text::Table::Sprintf: participant=Text::Table::Sprintf
   Text::Table::Tiny: participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: participant=Text::Table::TinyWide
   Text::TabularDisplay: participant=Text::TabularDisplay
   Text::UnicodeBox::Table: participant=Text::UnicodeBox::Table


Benchmark module startup overhead (C<< bencher --cpanmodules-module TextTable --module-startup >>):

 #table6#
 +-------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | participant                   | time (ms) | mod_overhead_time | pct_faster_vs_slowest | pct_slower_vs_fastest |  errors   | samples |
 +-------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+
 | Text::UnicodeBox::Table       |     130   |             124   |                 0.00% |              1920.11% |   0.00018 |      22 |
 | Text::Table::Manifold         |      68   |              62   |                89.17% |               967.88% |   0.00021 |      20 |
 | Text::ANSITable               |      38   |              32   |               240.29% |               493.63% |   0.00016 |      20 |
 | Text::MarkdownTable           |      35   |              29   |               268.03% |               448.90% |   0.00015 |      20 |
 | Text::Table::TinyColorWide    |      27   |              21   |               377.65% |               322.92% |   0.00012 |      20 |
 | Text::Table::TinyWide         |      25   |              19   |               414.71% |               292.48% |   0.00019 |      20 |
 | Text::Table::More             |      20   |              14   |               531.10% |               220.09% |   0.00016 |      20 |
 | Text::Table                   |      18   |              12   |               594.97% |               190.68% |   0.00018 |      20 |
 | Text::ASCIITable              |      10   |               4   |               775.30% |               130.79% |   0.00016 |      20 |
 | Text::Table::Tiny             |      14   |               8   |               793.38% |               126.12% | 8.7e-05   |      21 |
 | Text::Table::TinyColor        |      10   |               4   |               994.03% |                84.65% |   0.00024 |      21 |
 | Text::FormatTable             |      10   |               4   |              1022.66% |                79.94% |   0.00017 |      20 |
 | Text::Table::TinyBorderStyle  |       9   |               3   |              1299.43% |                44.35% |   0.00013 |      20 |
 | Text::TabularDisplay          |       9   |               3   |              1356.49% |                38.70% |   0.00017 |      21 |
 | Text::Table::HTML             |       9   |               3   |              1390.38% |                35.54% |   0.00013 |      24 |
 | Text::Table::HTML::DataTables |       8   |               2   |              1501.89% |                26.11% | 9.6e-05   |      20 |
 | Text::Table::Any              |       7.6 |               1.6 |              1579.81% |                20.26% | 4.1e-05   |      20 |
 | Text::Table::CSV              |       7   |               1   |              1756.55% |                 8.81% |   0.00012 |      21 |
 | perl -e1 (baseline)           |       6   |               0   |              1895.80% |                 1.22% |   0.00013 |      20 |
 | Text::Table::Sprintf          |       6   |               0   |              1899.53% |                 1.03% |   0.00019 |      20 |
 | Text::Table::Org              |       6   |               0   |              1920.11% |                 0.00% |   0.0002  |      20 |
 +-------------------------------+-----------+-------------------+-----------------------+-----------------------+-----------+---------+


Formatted as L<Benchmark.pm|Benchmark> result:

                                  Rate  Text::UnicodeBox::Table  Text::Table::Manifold  Text::ANSITable  Text::MarkdownTable  Text::Table::TinyColorWide  Text::Table::TinyWide  Text::Table::More  Text::Table  Text::Table::Tiny  Text::ASCIITable  Text::Table::TinyColor  Text::FormatTable  Text::Table::TinyBorderStyle  Text::TabularDisplay  Text::Table::HTML  Text::Table::HTML::DataTables  Text::Table::Any  Text::Table::CSV  perl -e1 (baseline)  Text::Table::Sprintf  Text::Table::Org 
  Text::UnicodeBox::Table        0.0/s                       --                   -47%             -70%                 -73%                        -79%                   -80%               -84%         -86%               -89%              -92%                    -92%               -92%                          -93%                  -93%               -93%                           -93%              -94%              -94%                 -95%                  -95%              -95% 
  Text::Table::Manifold          0.0/s                      91%                     --             -44%                 -48%                        -60%                   -63%               -70%         -73%               -79%              -85%                    -85%               -85%                          -86%                  -86%               -86%                           -88%              -88%              -89%                 -91%                  -91%              -91% 
  Text::ANSITable                0.0/s                     242%                    78%               --                  -7%                        -28%                   -34%               -47%         -52%               -63%              -73%                    -73%               -73%                          -76%                  -76%               -76%                           -78%              -80%              -81%                 -84%                  -84%              -84% 
  Text::MarkdownTable            0.0/s                     271%                    94%               8%                   --                        -22%                   -28%               -42%         -48%               -60%              -71%                    -71%               -71%                          -74%                  -74%               -74%                           -77%              -78%              -80%                 -82%                  -82%              -82% 
  Text::Table::TinyColorWide     0.0/s                     381%                   151%              40%                  29%                          --                    -7%               -25%         -33%               -48%              -62%                    -62%               -62%                          -66%                  -66%               -66%                           -70%              -71%              -74%                 -77%                  -77%              -77% 
  Text::Table::TinyWide          0.0/s                     420%                   172%              52%                  39%                          8%                     --               -19%         -28%               -43%              -60%                    -60%               -60%                          -64%                  -64%               -64%                           -68%              -69%              -72%                 -76%                  -76%              -76% 
  Text::Table::More              0.1/s                     550%                   240%              89%                  75%                         35%                    25%                 --          -9%               -30%              -50%                    -50%               -50%                          -55%                  -55%               -55%                           -60%              -62%              -65%                 -70%                  -70%              -70% 
  Text::Table                    0.1/s                     622%                   277%             111%                  94%                         50%                    38%                11%           --               -22%              -44%                    -44%               -44%                          -50%                  -50%               -50%                           -55%              -57%              -61%                 -66%                  -66%              -66% 
  Text::Table::Tiny              0.1/s                     828%                   385%             171%                 150%                         92%                    78%                42%          28%                 --              -28%                    -28%               -28%                          -35%                  -35%               -35%                           -42%              -45%              -50%                 -57%                  -57%              -57% 
  Text::ASCIITable               0.1/s                    1200%                   580%             280%                 250%                        170%                   150%               100%          80%                39%                --                      0%                 0%                           -9%                   -9%                -9%                           -19%              -24%              -30%                 -40%                  -40%              -40% 
  Text::Table::TinyColor         0.1/s                    1200%                   580%             280%                 250%                        170%                   150%               100%          80%                39%                0%                      --                 0%                           -9%                   -9%                -9%                           -19%              -24%              -30%                 -40%                  -40%              -40% 
  Text::FormatTable              0.1/s                    1200%                   580%             280%                 250%                        170%                   150%               100%          80%                39%                0%                      0%                 --                           -9%                   -9%                -9%                           -19%              -24%              -30%                 -40%                  -40%              -40% 
  Text::Table::TinyBorderStyle   0.1/s                    1344%                   655%             322%                 288%                        200%                   177%               122%         100%                55%               11%                     11%                11%                            --                    0%                 0%                           -11%              -15%              -22%                 -33%                  -33%              -33% 
  Text::TabularDisplay           0.1/s                    1344%                   655%             322%                 288%                        200%                   177%               122%         100%                55%               11%                     11%                11%                            0%                    --                 0%                           -11%              -15%              -22%                 -33%                  -33%              -33% 
  Text::Table::HTML              0.1/s                    1344%                   655%             322%                 288%                        200%                   177%               122%         100%                55%               11%                     11%                11%                            0%                    0%                 --                           -11%              -15%              -22%                 -33%                  -33%              -33% 
  Text::Table::HTML::DataTables  0.1/s                    1525%                   750%             375%                 337%                        237%                   212%               150%         125%                75%               25%                     25%                25%                           12%                   12%                12%                             --               -5%              -12%                 -25%                  -25%              -25% 
  Text::Table::Any               0.1/s                    1610%                   794%             400%                 360%                        255%                   228%               163%         136%                84%               31%                     31%                31%                           18%                   18%                18%                             5%                --               -7%                 -21%                  -21%              -21% 
  Text::Table::CSV               0.1/s                    1757%                   871%             442%                 400%                        285%                   257%               185%         157%               100%               42%                     42%                42%                           28%                   28%                28%                            14%                8%                --                 -14%                  -14%              -14% 
  perl -e1 (baseline)            0.2/s                    2066%                  1033%             533%                 483%                        350%                   316%               233%         200%               133%               66%                     66%                66%                           50%                   50%                50%                            33%               26%               16%                   --                    0%                0% 
  Text::Table::Sprintf           0.2/s                    2066%                  1033%             533%                 483%                        350%                   316%               233%         200%               133%               66%                     66%                66%                           50%                   50%                50%                            33%               26%               16%                   0%                    --                0% 
  Text::Table::Org               0.2/s                    2066%                  1033%             533%                 483%                        350%                   316%               233%         200%               133%               66%                     66%                66%                           50%                   50%                50%                            33%               26%               16%                   0%                    0%                -- 
 
 Legends:
   Text::ANSITable: mod_overhead_time=32 participant=Text::ANSITable
   Text::ASCIITable: mod_overhead_time=4 participant=Text::ASCIITable
   Text::FormatTable: mod_overhead_time=4 participant=Text::FormatTable
   Text::MarkdownTable: mod_overhead_time=29 participant=Text::MarkdownTable
   Text::Table: mod_overhead_time=12 participant=Text::Table
   Text::Table::Any: mod_overhead_time=1.6 participant=Text::Table::Any
   Text::Table::CSV: mod_overhead_time=1 participant=Text::Table::CSV
   Text::Table::HTML: mod_overhead_time=3 participant=Text::Table::HTML
   Text::Table::HTML::DataTables: mod_overhead_time=2 participant=Text::Table::HTML::DataTables
   Text::Table::Manifold: mod_overhead_time=62 participant=Text::Table::Manifold
   Text::Table::More: mod_overhead_time=14 participant=Text::Table::More
   Text::Table::Org: mod_overhead_time=0 participant=Text::Table::Org
   Text::Table::Sprintf: mod_overhead_time=0 participant=Text::Table::Sprintf
   Text::Table::Tiny: mod_overhead_time=8 participant=Text::Table::Tiny
   Text::Table::TinyBorderStyle: mod_overhead_time=3 participant=Text::Table::TinyBorderStyle
   Text::Table::TinyColor: mod_overhead_time=4 participant=Text::Table::TinyColor
   Text::Table::TinyColorWide: mod_overhead_time=21 participant=Text::Table::TinyColorWide
   Text::Table::TinyWide: mod_overhead_time=19 participant=Text::Table::TinyWide
   Text::TabularDisplay: mod_overhead_time=3 participant=Text::TabularDisplay
   Text::UnicodeBox::Table: mod_overhead_time=124 participant=Text::UnicodeBox::Table
   perl -e1 (baseline): mod_overhead_time=0 participant=perl -e1 (baseline)

To display as an interactive HTML table on a browser, you can add option C<--format html+datatables>.

=head1 SAMPLE OUTPUTS

This section shows what the output is like for (some of the) modules:

=over

=item * Text::Table::Any

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * Text::Table::Manifold

 +--------+--------+--------+
 |  col1  |  col2  |  col3  |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * Text::ANSITable

 .--------+--------+--------.
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 `--------+--------+--------'

=item * Text::ASCIITable

 .--------------------------.
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 '--------+--------+--------'

=item * Text::FormatTable

 col1  |col2  |col3  
 row1.1|row1.2|row1.3
 row2.1|row2.2|row2.3
 row3.1|row3.2|row3.3
 row4.1|row4.2|row4.3
 row5.1|row5.2|row5.3

=item * Text::MarkdownTable

 | col1   | col2   | col3   |
 |--------|--------|--------|
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |

=item * Text::Table

 col1   col2   col3  
 row1.1 row1.2 row1.3
 row2.1 row2.2 row2.3
 row3.1 row3.2 row3.3
 row4.1 row4.2 row4.3
 row5.1 row5.2 row5.3

=item * Text::Table::Tiny

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * Text::Table::TinyBorderStyle

 .--------+--------+--------.
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 `--------+--------+--------'

=item * Text::Table::More

 .--------+--------+--------.
 | col1   | col2   | col3   |
 +========+========+========+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 `--------+--------+--------'

=item * Text::Table::Sprintf

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * Text::Table::TinyColor

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * Text::Table::TinyColorWide

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * Text::Table::TinyWide

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=item * Text::Table::Org

 | col1   | col2   | col3   |
 |--------+--------+--------|
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |

=item * Text::Table::CSV

 "col1","col2","col3"
 "row1.1","row1.2","row1.3"
 "row2.1","row2.2","row2.3"
 "row3.1","row3.2","row3.3"
 "row4.1","row4.2","row4.3"
 "row5.1","row5.2","row5.3"

=item * Text::Table::HTML

 <table>
 <thead>
 <tr><th>col1</th><th>col2</th><th>col3</th></tr>
 </thead>
 <tbody>
 <tr><td>row1.1</td><td>row1.2</td><td>row1.3</td></tr>
 <tr><td>row2.1</td><td>row2.2</td><td>row2.3</td></tr>
 <tr><td>row3.1</td><td>row3.2</td><td>row3.3</td></tr>
 <tr><td>row4.1</td><td>row4.2</td><td>row4.3</td></tr>
 <tr><td>row5.1</td><td>row5.2</td><td>row5.3</td></tr>
 </tbody>
 </table>

=item * Text::Table::HTML::DataTables

 <html>
 <head>
 <link rel="stylesheet" type="text/css" href="file:///home/s1/perl5/perlbrew/perls/perl-5.30.2/lib/site_perl/5.30.2/auto/share/dist/Text-Table-HTML-DataTables/datatables-1.10.22/datatables.css">
 <script src="file:///home/s1/perl5/perlbrew/perls/perl-5.30.2/lib/site_perl/5.30.2/auto/share/dist/Text-Table-HTML-DataTables/jquery-2.2.4/jquery-2.2.4.min.js"></script>
 <script src="file:///home/s1/perl5/perlbrew/perls/perl-5.30.2/lib/site_perl/5.30.2/auto/share/dist/Text-Table-HTML-DataTables/datatables-1.10.22/datatables.js"></script>
 <script>var dt_opts = {"buttons":["colvis","print"],"dom":"lQfrtip"}; $(document).ready(function() { $("table").DataTable(dt_opts); });</script>
 
 </head>
 
 <body>
 <table>
 <thead>
 <tr><th>col1</th><th>col2</th><th>col3</th></tr>
 </thead>
 <tbody>
 <tr><td>row1.1</td><td>row1.2</td><td>row1.3</td></tr>
 <tr><td>row2.1</td><td>row2.2</td><td>row2.3</td></tr>
 <tr><td>row3.1</td><td>row3.2</td><td>row3.3</td></tr>
 <tr><td>row4.1</td><td>row4.2</td><td>row4.3</td></tr>
 <tr><td>row5.1</td><td>row5.2</td><td>row5.3</td></tr>
 </tbody>
 </table>
 </body>
 
 </html>

=item * Text::TabularDisplay

 +--------+--------+--------+
 | col1   | col2   | col3   |
 +--------+--------+--------+
 | row1.1 | row1.2 | row1.3 |
 | row2.1 | row2.2 | row2.3 |
 | row3.1 | row3.2 | row3.3 |
 | row4.1 | row4.2 | row4.3 |
 | row5.1 | row5.2 | row5.3 |
 +--------+--------+--------+

=back

=head1 FAQ

=head2 What is an Acme::CPANModules::* module?

An Acme::CPANModules::* module, like this module, contains just a list of module
names that share a common characteristics. It is a way to categorize modules and
document CPAN. See L<Acme::CPANModules> for more details.

=head2 What are ways to use this Acme::CPANModules module?

Aside from reading this Acme::CPANModules module's POD documentation, you can
install all the listed modules (entries) using L<cpanm-cpanmodules> script (from
L<App::cpanm::cpanmodules> distribution):

 % cpanm-cpanmodules -n TextTable

Alternatively you can use the L<cpanmodules> CLI (from L<App::cpanmodules>
distribution):

    % cpanmodules ls-entries TextTable | cpanm -n

or L<Acme::CM::Get>:

    % perl -MAcme::CM::Get=TextTable -E'say $_->{module} for @{ $LIST->{entries} }' | cpanm -n

or directly:

    % perl -MAcme::CPANModules::TextTable -E'say $_->{module} for @{ $Acme::CPANModules::TextTable::LIST->{entries} }' | cpanm -n

This Acme::CPANModules module contains benchmark instructions. You can run a
benchmark for some/all the modules listed in this Acme::CPANModules module using
the L<bencher> CLI (from L<Bencher> distribution):

    % bencher --cpanmodules-module TextTable

This Acme::CPANModules module also helps L<lcpan> produce a more meaningful
result for C<lcpan related-mods> command when it comes to finding related
modules for the modules listed in this Acme::CPANModules module.

=head1 HOMEPAGE

Please visit the project's homepage at L<https://metacpan.org/release/Acme-CPANModules-TextTable>.

=head1 SOURCE

Source repository is at L<https://github.com/perlancar/perl-Acme-CPANModules-TextTable>.

=head1 BUGS

Please report any bugs or feature requests on the bugtracker website L<https://rt.cpan.org/Public/Dist/Display.html?Name=Acme-CPANModules-TextTable>

When submitting a bug or request, please include a test-file or a
patch to an existing test-file that illustrates the bug or desired
feature.

=head1 SEE ALSO

L<Acme::CPANModules::HTMLTable>

L<Acme::CPANModules> - about the Acme::CPANModules namespace

L<cpanmodules> - CLI tool to let you browse/view the lists

=head1 AUTHOR

perlancar <perlancar@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2021, 2020, 2019 by perlancar@cpan.org.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
