#!/usr/bin/perl

##################################################################
# Copyright (c) 2002 Jaap G Karssenberg. All rights reserved.    #
# This program is free software; you can redistribute it and/or  #
# modify it under the same terms as Perl itself.                 #
#                                                                #
# This is the install script for the zoidberg program files.     #
# You'll also need the Zoidberg module.                          #
#                                                                #
# mailto:j.g.karssenberg@student.utwente.nl                      #
# http://zoidberg.sourceforge.net                                #
##################################################################

use strict;
use Getopt::Long qw(:config gnu_compat no_getopt_compat no_ignore_case);
use Pod::Usage;

##### default defaults #####
my $def_perl = '/usr/bin/perl';
my $def_prefix = '/usr/local/';
my $def_config = '/etc/';
my $log_file = 'zoid_install.log';
my $uninstall_file = 'uninstall.sh';
############################

my $help = 0;
my $silent = 0;
my $message = "
This is the install script for the zoidberg program files.
These are useless without the Zoidberg modules.

See also <http://zoidberg.sourceforge.net>
";
GetOptions(
	'help|?|usage'	=> \$help,
	'silent'	=> \$silent,
	'prefix=s'	=> \$def_prefix,
	'config_dir=s'	=> \$def_config,
	'perl=s'	=> \$def_perl,
	'log_file=s'	=> \$log_file,
) || pod2usage( {
	'-message' => $message,
	'-exitval' => 1,
	'-verbose' => 0,
} );

if ($help) {
	pod2usage( {
		'-message' => $message,
		'-exitval' => 0,
		'-verbose' => 1,
	} );
}

$def_prefix =~ s/\/?$//; #/
$def_config =~ s/\/?$//; #/

open LOG, ">$log_file" || die "could not open log file - do you have permissions for the current directory?";
open UNI, ">$uninstall_file" || die "could not open uninstall file - do you have permissions for the current directory?";
print UNI << "EOH"
#!/bin/sh
################################################################
# This is the uninstall script for the zoidberg program files. #
# This script was autogenerated by the install.pl script       #
#                                                              #
# mailto:j.g.karssenberg\@student.utwente.nl                    #
# http://zoidberg.sourceforge.net                              #
################################################################
EOH
;

print "Installing Zoidberg shell...\n";

# get shebang
unless ($silent) { print "First of all we need to know where your perl binary is.\n"; }
my $perl = ask("Where is your perl installed?", $def_perl);
my $shebang = '#!'.$perl;
print LOG "Shebang is \'$shebang\'\n";

# get prefix
unless ($silent) { print "Next we need a prefix for \'bin/zoid\' and \'share/zoid/\' files.\n"; }
my $prefix = ask("Where would you like to install the zoid?", $def_prefix);
$prefix =~ s/\/?$//; #/
print LOG "Prefix is \"$prefix\"\n";

# get config dir
unless ($silent) { print "And last we need a place to store the (default) config files.\n"; }
my $config = ask("Where would you like to config files?", $def_config);
$config =~ s/\/?$//; #/
print LOG "Config in \"$config\"\n";

# some checks
unless (-e $perl && -x $perl) {
	unless ($silent) {
		print "## WARNING: the perl binary you specified \"$perl\" does not exist or is not executable.\n";
		my $ding = ask("Press \'f\' to continu anyway", "do not continu");
		unless ($ding eq 'f') {
			print LOG "Installation cancelled";
			die "Installation cancelled";
		}
	}
	else { die "No executable found at \"$perl\""; }
}
for ($prefix.'/bin', $prefix.'/share', $config) {
	unless (-e $_) { die "No such directory $_ -- please create it first"; }
}

# bin
print "Copying ./bin/fluff to $prefix/bin/zoid\n";
open IN, './bin/fluff' || "die Could not open file ./bin/fluff to read";
open OUT, '>'.$prefix.'/bin/zoid'  || "die Could not open file to $prefix/bin/zoid write";
print LOG "Created $prefix/bin/zoid\n";
print UNI "rm -f $prefix/bin/zoid &&\necho \"removed $prefix/bin/zoid\" ;\n";
while (<IN>) {
	s/^\#\!\/usr\/bin\/perl/$shebang/;
	s/^(my\s\$prefix\s=\s)\'\/usr\/local\/\'/$1\'$prefix\/\'/;
	s/^(my\s\$skel_dir\s=\s)\'\/etc\/zoid\/\'/$1\'$config\/zoid\/\'/; #/
	print OUT $_;
}
close OUT;
close IN;

# etc
dircopy('./etc/', $config.'/zoid/');

# share
dircopy('./share/', $prefix.'/share/zoid/');

chmod(0755, $prefix."/bin/zoid");

my $slink = ask("Would you like to create the symlink $prefix/bin/zoidberg?", "Y/n");
if ($slink =~ /y/i) {
	symlink $prefix.'/bin/zoid', $prefix.'/bin/zoidberg';
	print LOG "Created $prefix/bin/zoidberg\n";
	print UNI "rm -f $prefix/bin/zoidberg &&\necho \"removed $prefix/bin/zoidberg\" ;\n";
}

print LOG "Done - all seems well\n";
close LOG;
close UNI;
print "Done.\tInstall log in \"$log_file\"\n$uninstall_file was created.\n## WARNING: do not forget to flush ~/.zoid if you update to a newer version\n";

unless ($silent) { # start the shell
	select(undef,undef,undef,0.8);
	print "Starting Zoidberg shell.\n";
	sleep 1;
	exec($prefix."/bin/zoid");
}

sub dircopy {
	# dir from, dir to
	my ($from, $to) = @_;
	$from =~ s/\/?$/\//;
	$to =~ s/\/?$/\//;
	print "Copying $from to $to\n";
	unless (-e $to) {
		mkdir($to) || die "Could not create dir $to";
		print LOG "Created dir ".$to."\n";
		print UNI "rm -fr ".$to." &&\necho \"removed ".$to."\" ;\n";
	}
	opendir FROM, $from || die "Could not open dir $from";
	my @files = readdir FROM;
	closedir FROM;
	shift @files; #.
	shift @files; #..
	foreach my $file (grep {-f $from.$_} @files) {
		open IN, $from.$file || die "Could not open file ".$from.$file." to read";
		open OUT, '>'.$to.$file  || die "Could not open file to ".$to.$file." write";
		print LOG "Created ".$to.$file."\n";
		print UNI "rm -f ".$to.$file." &&\necho \"removed ".$to.$file."\" ;\n";
		while (<IN>) { print OUT $_; }
		close OUT;
		close IN;
	}
	foreach my $dir (grep {(-d $from.$_)&&($_ ne 'CVS')} @files) {
		dircopy( $from.$dir, $to.$dir ); #recurs
	}
}

sub ask {
	my ($question, $default) = @_;
	my $answer = '';
	unless ($silent) {
		print "$question [$default] ";
		$answer = <>;
		chomp($answer);
	}
	return $answer ? $answer : $default;
}

__END__

=head1 NAME

    Install.pl -- an install script for the Zoidberg ProgramFiles

=head1 SYNOPSIS

    install.pl [options]

     Options:
       --help        Detailed help message
       --silent      Non-interactive mode
       --prefix      Set prefix
       --config_dir  Set config prefix
       --perl        Set location of the perl binary

=head1 OPTIONS

Abbreviations of options are also allowed.

=over 4

=item B<-h, --help>

    Print a this help message and exits.

=item B<-s, --silent>

    Non-interactive installation,
    a.k.a. "just use the default values".

=item B<--prefix>

    Set prefix, default is "/usr/local/".
    In this dir the subdirs "bin" and "share" are expected.
    Subdir "share/zoid" will be created.

=item B<--config_dir>

    Set config prefix, default is "/etc/".
    Subdir "zoid" will be created,
    here the default config will be.

=item B<--perl>

    Set the location of the perl binary.
    Default is "/usr/bin/perl".

=item B<--log_file>

    Set the name for the log file.

=back

=cut


