use ExtUtils::MakeMaker;
use strict;
use Config;

my $CC = $ENV{"CXX"} || 'g++';
my $LD = '$(CC)';
if ($^O eq 'cygwin' and $CC eq 'g++') {
    # Cygwin packages of Perl < 5.9.5 used "ld2" for $Config{ld} and
    # $Config{lddlflags} didn't contain -shared so we need to specify
    # this explicitly.  Perl >= 5.9.5 package do away with "ld2", but
    # it should be harmless to specify "-shared" there.
    $LD = 'g++ -shared';
}

my $xapian_config = $ENV{XAPIAN_CONFIG} || 'xapian-config';

my $xver = `$xapian_config --version`;
if ($xver eq '') {
    die "$xapian_config not found - set XAPIAN_CONFIG if Xapian isn't installed on your PATH.\n";
}
chomp($xver);
$xver =~ s/.*\s//; # "xapian 0.9.3" -> "0.9.3"

my $inc = `$xapian_config --cxxflags`;
chomp($inc);

my $libsvar = 'LIBS';
my $libs = `$xapian_config --libs 2> /dev/null`;
chomp($libs);
my ($xapian_config_dir) = $xapian_config =~ /^(.*?)[^\/]*$/;
if ($? || -f "$xapian_config_dir/Makefile") {
    # Assume we're being asked to build against an uninstalled xapian-core.
    my $libtool = "$xapian_config_dir/libtool";
    unless (-x $libtool) {
	die "You've asked me to link against what appears to be an uninstalled xapian-core tree, but I can't find libtool in that tree\n";
    }

    # We can't pass a .la file in LIBS since MakeMaker "knows better" and
    # ignores it.  Passing it in LDLOADLIBS works, but generates a warning.
    # We can avoid the warning by setting LDLOADLIBS using 'macro'.
    $libsvar = 'macro';
    $libs = `$xapian_config --ltlibs`;
    chomp($libs);
    $libs = {'LDLOADLIBS' => $libs};
    $LD = "$libtool --tag=CXX --mode=link $CC -avoid-version -module -no-install";
    $CC = "$libtool --mode=compile $CC";
}

# Filter out some gcc options which g++ doesn't support.
my $CCFLAGS = $Config{'ccflags'};
# Perl is built with -Wdeclaration-after-statement on RHEL5 - this isn't
# meaningful for C++ - it only emits a warning but it's easy to fix.
$CCFLAGS =~ s/(?:^|\s+)-Wdeclaration-after-statement(?:\s+|$)/ /;
# The generated code causes "variable may be used uninitialized" warnings
# if Perl was built with -Wall.
$CCFLAGS =~ s/(^|\s+)-Wall(\s+|$)/$1-Wall -Wno-uninitialized$2/;

# See lib/ExtUtils/MakeMaker.pm for details of how to influence
# the contents of the Makefile that is written.
WriteMakefile(
    'NAME'		=> 'Search::Xapian',
    'VERSION_FROM'	=> 'Xapian.pm', # finds $VERSION
    'PREREQ_PM'		=> {}, # e.g., Module::Name => 1.1
    ($] >= 5.005 ?    ## Add these new keywords supported since 5.005
      (ABSTRACT_FROM => 'Xapian.pm', # retrieve abstract from module
       AUTHOR     => 'Alex Bowley <xapian-discuss@lists.xapian.org>') : ()),
#       AUTHOR     => 'Alex Bowley <kilinrax@cpan.org>') : ()),
    $libsvar		=> $libs, # e.g., '-lm'
    'DEFINE'		=> '', # e.g., '-DHAVE_SOMETHING'
    'CC'                => $CC,
    'CCFLAGS'		=> $CCFLAGS,
    'LD'                => $LD,
    'INC'		=> $inc, # e.g., '-I/usr/include/other'
    'XSOPT'             => '-C++',
    'TYPEMAPS'          => ['perlobject.map','typemap'],
    # Add "make check" as alias for "make test".
    'depend'		=> { 'check' => 'test' },
);

my $VERSION = "unknown";
open F, "Makefile" or die $!;
while (<F>) {
    if (/^VERSION\s*=\s*(\S*)/) {
	$VERSION = $1;
	last;
    }
}
close F;

my ($BASEVERSION) = $VERSION =~ /^([0-9]+\.[0-9]+\.[0-9]+)/;
if ($xver !~ /^\Q$BASEVERSION\E(?:_svn[0-9]+)?$/) {
    warn "Warning: Xapian version $xver may be incompatible with Search::Xapian $VERSION\n";
}

my @bad;
for my $file (qw(Changes README)) {
    open F, $file or next;
    my $ok;
    while (<F>) {
	if (/\b\Q$VERSION\E\b/) {
	    $ok = 1;
	    last;
	}
    }
    close F;
    if (!$ok) {
	push @bad, $file;
    }
}
if (scalar @bad) {
    unlink "Makefile";
    die(join(",",@bad).": No mention current version: $VERSION\n");
}


sub MY::postamble {
  return "\$(XS_FILES): ".join(" ", <XS/*.xs>)."\n\ttouch \$(XS_FILES)";
}
