package Bif::Sync;
use strict;
use warnings;
use Bif::DB::Plugin::ChangeUUIDv1;
use Bif::Mo;
use Coro;
use DBIx::ThinSQL qw/qv/;
use Log::Any '$log';
use JSON;

our $VERSION = '0.1.5_4';

has changes_dup => (
    is      => 'rw',
    default => 0
);

has changes_sent => (
    is      => 'rw',
    default => 0
);

has changes_torecv => (
    is      => 'rw',
    default => 0
);

has changes_tosend => (
    is      => 'rw',
    default => 0
);

has changes_recv => (
    is      => 'rw',
    default => 0
);

has debug => (
    is      => 'rw',
    default => 0,
);

has db => (
    is       => 'ro',
    required => 1,
);

has hub_id => ( is => 'rw', );

has on_error => ( is => 'ro', required => 1 );

has on_update => (
    is      => 'rw',
    default => sub { },
);

has rh => ( is => 'rw', );

has wh => ( is => 'rw', );

has json => (
    is      => 'rw',
    default => sub { JSON->new->utf8 },
);

has temp_table => (
    is       => 'rw',
    init_arg => undef,
);

sub new_temp_table {
    my $self = shift;
    my $tmp = 'sync_' . $$ . sprintf( "%08x", rand(0xFFFFFFFF) );

    $self->db->do( 'CREATE TEMPORARY TABLE '
          . $tmp . '('
          . 'id INTEGER NOT NULL UNIQUE ON CONFLICT IGNORE,'
          . 'ucount INTEGER'
          . ')' );

    $self->temp_table($tmp);
    return $tmp;
}

sub read {
    my $self = shift;

    my $json = $self->rh->readline("\n");

    if ( !defined $json ) {
        $self->on_error->('connection close/timeout');
        $self->write('EOF/Timeout');
        return 'EOF';
    }

    my $msg = eval { $self->json->decode($json) };

    if ($@) {
        $self->on_error->($@);
        $self->write('InvalidEncoding');
        return 'INVALID';
    }
    elsif ( !defined $msg ) {
        $self->on_error->('no message received');
        $self->write('NoMessage');
        return 'INVALID';
    }

    $log->debugf( 'r: %s', $msg );

    return @$msg;
}

sub write {
    my $self = shift;

    $log->debugf( 'w: %s', \@_ );

    return $self->wh->print( $self->json->encode( \@_ ) . "\n" );
}

# Let sub classes override if necessary
sub trigger_on_update { }

sub real_sync {
    my $self      = shift;
    my $kind      = shift || die 'real_sync($KIND,$id,$prefix,$change_id)';
    my $id        = shift;
    my $prefix    = shift // '';
    my $change_id = shift;

    if ( not defined $id ) {
        $self->write( 'ProtocolError', 'real_sync($id)' );
        return 'ProtocolError';
    }

    my $db        = $self->db;
    my $tmp       = $self->temp_table;
    my $hub_id    = $self->hub_id;
    my $on_update = $self->on_update;
    my $prefix2   = $prefix . '_';

    $on_update->( 'matching: ' . $prefix2 ) if $on_update;

    my @refs;

    # Case 2b (see below): this end is a leaf node so fake the prefix stuff
    if ($change_id) {
        @refs = $db->xhashrefs(
            select => [
                'SUBSTR(c.uuid, 1, LENGTH(Xc.prefix)+1) AS prefix',
                'Xc.hash AS hash',
                'Xc.change_id AS change_id'
            ],
            from       => "${kind}_changes Xc",
            inner_join => 'changes c',
            on         => 'c.id = Xc.change_id',
            where      => {
                "Xc.${kind}_id" => $id,
                'Xc.change_id'  => $change_id,
                'Xc.prefix'     => $prefix,
            },
        );
    }

    # Branch node comparison
    else {
        @refs = $db->xhashrefs(
            select => [
                'Xc.prefix AS prefix',
                'Xc.hash AS hash',
                'Xc.change_id AS change_id'
            ],
            from  => "${kind}_changes Xc",
            where => {
                "Xc.${kind}_id"  => $id,
                'Xc.prefix LIKE' => $prefix2,
            },
            order_by => 'Xc.prefix',
        );
    }

    $self->write(
        'MATCH', $prefix2,
        {
            map { $_->{prefix} => [ $_->{hash}, $_->{change_id} ? 1 : 0 ] }
              @refs
        }
    );

    my ( $action, $mprefix, $there ) = $self->read;

    return "expected MATCH $prefix2 {} (not $action $mprefix ...)"
      unless $action eq 'MATCH'
      and $mprefix eq $prefix2
      and ref $there eq 'HASH';

    my @next;
    my @prefix;

    foreach my $ref (@refs) {

        # Case 1: Prefix does not exist at other end
        if ( !exists $there->{ $ref->{prefix} } ) {
            push( @prefix, ' OR ' ) if @prefix;
            push( @prefix,
                "Xc.prefix LIKE ",
                qv( $ref->{prefix} . '%' ),
                ' AND Xc.change_id IS NOT NULL' );
        }

        # Case 2: Prefix hash is not the same
        elsif ( $there->{ $ref->{prefix} }->[0] ne $ref->{hash} ) {

            # Case 2a: both ends are leaf nodes
            if ( $there->{ $ref->{prefix} }->[1] and $ref->{change_id} ) {
                push( @prefix, ' OR ' ) if @prefix;
                push( @prefix, "Xc.change_id = ", qv( $ref->{change_id} ) );
            }

            # Case 2b: this end is a leaf node
            elsif ( $ref->{change_id} ) {
                push( @next, [ $ref->{prefix}, $ref->{change_id} ] );
            }

            # Case 2c: this end, and *maybe* other end are branch nodes
            else {
                push( @next, [ $ref->{prefix} ] );
            }
        }
    }

    if (@prefix) {
        $self->db->xdo(
            insert_into => "$tmp(id)",
            select      => 'Xc.change_id',
            from        => "${kind}_changes Xc",
            where => [ "Xc.${kind}_id = ", qv($id), ' AND (', @prefix, ')' ],
        );
    }

    if (@next) {
        foreach my $next ( sort { $a->[0] cmp $b->[0] } @next ) {
            $self->real_sync( $kind, $id, @$next );
        }
    }

    return ucfirst($kind) . 'Sync' . $prefix;
}

sub real_send_changesets {
    my $self       = shift;
    my $total      = shift;
    my $statements = shift;

    $self->changes_tosend( $self->changes_tosend + $total );

    my $db  = $self->db;
    my $sth = $db->xprepare(@$statements);
    $sth->execute;

    my $sent = 0;

    while ( my $id = $sth->val ) {
        my $changeset = $db->uchangeset_v1($id);
        return 'SendFailure' unless $self->write( 'CHANGESET', $changeset );
        $sent++;
        $self->changes_sent( $self->changes_sent + 1 );
        $self->trigger_on_update;
    }

    return 'ChangesetCountMismatch' unless $sent == $total;
    return 'SendChangesets';
}

sub send_changesets {
    my $self       = shift;
    my $total      = shift;
    my $statements = shift;

    $self->write( 'TOTAL', $total );
    my @r = $self->real_send_changesets( $total, $statements );
    return @r unless $r[0] eq 'SendChangesets';

    @r = $self->read;
    return 'SendChangesets' if $r[0] eq 'Recv' and $r[1] == $total;
    return @r;
}

sub recv_changesets {
    my $self = shift;
    my $db   = $self->db;

    my ( $action, $total ) = $self->read;
    $total //= '*undef*';

    if ( $action ne 'TOTAL' or $total !~ m/^\d+$/ ) {
        return "expected TOTAL <int> (not $action $total)";
    }

    my $ucount;
    my $i   = $total;
    my $got = 0;

    $self->changes_torecv( $self->changes_torecv + $total );
    $self->trigger_on_update;

    my %import_functions = (
        CHANGESET => {},
        QUIT      => {},
        CANCEL    => {},
    );

    while ( $got < $total ) {
        my ( $action, $changeset ) = $self->read;

        return "expected CHANGSET not: $action"
          unless $action eq 'CHANGESET';

        my ( $res, $uuid ) = $db->save_uchangeset_v1($changeset);
        if ( 0 == $res ) {
            $self->changes_dup( $self->changes_dup + 1 );
        }
        elsif ( $res < 0 ) {
            $self->write( 'UUIDMismatch', $uuid );
            return ( 'UUIDMismatch', $uuid );
        }

        $got++;
        $self->changes_recv( $self->changes_recv + 1 );
        $self->trigger_on_update;

    }

    $db->xdo(
        insert_into => 'func_merge_changes',
        values      => { merge => 1 },
    );

    $self->write( 'Recv', $got );
    return 'RecvChangesets';
}

sub exchange_changesets {
    my $self            = shift;
    my $send_total      = shift;
    my $send_statements = shift;
    my $db              = $self->db;

    # Ensure that this goes out before we get properly asynchronous,
    # particularly before recv_changesets() sends out a Recv message.
    $self->write( 'TOTAL', $send_total );

    # Kick off receiving changesets as a separate Coro thread
    my $fh   = select;
    my $coro = async {
        select $fh;

        $self->recv_changesets();
    };

    # Now send receiving changesets
    my $send_status =
      $self->real_send_changesets( $send_total, $send_statements );

    # Cancel the $coro?
    return $send_status unless $send_status eq 'SendChangesets';

    # Collect the recv status
    my $recv_status = $coro->join;
    return $recv_status unless $recv_status eq 'RecvChangesets';

    # Only now can we read the sending status
    my ( $recv, $count ) = $self->read;
    return $recv unless $recv eq 'Recv' and $count == $send_total;

    return 'ExchangeChangesets';
}

1;

=head1 NAME

=for bif-doc #perl

Bif::Sync - synchronisation role

