########################################################################
# housekeeping
########################################################################

package mro::EVERY v0.1.0;
use v5.22;
use mro;

use Carp            qw( croak           );
use List::Util      qw( uniq            );
use Symbol          qw( qualify_to_ref  );

########################################################################
# package varaibles
########################################################################

our @CARP_NOT   = ( __PACKAGE__, qw( mro ) );
my $find_subs   = '';
my $with_auto   = '';

########################################################################
# utility subs
########################################################################

my $find_name
= sub
{
    my $proto   = shift;
    my $auto    = shift;
    my ($name)  = $auto =~ m{ (\w+) $}x;

    $proto->can( $name )
    or croak "Botched EVERY: '$proto' cannot '$name'";

    # if they handle this via AUTOLOAD then we have 
    # problem at this point, see find_with_autoload.
    
    local $"    = ',';
    my @isa     = $proto->mro::get_linear_isa->@*;

    # uniq avoids case of multiple-dispatch of 
    # hardwired inherited methods at multiple 
    # places in the tree. 

    my @found
    = uniq
    grep
    {
        $_
    }
    map
    {
        *{ qualify_to_ref $name => $_ }{ CODE }
    }
    @isa
    or
    croak "Bogus $proto: '$name' not in @isa";

    @found
};

my $find_with_autoload
= sub
{
    my $proto   = shift;
    my $auto    = shift;
    my ($name)  = $auto =~ m{ (\w+) $}x;

    $proto->can( $name )
    or croak "Botched EVERY: '$proto' cannot '$name'";
 
    local $"    = ',';
    my @isa     = $proto->mro::get_linear_isa->@*;

    # uniq avoids multiple-dispatch in case where
    # AUTOLOAD handling $name is inherited.

    my @found
    = uniq
    grep
    {
        $_
    }
    map
    {
        *{ qualify_to_ref $name => $_ }{ CODE }
        or 
        do
        {
            my $isa = qualify_to_ref ISA        => $_;
            my $ref = qualify_to_ref AUTOLOAD   => $_;

            local *$isa = [];

            # at this point can is isolated to the 
            # single pacakge.

            my $al
            = $_->can( $name )
            ? *{ $ref }{ CODE }
            : ''
            ;

            $al
            ?   sub
                {
                    # at this point if package can $name and
                    # has an AUTOLOAD but not the named sub.
                    #
                    # install $AUTOLOAD and bon voyage!

                    local *{ $ref } = $auto;
                    goto &$al;
                }
            : ()
            ;
        }
    }
    @isa
    or
    croak "Bogus $proto: '$name' & AUTOLOAD not in @isa";

    @found
};

sub import
{
    shift;  # discard this package

    for( @_ )
    {
        m{^   autoload $}x and $with_auto = 1;
        m{^ noautoload $}x and $with_auto = '';
    }

    $find_subs
    = $with_auto
    ? $find_with_autoload
    : $find_name
    ;

    return
}

########################################################################
# pseudo-packages
########################################################################

package EVERY;
use v5.22;
use Carp    qw( croak );

our @CARP_NOT   = ( __PACKAGE__, qw( mro ) );
our $AUTOLOAD   = '';

AUTOLOAD
{
    my $proto   = shift
    or croak "Bogus EVERY, called without an object.";

    # remaining arguments left on the stack.

    $proto->$_( @_ )
    for $proto->$find_subs( $AUTOLOAD );
}

package EVERY::LAST;
use v5.22;
use Carp    qw( croak );

our @CARP_NOT   = ( __PACKAGE__, qw( mro ) );
our $AUTOLOAD   = '';

AUTOLOAD
{
    my $proto   = shift
    or croak "Bogus EVERY::LAST, called without an object.";

    # remaining arguments left on the stack.

    $proto->$_( @_ )
    for reverse $proto->$find_subs( $AUTOLOAD );
}

# keep require happy
1
__END__

=head1 NAME

mro::EVERY - EVERY & EVERY::LAST pseudo-packages using mro. 

=head1 SYNOPSIS

    # EVERY & EVERY::LAST redispatch the named method into
    # all classes in the object/class hierarchy which 
    # implement the method or have a suitable can() and 
    # AUTOLOAD to handdle the method.

    package Giant;
    use mro;    
    use parent qw( fee fie foe fum );
    use mro::EVERY;

    # construct an object then dispatch the 'initialize'
    # method into each class from least-to-most derived 
    # that declares its own 'initialize' method.

    sub new
    {
        my $object  = &construct;

        $object->EVERY::LAST::initialize( @_ );
        $object
    }

    # tear down an object from the top down, calling
    # 'cleanup' for most-to-least derived classes.

    package Thingy;
    use mro qw( c3 );
    use parent qw( this that other );
    use mro::EVERY;

    DESTROY
    {
        my $object  = shift;

        $object->EVERY::cleanup;
    }

    # the "autoload" switch turns on scanning for 
    # $proto->can( $name ) and checking for AUTOLOAD
    # subs (vs. simply checking for a defined coderef
    # in the package). 
    # 
    # using this approach requires properly overloading
    # can() in the package. 
    # 
    # note that AUTOLOAD's can have all sorts of side
    # effects, this should be used with care and where
    # the handling classes really do have overloaded
    # "can" methods and really do handle the named
    # operation properly.
    #
    # lacking an overloaded can() and appropriate 
    # AUTOLOAD, this is a waste. nu, don't say I
    # didn't warn you.

    package Derived;
    use mro;
    use parent qw( Base1 );

    use mro::EVERY  qw( autoload );

    sub randomize
    {
        my $obj = shift;

        $obj->EVERY::methodical;
    }

    sub methodical
    {
        # first call from EVERY::methodical ends up 
        # here.

        ...
    }

    package Base1;

    # can has to return something for every method
    # the class can handle -- including UNIVERSAL
    # and any other base classes. this is a trivial
    # example that works because there are no other
    # bases classes than UNIVERSAL here.

    my %can = 
    (
        methodical  => \&AUTOLOAD
      , AUTOLOAD    => \&AUTOLOAD
      , bottle      => \&bottle
    );

    sub can
    {
        %can{ $_[1] }
        or
        UNIVERSAL->can( $_[1] )
    }

    sub bottle
    {
        # autoloaded or not, this has to be 
        # handled by can(), above.

        ...
    }

    our $AUTOLOAD   = '';
    AUTOLOD
    {
        # call ends up here becuase mro::EVERY can 
        # find that $pkg->can( 'methodical' ) and
        # also that there is an AUTOLOAD in the package
        # (not just inherited). 

        ...
    }

=head1 DESCRIPTION

The main use of both pseudo-classes is dispatching an arbitrary 
method up or down the inheritence stack without each of them having
to do their own re-dispatch to another. One common use of this is 
in initializers, which can use EVERY::LAST to walk up the tree from
least-to-most derived classes calling the method where it is 
declared in the class (vs. simply inherited). 

Initializers will usually go "up" the inheritence stack using
EVERY::LAST so that more derived classes can depend on their
base class being set up. Destructors will use EVERY and tear down
the object from most-derived to least derived.

With autoloads it gets a bit tricky becuase $package->can( 'X' )
needs to return true for X, there has to be an autoload, and 
the package has to be examined in isolation (i.e., with an empty
@ISA so that we don't get an inherited value from can()). If the 
package is using an AUTOLOAD to do something it'll need to overload
can() to hand back true for any method names dealt with via AUTOLOAD.

Without autoload this is quite simple: Walk down mro::get_linear_isa
looking for packages that define their own code for the method name.
Get a list back and call it in order for EVERY or reverse order for
EVERY::LAST. This is pretty much the same guts as NEXT, just using
mro for the package names rather than iteratingon @ISA. 

=head1 SEE ALSO

=over 4

=item mro 

This describes the use of "dfs" & "c3" methologies for 
resolving class inheritence order. 

=item NEXT

Further description EVERY & EVERY::LAST.

Note: This module is not compaitible with NEXT as they both
attempt to define the same pseudo-classes EVERY & EVERY::LAST. If
you are not working with mro, use NEXT; if you are use this. 

=back
