#!/usr/bin/perl -w

#######################################################################
#
# pod2wiki - A utility to convert Pod documents to Wiki format.
#
# reverse(''), August 2004, John McNamara, jmcnamara@cpan.org
#
# Documentation after __END__
#

use strict;
use Pod::Simple::Wiki;
use Getopt::Long;
use Pod::Usage;

my $man      = 0;
my $help     = 0;
my $style    = 'wiki';
my $whine    = 1;
my $errata   = 1;
my $complain = 0;

GetOptions(
            'help|?'    => \$help,
            'man'       => \$man,
            'style=s'   => \$style,
            'whine!'    => \$whine,
            'errata!'   => \$errata,
            'complain!' => \$complain,
          ) or pod2usage(2);

pod2usage(1) if $help;
pod2usage(-verbose => 2) if $man;


# From the Pod::Usage pod:
pod2usage() if @ARGV == 0 && -t STDIN;


my $parser = Pod::Simple::Wiki->new($style);
   $parser->no_whining(not $whine);
   $parser->no_errata_section(not $errata);
   $parser->complain_stderr($complain);

if (defined $ARGV[0]) {
    open IN, $ARGV[0]  or die "Couldn't open $ARGV[0]: $!\n";
} else {
    *IN = *STDIN;
}

if (defined $ARGV[1]) {
    open OUT, ">$ARGV[1]" or die "Couldn't open $ARGV[1]: $!\n";
    $parser->output_fh(*OUT);
}


$parser->parse_file(*IN);


__END__

=head1 NAME

pod2wiki - A utility to convert Pod documents to Wiki format.

=head1 SYNOPSIS

pod2wiki [--style --noerrata --help --man] podfile [outfile]

    Options:
        --style      wiki style (defaults to wiki. See --help)
        --noerrata   don't generate a "POD ERRORS" section
        --help       brief help message
        --man        full documentation


=head1 OPTIONS

=over 4

=item B<podfile>

The input file that contains the Pod file to be converted. It can also be stdin.

=item B<outfile>

The converted output file in wiki format. Defaults to stdout if not specified.

=item B<--style or -s>

Sets the wiki style of the output. If no C<style> is specified the program defaults to C<wiki>. The available options are:

=over 4

=item wiki

This is the original Wiki format as used on Ward Cunningham's Portland repository of Patterns. The formatting rules are given at http://c2.com/cgi/wiki?TextFormattingRules

=item kwiki

This is the format as used by Brian Ingerson's CGI::Kwiki: http://search.cpan.org/dist/CGI-Kwiki/

=item usemod

This is the format used by the Usemod wikis. See: http://www.usemod.com/cgi-bin/wiki.pl?WikiFormat

=item twiki

This is the format used by TWiki wikis.  See: http://www.twiki.org/

=item wikipedia or mediawiki

This is the format used by Wikipedia and MediaWiki wikis.  See: http://www.wikipedia.org/

=item moinmoin

This is the format used by MoinMoin wikis.  See: http://moinmoin.wikiwikiweb.de/

=back


=item B<--noerrata or -noe>

Don't generate a "POD ERRORS" section at the end of the document. Equivalent to the C<Pod::Simple::no_errata_section()> method.


=item B<--help or -h>

Print a brief help message and exits.


=item B<--man or -m>

Prints the manual page and exits.


=back

=head1 DESCRIPTION

This program is used for converting Pod text to Wiki text.

Pod is Perl's I<Plain Old Documentation> format. See C<man perlpod> or C<perldoc perlpod>.

A Wiki is a user extensible web site. It uses very simple mark-up that is converted to Html.

For an introduction to Wikis see: http://c2.com/cgi/wiki?WikiGettingStartedFaq and http://c2.com/cgi/wiki?WikiWikiWebFaq

=cut
