#include <string.h>
#include <stdlib.h>

#include <arch/neuron.h>
#include <arch/synaptic_group.h>

synaptic_arc::synaptic_arc(neuron *dendrilitic, neuron *fireing) {
    dendrilitic_neuron = dendrilitic;
    fireing_neuron  = fireing;

    weight      = 0.0;
    eligibility = 0.0;
    fireing->axon_fires_at->add_existing_arc(this);

    sprintf(name, "%s->%s", fireing->query_name(), dendrilitic->query_name());
#if (DEBUG || WEIGHT_DEBUG || CONNECTION_DEBUG)
    fprintf(stderr, "    created [%s], weight=%f.\n", name, weight);
#endif
}

synaptic_node::synaptic_node(neuron *dendrilitic, neuron *fireing) {
    next = 0;
    my_arc = new synaptic_arc(dendrilitic, fireing);
}

synaptic_node::synaptic_node(synaptic_arc *arc) {
    next = 0;
    my_arc = arc;
}

synaptic_group::synaptic_group(char *n) {
    size    = 0;
    head    = 0;
    weights = 0;

    strcpy(name, n);
    while(weights) {
        fprintf(stderr, "weights are supposed to be 0 now ... [%i,%s]\n", (int)weights, name);
        weights = 0;
    }
}

char *synaptic_arc::query_name() {
    return name;
}

void synaptic_group::add_existing_arc(synaptic_arc *arc) {
    synaptic_node *temp = head;
    size++;

    if(weights != 0) {
        fprintf(stderr, "Arcs cannot be added to a synaptic group, after \n");
        fprintf(stderr, "the weights have been queried [%i].\n", (int)weights);
        fprintf(stderr, "This happened in [%s].\n", name);
        exit(1);
    }

    if(head) {
        while(temp->next) {
            temp = temp->next;
        }
        temp->next = new synaptic_node(arc);
    } else {
        head       = new synaptic_node(arc);
    }
}

void synaptic_group::add_weight_for(neuron *dendrilitic, neuron *fireing) {
    synaptic_node *temp = head;
    size++;

    if(weights != 0) {
        fprintf(stderr, "Neurons cannot be added to a synaptic group, after \n");
        fprintf(stderr, "the weights have been queried [%i].\n", (int)weights);
        fprintf(stderr, "This happened in [%s].\n", name);
        exit(1);
    }

    if(head) {
        while(temp->next) {
            temp = temp->next;
        }
        temp->next = new synaptic_node(dendrilitic, fireing);
    } else {
        head       = new synaptic_node(dendrilitic, fireing);
    }
}

real synaptic_group::sum_of_outputs() {
    synaptic_node *temp = head;
    real ret = 0.0;

    while(temp) {
        ret += temp->my_arc->fireing_neuron->query_output() 
             * temp->my_arc->weight;
        temp = temp->next;
    }

    return ret;
}

real synaptic_group::weighted_sum_of_delta_from_above() {
    synaptic_node *temp = head;
    real ret = 0.0;

    while(temp) {
#if (DEBUG || ERROR_DEBUG)
        fprintf(stderr, "(%s.w=%f*%s.err=%f) = %f \n",
            temp->my_arc->query_name(),
            temp->my_arc->weight,
            temp->my_arc->dendrilitic_neuron->query_name(),
            temp->my_arc->dendrilitic_neuron->query_delta(),
            temp->my_arc->dendrilitic_neuron->query_delta()
                * temp->my_arc->weight
        );
#endif
        ret += temp->my_arc->dendrilitic_neuron->query_delta() 
             * temp->my_arc->weight;
        temp = temp->next;
    }
#if (DEBUG || ERROR_DEBUG)
        fprintf(stderr, "      total = %f \n",
            ret
        );
#endif
    return ret;
}

void synaptic_group::start_foreach() {
    current_node = head;
}

int  synaptic_group::has_a_weight() {
    return (current_node) ? 1:0;  // to prevent shitty scope hacking
}

int  synaptic_group::has_an_eligibility() {
    return (current_node) ? 1:0;  // to prevent shitty scope hacking
}

void synaptic_group::next_weight() {
    current_node = current_node->next;
}

void synaptic_group::next_eligibility() {
    current_node = current_node->next;
}

char *synaptic_group::query_current_name() {
    return current_node->my_arc->query_name();
}

real synaptic_group::query_current_weight() {
    return current_node->my_arc->weight;
}

real synaptic_group::query_current_eligibility() {
    return current_node->my_arc->eligibility;
}

void synaptic_group::set_current_weight_to(real d) {
#if (DEBUG || WEIGHT_DEBUG || IO_RECALC_DEBUG)
    fprintf(stderr, "  %20s::set_current_weight_to(%f)\n",
        query_current_name(),
        d
    );
#endif
    current_node->my_arc->weight = d;
    current_node->my_arc->dendrilitic_neuron->recalc_needed();
}

void synaptic_group::set_current_eligibility_to(real d) {
    current_node->my_arc->eligibility = d;
}

void synaptic_group::change_current_weight_by(real d) {
#if (DEBUG || WEIGHT_DEBUG || IO_RECALC_DEBUG)
    fprintf(stderr, "  %20s::change_current_weight_by(%f)\n",
        query_current_name(),
        d
    );
#endif
    current_node->my_arc->weight += d;
    current_node->my_arc->dendrilitic_neuron->recalc_needed();
}

void synaptic_group::change_current_eligibility_by(real d) {
    current_node->my_arc->eligibility += d;
}

void synaptic_group::set_weights_to(real *d) {
    int counter = 0;

    start_foreach();
    while(has_a_weight()) {
        set_current_weight_to(d[counter]);
        next_weight();
        counter++;
    }
}

void synaptic_group::change_weights_by(real *d) {
    int counter = 0;

    start_foreach();
    while(has_a_weight()) {
        change_current_weight_by(d[counter]);
        next_weight();
        counter++;
    }
}

void synaptic_group::recalc_needed_upward() {
    start_foreach();
    while(has_a_weight()) {
        current_node->my_arc->dendrilitic_neuron->recalc_needed();
        next_weight();
    }
}

real synaptic_group::query_current_fireing_output() {
    return current_node->my_arc->fireing_neuron->query_output();
}

real synaptic_group::query_current_dendrilitic_output() {
    /*
    ** change suggested by Kirill Erofeev
    ** seems good so we're keepin' it... 
    ** return current_node->my_arc->fireing_neuron->query_output();
    */

    return current_node->my_arc->dendrilitic_neuron->query_output();
}

real *synaptic_group::query_weights() {
    int counter = 0;

    if(!weights) {
        weights = new real[size];
    }

    start_foreach();
    while(has_a_weight()) {
        weights[counter] = current_node->my_arc->weight;
        counter++;
        next_weight();
    }

    return weights;
}

void synaptic_group::reinitialize_weights_with(real max, real min) {
    start_foreach();
    while(has_a_weight()) {
        current_node->my_arc->weight =
            ((real) rand() / RAND_MAX)*(max-min)+min;
#if DEBUG || WEIGHT_DEBUG
        fprintf(stderr, "%s's weight re-initialized to %f\n", 
            current_node->my_arc->query_name(),
            current_node->my_arc->weight
        );
#endif
        next_weight();
    }
}

void synaptic_group::save_weights(FILE *F) {
    real *r = query_weights();
    fwrite(r, sizeof(real), size, F);
}

void synaptic_group::restore_weights(FILE *F) {
    real *r = new real[size];
    fread(r, sizeof(real), size, F);
    set_weights_to(r);
    delete r;
}
