#include <string.h>
#include <stdlib.h>

#include <arch/neuron.h>
#include <utils/transfer.h>

neuron::neuron(char n[20]) {
    char temp[40];
    strcpy(name, n);
    is_biased = 0;
    my_input  = -1.0;
    transfer_type = SIGMOID;
    input_same      = 0;
    output_same     = 0;
    output_dot_same = 0;

#if DEBUG
    fprintf(stderr, "Created neuron::%s\n", name);
#endif

    sprintf(temp, "%s-dendrites", name);
    dendrites     = new synaptic_group(temp);

    sprintf(temp, "%s-axons", name);
    axon_fires_at = new synaptic_group(temp);
}

char *neuron::query_name() {
    return name;
}

int neuron::query_biased() {
    return (is_biased) ? 1:0;
}

void neuron::set_transfer_function(int type) {
    transfer_type = type;
}

void neuron::set_input(real in) {
    if(dendrites->size) {
        fprintf(stderr, "Input given to %s, but it already had dendrites!\n", name);
        exit(1);
    }
    if(is_biased) {
        fprintf(stderr, "Neuron %s is set to be a bias unit, but it was given input!\n",
            name
        );
        exit(1);
    }
#if IO_DEBUG || DEBUG
        fprintf(stderr, "            %20s::set_input(%f)\n",
            name,
            in
        );
#endif
    recalc_needed();
    my_input = in;
}

void neuron::set_input_as_bias(real in) {
    if(dendrites->size) {
        fprintf(stderr, "%s tried to bias, but it already had dendrites!\n", name);
        exit(1);
    }
    if(is_biased) {
        fprintf(stderr, "Neuron %s is already set to be a bias unit!\n",
            name
        );
        exit(1);
    }
#if IO_DEBUG || DEBUG
        fprintf(stderr, "            %20s::set_input_as_bias(%f)\n",
            name,
            in
        );
#endif
    recalc_needed();
    my_input  = in;
    is_biased = 1;
}

real neuron::query_input() {
#ifndef use_instantaneous_propagation
    if(!input_same) 
#endif
        recalc_input();
#if IO_DEBUG || DEBUG
        fprintf(stderr, "            %20s::query_input()      = %f\n",
            name,
            my_input
        );
#endif
    return my_input;
}

real neuron::query_output() {
#ifndef use_instantaneous_propagation
    if(!output_same) 
#endif
        recalc_output();
#if IO_DEBUG || DEBUG
        fprintf(stderr, "            %20s::query_output()     = %f\n",
            name,
            my_output
        );
#endif
    return my_output;
}

real neuron::query_output_dot() {
#ifndef use_instantaneous_propagation
    if(!output_dot_same) 
#endif
        recalc_output_dot();
#if IO_DEBUG || DEBUG
        fprintf(stderr, "            %20s::query_output_dot() = %f\n",
            name,
            my_output_dot
        );
#endif
    return my_output_dot;
}

void neuron::recalc_input() {
#if IO_DEBUG || IO_RECALC_DEBUG || DEBUG
    fprintf(stderr, "%20s::recalc_input()\n", name);
#endif
    if(dendrites->size) {
         my_input = dendrites->sum_of_outputs();
    }

    input_same = 1;
}

void neuron::recalc_output() {
#if IO_DEBUG || IO_RECALC_DEBUG || DEBUG
    fprintf(stderr, "%20s::recalc_output()\n", name);
#endif
    my_output = transfer(query_input(), transfer_type);

    output_same = 1;
}

void neuron::recalc_output_dot() {
    my_output_dot = transfer_dot(query_input(), transfer_type);
#if IO_DEBUG || IO_RECALC_DEBUG || DEBUG
    fprintf(stderr, "%20s::recalc_output_dot(%1.2f, %i) = %1.2f\n", 
        name,
        query_input(),
        transfer_type,
        my_output_dot
    );
#endif
    output_dot_same = 1;
}

void neuron::recalc_needed() {
    if(input_same || output_same || output_dot_same) {
         input_same     = 0;
        output_same     = 0;
        output_dot_same = 0;
        axon_fires_at->recalc_needed_upward();
#if IO_RECALC_DEBUG || DEBUG
    fprintf(stderr, "d       %20s::recalc_needed()\n", name);
#endif
    }
#if IO_RECALC_DEBUG || DEBUG
    else {
    fprintf(stderr, "s       %20s::recalc_needed()\n", name);
    }
#endif
}

void neuron::connect_to(neuron *n) {
#if DEBUG || CONNECTION_DEBUG
    fprintf(stderr, "        %s::connect_to(%s)\n",
        name,
        n->query_name()
    );
#endif
    if(!is_biased) {
        dendrites->add_weight_for(this,n); // (dendrilitic, fireing)
    }
#if DEBUG || CONNECTION_DEBUG
    else {
        fprintf(stderr, "        attempt to give %s dendrites... bypassed.\n", name);
    }
#endif
}

void neuron::dendrites_touch(int n, ...) {
    neuron **k;

    for(int i=1; i<=n; i++) {
        k = (neuron **)(&n+i);
        connect_to(*k);
    }
}

void neuron::dendrites_touch(layer *l) {
    l->start_foreach();
    while (l->has_a_neuron()) {
        connect_to(l->query_current_neuron());
        l->next_neuron();
    }
}

void neuron::set_delta(real d) { 
#if DEBUG || ERROR_DEBUG
    fprintf(stderr, "%s::set_delta(%f)\n", name, d);
#endif
    my_delta = d; 
}

real neuron::query_delta() { 
    return my_delta; 
}

void neuron::set_is_in_layer_node(layer_node *l) {
    my_layer_node = l;
}

neuron *neuron::query_left_neighbor() {
    layer_node *tmp = my_layer_node->last;
    return (tmp) ? tmp->my_neuron : 0;
}

neuron *neuron::query_right_neighbor() {
    layer_node *tmp = my_layer_node->next;
    return (tmp) ? tmp->my_neuron : 0;
}

neuron *neuron::query_above_neighbor() {
    if(!(my_layer_node->above) && !(my_layer_node->below)) {
        fprintf(stderr, "Set_matrix_size() was never called for %s's layer.\n", name);
        fprintf(stderr, "But, query_above_neighbor was called anyway.\n");
        exit(1);
    }
    layer_node *tmp = my_layer_node->above;
    return (tmp) ? tmp->my_neuron : 0;
}

neuron *neuron::query_below_neighbor() {
    if(!(my_layer_node->above) && !(my_layer_node->below)) {
        fprintf(stderr, "Set_matrix_size() was never called for %s's layer.\n", name);
        fprintf(stderr, "But, query_above_neighbor was called anyway.\n");
        exit(1);
    }
    layer_node *tmp = my_layer_node->below;
    return (tmp) ? tmp->my_neuron : 0;
}

void neuron::save_weights(FILE *F) { 
    dendrites->save_weights(F);
}

void neuron::restore_weights(FILE *F) { 
    dendrites->restore_weights(F);
}
