#include <stdlib.h>
#include <time.h>

#include <arch/neuron.h>
#include <nets/kohonen.h>
#include <arch/layer.h>
#include <utils/transfer.h>

kohonen::kohonen(real _alpha, int inputs, int outputs) {
    time_t t;
    srand((rand())*(unsigned)time(&t));

     input = new layer("input",  inputs);
    output = new layer("output", outputs);

     input->set_transfer_function(SUM);
    output->set_transfer_function(SUM);

    output->dendrites_touch(input);

     input->reinitialize_weights_with(0.1, -0.1);
    output->reinitialize_weights_with(0.1, -0.1);

    alpha = _alpha;
    num_outputs = outputs;
}

void kohonen::set_matrix_size(int rows, int cols) {
    output->set_matrix_size(rows, cols);
}

void kohonen::set_input(real *inputs) {
    input->set_input(inputs);
}

real *kohonen::query_output() {
    return output->query_output();
}

int kohonen::query_max_output() {
    real max     = -32000;
    int  counter = 1;
    int  mcounter;

    output->start_foreach();
    while(output->has_a_neuron()) {
        if(output->query_current_neuron()->query_output() > max) {
            max      = output->query_current_neuron()->query_output();
            mcounter = counter;
        }
        output->next_neuron();
        counter++;
    }

    return mcounter;
}

real kohonen::update_learning_rate() {
    alpha = 0.5 * alpha;
    return alpha;
}

real kohonen::update_learning_rate(real gamma) {
    alpha = gamma * alpha;
    return alpha;
}

real kohonen::D(neuron *n) {
    real ret    = 0.0;
    real tmp    = 0.0;

    n->dendrites->start_foreach();
    while(n->dendrites->has_a_weight()) {
        tmp  = (n->dendrites->query_current_weight() -
                n->dendrites->query_current_fireing_output());
        ret += tmp*tmp;
        n->dendrites->next_weight();
    }
    return ret;
}

neuron *kohonen::train() {
    real min      = 100000;
    neuron *min_n = 0;
    real Dj;

    output->start_foreach();
    while(output->has_a_neuron()) {
        Dj = D(output->query_current_neuron());
        if(Dj<min) {
            min = Dj;
            min_n = output->query_current_neuron();
        }
        output->next_neuron();
    }

    if(!min_n) {
        fprintf(stderr, "kohonen::train() malfunctioned.  No winner was found.\n");
        exit(1);
    }

    return min_n;
}

void kohonen::linear_train(int radius) {
    update_weights_linearly(train(), radius);
}

void kohonen::rectangular_train(int radius) {
    update_weights_rectangularly(train(), radius);
}

void kohonen::update_weights_for_single_neuron(neuron *n) {
    if(!n) return;

    n->dendrites->start_foreach();
    while(n->dendrites->has_a_weight()) {
        n->dendrites->change_current_weight_by(
            (n->dendrites->query_current_fireing_output() - 
             n->dendrites->query_current_weight()) * alpha
        );
        n->dendrites->next_weight();
    }
}

void kohonen::go_left_and_right_to_radius(neuron *n, int radius) {
    neuron *left  = n;
    neuron *right = n;

    update_weights_for_single_neuron(n);
    for(int i=0; i<radius; i++) {
        if(left)  left  =  left->query_left_neighbor();
        if(right) right = right->query_right_neighbor();

        update_weights_for_single_neuron(left );
        update_weights_for_single_neuron(right);
    }
}

void kohonen::update_weights_linearly(neuron *n, int radius) {
    go_left_and_right_to_radius(n, radius);
}

void kohonen::update_weights_rectangularly(neuron *n, int radius) {
    neuron *above = n; neuron *below = n;

    go_left_and_right_to_radius(n, radius);

    for(int j=0; j<radius; j++) {
        if(above) above = above->query_above_neighbor();
        if(below) below = below->query_below_neighbor();

        go_left_and_right_to_radius(above, radius);
        go_left_and_right_to_radius(below, radius);
    }
}

void kohonen::save_net(char *filename) {
    FILE *F = fopen(filename, "w");

    output->save_weights(F);

    fclose(F);
}

void kohonen::restore_net(char *filename) {
    FILE *F = fopen(filename, "r");

    output->restore_weights(F);

    fclose(F);
}
