#!perl
# -*-cperl-*-
#
# 02-rsabssa.t - Test RSABSSA-PSS as per RFC 9474
# Copyright (c) 2024 Ashish Gulhati <crypt-rsab at hash.neo.email>

use Test::More tests => 33;
use Crypt::RSA::DataFormat qw(i2osp os2ip octet_xor);
use Try::Tiny;

BEGIN {
    use_ok( 'Crypt::RSA::Blind' ) || print "Bail out!\n";
}

diag( "Testing Crypt::RSA::Blind $Crypt::RSA::Blind::VERSION, Perl $], $^X" );

my $rsab = new Crypt::RSA::Blind;
my @testvectors = testvecs();

my ($pubkey, $seckey);

for (@testvectors) {
  diag ($_->{name});
  ok (($pubkey, $seckey) = $rsab->keygen (e => $_->{e}, p => $_->{p}, q => $_->{q}), "Key generation");
  my $init = $rsab->init;
  my $msg = $_->{randomize} ? $rsab->ssa_randomize($_->{msg}) : $_->{msg};
  ok (my ($blinded_msg, $inv) = $rsab->ssa_blind( PublicKey => $pubkey, Message => $msg, sLen => $_->{slen},
						  Init => $init, R_inv => $_->{r}, Salt => $_->{salt}),
      'Create blind signing request');
  ok (os2ip($blinded_msg) == $_->{blm}, "Check blind signing request") unless $_->{randomize};
  ok (my $blind_sig = $rsab->ssa_blind_sign( SecretKey => $seckey, BlindedMessage => $blinded_msg),
      'Create blind signature');
  ok (os2ip($blind_sig) == $_->{bls}, 'Check blind signature') unless $_->{randomize};
  ok (my $sig = $rsab->ssa_finalize( PublicKey => $pubkey, BlindSig => $blind_sig, Blinding => $inv,
				     Message => $msg, sLen => $_->{slen}), 'Unblind signature');
  ok ($sig = $rsab->ssa_finalize( PublicKey => $pubkey, BlindSig => $blind_sig, Init => $init,
				  Message => $msg, sLen => $_->{slen}), 'Unblind signature with Init vector');
  ok (os2ip($sig) == $_->{sg}, 'Check signature') unless $_->{randomize};
}

diag ('Without test vectors');

ok (($pubkey, $seckey) = $rsab->keygen(Size => 1024), "Key generation (1024 bits)");

my $msg = "Hello, world!";
my $slen = 0;
my $init = $rsab->init;
my ($blinded_msg, $inv) = $rsab->ssa_blind (PublicKey => $pubkey, Message => $msg, sLen => $slen, Init => $init);
my $blind_sig = $rsab->ssa_blind_sign(SecretKey => $seckey, BlindedMessage => $blinded_msg);
my $sig = $rsab->ssa_finalize(PublicKey => $pubkey, BlindSig => $blind_sig, Init => $init, Message => $msg, sLen => $slen);
my $sig = $rsab->ssa_finalize(PublicKey => $pubkey, BlindSig => $blind_sig, Blinding => $inv, Message => $msg, sLen => $slen);
ok (try { $rsab->pss_verify(PublicKey => $pubkey, Signature => $sig, Message => $msg, sLen => $slen) }, 'Deterministic, no salt');

$slen = 48;
$init = $rsab->init;
my ($blinded_msg, $inv) = $rsab->ssa_blind (PublicKey => $pubkey, Message => $msg, sLen => $slen, Init => $init);
my $blind_sig = $rsab->ssa_blind_sign(SecretKey => $seckey, BlindedMessage => $blinded_msg);
my $sig = $rsab->ssa_finalize(PublicKey => $pubkey, BlindSig => $blind_sig, Init => $init, Message => $msg, sLen => $slen);
my $sig = $rsab->ssa_finalize(PublicKey => $pubkey, BlindSig => $blind_sig, Blinding => $inv, Message => $msg, sLen => $slen);
ok (try { $rsab->pss_verify(PublicKey => $pubkey, Signature => $sig, Message => $msg, sLen => $slen) }, 'Deterministic, with salt');

ok (($pubkey, $seckey) = $rsab->keygen(Size => 1024), "Key generation (2048 bits)");

$msg = $rsab->ssa_randomize($msg);
my $slen = 0;
$init = $rsab->init;
my ($blinded_msg, $inv) = $rsab->ssa_blind (PublicKey => $pubkey, Message => $msg, sLen => $slen, Init => $init);
my $blind_sig = $rsab->ssa_blind_sign(SecretKey => $seckey, BlindedMessage => $blinded_msg);
my $sig = $rsab->ssa_finalize(PublicKey => $pubkey, BlindSig => $blind_sig, Init => $init, Message => $msg, sLen => $slen);
my $sig = $rsab->ssa_finalize(PublicKey => $pubkey, BlindSig => $blind_sig, Blinding => $inv, Message => $msg, sLen => $slen);
ok (try { $rsab->pss_verify(PublicKey => $pubkey, Signature => $sig, Message => $msg, sLen => $slen) }, 'Randomized, no salt');

$msg = "Hello, world!";
$msg = $rsab->ssa_randomize($msg);
$slen = 48;
$init = $rsab->init;
my ($blinded_msg, $inv) = $rsab->ssa_blind (PublicKey => $pubkey, Message => $msg, sLen => $slen, Init => $init);
my $blind_sig = $rsab->ssa_blind_sign(SecretKey => $seckey, BlindedMessage => $blinded_msg);
my $sig = $rsab->ssa_finalize(PublicKey => $pubkey, BlindSig => $blind_sig, Init => $init, Message => $msg, sLen => $slen);
my $sig = $rsab->ssa_finalize(PublicKey => $pubkey, BlindSig => $blind_sig, Blinding => $inv, Message => $msg, sLen => $slen);
ok (try { $rsab->pss_verify(PublicKey => $pubkey, Signature => $sig, Message => $msg, sLen => $slen) }, 'Randomized, with salt');

exit;

sub testvecs {
  my @testvectors = ( { name => 'RSABSSA-SHA384-PSS-Deterministic',
			msg => i2osp(Math::Pari::_hex_cvt('0x8F3DC6FB8C4A02F4D6352EDF0907822C1210A9B32F9BDDA4C45A698C80023AA6B59F8CFEC5FDBB36331372EBEFEDAE7D'), 48),
			n => '0xAEC4D69ADDC70B990EA66A5E70603B6FEE27AAFEBD08F2D94CBE1250C556E047A928D635C3F45EE9B66D1BC628A03BAC9B7C3F416FE20DABEA8F3D7B4BBF7F963BE335D2328D67E6C13EE4A8F955E05A3283720D3E1F139C38E43E0338AD058A9495C53377FC35BE64D208F89B4AA721BF7F7D3FEF837BE2A80E0F8ADF0BCD1EEC5BB040443A2B2792FDCA522A7472AED74F31A1EBE1EEBC1F408660A0543DFE2A850F106A617EC6685573702EAAA21A5640A5DCAF9B74E397FA3AF18A2F1B7C03BA91A6336158DE420D63188EE143866EE415735D155B7C2D854D795B7BC236CFFD71542DF34234221A0413E142D8C61355CC44D45BDA94204974557AC2704CD8B593F035A5724B1ADF442E78C542CD4414FCE6F1298182FB6D8E53CEF1ADFD2E90E1E4DEEC52999BDC6C29144E8D52A125232C8C6D75C706EA3CC06841C7BDA33568C63A6C03817F722B50FCF898237D788A4400869E44D90A3020923DC646388ABCC914315215FCD1BAE11B1C751FD52443AAC8F601087D8D42737C18A3FA11ECD4131ECAE017AE0A14ACFC4EF85B83C19FED33CFD1CD629DA2C4C09E222B398E18D822F77BB378DEA3CB360B605E5AA58B20EDC29D000A66BD177C682A17E7EB12A63EF7C2E4183E0D898F3D6BF567BA8AE84F84F1D23BF8B8E261C3729E2FA6D07B832E07CDDD1D14F55325C6F924267957121902DC19B3B32948BDEAD5',
			e => '0x010001',
			d => '0x0D43242AEFE1FB2C13FBC66E20B678C4336D20B1808C558B6E62AD16A287077180B177E1F01B12F9C6CD6C52630257CCEF26A45135A990928773F3BD2FC01A313F1DAC97A51CEC71CB1FD7EFC7ADFFDEB05F1FB04812C924ED7F4A8269925DAD88BD7DCFBC4EF01020EBFC60CB3E04C54F981FDBD273E69A8A58B8CEB7C2D83FBCBD6F784D052201B88A9848186F2A45C0D2826870733E6FD9AA46983E0A6E82E35CA20A439C5EE7B502A9062E1066493BDADF8B49EB30D9558ED85ABC7AFB29B3C9BC644199654A4676681AF4BABCEA4E6F71FE4565C9C1B85D9985B84EC1ABF1A820A9BBEBEE0DF1398AAE2C85AB580A9F13E7743AFD3108EB32100B870648FA6BC17E8ABAC4D3C99246B1F0EA9F7F93A5DD5458C56D9F3F81FF2216B3C3680A13591673C43194D8E6FC93FC1E37CE2986BD628AC48088BC723D8FBE293861CA7A9F4A73E9FA63B1B6D0074F5DEA2A624C5249FF3AD811B6255B299D6BC5451BA7477F19C5A0DB690C3E6476398B1483D10314AFD38BBAF6E2FBDBCD62C3CA9797A420CA6034EC0A83360A3EE2ADF4B9D4BA29731D131B099A38D6A23CC463DB754603211260E99D19AFFC902C915D7854554AABF608E3AC52C19B8AA26AE042249B17B2D29669B5C859103EE53EF9BDC73BA3C6B537D5C34B6D8F034671D7F3A8A6966CC4543DF223565343154140FD7391C7E7BE03E241F4ECFEB877A051',
			p => '0xE1F4D7A34802E27C7392A3CEA32A262A34DC3691BD87F3F310DC75673488930559C120FD0410194FB8A0DA55BD0B81227E843FDCA6692AE80E5A5D414116D4803FCA7D8C30EAAAE57E44A1816EBB5C5B0606C536246C7F11985D731684150B63C9A3AD9E41B04C0B5B27CB188A692C84696B742A80D3CD00AB891F2457443DADFEBA6D6DAF108602BE26D7071803C67105A5426838E6889D77E8474B29244CEFAF418E381B312048B457D73419213063C60EE7B0D81820165864FEF93523C9635C22210956E53A8D96322493FFC58D845368E2416E078E5BCB5D2FD68AE6ACFA54F9627C42E84A9D3F2774017E32EBCA06308A12ECC290C7CD1156DCCCFB2311',
			q => '0xC601A9CAEA66DC3835827B539DB9DF6F6F5AE77244692780CD334A006AB353C806426B60718C05245650821D39445D3AB591ED10A7339F15D83FE13F6A3DFB20B9452C6A9B42EAA62A68C970DF3CADB2139F804AD8223D56108DFDE30BA7D367E9B0A7A80C4FDBA2FD9DDE6661FC73FC2947569D2029F2870FC02D8325ACF28C9AFA19ECF962DAA7916E21AFAD09EB62FE9F1CF91B77DC879B7974B490D3EBD2E95426057F35D0A3C9F45F79AC727AB81A519A8B9285932D9B2E5CCD347E59F3F32AD9CA359115E7DA008AB7406707BD0E8E185A5ED8758B5BA266E8828F8D863AE133846304A2936AD7BC7C9803879D2FC4A28E69291D73DBD799F8BC238385',
			r => Math::Pari::_hex_cvt('0x80682C48982407B489D53D1261B19EC8627D02B8CDA5336750B8CEE332AE260DE57B02D72609C1E0E9F28E2040FC65B6F02D56DBD6AA9AF8FDE656F70495DFB723BA01173D4707A12FDDAC628CA29F3E32340BD8F7DDB557CF819F6B01E445AD96F874BA235584EE71F6581F62D4F43BF03F910F6510DEB85E8EF06C7F09D9794A008BE7FF2529F0EBB69DECEF646387DC767B74939265FEC0223AA6D84D2A8A1CC912D5CA25B4E144AB8F6BA054B54910176D5737A2CFF011DA431BD5F2A0D2D66B9E70B39F4B050E45C0D9C16F02DEDA9DDF2D00F3E4B01037D7029CD49C2D46A8E1FC2C0C17520AF1F4B5E25BA396AFC4CD60C494A4C426448B35B49635B337CFB08E7C22A39B256DD032C00ADDDAFB51A627F99A0E1704170AC1F1912E49D9DB10EC04C19C58F420212973E0CB329524223A6AA56C7937C5DFFDB5D966B6CD4CBC26F3201DD25C80960A1A111B32947BB78973D269FAC7F5186530930ED19F68507540EED9E1BAB8B00F00D8CA09B3F099AAE46180E04E3584BD7CA054DF18A1504B89D1D1675D0966C4AE1407BE325CDF623CF13FF13E4A28B594D59E3EADBADF6136EEE7A59D6A444C9EB4E2198E8A974F27A39EB63AF2C9AF3870488B8ADAAD444674F512133AD80B9220E09158521614F1FAADFE8505EF57B7DF6813048603F0DD04F4280177A11380FBFC861DBCBD7418D62155248DAD5FDEC0991F'),
			slen => 48,
			salt => '051722B35F458781397C3A671A7D3BD3096503940E4C4F1AAA269D60300CE449555CD7340100DF9D46944C5356825ABF',
			randomize => 0,
			blm => Math::Pari::_hex_cvt('0x10c166c6a711e81c46f45b18e5873cc4f494f003180dd7f115585d871a28930259654fe28a54dab319cc5011204c8373b50a57b0fdc7a678bd74c523259dfe4fd5ea9f52f170e19dfa332930ad1609fc8a00902d725cfe50685c95e5b2968c9a2828a21207fcf393d15f849769e2af34ac4259d91dfd98c3a707c509e1af55647efaa31290ddf48e0133b798562af5eabd327270ac2fb6c594734ce339a14ea4fe1b9a2f81c0bc230ca523bda17ff42a377266bc2778a274c0ae5ec5a8cbbe364fcf0d2403f7ee178d77ff28b67a20c7ceec009182dbcaa9bc99b51ebbf13b7d542be337172c6474f2cd3561219fe0dfa3fb207cff89632091ab841cf38d8aa88af6891539f263adb8eac6402c41b6ebd72984e43666e537f5f5fe27b2b5aa114957e9a580730308a5f5a9c63a1eb599f093ab401d0c6003a451931b6d124180305705845060ebba6b0036154fcef3e5e9f9e4b87e8f084542fd1dd67e7782a5585150181c01eb6d90cb95883837384a5b91dbb606f266059ecc51b5acbaa280e45cfd2eec8cc1cdb1b7211c8e14805ba683f9b78824b2eb005bc8a7d7179a36c152cb87c8219e5569bba911bb32a1b923ca83de0e03fb10fba75d85c55907dda5a2606bf918b056c3808ba496a4d95532212040a5f44f37e1097f26dc27b98a51837daa78f23e532156296b64352669c94a8a855acf30533d8e0594ace7c442'),
			bls => Math::Pari::_hex_cvt('0x364f6a40dbfbc3bbb257943337eeff791a0f290898a6791283bba581d9eac90a6376a837241f5f73a78a5c6746e1306ba3adab6067c32ff69115734ce014d354e2f259d4cbfb890244fd451a497fe6ecf9aa90d19a2d441162f7eaa7ce3fc4e89fd4e76b7ae585be2a2c0fd6fb246b8ac8d58bcb585634e30c9168a434786fe5e0b74bfe8187b47ac091aa571ffea0a864cb906d0e28c77a00e8cd8f6aba4317a8cc7bf32ce566bd1ef80c64de041728abe087bee6cadd0b7062bde5ceef308a23bd1ccc154fd0c3a26110df6193464fc0d24ee189aea8979d722170ba945fdcce9b1b4b63349980f3a92dc2e5418c54d38a862916926b3f9ca270a8cf40dfb9772bfbdd9a3e0e0892369c18249211ba857f35963d0e05d8da98f1aa0c6bba58f47487b8f663e395091275f82941830b050b260e4767ce2fa903e75ff8970c98bfb3a08d6db91ab1746c86420ee2e909bf681cac173697135983c3594b2def673736220452fde4ddec867d40ff42dd3da36c84e3e52508b891a00f50b4f62d112edb3b6b6cc3dbd546ba10f36b03f06c0d82aeec3b25e127af545fac28e1613a0517a6095ad18a98ab79f68801e05c175e15bae21f821e80c80ab4fdec6fb34ca315e194502b8f3dcf7892b511aee45060e3994cd15e003861bc7220a2babd7b40eda03382548a34a7110f9b1779bf3ef6011361611e6bc5c0dc851e1509de1a'),
			sg => Math::Pari::_hex_cvt('0x6fef8bf9bc182cd8cf7ce45c7dcf0e6f3e518ae48f06f3c670c649ac737a8b8119a34d51641785be151a697ed7825fdfece82865123445eab03eb4bb91cecf4d6951738495f8481151b62de869658573df4e50a95c17c31b52e154ae26a04067d5ecdc1592c287550bb982a5bb9c30fd53a768cee6baabb3d483e9f1e2da954c7f4cf492fe3944d2fe456c1ecaf0840369e33fb4010e6b44bb1d721840513524d8e9a3519f40d1b81ae34fb7a31ee6b7ed641cb16c2ac999004c2191de0201457523f5a4700dd649267d9286f5c1d193f1454c9f868a57816bf5ff76c838a2eeb616a3fc9976f65d4371deecfbab29362caebdff69c635fe5a2113da4d4d8c24f0b16a0584fa05e80e607c5d9a2f765f1f069f8d4da21f27c2a3b5c984b4ab24899bef46c6d9323df4862fe51ce300fca40fb539c3bb7fe2dcc9409e425f2d3b95e70e9c49c5feb6ecc9d43442c33d50003ee936845892fb8be475647da9a080f5bc7f8a716590b3745c2209fe05b17992830ce15f32c7b22cde755c8a2fe50bd814a0434130b807dc1b7218d4e85342d70695a5d7f29306f25623ad1e8aa08ef71b54b8ee447b5f64e73d09bdd6c3b7ca224058d7c67cc7551e9241688ada12d859cb7646fbd3ed8b34312f3b49d69802f0eaa11bc4211c2f7a29cd5c01ed01a39001c5856fab36228f5ee2f2e1110811872fe7c865c42ed59029c706195d52')
		      },
		      { name => 'RSABSSA_SHA384_PSSZERO_Deterministic',
			msg => i2osp(Math::Pari::_hex_cvt('0x8F3DC6FB8C4A02F4D6352EDF0907822C1210A9B32F9BDDA4C45A698C80023AA6B59F8CFEC5FDBB36331372EBEFEDAE7D'), 48),
			n => '0xAEC4D69ADDC70B990EA66A5E70603B6FEE27AAFEBD08F2D94CBE1250C556E047A928D635C3F45EE9B66D1BC628A03BAC9B7C3F416FE20DABEA8F3D7B4BBF7F963BE335D2328D67E6C13EE4A8F955E05A3283720D3E1F139C38E43E0338AD058A9495C53377FC35BE64D208F89B4AA721BF7F7D3FEF837BE2A80E0F8ADF0BCD1EEC5BB040443A2B2792FDCA522A7472AED74F31A1EBE1EEBC1F408660A0543DFE2A850F106A617EC6685573702EAAA21A5640A5DCAF9B74E397FA3AF18A2F1B7C03BA91A6336158DE420D63188EE143866EE415735D155B7C2D854D795B7BC236CFFD71542DF34234221A0413E142D8C61355CC44D45BDA94204974557AC2704CD8B593F035A5724B1ADF442E78C542CD4414FCE6F1298182FB6D8E53CEF1ADFD2E90E1E4DEEC52999BDC6C29144E8D52A125232C8C6D75C706EA3CC06841C7BDA33568C63A6C03817F722B50FCF898237D788A4400869E44D90A3020923DC646388ABCC914315215FCD1BAE11B1C751FD52443AAC8F601087D8D42737C18A3FA11ECD4131ECAE017AE0A14ACFC4EF85B83C19FED33CFD1CD629DA2C4C09E222B398E18D822F77BB378DEA3CB360B605E5AA58B20EDC29D000A66BD177C682A17E7EB12A63EF7C2E4183E0D898F3D6BF567BA8AE84F84F1D23BF8B8E261C3729E2FA6D07B832E07CDDD1D14F55325C6F924267957121902DC19B3B32948BDEAD5',
			e => '0x010001',
			d => '0x0D43242AEFE1FB2C13FBC66E20B678C4336D20B1808C558B6E62AD16A287077180B177E1F01B12F9C6CD6C52630257CCEF26A45135A990928773F3BD2FC01A313F1DAC97A51CEC71CB1FD7EFC7ADFFDEB05F1FB04812C924ED7F4A8269925DAD88BD7DCFBC4EF01020EBFC60CB3E04C54F981FDBD273E69A8A58B8CEB7C2D83FBCBD6F784D052201B88A9848186F2A45C0D2826870733E6FD9AA46983E0A6E82E35CA20A439C5EE7B502A9062E1066493BDADF8B49EB30D9558ED85ABC7AFB29B3C9BC644199654A4676681AF4BABCEA4E6F71FE4565C9C1B85D9985B84EC1ABF1A820A9BBEBEE0DF1398AAE2C85AB580A9F13E7743AFD3108EB32100B870648FA6BC17E8ABAC4D3C99246B1F0EA9F7F93A5DD5458C56D9F3F81FF2216B3C3680A13591673C43194D8E6FC93FC1E37CE2986BD628AC48088BC723D8FBE293861CA7A9F4A73E9FA63B1B6D0074F5DEA2A624C5249FF3AD811B6255B299D6BC5451BA7477F19C5A0DB690C3E6476398B1483D10314AFD38BBAF6E2FBDBCD62C3CA9797A420CA6034EC0A83360A3EE2ADF4B9D4BA29731D131B099A38D6A23CC463DB754603211260E99D19AFFC902C915D7854554AABF608E3AC52C19B8AA26AE042249B17B2D29669B5C859103EE53EF9BDC73BA3C6B537D5C34B6D8F034671D7F3A8A6966CC4543DF223565343154140FD7391C7E7BE03E241F4ECFEB877A051',
			p => '0xE1F4D7A34802E27C7392A3CEA32A262A34DC3691BD87F3F310DC75673488930559C120FD0410194FB8A0DA55BD0B81227E843FDCA6692AE80E5A5D414116D4803FCA7D8C30EAAAE57E44A1816EBB5C5B0606C536246C7F11985D731684150B63C9A3AD9E41B04C0B5B27CB188A692C84696B742A80D3CD00AB891F2457443DADFEBA6D6DAF108602BE26D7071803C67105A5426838E6889D77E8474B29244CEFAF418E381B312048B457D73419213063C60EE7B0D81820165864FEF93523C9635C22210956E53A8D96322493FFC58D845368E2416E078E5BCB5D2FD68AE6ACFA54F9627C42E84A9D3F2774017E32EBCA06308A12ECC290C7CD1156DCCCFB2311',
			q => '0xC601A9CAEA66DC3835827B539DB9DF6F6F5AE77244692780CD334A006AB353C806426B60718C05245650821D39445D3AB591ED10A7339F15D83FE13F6A3DFB20B9452C6A9B42EAA62A68C970DF3CADB2139F804AD8223D56108DFDE30BA7D367E9B0A7A80C4FDBA2FD9DDE6661FC73FC2947569D2029F2870FC02D8325ACF28C9AFA19ECF962DAA7916E21AFAD09EB62FE9F1CF91B77DC879B7974B490D3EBD2E95426057F35D0A3C9F45F79AC727AB81A519A8B9285932D9B2E5CCD347E59F3F32AD9CA359115E7DA008AB7406707BD0E8E185A5ED8758B5BA266E8828F8D863AE133846304A2936AD7BC7C9803879D2FC4A28E69291D73DBD799F8BC238385',
			r => Math::Pari::_hex_cvt('0x80682C48982407B489D53D1261B19EC8627D02B8CDA5336750B8CEE332AE260DE57B02D72609C1E0E9F28E2040FC65B6F02D56DBD6AA9AF8FDE656F70495DFB723BA01173D4707A12FDDAC628CA29F3E32340BD8F7DDB557CF819F6B01E445AD96F874BA235584EE71F6581F62D4F43BF03F910F6510DEB85E8EF06C7F09D9794A008BE7FF2529F0EBB69DECEF646387DC767B74939265FEC0223AA6D84D2A8A1CC912D5CA25B4E144AB8F6BA054B54910176D5737A2CFF011DA431BD5F2A0D2D66B9E70B39F4B050E45C0D9C16F02DEDA9DDF2D00F3E4B01037D7029CD49C2D46A8E1FC2C0C17520AF1F4B5E25BA396AFC4CD60C494A4C426448B35B49635B337CFB08E7C22A39B256DD032C00ADDDAFB51A627F99A0E1704170AC1F1912E49D9DB10EC04C19C58F420212973E0CB329524223A6AA56C7937C5DFFDB5D966B6CD4CBC26F3201DD25C80960A1A111B32947BB78973D269FAC7F5186530930ED19F68507540EED9E1BAB8B00F00D8CA09B3F099AAE46180E04E3584BD7CA054DF18A1504B89D1D1675D0966C4AE1407BE325CDF623CF13FF13E4A28B594D59E3EADBADF6136EEE7A59D6A444C9EB4E2198E8A974F27A39EB63AF2C9AF3870488B8ADAAD444674F512133AD80B9220E09158521614F1FAADFE8505EF57B7DF6813048603F0DD04F4280177A11380FBFC861DBCBD7418D62155248DAD5FDEC0991F'),
			slen => 0,
			salt => '',
			randomize => 0,
			blm => Math::Pari::_hex_cvt('0x982790826556aabe6004467671a864397eea3b95740e9a11c8b80b99ee0cf4dbc50af860bda81b601a2eceaa6943ef104f13325ad0be2e37f42030b3120e87cfee8cfe59cde1acfb25485a43275ebe777292e2518181ae531e596f988ff16f458daa5a42408939cbe60e7271391a21657276427d195bee6a20054101d4ceb892ecdea402ea1a866acf0e451a3336f07e7589330d96c3883fd5bc1a829a715b618b74a86b2a898764246ad081d4c9f1edb8ab5077e315fde2417ec2dd33cad93e120340b49be89c18a63e62c6bb289037283d3bf18608be11ee4c823c710b0c6b89235fed3f03a7b96ddd25a8f54f20dac37ce8905093ad8e066810f354fb1773236e3d3788ba755de2c9bce8d340078bb1831ddc7314a5018673427ced65cb356281aae08b5e6636f3eb2417e09d6ae476a9abcc410bc8c90813d0740e39ae75efae4c02eed49dbb7aa51258bb71197445d17a6029bf566ba6b36282173af2c42e9b9631366f22eb6a19ef1d92bd3ce0631d3a7fb3288195b0ba380a3828d5411cefd5eba83e52198c001ac9946a333a33d89d4d235fc833239d59837f04eaf065e9563659b00c7624a6263b727d8f2c07959ba2bb592e7ff251b8f09c85995fd2e4474e743586576b518230986b6076b762ae77088a37e4bffd2ef41ae68d6d4e79205290b4f76c42ef039638c41cdc6fe8af9b429c0dee45b2942e3861da2a'),
			bls => Math::Pari::_hex_cvt('0x362ef369f9b8c1487e285514702a7cd6fe03e4a2fb854881f3d3f986b7742a0c9bfab6562a6cd5ed71c574af67d7e77e71b33420c08ebb0ff37886b858297f9562fc366066c6d8e77bad1918b04756ba03f5c385d44f06759daf1b7a38b2a64248dee95d0e3886c8afa1f74afd8ac3c56520d0f3fd206df8e0d257312756803b09a79d0cc38112592c3aec32de5a9bc3284c5a0a2d0808b102deafa5cc60f04e3d71c0284cba04f17f88aa8e07d5544fe0265807d515877f79d30ed26d522b9d9c56597647b0dbca5a69d6418f8d1b51481723f272c2a3d48f6f4fd6beeac3576c3edb00e8779964548aeab8e004c7c4f8ef9cb6e680e2d2d49792004bb3e6974fa48f241a361ca449c02bd4c0ad4e66252c55e656f16049908efe59acbafa1171895dfac64d909808e5420469d622c7253ec1de7522b41634d383bf8786bf881cbf1561627f1e62b2d93300ec30ec0f5f0ab32036fce068bc76b0b0c6452079537f8d7f8dcee4b42bbf2d9ad7499d3835cd93cfc7e8ebea3554ab5241e181e5d73241b7bebf0a281b63594a35f4993e2b416d60db966b58b648cfcba2c4bee4c2830aae4a70ff55012480298f549c13b1b2684277bca12f592471b8a99285174f1c0ebb38fc80e74a10b3f02ec3e6682ba873f7ff0e1e79718b470927c74ed754d4f7c3d9a55e22246e829cdb5a1c6fb2a0a6c896df303063c918bcf5eb0017'),
			sg => Math::Pari::_hex_cvt('0x4454b6983ff01cb28545329f394936efa42ed231e15efbc025fdaca00277acf0c8e00e3d8b0ecebd35b057b8ebfc14e1a7097368a4abd20b555894ccef3d1b9528c6bcbda6b95376bef230d0f1feff0c1064c62c60a7ae7431d1fdfa43a81eed9235e363e1ffa0b2797aba6aad6082fcd285e14fc8b71de6b9c87cb4059c7dc1e96ae1e63795a1e9af86b9073d1d848aef3eca8a03421bcd116572456b53bcfd4dabb0a9691f1fabda3ed0ce357aee2cfee5b1a0eb226f69716d4e011d96eede5e38a9acb531a64336a0d5b0bae3ab085b658692579a376740ff6ce69e89b06f360520b864e33d82d029c808248a19e18e31f0ecd16fac5cd4870f8d3ebc1c32c718124152dc905672ab0b7af48bf7d1ac1ff7b9c742549c91275ab105458ae37621757add83482bbcf779e777bbd61126e93686635d4766aedf5103cf7978f3856ccac9e28d21a850dbb03c811128616d315d717be1c2b6254f8509acae862042c034530329ce15ca2e2f6b1f5fd59272746e3918c748c0eb810bf76884fa10fcf749326bbfaa5ba285a0186a22e4f628dbf178d3bb5dc7e165ca73f6a55ecc14c4f5a26c4693ce5da032264cbec319b12ddb9787d0efa4fcf1e5ccee35ad85ecd453182df9ed735893f830b570faae8be0f6fe2e571a4e0d927cba4debd368d3b4fca33ec6251897a137cf75474a32ac8256df5e5ffa518b88b43fb6f63a24')
		      },
		      { name => 'RSABSSA_SHA384_PSS_Randomized',
			msg => i2osp(Math::Pari::_hex_cvt('0x8F3DC6FB8C4A02F4D6352EDF0907822C1210A9B32F9BDDA4C45A698C80023AA6B59F8CFEC5FDBB36331372EBEFEDAE7D'), 48),
			n => '0xAEC4D69ADDC70B990EA66A5E70603B6FEE27AAFEBD08F2D94CBE1250C556E047A928D635C3F45EE9B66D1BC628A03BAC9B7C3F416FE20DABEA8F3D7B4BBF7F963BE335D2328D67E6C13EE4A8F955E05A3283720D3E1F139C38E43E0338AD058A9495C53377FC35BE64D208F89B4AA721BF7F7D3FEF837BE2A80E0F8ADF0BCD1EEC5BB040443A2B2792FDCA522A7472AED74F31A1EBE1EEBC1F408660A0543DFE2A850F106A617EC6685573702EAAA21A5640A5DCAF9B74E397FA3AF18A2F1B7C03BA91A6336158DE420D63188EE143866EE415735D155B7C2D854D795B7BC236CFFD71542DF34234221A0413E142D8C61355CC44D45BDA94204974557AC2704CD8B593F035A5724B1ADF442E78C542CD4414FCE6F1298182FB6D8E53CEF1ADFD2E90E1E4DEEC52999BDC6C29144E8D52A125232C8C6D75C706EA3CC06841C7BDA33568C63A6C03817F722B50FCF898237D788A4400869E44D90A3020923DC646388ABCC914315215FCD1BAE11B1C751FD52443AAC8F601087D8D42737C18A3FA11ECD4131ECAE017AE0A14ACFC4EF85B83C19FED33CFD1CD629DA2C4C09E222B398E18D822F77BB378DEA3CB360B605E5AA58B20EDC29D000A66BD177C682A17E7EB12A63EF7C2E4183E0D898F3D6BF567BA8AE84F84F1D23BF8B8E261C3729E2FA6D07B832E07CDDD1D14F55325C6F924267957121902DC19B3B32948BDEAD5',
			e => '0x010001',
			d => '0x0D43242AEFE1FB2C13FBC66E20B678C4336D20B1808C558B6E62AD16A287077180B177E1F01B12F9C6CD6C52630257CCEF26A45135A990928773F3BD2FC01A313F1DAC97A51CEC71CB1FD7EFC7ADFFDEB05F1FB04812C924ED7F4A8269925DAD88BD7DCFBC4EF01020EBFC60CB3E04C54F981FDBD273E69A8A58B8CEB7C2D83FBCBD6F784D052201B88A9848186F2A45C0D2826870733E6FD9AA46983E0A6E82E35CA20A439C5EE7B502A9062E1066493BDADF8B49EB30D9558ED85ABC7AFB29B3C9BC644199654A4676681AF4BABCEA4E6F71FE4565C9C1B85D9985B84EC1ABF1A820A9BBEBEE0DF1398AAE2C85AB580A9F13E7743AFD3108EB32100B870648FA6BC17E8ABAC4D3C99246B1F0EA9F7F93A5DD5458C56D9F3F81FF2216B3C3680A13591673C43194D8E6FC93FC1E37CE2986BD628AC48088BC723D8FBE293861CA7A9F4A73E9FA63B1B6D0074F5DEA2A624C5249FF3AD811B6255B299D6BC5451BA7477F19C5A0DB690C3E6476398B1483D10314AFD38BBAF6E2FBDBCD62C3CA9797A420CA6034EC0A83360A3EE2ADF4B9D4BA29731D131B099A38D6A23CC463DB754603211260E99D19AFFC902C915D7854554AABF608E3AC52C19B8AA26AE042249B17B2D29669B5C859103EE53EF9BDC73BA3C6B537D5C34B6D8F034671D7F3A8A6966CC4543DF223565343154140FD7391C7E7BE03E241F4ECFEB877A051',
			p => '0xE1F4D7A34802E27C7392A3CEA32A262A34DC3691BD87F3F310DC75673488930559C120FD0410194FB8A0DA55BD0B81227E843FDCA6692AE80E5A5D414116D4803FCA7D8C30EAAAE57E44A1816EBB5C5B0606C536246C7F11985D731684150B63C9A3AD9E41B04C0B5B27CB188A692C84696B742A80D3CD00AB891F2457443DADFEBA6D6DAF108602BE26D7071803C67105A5426838E6889D77E8474B29244CEFAF418E381B312048B457D73419213063C60EE7B0D81820165864FEF93523C9635C22210956E53A8D96322493FFC58D845368E2416E078E5BCB5D2FD68AE6ACFA54F9627C42E84A9D3F2774017E32EBCA06308A12ECC290C7CD1156DCCCFB2311',
			q => '0xC601A9CAEA66DC3835827B539DB9DF6F6F5AE77244692780CD334A006AB353C806426B60718C05245650821D39445D3AB591ED10A7339F15D83FE13F6A3DFB20B9452C6A9B42EAA62A68C970DF3CADB2139F804AD8223D56108DFDE30BA7D367E9B0A7A80C4FDBA2FD9DDE6661FC73FC2947569D2029F2870FC02D8325ACF28C9AFA19ECF962DAA7916E21AFAD09EB62FE9F1CF91B77DC879B7974B490D3EBD2E95426057F35D0A3C9F45F79AC727AB81A519A8B9285932D9B2E5CCD347E59F3F32AD9CA359115E7DA008AB7406707BD0E8E185A5ED8758B5BA266E8828F8D863AE133846304A2936AD7BC7C9803879D2FC4A28E69291D73DBD799F8BC238385',
			r => Math::Pari::_hex_cvt('0x80682C48982407B489D53D1261B19EC8627D02B8CDA5336750B8CEE332AE260DE57B02D72609C1E0E9F28E2040FC65B6F02D56DBD6AA9AF8FDE656F70495DFB723BA01173D4707A12FDDAC628CA29F3E32340BD8F7DDB557CF819F6B01E445AD96F874BA235584EE71F6581F62D4F43BF03F910F6510DEB85E8EF06C7F09D9794A008BE7FF2529F0EBB69DECEF646387DC767B74939265FEC0223AA6D84D2A8A1CC912D5CA25B4E144AB8F6BA054B54910176D5737A2CFF011DA431BD5F2A0D2D66B9E70B39F4B050E45C0D9C16F02DEDA9DDF2D00F3E4B01037D7029CD49C2D46A8E1FC2C0C17520AF1F4B5E25BA396AFC4CD60C494A4C426448B35B49635B337CFB08E7C22A39B256DD032C00ADDDAFB51A627F99A0E1704170AC1F1912E49D9DB10EC04C19C58F420212973E0CB329524223A6AA56C7937C5DFFDB5D966B6CD4CBC26F3201DD25C80960A1A111B32947BB78973D269FAC7F5186530930ED19F68507540EED9E1BAB8B00F00D8CA09B3F099AAE46180E04E3584BD7CA054DF18A1504B89D1D1675D0966C4AE1407BE325CDF623CF13FF13E4A28B594D59E3EADBADF6136EEE7A59D6A444C9EB4E2198E8A974F27A39EB63AF2C9AF3870488B8ADAAD444674F512133AD80B9220E09158521614F1FAADFE8505EF57B7DF6813048603F0DD04F4280177A11380FBFC861DBCBD7418D62155248DAD5FDEC0991F'),
			slen => 48,
			salt => '051722B35F458781397C3A671A7D3BD3096503940E4C4F1AAA269D60300CE449555CD7340100DF9D46944C5356825ABF',
			randomize => 1,
		      },
		      { name => 'RSABSSA_SHA384_PSSZERO_Randomized',
			msg => i2osp(Math::Pari::_hex_cvt('0x8F3DC6FB8C4A02F4D6352EDF0907822C1210A9B32F9BDDA4C45A698C80023AA6B59F8CFEC5FDBB36331372EBEFEDAE7D'), 48),
			n => '0xAEC4D69ADDC70B990EA66A5E70603B6FEE27AAFEBD08F2D94CBE1250C556E047A928D635C3F45EE9B66D1BC628A03BAC9B7C3F416FE20DABEA8F3D7B4BBF7F963BE335D2328D67E6C13EE4A8F955E05A3283720D3E1F139C38E43E0338AD058A9495C53377FC35BE64D208F89B4AA721BF7F7D3FEF837BE2A80E0F8ADF0BCD1EEC5BB040443A2B2792FDCA522A7472AED74F31A1EBE1EEBC1F408660A0543DFE2A850F106A617EC6685573702EAAA21A5640A5DCAF9B74E397FA3AF18A2F1B7C03BA91A6336158DE420D63188EE143866EE415735D155B7C2D854D795B7BC236CFFD71542DF34234221A0413E142D8C61355CC44D45BDA94204974557AC2704CD8B593F035A5724B1ADF442E78C542CD4414FCE6F1298182FB6D8E53CEF1ADFD2E90E1E4DEEC52999BDC6C29144E8D52A125232C8C6D75C706EA3CC06841C7BDA33568C63A6C03817F722B50FCF898237D788A4400869E44D90A3020923DC646388ABCC914315215FCD1BAE11B1C751FD52443AAC8F601087D8D42737C18A3FA11ECD4131ECAE017AE0A14ACFC4EF85B83C19FED33CFD1CD629DA2C4C09E222B398E18D822F77BB378DEA3CB360B605E5AA58B20EDC29D000A66BD177C682A17E7EB12A63EF7C2E4183E0D898F3D6BF567BA8AE84F84F1D23BF8B8E261C3729E2FA6D07B832E07CDDD1D14F55325C6F924267957121902DC19B3B32948BDEAD5',
			e => '0x010001',
			d => '0x0D43242AEFE1FB2C13FBC66E20B678C4336D20B1808C558B6E62AD16A287077180B177E1F01B12F9C6CD6C52630257CCEF26A45135A990928773F3BD2FC01A313F1DAC97A51CEC71CB1FD7EFC7ADFFDEB05F1FB04812C924ED7F4A8269925DAD88BD7DCFBC4EF01020EBFC60CB3E04C54F981FDBD273E69A8A58B8CEB7C2D83FBCBD6F784D052201B88A9848186F2A45C0D2826870733E6FD9AA46983E0A6E82E35CA20A439C5EE7B502A9062E1066493BDADF8B49EB30D9558ED85ABC7AFB29B3C9BC644199654A4676681AF4BABCEA4E6F71FE4565C9C1B85D9985B84EC1ABF1A820A9BBEBEE0DF1398AAE2C85AB580A9F13E7743AFD3108EB32100B870648FA6BC17E8ABAC4D3C99246B1F0EA9F7F93A5DD5458C56D9F3F81FF2216B3C3680A13591673C43194D8E6FC93FC1E37CE2986BD628AC48088BC723D8FBE293861CA7A9F4A73E9FA63B1B6D0074F5DEA2A624C5249FF3AD811B6255B299D6BC5451BA7477F19C5A0DB690C3E6476398B1483D10314AFD38BBAF6E2FBDBCD62C3CA9797A420CA6034EC0A83360A3EE2ADF4B9D4BA29731D131B099A38D6A23CC463DB754603211260E99D19AFFC902C915D7854554AABF608E3AC52C19B8AA26AE042249B17B2D29669B5C859103EE53EF9BDC73BA3C6B537D5C34B6D8F034671D7F3A8A6966CC4543DF223565343154140FD7391C7E7BE03E241F4ECFEB877A051',
			p => '0xE1F4D7A34802E27C7392A3CEA32A262A34DC3691BD87F3F310DC75673488930559C120FD0410194FB8A0DA55BD0B81227E843FDCA6692AE80E5A5D414116D4803FCA7D8C30EAAAE57E44A1816EBB5C5B0606C536246C7F11985D731684150B63C9A3AD9E41B04C0B5B27CB188A692C84696B742A80D3CD00AB891F2457443DADFEBA6D6DAF108602BE26D7071803C67105A5426838E6889D77E8474B29244CEFAF418E381B312048B457D73419213063C60EE7B0D81820165864FEF93523C9635C22210956E53A8D96322493FFC58D845368E2416E078E5BCB5D2FD68AE6ACFA54F9627C42E84A9D3F2774017E32EBCA06308A12ECC290C7CD1156DCCCFB2311',
			q => '0xC601A9CAEA66DC3835827B539DB9DF6F6F5AE77244692780CD334A006AB353C806426B60718C05245650821D39445D3AB591ED10A7339F15D83FE13F6A3DFB20B9452C6A9B42EAA62A68C970DF3CADB2139F804AD8223D56108DFDE30BA7D367E9B0A7A80C4FDBA2FD9DDE6661FC73FC2947569D2029F2870FC02D8325ACF28C9AFA19ECF962DAA7916E21AFAD09EB62FE9F1CF91B77DC879B7974B490D3EBD2E95426057F35D0A3C9F45F79AC727AB81A519A8B9285932D9B2E5CCD347E59F3F32AD9CA359115E7DA008AB7406707BD0E8E185A5ED8758B5BA266E8828F8D863AE133846304A2936AD7BC7C9803879D2FC4A28E69291D73DBD799F8BC238385',
			r => Math::Pari::_hex_cvt('0x80682C48982407B489D53D1261B19EC8627D02B8CDA5336750B8CEE332AE260DE57B02D72609C1E0E9F28E2040FC65B6F02D56DBD6AA9AF8FDE656F70495DFB723BA01173D4707A12FDDAC628CA29F3E32340BD8F7DDB557CF819F6B01E445AD96F874BA235584EE71F6581F62D4F43BF03F910F6510DEB85E8EF06C7F09D9794A008BE7FF2529F0EBB69DECEF646387DC767B74939265FEC0223AA6D84D2A8A1CC912D5CA25B4E144AB8F6BA054B54910176D5737A2CFF011DA431BD5F2A0D2D66B9E70B39F4B050E45C0D9C16F02DEDA9DDF2D00F3E4B01037D7029CD49C2D46A8E1FC2C0C17520AF1F4B5E25BA396AFC4CD60C494A4C426448B35B49635B337CFB08E7C22A39B256DD032C00ADDDAFB51A627F99A0E1704170AC1F1912E49D9DB10EC04C19C58F420212973E0CB329524223A6AA56C7937C5DFFDB5D966B6CD4CBC26F3201DD25C80960A1A111B32947BB78973D269FAC7F5186530930ED19F68507540EED9E1BAB8B00F00D8CA09B3F099AAE46180E04E3584BD7CA054DF18A1504B89D1D1675D0966C4AE1407BE325CDF623CF13FF13E4A28B594D59E3EADBADF6136EEE7A59D6A444C9EB4E2198E8A974F27A39EB63AF2C9AF3870488B8ADAAD444674F512133AD80B9220E09158521614F1FAADFE8505EF57B7DF6813048603F0DD04F4280177A11380FBFC861DBCBD7418D62155248DAD5FDEC0991F'),
			slen => 0,
			salt => '',
			randomize => 1
		      }
		    );
}
