##@file
# Zimbra preauthentication

##@class
# Zimbra preauthentication
#
# It will build Zimbra preauth URL

# This specific handler is intended to be called directly by Apache

package Lemonldap::NG::Handler::Lib::ZimbraPreAuth;

use strict;
use base qw(Lemonldap::NG::Handler::ApacheMP2::Main);
use Digest::HMAC_SHA1 qw(hmac_sha1 hmac_sha1_hex);

our $VERSION = '1.9.99_2.0alpha1';

# Overload main run method
sub run {
    my ( $class, $req ) = @_;
    my $ret = $class->SUPER::run($req);

    # Continue only if user is authorized
    return $ret unless ( $ret == $class->OK );

    # Get current URI
    my $uri = $class->uri_with_args($req);

    # Get Zimbra parameters
    my $localConfig      = $class->localConfig;
    my $zimbraPreAuthKey = $localConfig->{zimbraPreAuthKey};
    my $zimbraAccountKey = $localConfig->{zimbraAccountKey} || 'uid';
    my $zimbraBy         = $localConfig->{zimbraBy} || 'id';
    my $zimbraUrl        = $localConfig->{zimbraUrl} || '/service/preauth';
    my $zimbraSsoUrl     = $localConfig->{zimbraSsoUrl} || '^/zimbrasso$';
    my $timeout          = $localConfig->{'timeout'} || '0';

    # Display found values in debug mode
    $class->logger->debug("zimbraPreAuthKey: $zimbraPreAuthKey");
    $class->logger->debug("zimbraAccountKey: $zimbraAccountKey");
    $class->logger->debug("zimbraBy: $zimbraBy");
    $class->logger->debug("zimbraUrl: $zimbraUrl");
    $class->logger->debug("zimbraSsoUrl: $zimbraSsoUrl");
    $class->logger->debug("timeout: $timeout");

    # Return if we are not on a Zimbra SSO URI
    return $class->OK unless ( $uri =~ $zimbraSsoUrl );

    # Check mandatory parameters
    unless ($zimbraPreAuthKey) {
        $class->logger->error("No Zimbra preauth key configured");
        return $class->SERVER_ERROR;
    }

    # Build URL
    my $zimbra_url =
      $class->_buildZimbraPreAuthUrl( $zimbraPreAuthKey, $zimbraUrl,
        $class->datas->{$zimbraAccountKey},
        $zimbraBy, $timeout );

    # Header location
    Lemonldap::NG::Handler::API->set_header_out( 'Location' => $zimbra_url );

    # Return $class->REDIRECT
    return $class->REDIRECT;
}

## @method private string _buildZimbraPreAuthUrl(string key, string url, string account, string by, int timeout)
# Build Zimbra PreAuth URL
# @param key PreAuthKey
# @param url URL
# @param account User account
# @param by Account type
# @param timeout Timout
# @return Zimbra PreAuth URL
sub _buildZimbraPreAuthUrl {
    my ( $class, $key, $url, $account, $by, $timeout ) = @_;

    # Expiration time is calculated with _utime and timeout
    my $expires =
      $timeout ? ( $class->datas->{_utime} + $timeout ) * 1000 : $timeout;

    # Timestamp
    my $timestamp = time() * 1000;

    # Compute preauth value
    my $computed_value =
      hmac_sha1_hex( "$account|$by|$expires|$timestamp", $key );

    $class->logger->debug(
        "Compute value $account|$by|$expires|$timestamp into $computed_value");

    # Build PreAuth URL
    my $zimbra_url =
"$url?account=$account&by=$by&timestamp=$timestamp&expires=$expires&preauth=$computed_value";

    $class->logger->debug("Build Zimbra URL: $zimbra_url");

    return $zimbra_url;
}

1;
