package Module::Install::PPBuild;
use base 'Module::Install::Base';
use App::PPBuild::Makefile qw//;

#{{{ POD

=pod

=head1 NAME

Module::Install::PPBuild - A Module::Install extension for PPBuild based
projects.

=head1 DESCRIPTION

This extension is used to add rules to the Makefile generated by
Module::Install for PPBuild tasks. You can also map PPBuild tasks to MakeMaker
rules to override some of the autogenerated rules such as test or install. By
default all PPBuild tasks are prefixed with ppbuild_ in the Makefile rule
defenition. When a MakeMaker rule is overriden it will simply call the
ppbuild_task rule.

=head1 SYNOPSIS

Makefile.PL:

    use inc::Module::Install;

    ppb_prefix 'ppbuild_';

    # Load the PPBFile, map PPBuild tasks to MakeMaker rules. ( TASK => RULE );
    ppbfile 'PPBFile', 'my_install' => 'install', 'my_test' => 'test';

    # Like any Module::Install Makefile.PL.
    WriteAll;

=head1 HOW IT WORKS

Module::Install uses ExtUtils::MakeMaker to do most of the work. This module
makes use of the sub MY::[FUNCTION] {} system to override specified MakeMaker
rules replacing them with rules from App::PPBuild::Makefile. It also Adds all
the PPBuild tasks as makefile rules with the ppbuild_ prefix. This is done
using the MY::postamble() method.

=head1 EXPORTED FUNCTIONS

=over 4

=cut

#}}}

use Exporter 'import';
our @EXPORT = qw/ ppbfile ppb_prefix /;
our $PREFIX = 'ppbuild_';

=item ppb_prefix()

Set the prefix for PPBuild task-rules in the makefile.

=cut

sub ppb_prefix {
    $PREFIX = shift if @_;
    return $PREFIX;
}

=item ppbfile()

Used to specify the PPBFile to use, as well as which Tasks map to which MakeMaker rules.

    # Load the PPBFile, map PPBuild tasks to MakeMaker rules. ( TASK => RULE );
    ppbfile 'PPBFile', 'my_install' => 'install', 'my_test' => 'test';

    ppbfile(
        'File',
        SomeTask => MakeMaker_Rule,
        ...
    );

For a list of MakeMaker rules that can be overridden see the POD for
ExtUtils::MM_Unix. The obvious ones are install and test.

=cut

sub ppbfile {
    require ExtUtils::MakeMaker;
    my $object = shift;
    my $file = shift;
    my %overrides = @_;
    App::PPBuild::Makefile::ppbfile( $file );

    my $post = App::PPBuild::Makefile::helpers( $PREFIX );
    #Override any rules that have a corresponding task
    {
        for my $task ( App::PPBuild::Makefile::tasklist() ) {
            if ( my $rule = $overrides{ $task } ) {
                task2function( $task, $rule );
            }
            $post .= App::PPBuild::Makefile::rule( $task, $PREFIX );
        }
    }

    my $code = <<EOT;
        sub MY::postamble {
            return '$post';
        };
EOT

    eval $code;

}

=back

=head1 PRIVATE FUNCTIONS

=over 4

=item task2function()

Used internally to create the MY::rule() function that overrides MakeMaker rules.

=cut

sub task2function {
    my ( $task, $rule ) = @_;
    no strict 'refs';
    my $code = <<EOT;
    sub MY::$rule {
        return "$rule: ${PREFIX}${task}\n"
    }
EOT

    eval $code;
}

__END__

=back

=head1 AUTHOR

Chad Granum E<lt>exodist7@gmail.comE<gt>

=head1 COPYRIGHT

Copyright 2008 Chad Granum

licensed under the GPL version 3.
You should have received a copy of the GNU General Public License
along with this.  If not, see <http://www.gnu.org/licenses/>.

=cut

