#include "../lib/test.h"

#define TEST(name) TEST_CASE("parse-compression: " name, "[parse-compression]")

/* ACCEPT-ENCODING */
TEST("Accept-Encoding: gzip") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->headers.get("Accept-Encoding") == "gzip");
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

TEST("Accept-Encoding: gzip;q=1.0") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=1.0\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

TEST("Accept-Encoding: gzip;q=0.002") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=0.002\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

TEST("Accept-Encoding: gzip;q=0.0") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=0.0\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(!(req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

TEST("Accept-Encoding: gzip;q=0.000") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=0.000\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(!(req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

TEST("Accept-Encoding: (empty)") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding:\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() == Compression::IDENTITY));
}

TEST("Accept-Encoding: gzip, deflate, br, *;q=0") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip, deflate, br, *;q=0\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::DEFLATE));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}


TEST("Accept-Encoding: gzip;q=1.0, deflate;q=0") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=1.0, deflate;q=0\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK(!(req->allowed_compression() & Compression::DEFLATE));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}


TEST("Accept-Encoding: gzip;q=0.123, deflate;q=0.120") {
    RequestParser p;
    string raw = "GET / HTTP/1.1\r\n"
        "Accept-Encoding: gzip;q=0.123, deflate;q=0.120\r\n"
    "\r\n";
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK((req->allowed_compression() & Compression::GZIP));
    CHECK((req->allowed_compression() & Compression::DEFLATE));
    CHECK((req->allowed_compression() & Compression::IDENTITY));
}

/* CONTENT-ENCODING */

TEST("Content-Encoding: empty") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::IDENTITY);
}

TEST("Content-Encoding: identity") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: identity\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::IDENTITY);
}

TEST("Content-Encoding: gzip") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: gzip\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::GZIP);
}

TEST("Content-Encoding: deflate") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: deflate\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::DEFLATE);
}

TEST("Content-Encoding: rar") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: rar\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::error);
    CHECK(result.error == errc::unsupported_compression);
}

TEST("Content-Encoding: rar (disabled uncompression)") {
    RequestParser p;
    p.uncompress_content = false;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: rar\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::IDENTITY);
}

TEST("Content-Encoding: gzip, identity") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: gzip, identity\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(req->compression.type == Compression::GZIP);
}

TEST("Content-Encoding: gzip, gzip") {
    RequestParser p;
    string raw =
        "POST /upload HTTP/1.1\r\n"
        "Content-Encoding: gzip, gzip\r\n"
        "\r\n"
        ;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::error);
    CHECK(result.error == errc::unsupported_compression);
}

/* compression gzip */
unsigned char gzip_sample[] = {
    0x50, 0x4f, 0x53, 0x54, 0x20, 0x2f, 0x20, 0x48,
    0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31, 0x0d,
    0x0a, 0x48, 0x6f, 0x73, 0x74, 0x3a, 0x20, 0x31,
    0x32, 0x37, 0x2e, 0x30, 0x2e, 0x30, 0x2e, 0x31,
    0x0d, 0x0a, 0x55, 0x73, 0x65, 0x72, 0x2d, 0x41,
    0x67, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x63, 0x75,
    0x72, 0x6c, 0x2f, 0x37, 0x2e, 0x36, 0x37, 0x2e,
    0x30, 0x0d, 0x0a, 0x41, 0x63, 0x63, 0x65, 0x70,
    0x74, 0x3a, 0x20, 0x2a, 0x2f, 0x2a, 0x0d, 0x0a,
    0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
    0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67,
    0x3a, 0x20, 0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a,
    0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
    0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x3a, 0x20,
    0x33, 0x38, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
    0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
    0x3a, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63,
    0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x78, 0x2d,
    0x77, 0x77, 0x77, 0x2d, 0x66, 0x6f, 0x72, 0x6d,
    0x2d, 0x75, 0x72, 0x6c, 0x65, 0x6e, 0x63, 0x6f,
    0x64, 0x65, 0x64, 0x0d, 0x0a, 0x0d, 0x0a, 0x1f,
    0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x03, 0xf3, 0xc9, 0x2f, 0x4a, 0xcd, 0x55, 0xc8,
    0x2c, 0x28, 0x2e, 0xcd, 0x55, 0x48, 0xc9, 0xcf,
    0xc9, 0x2f, 0xe2, 0x02, 0x00, 0xe8, 0xd0, 0x4e,
    0xfa, 0x12, 0x00, 0x00, 0x00
};

TEST("request with gzip payload (small piece)") {

/*
wireshark captured piece. Generated via

echo 'Lorem ipsum dolor' | gzip | curl -v -i --data-binary @- -H "Content-Encoding: gzip" http://127.0.0.1/

POST / HTTP/1.1
Host: 127.0.0.1
User-Agent: curl/7.67.0
Content-Encoding: gzip
Content-Length: 38
Content-Type: application/x-www-form-urlencoded

............/J.U.,(..UH.../.....N.....
*/
    string raw(reinterpret_cast<char*>(gzip_sample), sizeof (gzip_sample));
    RequestParser p;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(result.error.value() == 0);
    CHECK(req->compression.type == Compression::GZIP);
    REQUIRE(req->body.parts.size() == 1);
    CHECK(req->body.parts[0] == "Lorem ipsum dolor\n");

    SECTION("check reset") {
        // p.reset() is implicitly called
        auto req = result.request;
        CHECK(result.state == State::done);
        CHECK(result.error.value() == 0);
        CHECK(req->compression.type == Compression::GZIP);
        REQUIRE(req->body.parts.size() == 1);
        CHECK(req->body.parts[0] == "Lorem ipsum dolor\n");
    }
}

TEST("request with corrupted gzip payload") {
    // the same sample as above, but changed byte in body
    unsigned char data[] = {
        0x50, 0x4f, 0x53, 0x54, 0x20, 0x2f, 0x20, 0x48,
        0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31, 0x0d,
        0x0a, 0x48, 0x6f, 0x73, 0x74, 0x3a, 0x20, 0x31,
        0x32, 0x37, 0x2e, 0x30, 0x2e, 0x30, 0x2e, 0x31,
        0x0d, 0x0a, 0x55, 0x73, 0x65, 0x72, 0x2d, 0x41,
        0x67, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x63, 0x75,
        0x72, 0x6c, 0x2f, 0x37, 0x2e, 0x36, 0x37, 0x2e,
        0x30, 0x0d, 0x0a, 0x41, 0x63, 0x63, 0x65, 0x70,
        0x74, 0x3a, 0x20, 0x2a, 0x2f, 0x2a, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67,
        0x3a, 0x20, 0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x3a, 0x20,
        0x33, 0x38, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63,
        0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x78, 0x2d,
        0x77, 0x77, 0x77, 0x2d, 0x66, 0x6f, 0x72, 0x6d,
        0x2d, 0x75, 0x72, 0x6c, 0x65, 0x6e, 0x63, 0x6f,
        0x64, 0x65, 0x64, 0x0d, 0x0a, 0x0d, 0x0a, 0x1f,
        0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0xf3, 0xc9, 0x2f, 0x4a, 0xcd, 0x55, 0xc8,
        0x2c, 0x28, 0x2e, 0xcd, 0x55, 0x48, 0xc9, 0xcf,
        0xc9, 0x2f, 0xe2, 0x02, 0x00, 0xe8, 0xd0, 0x4e,
        0xfa, 0x99, 0x00, 0x00, 0x00 };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    RequestParser p;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::error);
    CHECK(req->compression.type == Compression::GZIP);
    CHECK(result.error == errc::uncompression_failure);
}

TEST("request with redundant gzip payload") {
    // the same sample as above + extra byte on
    unsigned char data[] = {
        0x50, 0x4f, 0x53, 0x54, 0x20, 0x2f, 0x20, 0x48,
        0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31, 0x0d,
        0x0a, 0x48, 0x6f, 0x73, 0x74, 0x3a, 0x20, 0x31,
        0x32, 0x37, 0x2e, 0x30, 0x2e, 0x30, 0x2e, 0x31,
        0x0d, 0x0a, 0x55, 0x73, 0x65, 0x72, 0x2d, 0x41,
        0x67, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x63, 0x75,
        0x72, 0x6c, 0x2f, 0x37, 0x2e, 0x36, 0x37, 0x2e,
        0x30, 0x0d, 0x0a, 0x41, 0x63, 0x63, 0x65, 0x70,
        0x74, 0x3a, 0x20, 0x2a, 0x2f, 0x2a, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67,
        0x3a, 0x20, 0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x3a, 0x20,
        0x33, 0x39, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63,
        0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x78, 0x2d,
        0x77, 0x77, 0x77, 0x2d, 0x66, 0x6f, 0x72, 0x6d,
        0x2d, 0x75, 0x72, 0x6c, 0x65, 0x6e, 0x63, 0x6f,
        0x64, 0x65, 0x64, 0x0d, 0x0a, 0x0d, 0x0a, 0x1f,
        0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0xf3, 0xc9, 0x2f, 0x4a, 0xcd, 0x55, 0xc8,
        0x2c, 0x28, 0x2e, 0xcd, 0x55, 0x48, 0xc9, 0xcf,
        0xc9, 0x2f, 0xe2, 0x02, 0x00, 0xe8, 0xd0, 0x4e,
        0xfa, 0x12, 0x00, 0x00, 0x00, 0x00 };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    RequestParser p;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::error);
    CHECK(req->compression.type == Compression::GZIP);
    CHECK(result.error == errc::uncompression_failure);
}

TEST("request with gzip payload (byte-by-byte consumption)") {
    string raw(reinterpret_cast<char*>(gzip_sample), sizeof (gzip_sample));
    RequestParser p;
    RequestParser::Result result;
    for(size_t i = 1; i <= raw.size(); ++i){
        auto piece = raw.substr(i - 1, 1);
        result = p.parse(piece);
        if ((result.state == State::done) || (result.state == State::error)) { break; }
    }
    auto req = result.request;
    CHECK(result.state == State::done);
    CHECK(result.error.value() == 0);
    CHECK(req->compression.type == Compression::GZIP);

    string joined;
    for(auto& it: req->body.parts) { joined += it; }
    CHECK(joined == "Lorem ipsum dolor\n");
}

TEST("request with gzip payload, no content-length") {
    unsigned char data[] = {
        0x48, 0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31,
        0x20, 0x32, 0x30, 0x30, 0x20, 0x4f, 0x4b, 0x0d,
        0x0a,
        0x48, 0x6f, 0x73, 0x74, 0x3a, 0x20, 0x31,
        0x32, 0x37, 0x2e, 0x30, 0x2e, 0x30, 0x2e, 0x31,
        0x0d, 0x0a, 0x55, 0x73, 0x65, 0x72, 0x2d, 0x41,
        0x67, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x63, 0x75,
        0x72, 0x6c, 0x2f, 0x37, 0x2e, 0x36, 0x37, 0x2e,
        0x30, 0x0d, 0x0a, 0x41, 0x63, 0x63, 0x65, 0x70,
        0x74, 0x3a, 0x20, 0x2a, 0x2f, 0x2a, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67,
        0x3a, 0x20, 0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63,
        0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x78, 0x2d,
        0x77, 0x77, 0x77, 0x2d, 0x66, 0x6f, 0x72, 0x6d,
        0x2d, 0x75, 0x72, 0x6c, 0x65, 0x6e, 0x63, 0x6f,
        0x64, 0x65, 0x64, 0x0d, 0x0a, 0x0d, 0x0a, 0x1f,
        0x8b, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x03, 0xf3, 0xc9, 0x2f, 0x4a, 0xcd, 0x55, 0xc8,
        0x2c, 0x28, 0x2e, 0xcd, 0x55, 0x48, 0xc9, 0xcf,
        0xc9, 0x2f, 0xe2, 0x02, 0x00, 0xe8, 0xd0, 0x4e,
        0xfa, 0x12, 0x00, 0x00, 0x00 };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    ResponseParser p;
    p.set_context_request(new Request(Method::GET, new URI("/")));
    auto result = p.parse(raw);
    auto res = result.response;
    CHECK(result.state == State::body);
    CHECK(res->compression.type == Compression::GZIP);
    REQUIRE(res->body.parts.size() == 1);
    CHECK(res->body.parts[0] == "Lorem ipsum dolor\n");
    CHECK(p.eof().state == State::done);
}

TEST("request with gzip payload (max body size)") {
    unsigned char data[] = {
        0x50, 0x4f, 0x53, 0x54, 0x20, 0x2f, 0x20, 0x48,
        0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31, 0x0d,
        0x0a, 0x48, 0x6f, 0x73, 0x74, 0x3a, 0x20, 0x31,
        0x32, 0x37, 0x2e, 0x30, 0x2e, 0x30, 0x2e, 0x31,
        0x0d, 0x0a, 0x55, 0x73, 0x65, 0x72, 0x2d, 0x41,
        0x67, 0x65, 0x6e, 0x74, 0x3a, 0x20, 0x63, 0x75,
        0x72, 0x6c, 0x2f, 0x37, 0x2e, 0x36, 0x37, 0x2e,
        0x30, 0x0d, 0x0a, 0x41, 0x63, 0x63, 0x65, 0x70,
        0x74, 0x3a, 0x20, 0x2a, 0x2f, 0x2a, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x45, 0x6e, 0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67,
        0x3a, 0x20, 0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d,
        0x4c, 0x65, 0x6e, 0x67, 0x74, 0x68, 0x3a, 0x20,
        0x32, 0x35, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x61, 0x70, 0x70, 0x6c, 0x69, 0x63,
        0x61, 0x74, 0x69, 0x6f, 0x6e, 0x2f, 0x78, 0x2d,
        0x77, 0x77, 0x77, 0x2d, 0x66, 0x6f, 0x72, 0x6d,
        0x2d, 0x75, 0x72, 0x6c, 0x65, 0x6e, 0x63, 0x6f,
        0x64, 0x65, 0x64, 0x0d, 0x0a, 0x0d, 0x0a, 0x1f,
        0x8b, 0x08, 0x00, 0xa3, 0x9d, 0xf0, 0x5d, 0x02,
        0x03, 0x73, 0x74, 0xa4, 0x02, 0xe0, 0x02, 0x00,
        0xef, 0xa5, 0xc2, 0x7b, 0x46, 0x00, 0x00, 0x00 };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    RequestParser p;
    p.max_body_size = 30;
    auto result = p.parse(raw);
    auto req = result.request;
    CHECK(result.state == State::error);
    CHECK(req->compression.type == Compression::GZIP);
    CHECK(result.error == errc::body_too_large);
}

TEST("response with gzipped chunked response") {
    unsigned char data[] = {
        0x48, 0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31,
        0x20, 0x32, 0x30, 0x30, 0x20, 0x4f, 0x4b, 0x0d,
        0x0a, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x3a,
        0x20, 0x6e, 0x67, 0x69, 0x6e, 0x78, 0x2f, 0x31,
        0x2e, 0x31, 0x36, 0x2e, 0x31, 0x0d, 0x0a, 0x44,
        0x61, 0x74, 0x65, 0x3a, 0x20, 0x57, 0x65, 0x64,
        0x2c, 0x20, 0x31, 0x31, 0x20, 0x44, 0x65, 0x63,
        0x20, 0x32, 0x30, 0x31, 0x39, 0x20, 0x31, 0x30,
        0x3a, 0x30, 0x39, 0x3a, 0x35, 0x36, 0x20, 0x47,
        0x4d, 0x54, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x74, 0x65, 0x78, 0x74, 0x2f, 0x70,
        0x6c, 0x61, 0x69, 0x6e, 0x0d, 0x0a, 0x4c, 0x61,
        0x73, 0x74, 0x2d, 0x4d, 0x6f, 0x64, 0x69, 0x66,
        0x69, 0x65, 0x64, 0x3a, 0x20, 0x57, 0x65, 0x64,
        0x2c, 0x20, 0x31, 0x31, 0x20, 0x44, 0x65, 0x63,
        0x20, 0x32, 0x30, 0x31, 0x39, 0x20, 0x31, 0x30,
        0x3a, 0x30, 0x39, 0x3a, 0x31, 0x38, 0x20, 0x47,
        0x4d, 0x54, 0x0d, 0x0a, 0x54, 0x72, 0x61, 0x6e,
        0x73, 0x66, 0x65, 0x72, 0x2d, 0x45, 0x6e, 0x63,
        0x6f, 0x64, 0x69, 0x6e, 0x67, 0x3a, 0x20, 0x63,
        0x68, 0x75, 0x6e, 0x6b, 0x65, 0x64, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69,
        0x6f, 0x6e, 0x3a, 0x20, 0x6b, 0x65, 0x65, 0x70,
        0x2d, 0x61, 0x6c, 0x69, 0x76, 0x65, 0x0d, 0x0a,
        0x56, 0x61, 0x72, 0x79, 0x3a, 0x20, 0x41, 0x63,
        0x63, 0x65, 0x70, 0x74, 0x2d, 0x45, 0x6e, 0x63,
        0x6f, 0x64, 0x69, 0x6e, 0x67, 0x0d, 0x0a, 0x45,
        0x54, 0x61, 0x67, 0x3a, 0x20, 0x57, 0x2f, 0x22,
        0x35, 0x64, 0x66, 0x30, 0x63, 0x30, 0x34, 0x65,
        0x2d, 0x34, 0x35, 0x22, 0x0d, 0x0a, 0x43, 0x6f,
        0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d, 0x45, 0x6e,
        0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67, 0x3a, 0x20,
        0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a, 0x0d, 0x0a,
        0x31, 0x38, 0x0d, 0x0a, 0x1f, 0x8b, 0x08, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x73, 0x74,
        0xa4, 0x02, 0x00, 0x00, 0xe6, 0x31, 0xe4, 0x7c,
        0x45, 0x00, 0x00, 0x00, 0x0d, 0x0a, 0x30, 0x0d,
        0x0a, 0x0d, 0x0a };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    ResponseParser p;
    p.set_context_request(new Request(Method::GET, new URI("/")));
    auto result = p.parse(raw);
    auto res = result.response;
    CHECK(result.state == State::done);
    CHECK(res->compression.type == Compression::GZIP);
    REQUIRE(res->body.parts.size() == 1);
    CHECK(res->body.to_string() == "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA");
}

TEST("response with corrupted gzipped chunked response") {
    unsigned char data[] = {
        0x48, 0x54, 0x54, 0x50, 0x2f, 0x31, 0x2e, 0x31,
        0x20, 0x32, 0x30, 0x30, 0x20, 0x4f, 0x4b, 0x0d,
        0x0a, 0x53, 0x65, 0x72, 0x76, 0x65, 0x72, 0x3a,
        0x20, 0x6e, 0x67, 0x69, 0x6e, 0x78, 0x2f, 0x31,
        0x2e, 0x31, 0x36, 0x2e, 0x31, 0x0d, 0x0a, 0x44,
        0x61, 0x74, 0x65, 0x3a, 0x20, 0x57, 0x65, 0x64,
        0x2c, 0x20, 0x31, 0x31, 0x20, 0x44, 0x65, 0x63,
        0x20, 0x32, 0x30, 0x31, 0x39, 0x20, 0x31, 0x30,
        0x3a, 0x30, 0x39, 0x3a, 0x35, 0x36, 0x20, 0x47,
        0x4d, 0x54, 0x0d, 0x0a, 0x43, 0x6f, 0x6e, 0x74,
        0x65, 0x6e, 0x74, 0x2d, 0x54, 0x79, 0x70, 0x65,
        0x3a, 0x20, 0x74, 0x65, 0x78, 0x74, 0x2f, 0x70,
        0x6c, 0x61, 0x69, 0x6e, 0x0d, 0x0a, 0x4c, 0x61,
        0x73, 0x74, 0x2d, 0x4d, 0x6f, 0x64, 0x69, 0x66,
        0x69, 0x65, 0x64, 0x3a, 0x20, 0x57, 0x65, 0x64,
        0x2c, 0x20, 0x31, 0x31, 0x20, 0x44, 0x65, 0x63,
        0x20, 0x32, 0x30, 0x31, 0x39, 0x20, 0x31, 0x30,
        0x3a, 0x30, 0x39, 0x3a, 0x31, 0x38, 0x20, 0x47,
        0x4d, 0x54, 0x0d, 0x0a, 0x54, 0x72, 0x61, 0x6e,
        0x73, 0x66, 0x65, 0x72, 0x2d, 0x45, 0x6e, 0x63,
        0x6f, 0x64, 0x69, 0x6e, 0x67, 0x3a, 0x20, 0x63,
        0x68, 0x75, 0x6e, 0x6b, 0x65, 0x64, 0x0d, 0x0a,
        0x43, 0x6f, 0x6e, 0x6e, 0x65, 0x63, 0x74, 0x69,
        0x6f, 0x6e, 0x3a, 0x20, 0x6b, 0x65, 0x65, 0x70,
        0x2d, 0x61, 0x6c, 0x69, 0x76, 0x65, 0x0d, 0x0a,
        0x56, 0x61, 0x72, 0x79, 0x3a, 0x20, 0x41, 0x63,
        0x63, 0x65, 0x70, 0x74, 0x2d, 0x45, 0x6e, 0x63,
        0x6f, 0x64, 0x69, 0x6e, 0x67, 0x0d, 0x0a, 0x45,
        0x54, 0x61, 0x67, 0x3a, 0x20, 0x57, 0x2f, 0x22,
        0x35, 0x64, 0x66, 0x30, 0x63, 0x30, 0x34, 0x65,
        0x2d, 0x34, 0x35, 0x22, 0x0d, 0x0a, 0x43, 0x6f,
        0x6e, 0x74, 0x65, 0x6e, 0x74, 0x2d, 0x45, 0x6e,
        0x63, 0x6f, 0x64, 0x69, 0x6e, 0x67, 0x3a, 0x20,
        0x67, 0x7a, 0x69, 0x70, 0x0d, 0x0a, 0x0d, 0x0a,
        0x31, 0x38, 0x0d, 0x0a, 0x1f, 0x8b, 0x08, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x73, 0x74,
        0xa4, 0x02, 0x00, 0x00, 0xe6, 0x31, 0xe4, 0x9c,
        0x45, 0x00, 0x00, 0x00, 0x0d, 0x0a, 0x30, 0x0d,
        0x0a, 0x0d, 0x0a };

    string raw(reinterpret_cast<char*>(data), sizeof (data));
    ResponseParser p;
    p.set_context_request(new Request(Method::GET, new URI("/")));
    auto result = p.parse(raw);
    auto res = result.response;
    CHECK(result.state == State::error);
    CHECK(res->compression.type == Compression::GZIP);
    CHECK(result.error == errc::uncompression_failure);
}
