# NAME

CXC::Exporter::Util - Tagged Based Exporting

# VERSION

version 0.01

# SYNOPSIS

In the exporting code:

    package My::Exporter;
    use CXC::Exporter::Util ':all';

    use parent 'Exporter' # or Exporter::Tiny

    # install sets of constants, with automatically generated
    # enumerating functions
    install_CONSTANTS( {
          DETECTORS => {
              ACIS => 'ACIS',
              HRC  => 'HRC',
          },

          AGGREGATES => {
              ALL  => 'all',
              NONE => 'none',
              ANY  => 'any',
          },
      } );

    # install some functions
    install_EXPORTS(
              { fruit => [ 'tomato', 'apple' ],
                nut   => [ 'almond', 'walnut' ],
              } );

In importing code:

    # import all of the fruit functions, all of the DETECTORS constants,
    # as well as a function enumerating the DETECTORS constants
    use My::Exporter ':fruit', ':detector', 'DETECTORS';

    # print the DETECTORS constants' values;
    say $_ for DETECTORS;

# DESCRIPTION

`CXC::Exporter::Util` provides _tag-centric_ utilities for modules
which export symbols.  It doesn't provide exporting services; its sole
purpose is to manipulate the data structures used by exporting modules
which follow the API provided by Perl's core [Exporter](https://metacpan.org/pod/Exporter) module
(e.g. [Exporter::Tiny](https://metacpan.org/pod/Exporter%3A%3ATiny)).

In particular, it treats `%EXPORT_TAGS` as the definitive source for
information about exportable symbols and uses it to generate
`@EXPORT_OK` and `@EXPORT`.  Consolidation of symbol information in
one place avoids errors of omission.

## Exporting Symbols

At it simplest, the exporting module calls ["install\_EXPORTS"](#install_exports) with a
hash specifying tags and their symbols sets, e.g.,

    package My::Exporter;
    use CXC::Exporter::Util;

    use parent 'Exporter'; # or your favorite compatible exporter

    install_EXPORTS(
              { fruit => [ 'tomato', 'apple' ],
                nut   => [ 'almond', 'walnut' ],
              } );

    sub tomato {...}
    sub apple  {...}
    sub almond {...}
    sub walnut {...}

An importing module could use this via

    use My::ExportingModule ':fruit'; # import tomato, apple
    use My::ExportingModule ':nut';   # import almond, walnut
    use My::ExportingModule ':all';   # import tomato, apple,
                                      #        almond, walnut,

For more complicated setups, `%EXPORT_TAGS` may be specified first:

    package My::ExportingModule;
    use CXC::Exporter::Util;

    use parent 'Exporter';
    our %EXPORT_TAGS = ( tag => [ 'Symbol1', 'Symbol2' ] );
    install_EXPORTS;

`install_EXPORTS` may be called multiple times

## Exporting Constants

[CXC::Exporter::Util](https://metacpan.org/pod/CXC%3A%3AExporter%3A%3AUtil) provides additional support for creating,
organizing and installing constants via ["install\_CONSTANTS"](#install_constants).
Constants are created via Perl's [constant](https://metacpan.org/pod/constant) pragma.

["install\_CONSTANTS"](#install_constants) is passed one or more hashes containing sets of
constants grouped by tags, e.g.:

    install_CONSTANTS( {
          DETECTORS => {
              ACIS => 'ACIS',
              HRC  => 'HRC',
          },

          AGGREGATES => {
              ALL  => 'all',
              NONE => 'none',
              ANY  => 'any',
          },
     });

     # A call to install_EXPORTS (with or without arguments) must follow
     # install_CONSTANTS;
     install_EXPORTS;

For each set an enumerating function is created which returns the set's values.

In the above example, constant functions `ACIS`, `HRC`, `ALL`,
`NONE`, `ANY`, tags `detectors` and `aggregates`, and constant
functions `DETECTORS` and `AGGREGATES` are created.

The enumerating functions are useful for generating enumerated types
via e.g. [Type::Tiny](https://metacpan.org/pod/Type%3A%3ATiny):

    Enum[ DETECTORS ]

or iterating:

    say $_ for DETECTORS;

If the constants are used later in the module for other purposes, constant definition
should be done in a [BEGIN](https://metacpan.org/pod/BEGIN) block:

    our %CONSTANTS;
    BEGIN { %CONSTANTS = (...);
            install_CONSTANTS( \%CONSTANTS );
    }

`install_CONSTANTS` may be called multiple times.

For more complex situations, the lower level ["install\_constant\_tag"](#install_constant_tag)
and ["install\_constant\_func"](#install_constant_func) routines may be useful.

# SUBROUTINES

## install\_EXPORTS

    install_EXPORTS( [\%export_tags], [$package], [\%options]  );

Populate `$package`'s `@EXPORT` and `@EXPORT_OK` arrays based upon
`%export_tags`.

If not specified, `%export_tags` defaults to `%EXPORT_TAGS` in `$package`.

If not specified,  `$package` defaults to the caller's package.

Available Options:

- overwrite => \[Boolean\]

    If the `overwrite` option is true, the contents of
    `%export_tags` will overwrite `%EXPORT_TAGS`, otherwise they are merged.
    Overwriting will wipe out tags installed by ["install\_CONSTANTS"](#install_constants).

    This defaults to false.

- package => \[Package Name\]

    This provides another means of indicating which package to install into.
    Setting this overrides the optional `$package` argument.

- all => \[Boolean | 'auto' \]

    This determines whether ["install\_EXPORTS"](#install_exports) creates an `all` tag
    based on the contents of `%EXPORT_TAGS`.  Some exporters, such as
    [Exporter::Tiny](https://metacpan.org/pod/Exporter%3A%3ATiny) and [Sub::Exporter](https://metacpan.org/pod/Sub%3A%3AExporter) automatically handle the `all`
    tag, but Perl's default [Exporter](https://metacpan.org/pod/Exporter) does not.

    If set to `auto` (the default), it will install the `all` tag if
    `$package` is _not_ a subclass of [Exporter::Tiny](https://metacpan.org/pod/Exporter%3A%3ATiny).

    (At present I don't know how to determine if [Sub::Exporter](https://metacpan.org/pod/Sub%3A%3AExporter) is used).

This routine does the following in `$package`, which defaults to the
caller's package.

- Install the symbols specified via the `default` tag into `$package`'s `@EXPORT`.
- Install all of the symbols in `%EXPORT_TAGS` into `@EXPORT_OK`.

## install\_CONSTANTS

    install_CONSTANTS( $spec0, [$spec1, ...,  \$specN ], ?$package  );

Create sets of constants and make them available for export in
`$package`.

If not specified,  `$package` defaults to the caller's package.

The passed `$specN` arguments are either hashrefs or arrayrefs and
contain one or more constant set specifications.

A constant set specification consists of a unique identifier and a
list of name-value pairs, specified either has a hash or an array.
The identifier is used to create a export tag for the set as well as a
enumerating function which returns the set's values.

If the set's name-value pairs are specified via a hash,
the enumerating function will return values in a random order; if
specified as an array the values will be returned n the specified
order.

A call of &lt;install\_EXPORTS> _must_ be made after calls to
`install_CONSTANTS` or they won't be added to the exports.

For example:

    package My::Exporter;

    use parent 'Exporter::Tiny';

    install_CONSTANTS(
       { ANIMALS        => { HORSE => 'horse', GOAT   => 'goat' },
         VEHICLES       => { CAR   => 'car',   PICKUP => 'pickup' },
         WATERBODIES    => [ OCEAN => 'ocean', SEA    => 'sea',
                             LAKE  => 'lake',  POND   => 'pond' ],
       }
    );

    install_EXPORTS;

will creates three sets of constants:

    Tag         Enumerating_Function Constant_Functions
    ---         -------------------- ------------------
    animals     ANIMALS              HORSE GOAT
    vehicles    VEHICLES             CAR PICKUP
    waterbodies WATERBODIES          OCEAN SEA LAKE POND

as well as a tag `constant_funcs` and its associated constant
enumerating functions `ANIMALS`, `VEHICLES`, and `WATERBODIES`.

Because the water bodies set was specified as an array, the
`WATERBODIES` function will always return the set's values in the
specified order of

    ocean sea lake pond

In use,

    use My::Expoter -animals, 'ANIMALS';

will import the `HORSE`, `GOAT`, and `ANIMALS` constant functions.

By default a set's tag name is set to the lower-case of its identifier
and its enumerating function name is set to the identifier.  It is
possible to specify them separately by passing the collection of sets
as an array and specifying the _tag_-_function_ pair in another
array. For example,

    install_CONSTANTS(
       [  ['Critters', 'Animals' ]  => { HORSE => 'horse', GOAT   => 'goat' },
          ...
       ]
    );

will create the tag `Critters` and the enumerating function
`Animals` for the first set of constants.

[install\_EXPORTS](https://metacpan.org/pod/install_EXPORTS) uses ["install\_constant\_tag"](#install_constant_tag) to install the individual
tagged sets.

## install\_constant\_tag

    insall_constant_tag( $id, $constants, [$package] )
    insall_constant_tag( [ $tag, $func_name ], $constants, [$package] )

`$constants` may be either a hashref or an arrayref ( containing
_name-value_ pairs ) specifying constant names and their values,
e.g.

    ( NAME1 => VALUE1, NAME2 => VALUE2 )

This routine does the following in `$package`, which defaults to the
caller's package.

1. installs the constants in `$package`,
2. installs a constant function (by default with the same name as the
identifier, `$id`) which returns the values and adds it to the
`$EXPORT_TAGS{contant_funcs}` tag.  To specify an alternate name, use
the second form (above) for invoking `install_constant_tag`.
3. adds the constant names to `%EXPORT_TAGS` with a tag which is by
default generated by lower-casing `$id`. To specify an alternate
tag, use the second form (above) for invoking `install_constant_tag`.

For example, after

    install_constant_tag( 'AGGREGATES',
       { ALL => 'all', NONE => 'none', ANY => 'any' } );

1. The constants, `ALL`, `NONE`, `ANY` will be created.
2. The function `AGGREGATES` will return `all`, `none`, `any`.
3. A package importing from `$package` can import the constants
`ALL`, `NONE`, `ANY` via the `aggregate` tag:

        use Package ':aggregate';

4. A package importing from `$package` can import the `AGGREGATES`
constant function via the `constant_funcs` tag:

        use Package ':constant_funcs';

    or directly

        use Package 'AGGREGATES';

`install_constant_tag` uses ["install\_constant\_func"](#install_constant_func) to create and install
the constant functions which return the constant values.

## install\_constant\_func( $name, \\@values )

This routine does the following in `$package`, which defaults to the
caller's package.

1. Create a constant subroutine named `$name` which returns `@values`;
2. Adds `$name` to the `constant_funcs` tag in `%EXPORT_TAGS`.

For example, after calling

    install_constant_func( 'AGGREGATES', [ 'all', 'none', 'any' ]  );

1. The function `AGGREGATES` will return `all`, `none`, `any`.
2. A package importing from `$package` can import the `AGGREGATE`
constant function via the `constant_funcs` tag:

        use Package ':constant_funcs';

    or directly

        use Package 'AGGREGATES';

# EXAMPLES

- Alternate constant generation modules.

    To use an alternate constant generation function bypass
    [install\_CONSTANTS](https://metacpan.org/pod/install_CONSTANTS) and load things manually.

    For example,  using [enum](https://metacpan.org/pod/enum):

        package My::Exporter;

        use CXC::Exporter::Util ':all';

        our @DaysOfWeek;
        BEGIN{ @DaysOfWeek = qw( Sun Mon Tue Wed Thu Fri Sat ) }
        use enum @DaysOfWeek;
        use constant DaysOfWeek => map { &$_ } @DaysOfWeek;
        install_EXPORTS( { days_of_week => \@DaysOfWeek,
                           constant_funcs => [ 'DaysOfWeek' ],
                          });

    and then

        use My::Exporter -days_of_week;

        say Sun | Mon;

# SUPPORT

## Bugs

Please report any bugs or feature requests to bug-cxc-exporter-util@rt.cpan.org  or through the web interface at: https://rt.cpan.org/Public/Dist/Display.html?Name=CXC-Exporter-Util

## Source

Source is available at

    https://gitlab.com/djerius/cxc-exporter-util

and may be cloned from

    https://gitlab.com/djerius/cxc-exporter-util.git

# SEE ALSO

Please see those modules/websites for more information related to this module.

- [Exporter](https://metacpan.org/pod/Exporter)

# AUTHOR

Diab Jerius <djerius@cpan.org>

# COPYRIGHT AND LICENSE

This software is Copyright (c) 2022 by Smithsonian Astrophysical Observatory.

This is free software, licensed under:

    The GNU General Public License, Version 3, June 2007
