package Archive::Har::Page;

use warnings;
use strict;
use Carp();
use Archive::Har::Page::PageTimings();

=head1 NAME

Archive::Har::Page - Represents a single page inside the HTTP Archive

=head1 VERSION

Version 0.06

=cut

our $VERSION = '0.06';

=head1 SYNOPSIS

    use Archive::Har();

    my $http_archive_string = '"log": { "version": "1.1", .... ';
    my $har = Archive::Har->new();
    $har->string($http_archive_string);
    foreach my $page ($har->pages()) {
        print "DateTime: " . $page->startedDateTime() . "\n";
        print "Id: " . $page->id() . "\n";
        print "Title: ". $page->title() . "\n";
        my $timing = $page->pageTimings();
        print "Comment: " . $page->comment() . "\n";
    }

=head1 SUBROUTINES/METHODS

=cut

my (%_fields) = (
			startedDateTime => undef,
			id => undef,
			title => undef,
			pageTimings => undef,
			comment => undef,
		);

=head2 startedDateTime

returns the date and time stamp for the beginning of the page load (ISO 8601 format)

=head2 id

returns the unique identifier of a page within the Archive. This is referenced by the Archive::Har::Entry objects

=head2 title

returns the page title

=head2 pageTimings

returns the page timings object

=head2 comment

returns the comment about the Page

=cut

sub new {
	my ($class, $params) = @_;
	my $self = {};
	bless $self, $class;
	foreach my $key (sort { $a cmp $b } keys %_fields) {
		if ($key eq 'pageTimings') {
			$self->$key(Archive::Har::Page::PageTimings->new($params->{$key}));
		} else {
			my $key_class = ref $params->{$key};
			if ((defined $key_class) && ($key_class =~ /^JSON.*?Boolean$/smx)) {
				$self->$key($params->{$key} ? 1 : 0);
			} else {
				$self->$key($params->{$key});
			}
		}
	}
	foreach my $key (sort { $a cmp $b } keys %$params) {
		if ($key =~ /^_/) { # check for private fields
			my $key_class = ref $params->{$key};
			if ((defined $key_class) && ($key_class =~ /^JSON.*?Boolean$/smx)) {
				$self->$key($params->{$key} ? 1 : 0);
			} else {
				$self->$key($params->{$key});
			}
		}
	}
	return $self;
}

sub startedDateTime {
	my ($self, $new) = @_;
	my $old = $self->{startedDateTime};
	$old //= '0000-00-00T00:00:00.0+00:00';
	if (@_ > 1) {
		if (defined $new) {
			if ($new =~ /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d+(?:\+\d{2}:\d{2}|Z)$/smx) {
				$self->{startedDateTime} = $new;
			} else {
				Carp::croak("startedDateTime is not formatted correctly:$new");
			}
		} else {
			$self->{startedDateTime} = undef;
		}
	}
	return $old;
}

sub id {
	my ($self, $new) = @_;
	my $old = $self->{id};
	if (@_ > 1) {
		$self->{id} = $new;
	}
	return $old;
}

sub title {
	my ($self, $new) = @_;
	my $old = $self->{title};
	$old //= 'unknown';
	if (@_ > 1) {
		$self->{title} = $new;
	}
	return $old;
}

sub AUTOLOAD {
	my $self = shift;
	my $type = ref($self) or Carp::croak "$self is not an object";

	my $name = $Archive::Har::Page::AUTOLOAD;
	$name =~ s/.*://;   # strip fully-qualified portion

	if ($name =~ /^[a-zA-Z]+$/) {
		if (not exists $_fields{$name} ) {
		    Carp::croak "Can't access '$name' field in class $type";
		}
		if (@_) {
		    return $self->{$name} = shift;
		} else {
		    return $self->{$name};
		}
	} elsif ($name =~ /^_[a-zA-Z]+$/) { # private fields
		if (@_) {
		    return $self->{$name} = shift;
		} else {
	            return $self->{$name};
		}
	}
	return;
}

sub TO_JSON {
	my ($self) = @_;
	my $json = {};
	foreach my $key (sort { $a cmp $b } keys %$self) {
		next if (!defined $self->$key());
		$json->{$key} = $self->$key();
	}
	return $json;
}

sub DESTROY {
}

1;
