#!perl
# lib/Build/Hopen.pm: utility routines for hopen(1).  This file is also the
# source of the repo's README.md, which is autogenerated from this POD.

package Build::Hopen;
use Build::Hopen::Base;
use parent 'Exporter';

our $VERSION = '0.000003'; # TRIAL

our @EXPORT = qw(boolify hnew hlog);
# Docs {{{1

=head1 NAME

Build::Hopen - A build generator with first-class edges and explicit dependencies

=head1 SYNOPSIS

Input is the last-sorting file in C<.> matching C<*.hopen>, unless you
specify otherwise.  That way you can call your build file C<.hopen> if
you want it hidden, or C<z.hopen> if you want it to sort below all your other
files.  Sort order is Lua's C<<>, which is by byte value.

Output is a build file for a build system (Ninja or Make will
be first).  You will eventually be able to pick a generator, a la CMake.
The invoker will put the selected generator's path
first in C<@INC>, but other than that it's all straight Perl.

=head1 INSTALLATION

Easiest: install C<cpanminus> if you don't have it - see
L<https://metacpan.org/pod/App::cpanminus#INSTALLATION>.  Then run
C<cpanm Build::Hopen>.

Manually: clone or untar into a working directory.  Then, in that directory,

    perl Makefile.PL
    make
    make test

... and if all the tests pass,

    make install

If some of the tests fail, please check the issues and file a new one if
no one else has reported the problem yet.

=head1 VARIABLES

=head2 $VERBOSE

Set to truthy to get debug output on stderr from hopen's internals.

=cut

# }}}1

our $VERBOSE = false;

=head1 FUNCTIONS

=head2 boolify

Convert a scalar to a Boolean as Perl, except:

=over

=item * Falsy

C</^(false|off|no)$/i>

=item * Truthy

C<"0">

=back

So C<false>, C<off>, C<no>, empty string, C<undef>, and numeric C<0> are falsy,
and all other values (including string C<'0'>) are truthy.

=cut

sub boolify {
    return false if $_[0] =~ /^(false|off|no)$/i;
    return true if $_[0] =~ /^0$/;
    return !!$_[0];
} #boolify()

=head2 hnew

Creates a new Build::Hopen instance.  For example:

    hnew DAG => 'foo';

is the same as

    Build::Hopen::G::DAG->new( name => 'foo' );

If the provided name does not include a double-colon, it is first tried after
C<Build::Hopen::G::>.  It is then tried in C<Build::Hopen::> and as a
complete package name.  The first one that succeeds is used.

The first parameter must be a part of a class name, and the second parameter
must be the name of the new instance.  All other parameters are passed
unchanged to the relevant constructor.

=cut

sub hnew {
    my $class = shift or croak 'Need a class';
    my @stems = ('Build::Hopen::G::', 'Build::Hopen::', '');
    shift @stems if $class =~ /::/;
    foreach my $stem (@stems) {
        my $instance = eval {
            eval "require $stem$class";
            "$stem$class"->new('name', @_)
                # put 'name' in front of the name parameter.
        };
        return $instance if $instance;
    }
    die "Could not find class for $class";
} #hnew()

=head2 hlog

Log information if L</$VERBOSE> is set.  Usage:

    hlog { <list of things to log> };

The items in the list are joined by C<' '> on output, and a C<'\n'> is added.
Each line is prefixed with C<'# '> for the benefit of test runs.

=cut

sub hlog (&) {
    return unless $VERBOSE;
    my $crLog = shift or croak 'Nothing to log!';
    my @log = &$crLog();
    say STDERR (join(' ', @log)) =~ s/^/# /gmr;
} #hlog()

1; # End of Build::Hopen
__END__

# Rest of docs {{{1

=head1 INTERNALS

 - C<Op>: A class representing an operation
   - C<Op:run()> takes a table of inputs and returns a table of outputs.
   - C<Op:describe()> returns a table listing those inputs and outputs.


=head2 Implementation

After the C<hopen> file is processed, cycles are detected and reported as
errors.  *(TODO change this to support LaTeX multi-run files?)*  Then the DAG
is traversed, and each operation writes the necessary information to the
file being generated.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Build::Hopen
    perldoc hopen

You can also look for information at:

=over 4

=item * GitHub (report bugs here)

L<https://github.com/cxw42/hopen>

=item * MetaCPAN

L<https://metacpan.org/release/Build-Hopen>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Build-Hopen>

=item * CPAN Ratings

L<https://cpanratings.perl.org/d/Build-Hopen>

=back

=head1 INSPIRED BY

=over

=item *

L<Luke|https://github.com/gvvaughan/luke>

=item *

a bit of L<Ant|https://ant.apache.org/>

=item *

a tiny bit of L<Buck|https://buckbuild.com/concept/what_makes_buck_so_fast.html>

=item *

my own frustrations working with CMake.

=back

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2018 Christopher White, C<< <cxwembedded at gmail.com> >>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this program; if not, write to the Free
Software Foundation, Inc.,
51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA

=cut

# }}}1
# vi: set fdm=marker:
