package OpenInteract2::Brick::News;

use strict;
use base qw( OpenInteract2::Brick );
use OpenInteract2::Exception;

my %INLINED_SUBS = (
    'news-2.25.zip' => 'NEWS225ZIP',
);

sub get_name {
    return 'news';
}

sub get_resources {
    return (
        'news-2.25.zip' => [ 'pkg news-2.25.zip', 'no' ],
    );
}

sub load {
    my ( $self, $resource_name ) = @_;
    my $inline_sub_name = $INLINED_SUBS{ $resource_name };
    unless ( $inline_sub_name ) {
        OpenInteract2::Exception->throw(
            "Resource name '$resource_name' not found ",
            "in ", ref( $self ), "; cannot load content." );
    }
    return $self->$inline_sub_name();
}

OpenInteract2::Brick->register_factory_type( get_name() => __PACKAGE__ );

=pod

=head1 NAME

OpenInteract2::Brick::News - Base-64 encoded OI2 package 'news-2.25.zip' shipped with distribution

=head1 SYNOPSIS

  oi2_manage create_website --website_dir=/path/to/site

=head1 DESCRIPTION

Are you sure you even need to be reading this? If you are just looking
to install a package just follow the instructions from the SYNOPSIS.

Still here? This class holds the Base64-encoded versions of package
file "news-2.25.zip" shipped with OpenInteract2. Once you decode them you
should store them as a ZIP file and then read them in with
Archive::Zip or some other utility.

A typical means to do this is:

 my $brick = OpenInteract2::Brick->new( 'news' );

 # there's only one resource in this brick...
 my ( $pkg_name ) = $brick->list_resources;
 my $pkg_info = $brick->load_resource( $pkg_name );
 my $pkg_file = OpenInteract2::Util->decode_base64_and_store(
     \$pkg_info->{content}
 );

 # $pkg_file now references a file on the filesystem that you can
 # manipulate as normal

These resources are associated with OpenInteract2 version 1.99_06.


=head2 Resources

You can grab resources individually using the names below and
C<load_resource()> and C<copy_resources_to()>, or you can copy all the
resources at once using C<copy_all_resources_to()> -- see
L<OpenInteract2::Brick> for details.

=over 4


=item B<news-2.25.zip>


=back

=head1 COPYRIGHT

Copyright (c) 2005 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS


Chris Winters E<lt>chris@cwinters.comE<gt>


=cut


sub NEWS225ZIP {
    return <<'SOMELONGSTRING';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SOMELONGSTRING
}

