package OpenInteract::BasicPage;

# $Id: BasicPage.pm,v 1.4 2002/01/02 02:43:56 lachoy Exp $

use strict;
use Data::Dumper   qw( Dumper );
use File::Basename qw();

$OpenInteract::BasicPage::VERSION = sprintf("%d.%02d", q$Revision: 1.4 $ =~ /(\d+)\.(\d+)/);


########################################
# CLASS METHODS
########################################

# Retrieve unique directory names and counts of member files from the
# system.
#
# Returns: arrayref of arrayrefs, first member is the directory name
# and the second is the number of files in the directory.

sub list_directories {
    my ( $class ) = @_;
    my $directory_list = eval { $class->db_select({ 
                                        from   => $class->CONFIG->{table_name},
                                        select => [ 'directory, count(*)' ],
                                        select_modifier => 'DISTINCT',
                                        order  => 'directory',
                                        group  => 'directory' }) };
    if ( $@ ) {
        my $R = OpenInteract::Request->instance;
        $R->scrib( 0, "Error trying to retrieve distinct directory names: $@",
                      Dumper( SPOPS::Error->get ) );
        $directory_list = [];
    }
    return $directory_list;
}


########################################
# OBJECT METHODS
########################################

# Just replace the generated 'url' method with one that just uses the
# location. This won't work if you do not have the BasicPage handler
# answering all 'unknown' requests.

sub object_description {
    my ( $self ) = @_;
    my $info = $self->SUPER::object_description();
    $info->{url} = $self->{location};
    return $info;
}


########################################
# RULESET METHODS
########################################

# Here we add a ruleset so we can set the value for 'directory'
# whenever we save the object

sub ruleset_behavior {
    my ( $class, $rs_table ) = @_;
    push @{ $rs_table->{pre_save_action} }, \&find_directory;
    return __PACKAGE__;
}

sub find_directory {
    my ( $self, $p ) = @_;
    $self->{directory} = File::Basename::dirname( $self->{location} );
    return 1;
}

1;

__END__

=pod

=head1 NAME

OpenInteract::BasicPage - Additional methods for the SPOPS BasicPage object

=head1 SYNOPSIS

 # Create a new basic page object

 my $page = OpenInteract::BasicPage->new();
 $page->{location} = '/mysite/home';
 $page->{is_file}  = 'no';
 $page->{pagetext} = "<h2>Headline</h2><p>This is the text for my page</p>";
 $page->{author}   = "Scooby Doo";
 $page->save;

 print "Directory for location $page->{location} is $page->{directory}\n";

 # Get the list of directories

 my $directory_info = OpenInteract::BasicPage->list_directories();
 foreach my $info ( @{ $directory_info } ) {
     print "Directory $info->[0] has $info->[1] entries\n";
 }

=head1 DESCRIPTION

This class adds methods to the SPOPS class that represents static pages. 

=head1 METHODS

B<find_directories()>

Class method that finds all unique directories currently used in the
system along with the number of files in each.

Returns an arrayref of arrayrefs. The first element in each arrayref
in the directory name, the second is the number of files in the
directory.

B<object_description()>

(Overrides method from SPOPS.)

Modify the C<url> value of the returned hashref of object information
to simply be the location of the basic page.

=head1 RULESETS

B<pre_save_action>

Set the C<directory> property from the C<location> property 

=head1 BUGS

None known.

=head1 TO DO

Nothing known.

=head1 SEE ALSO

=head1 COPYRIGHT

Copyright (c) 2001-2002 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
