package OpenInteract::Handler::SystemDoc;

# $Id: SystemDoc.pm,v 1.7 2001/08/24 21:00:23 lachoy Exp $

use strict;
use SPOPS::Secure qw( :level );
use Data::Dumper  qw( Dumper );
use OpenInteract::Package;
use OpenInteract::PackageRepository;
use Pod::POM;

my ( %POD_CACHE );

@OpenInteract::Handler::SystemDoc::ISA     = qw( OpenInteract::Handler::GenericDispatcher SPOPS::Secure );
$OpenInteract::Handler::SystemDoc::VERSION = sprintf("%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::SystemDoc::author            = 'chris@cwinters.com';
$OpenInteract::Handler::SystemDoc::default_method    = 'listing';
@OpenInteract::Handler::SystemDoc::forbidden_methods = qw( _colonify _uncolonify );
%OpenInteract::Handler::SystemDoc::security          = ( 
 listing => SEC_LEVEL_READ,  show   => SEC_LEVEL_READ, 
);


use constant MAIN_SCRIPT => '/SystemDoc';

sub listing {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $params = { error_msg   => $p->{error_msg}, 
                   main_script => MAIN_SCRIPT  };
    $R->{page}->{title} = 'OpenInteract System Documentation / Menu';
    return $R->template->handler( {}, $params, 
                                  { db      => 'system_doc_menu', 
                                    package => 'system_doc' } );
}



sub package_list {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $params = { error_msg   => $p->{error_msg}, 
                   main_script => MAIN_SCRIPT };

    # Grab the repository for thiw website and fetch a list of all the
    # packages so we can and find the 'doc/titles' document in each, or
    # so we can read the files from 'doc/' directly

    my $repos = OpenInteract::PackageRepository->fetch( 
                                     undef, 
                                     { directory => $R->CONFIG->{dir}->{base} } );
    my $pkg_list   = $repos->fetch_all_packages();

PACKAGE:
    foreach my $pkg ( @{ $pkg_list } ) {
        $R->DEBUG && $R->scrib( 1, "Finding documents for package $pkg->{name}-$pkg->{version}" );
        my $doc_dir = join( '/', $pkg->{website_dir}, $pkg->{package_dir}, 'doc' );
        unless ( -d $doc_dir ) {
            $doc_dir = join( '/', $pkg->{base_dir}, $pkg->{package_dir}, 'doc' );
            next PACKAGE unless ( -d $doc_dir );
        }
        $R->DEBUG && $R->scrib( 1, "Found doc dir: $doc_dir" );
        if ( -f "$doc_dir/titles" ) {
            $R->DEBUG && $R->scrib( 1, "Opening titles file in dir" );
            eval { open( TITLES, "$doc_dir/titles" ) || die $! };
            if ( $@ ) {
                $R->scrib( 0, "Error opening titles file; reading filenames directly. (Error: $@)" );
            }
            else {
                while ( <TITLES> ) {
                    chomp;
                    next if ( /^\s*\#/ );
                    next if ( /^\s*$/ );
                    my ( $filename, $title ) = /^([\w.]+)\s+(.*)$/;
                    $R->DEBUG && $R->scrib( 1, "Found $filename / $title from $_" );
                    push @{ $pkg->{tmp_doc_list} }, 
                                     { filename => $filename, title => $title };
                }
                close( TITLES );
                next PACKAGE;
            }
        }
        eval { opendir( DOC, $doc_dir ) || die $! };
        if ( $@ ) {
            $R->scrib( 0, "Could not open $doc_dir: $@" );
            next PACKAGE;
        }

        # Ensure that we get no tmp files and that the file exists

        my @doc_files = grep { -f "$doc_dir/$_" } 
                        grep ! /^(tmp|\.|\_)/, 
                        grep ! /~$/, 
                        readdir( DOC );
        closedir( DOC );
        $pkg->{tmp_doc_list} = [ map { { filename => $_, 
                                         title    => "$_ (no title)" } } 
                                 @doc_files ];
    }
    $params->{package_list} = $pkg_list;
    $R->{page}->{title} = 'OpenInteract Package Documentation';
    return $R->template->handler( {}, $params, 
                                  { db      => 'package_doc',
                                    package => 'system_doc' } );
}



sub module_list {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;
    my $params = { error_msg   => $p->{error_msg},
                   main_script => MAIN_SCRIPT  };
    my $view = $R->apache->param( 'view' ) || 'local';

    # Now sort INC and chop up the files into packages -- depending on
    # what option we were given, restrict the modules to either
    # SPOPS/OpenInteract or Everything

    my %this_inc = %INC;
    my @top = ();
    my $website_name = $R->CONFIG->{website_name};
    $R->DEBUG && $R->scrib( 1, "Website name being filtered: ($website_name)" );

    if ( $view eq 'local' ) {
        @top = ( [ 'OpenInteract', [ 'OpenInteract' ] ],
                 [ 'SPOPS', [ 'SPOPS' ] ],
                 [ $website_name, [] ] );
        my @local_modules = sort grep /^(OpenInteract|SPOPS|$website_name)/, keys %this_inc;
        foreach my $local_module ( @local_modules ) {
            my ( $module_type ) = $local_module =~ /^(\w+)\//;
            push @{ $top[0]->[1] }, _colonify( $local_module )  if ( $module_type eq 'OpenInteract' );
            push @{ $top[1]->[1] }, _colonify( $local_module )  if ( $module_type eq 'SPOPS' );
            push @{ $top[2]->[1] }, _colonify( $local_module )  if ( $module_type eq $website_name );
            $R->DEBUG && $R->scrib( 1, "Found package $local_module ($module_type)" );
        }
    }
    elsif ( $view eq 'general' ) {

        # Get rid of the SPOPS/OI/ThisApp packages

        foreach ( grep /^OpenInteract/, keys %this_inc ) { delete $this_inc{ $_ } }
        foreach ( grep /^SPOPS/, keys %this_inc ) { delete $this_inc{ $_ } }
        foreach ( grep /^$website_name/, keys %this_inc ) { delete $this_inc{ $_ } }

        my $count = -1;
        my $curr_parent = undef;
        foreach my $full_pkg ( sort keys %this_inc ) {
            next unless ( $full_pkg =~ /\.pm$/ );
            my ( $first ) = split /\//, $full_pkg;
            if ( $first ne $curr_parent ) {
                $count++;
                $R->DEBUG && $R->scrib( 1, "First item != parent: ($first) / ($curr_parent)" );
                $curr_parent   = $first;
                $curr_parent   =~ s/\.pm$//;
                $top[ $count ] = [ $curr_parent, [] ]; #  _colonify( $curr_parent ) 
            }
            $R->DEBUG && $R->scrib( 1, "Found package $full_pkg" );
            push @{ $top[ $count ]->[1] }, _colonify( $full_pkg );
        }
    }
    $params->{module_list}  = \@top;
    $R->DEBUG && $R->scrib( 1, "Number of module parents found: ", scalar @top );
  
    $R->{page}->{title} = 'OpenInteract System Documentation / Module Documentation';
    return $R->template->handler( {}, $params, 
                                  { db      => 'system_module_listing',
                                    package => 'system_doc' } );
}


sub _colonify {
    my ( $text ) = @_;
    $text =~ s|\.pm$||;
    $text =~ s|/|::|g;
    return $text;
}


sub _uncolonify {
    my ( $text ) = @_;
    $text .= '.pm';
    $text =~ s|::|/|g;
    return $text;
}


sub show {
    my ( $class, $p ) = @_;
    my $R = OpenInteract::Request->instance;

    my ( $pod_file, $html_file, $text_file );

    # If this is a package, display the doc

    my $package = $R->apache->param( 'package' );
    if ( $package ) {
        my $doc = $R->apache->param( 'doc' );
        my $repos = OpenInteract::PackageRepository->fetch(
                                   undef,
                                   { directory => $R->CONFIG->{dir}->{base} } );
        if ( $doc =~ /\.(html|txt|pod)$/ ) {
            my $full_filename = $repos->find_file( $package, "doc/$doc" );
            $R->DEBUG && $R->scrib( 1, "Found ($full_filename) using $doc and $package" );
            $pod_file  = $full_filename  if ( $doc =~ /\.pod$/ );
            $html_file = $full_filename  if ( $doc =~ /\.html$/ );
            $text_file = $full_filename  if ( $doc =~ /\.txt$/ );
        }
    }
    else {
        my $module = $p->{module} || $R->apache->param( 'module' );

        # ewww! ick!
        $pod_file = $POD_CACHE{ $module } || `perldoc -l $module`;
        chomp $pod_file;
        $R->DEBUG && $R->scrib( 1, "Found ($pod_file) from module ($module)" );
        unless ( $pod_file ) {
            $pod_file = $INC{ _uncolonify( $module ) };
            $R->DEBUG && $R->scrib( 1, "Found ($pod_file) from module ($module)" );
        }
        $POD_CACHE{ $module } = $pod_file  if ( -f $pod_file );
    }

    if ( -f $pod_file ) {
        $R->DEBUG && $R->scrib( 1, "Trying to view pod in ($pod_file)" );
        my $parser = Pod::POM->new();
        my $pom = $parser->parse( $pod_file );
        return '<p>Error parsing POD: ' . $parser->error() . '</p>' unless ( $pom );

        eval { require OpenInteract::PodView };
        return "<p>Error trying to load POD viewer: $@</p>"  if ( $@ );
        my $doc = OpenInteract::PodView->print( $pom );
        return '<p>Sorry, no docs for this module.</p>'      unless ( $doc );
        $doc =~ s/^.*<BODY>//;
        $doc =~ s|</BODY>.*$||;
        $R->{page}->{title} = 'OpenInteract System Documentation Display';
        return $doc;
    }

    if ( -f $html_file ) {
        eval { open( HTML, $html_file ) || die $! };
        return "<p>Error opening HTML file: $@"  if ( $@ );
        local $/ = undef;
        my $html_text = <HTML>;
        close( HTML );
        $html_text =~ s/^.*<BODY>//;
        $html_text =~ s|</BODY>.*$||;
        return '<p>Sorry, no docs for this module.' unless ( $html_text );
        $R->{page}->{title} = 'OpenInteract System Documentation Display';
        return $html_text;
    }

    if ( -f $text_file ) {
        eval { open( TEXT, $text_file ) || die $! };
        return "<p>Error opening TEXT file: $@"  if ( $@ );
        local $/ = undef;
        my $text_text = <TEXT>;
        close( TEXT );
        return '<p>Sorry, no docs for this module.' unless ( $text_text );
        $R->{page}->{title} = 'OpenInteract System Documentation Display';
        return qq(<pre><font size="-1">$text_text</font></pre>);
    }
    return '<p>Sorry, I cannot find documentation by that name.</p>';
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::SystemDoc - Display system documentation in HTML format

=head1 SYNOPSIS

=head1 DESCRIPTION

Display documentation for the OpenInteract system, SPOPS modules, and any
other perl modules used.

=head1 METHODS

B<listing()>

List the OpenInteract system documentation and all the modules used by
the system -- we display both the C<OpenInteract> modules and the
C<SPOPS> modules first.

B<package_list( \%params )>

B<module_list( \%params )>

B<show( \%params )>

Display a particular document or module, filtering through C<pod2html>.

Parameters:

=over 4

=item *

B<filename>: Full filename of document to extract POD from.

=item *

B<module>: Perl module to extract POD from; we match up the module to
a file using %INC

=back

=head1 TO DO

B<Get more meta information>

System documentation needs more meta information so we can better
display title and other information on the listing page.

=head1 BUGS

None known.

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
