#!/usr/bin/perl

# $Id: search_index.pl,v 1.2 2001/02/04 23:06:33 lachoy Exp $

# search_index.pl - Search the index for objects matching one or more words.

use strict;
use OpenInteract::Startup;
use Data::Dumper  qw( Dumper );
use Getopt::Long  qw( GetOptions );

{
 my $usage = "Usage: $0 --website_dir=/path/to/my_site term [ term term ... ]";

 # Get the options

 my ( $OPT_website_dir );
 GetOptions( 'website_dir=s' => \$OPT_website_dir );

 if ( ! $OPT_website_dir and $ENV{OIWEBSITE} ) {
   print "[oi_manage]: Using ($ENV{OIWEBSITE}) for 'website_dir'.\n";
   $OPT_website_dir = $ENV{OIWEBSITE};
 }

 unless ( -d $OPT_website_dir ) {
   die "$usage\n Parameter 'website_dir' must refer to an OpenInteract website directory!\n";
 }

 # Get the terms

 my @terms = @ARGV;
 die "$usage\n"  unless ( scalar @terms );

 # Setup the OI environment

 my $R = OpenInteract::Startup->setup_static_environment( 
                                    $OPT_website_dir );

 # Perform the search and dump the results
 
 my $object_list = OpenInteract::FullText->search_ft_index({
                                     search_terms => \@terms,
                                     skip_security => 1 });
 print "Search Results\n===============\n\n";
 my $pat = "Score %03d for object: %-20s (%-s)\n";
 foreach my $obj ( @{ $object_list } ) {
   printf( $pat, $obj->{tmp_ft_score}, ref $obj, $obj->id );
 }
 print "\nAction complete.\n";

 # Some DBD drivers complain about not explicitly disconnecting, which
 # worries people...

 $R->db->disconnect; 
}

__END__

=pod

=head1 NAME

search_index.pl - Simple script to search the full-text index for objects

=head1 SYNOPSIS

 > search_index.pl --website_dir=/path/to/my_site term1 term2

OR (using a bash shell):

 > export OIWEBSITE=/path/to/my_site
 > search_index.pl term1 term2

=head1 DESCRIPTION

Searches the full-text index in your application for objects that have
one or more terms in them. This is partly a demonstration of how to
search the full-text index and how to use the search results, and
partly for use as a command-line check to the index which is normally
only searchable via the web.

=head1 BUGS 

None yet!

=head1 TO DO

=head1 SEE ALSO

L<OpenInteract::FullText>

=head1 COPYRIGHT

Copyright (c) 2001 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
