#!/usr/bin/perl

# $Id: add_objects_to_index.pl,v 1.4 2000/11/18 23:16:35 cwinters Exp $

# add_objects_to_index.pl -- Re-index a particular class.

use strict;
use OpenInteract::Startup;
use OpenInteract::DBI;
use Data::Dumper  qw( Dumper );
use Getopt::Long  qw( GetOptions );

{
 my $usage = "Usage: $0 --application_dir=/path/to/my_app object-tag";
 my ( $OPT_application_dir );
 GetOptions( 'application_dir=s' => \$OPT_application_dir );
 unless ( -d $OPT_application_dir ) {
   die "$usage\n Parameter 'application_dir' must refer to an OpenInteract application directory!\n";
 }
 unshift @INC, $OPT_application_dir;
 my $base_conf = join( '/', $OPT_application_dir, 'conf', 'base.conf' );
 unless ( -f $base_conf ) {
   die "$usage\n Cannot find base configuration file in $OPT_application_dir!\n";
 }

 my $object_tag = shift;
 unless ( $object_tag ) {
   die "$usage\n";
 }

 my ( $init, $C ) = OpenInteract::Startup->main_initialize( { base_config_file => $base_conf,
                                                              alias_init => 1, spops_init => 1 } );
 my $REQUEST_CLASS = $C->{request_class};
 my $R = $REQUEST_CLASS->instance;
 $R->{stash_class} = $C->{stash_class};
 my $db = OpenInteract::DBI->connect( $C->{db_info} );
 $R->stash( 'db', $db );

 my $obj_class = $R->$object_tag();
 unless ( $obj_class ) {
   die "Cannot retrieve objects without a class! (no match for $object_tag)";
 }
 unless ( $obj_class->isa( 'OpenInteract::FullText' ) ) {
   die "Failed! The class ($obj_class) corresponding to tag \n",
       "($object_tag) does not currently use the full-text indexing\n",
       "engine. Change the 'isa' tag for the object.\n";
 }
 my $obj_list = $obj_class->fetch_group( { skip_security => 1 } );
 print "Objects found in group: ", scalar @{ $obj_list }, "\n";
 my $count = 0;
 foreach my $obj ( @{ $obj_list } ) {
   print " $count" if ( $count % 25 == 0 );
   $obj->reindex_object;
   $count++;
 }
 $db->disconnect;
}
 
__END__

=pod

=head1 NAME

add_objects_to_index.pl - Reindex objects in a particular class

=head1 SYNOPSIS

 add_objects_to_index.pl --application_dir=/path/to/my_app myobject

=head1 DESCRIPTION

Cycles through every available object in the class given and calls
'reindex_object' on it.

=head1 BUGS 

None yet!

=head1 TO DO

=head1 SEE ALSO

L<OpenInteract::FullText>

=head1 COPYRIGHT

Copyright (c) 2000 intes.net, inc.. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

 Chris Winters (chris@cwinters.com)

=cut
