use 5.10.1;
use strict;
use warnings;

package Dist::Zilla::Plugin::Changelog::Grouped;

# ABSTRACT: Simplify usage of a grouped changelog
our $VERSION = '0.0100';

use Moose;
use namespace::autoclean;

use Types::Standard qw/Str ArrayRef/;
use Path::Tiny;
use CPAN::Changes;
with qw/
    Dist::Zilla::Role::Plugin
    Dist::Zilla::Role::FileMunger
    Dist::Zilla::Role::AfterRelease
/;

has change_file => (
    is => 'ro',
    isa => Str,
    default => 'Changes',
);
has groups => (
    is => 'ro',
    isa => (ArrayRef[Str])->plus_coercions(Str, sub { [split m{\s*,\s*}, $_] }),
    traits => ['Array'],
    coerce => 1,
    default => sub { ['API Changes', 'Bug Fixes', 'Enhancements', 'Documentation'] },
    handles => {
        all_groups => 'elements',
    }
);

sub munge_files {
    my $self = shift;

    my($file) = grep { $_->name eq $self->change_file } @{ $self->zilla->files };

    my $changes = CPAN::Changes->load_string($file->content, next_token => qr/\{\{\$NEXT\}\}/);
    my $next = (reverse $changes->releases)[0];
    return if !defined $next;

    $next->delete_group($_) for grep { !@{ $next->changes($_) } } $next->groups;

    $self->log_debug(['Cleaning up %s in memory', $file->name]);
    $file->content($changes->serialize);
}

sub after_release {
    my $self = shift;

    my $changes = CPAN::Changes->load($self->change_file, next_token => qr/\{\{\$NEXT\}\}/);
    $changes->delete_empty_groups;
    my($next) = reverse $changes->releases;

    $next->add_group($self->all_groups);
    $self->log_debug(['Cleaning up %s on disk', $self->change_file]);

    path($self->change_file)->spew($changes->serialize);
}

__PACKAGE__->meta->make_immutable;

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Dist::Zilla::Plugin::Changelog::Grouped - Simplify usage of a grouped changelog



=begin HTML

<p><img src="https://img.shields.io/badge/perl-5.10.1+-brightgreen.svg" alt="Requires Perl 5.10.1+" /> <a href="https://travis-ci.org/Csson/p5-Dist-Zilla-Plugin-Changelog-Grouped"><img src="https://api.travis-ci.org/Csson/p5-Dist-Zilla-Plugin-Changelog-Grouped.svg?branch=master" alt="Travis status" /></a> <img src="https://img.shields.io/badge/coverage-94.6%-yellow.svg" alt="coverage 94.6%" /></p>

=end HTML


=begin markdown

![Requires Perl 5.10.1+](https://img.shields.io/badge/perl-5.10.1+-brightgreen.svg) [![Travis status](https://api.travis-ci.org/Csson/p5-Dist-Zilla-Plugin-Changelog-Grouped.svg?branch=master)](https://travis-ci.org/Csson/p5-Dist-Zilla-Plugin-Changelog-Grouped) ![coverage 94.6%](https://img.shields.io/badge/coverage-94.6%-yellow.svg)

=end markdown

=head1 VERSION

Version 0.0100, released 2016-02-06.

=head1 SYNOPSIS

    # in dist.ini (these are the defaults)
    [Changelog::Grouped]
    change_file = Changes
    groups = API Changes, Bug Fixes, Documentation, Enhancements

=head1 DESCRIPTION

This plugin does two things:

=over 4

=item *

During the build phase it removes empty groups from the changelog.

=item *

After a release it adds the configured groups to the changelog under C<{{$NEXT}}>.

=back

=head1 ACKNOWLEDGMENTS

This plugin is based on parts of L<Dist::Zilla::Plugin::NextVersion::Semantic>.

=head1 SOURCE

L<https://github.com/Csson/p5-Dist-Zilla-Plugin-Changelog-Grouped>

=head1 HOMEPAGE

L<https://metacpan.org/release/Dist-Zilla-Plugin-Changelog-Grouped>

=head1 AUTHOR

Erik Carlsson <info@code301.com>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2016 by Erik Carlsson.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
