#define CHAZ_USE_SHORT_NAMES

#include "Charmonizer/Core/HeadCheck.h"
#include "Charmonizer/Core/ModHandler.h"
#include "Charmonizer/Core/Util.h"
#include "Charmonizer/Probe/Headers.h"
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#include <stdarg.h>

/* keep track of which headers have succeeded */
static int keeper_count = 0;
#define MAX_KEEPER_COUNT 200
static const char *keepers[MAX_KEEPER_COUNT + 1] = { NULL };

/* Add a header to the keepers array.
 */
static void
keep(const char *header_name);

static size_t aff_buf_size = 0;
static char *aff_buf = NULL;

/* Transform "header.h" into "CHY_HAS_HEADER_H, storing the result in 
 * [aff_buf].
 */
static void
encode_affirmation(const char *header_name);

char *c89_headers[] = {
    "assert.h",
    "ctype.h",
    "errno.h",
    "float.h",
    "limits.h",
    "locale.h",
    "math.h",
    "setjmp.h",
    "signal.h",
    "stdarg.h",
    "stddef.h",
    "stdio.h",
    "stdlib.h",
    "string.h",
    "time.h",
    NULL
};

char *posix_headers[] = {
    "cpio.h",
    "dirent.h",
    "fcntl.h",
    "grp.h",
    "pwd.h",
    "sys/stat.h",
    "sys/times.h",
    "sys/types.h",
    "sys/utsname.h",
    "sys/wait.h",
    "tar.h",
    "termios.h",
    "unistd.h",
    "utime.h",
    NULL
};

chaz_bool_t
chaz_Headers_check(const char *header_name)
{
    return check_header(header_name);
}

void
chaz_Headers_run(void) 
{
    int i;
    chaz_bool_t has_posix = false;
    chaz_bool_t has_c89   = false;
    
    START_RUN("Headers");

    /* test for all POSIX headers in one blast */
    if (check_many_headers((const char**)posix_headers)) {
        has_posix = true;
        append_conf("#define CHY_HAS_POSIX\n");
        for (i = 0; posix_headers[i] != NULL; i++) {
            keep(posix_headers[i]);
        }
    }

    /* test for all c89 headers in one blast */
    if (check_many_headers((const char**)c89_headers)) {
        has_c89 = true;
        append_conf("#define CHY_HAS_C89\n");
        append_conf("#define CHY_HAS_C90\n");
        for (i = 0; c89_headers[i] != NULL; i++) {
            keep(c89_headers[i]);
        }
    }

    /* append the config with every header detected so far */
    for (i = 0; keepers[i] != NULL; i++) {
        encode_affirmation(keepers[i]);
        append_conf("#define CHY_%s\n", aff_buf);
    }

    /* shorten */
    START_SHORT_NAMES;
    if (has_posix)
        shorten_macro("HAS_POSIX");
    if (has_c89) {
        shorten_macro("HAS_C89");
        shorten_macro("HAS_C90");
    }
    for (i = 0; keepers[i] != NULL; i++) {
        encode_affirmation(keepers[i]);
        shorten_macro(aff_buf);
    }
    END_SHORT_NAMES;

    END_RUN;
}

static void
keep(const char *header_name)
{
    if (keeper_count >= MAX_KEEPER_COUNT)
        die("Too many keepers -- increase MAX_KEEPER_COUNT");
    keepers[keeper_count++] = header_name;
    keepers[keeper_count]   = NULL;
}

static void
encode_affirmation(const char *header_name) {
    char *buf, *buf_end;
    size_t len = strlen(header_name) + sizeof("HAS_");
    
    /* grow buffer and start off with "HAS_" */
    aff_buf_size = grow_buf(&aff_buf, aff_buf_size, len);
    strcpy(aff_buf, "HAS_");

    /* transform one char at a time */
    for(buf = aff_buf + sizeof("HAS_") - 1, buf_end = aff_buf + len; 
        buf < buf_end; 
        header_name++, buf++
    ) {
        if (*header_name == '\0') {
            *buf = '\0';
            break;
        }
        else if (isalnum(*header_name)){
            *buf = toupper(*header_name);
        }
        else {
            *buf = '_';
        }
    }
}


/* Copyright 2006-2007 Marvin Humphrey
 *
 * This program is free software; you can redistribute it and/or modify
 * under the same terms as Perl itself.
 */

