package PDK::Device::Radware;

use 5.030;
use strict;
use warnings;

use Moose;
use Expect qw'exp_continue';
use Carp   qw'croak';
with 'PDK::Device::Base';
use namespace::autoclean;

# 定义设备交互的提示符正则表达式
has prompt => (
  is       => 'ro',
  required => 1,
  default  => '^>>.*?#\s*$',    # 默认提示符格式
);

#------------------------------------------------------------------------------
# errCodes: 返回可能的错误模式数组
# 描述：用于检测命令执行时可能发生的错误模式
#------------------------------------------------------------------------------
sub errCodes {
  my $self = shift;

  return [
    qr/^Error:.*?$/m,    # 错误提示模式
  ];
}

#------------------------------------------------------------------------------
# waitfor: 等待特定提示符并捕获输出
# 描述：处理期望输出，管理分页、确认等操作，并返回捕获的输出
# 参数：
#   $prompt - 可选，指定等待的提示符
# 返回：捕获的命令输出
#------------------------------------------------------------------------------
sub waitfor {
  my ($self, $prompt) = @_;

  my $buff = "";                  # 存储命令输出
  $prompt //= $self->{prompt};    # 使用提供的提示符或默认提示符

  # 获取 Expect 对象实例
  my $exp = $self->{exp};

  # 处理期望输出
  my @ret = $exp->expect(
    15,
    [
      qr/Confirm saving without first applying changes/i => sub {
        $exp->send("y\r");                          # 发送确认操作
        $buff .= $exp->before() . $exp->match();    # 捕获输出
        exp_continue;                               # 继续处理
      }
    ],
    [
      qr/Confirm saving to FLASH/i => sub {
        $exp->send("y\r");                          # 发送确认选择
        $buff .= $exp->before() . $exp->match();    # 捕获输出
        exp_continue;                               # 继续处理
      }
    ],
    [
      qr/Confirm dumping all information/i => sub {
        $exp->send("y\r");                          # 发送回车确认
        $buff .= $exp->before() . $exp->match();    # 捕获输出
        exp_continue;                               # 继续处理
      }
    ],
    [
      qr/(Display|Include) private keys/i => sub {

        # 从环境变量加载 PDK_FTP_PASSPHRASE（如果未提供）
        my $passphrase = $self->{passphrase} || $ENV{PDK_FTP_PASSPHRASE};
        $self->{passphrase} = $passphrase if $self->{passphrase} ne $passphrase;

        # 根据是否提供 passphrase 发送确认
        $exp->send(defined $passphrase ? "y\r" : "n\r");
        $buff .= $exp->before() . $exp->match();    # 捕获输出
        exp_continue;                               # 继续处理
      }
    ],
    [
      qr/(Enter|Reconfirm) passphrase/i => sub {
        $exp->send("$self->{passphrase}\r");        # 发送 passphrase
        $buff .= $exp->before() . $exp->match();    # 捕获输出
        exp_continue;                               # 继续处理
      }
    ],
    [
      qr/$prompt/m => sub {
        $buff .= $exp->before() . $exp->match();    # 处理预期提示符，保存输出
      }
    ],
    [
      eof => sub {                                  # 会话丢失，连接被意外关闭
        croak("执行[waitfor]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {                              # 会话超时
        croak("执行[waitfor]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ],
  );

  # 检查期望结果，处理错误
  croak($ret[3]) if defined $ret[1];    # 抛出异常

  # 清理输出，移除多余的回车符
  $buff =~ s/\x1b\[\d+D\s+\x1b\[\d+D//g;

  return $buff;                         # 返回捕获的输出
}

#------------------------------------------------------------------------------
# runCommands: 执行一系列命令，配置下发模式
# 描述：检查命令列表，自动加载配置模式并在必要时添加保存配置的命令
# 参数：
#   $commands - 命令列表，数组参考
# 返回：无
#------------------------------------------------------------------------------
sub runCommands {
  my ($self, $commands) = @_;

  # 确保命令列表是一个数组引用
  croak "执行[runCommands]，必须提供一组待下发脚本" unless ref $commands eq 'ARRAY';

  # 自动加载配置模式
  unshift @$commands, 'cd' unless $commands->[0] =~ /^cd/mi;

  # 添加保存配置命令（如果尚未存在）
  push @$commands, 'apply', 'save' unless $commands->[-1] =~ /^(apply|save)/mi;

  # 执行命令
  $self->execCommands($commands);
}

#------------------------------------------------------------------------------
# getConfig: 获取设备的当前配置
# 返回：包含成功标志和配置内容的哈希引用
#------------------------------------------------------------------------------
sub getConfig {
  my $self = shift;

  # 定义要执行的命令
  my $commands = [
    "cfg/dump",    # 获取设备运行配置
    "cd",          # 切换到顶层视图
  ];

  # 执行命令并获取结果
  my $config = $self->execCommands($commands);

  # 返回错误信息（如果有）
  return $config if $config->{success} == 0;    # 若执行失败，直接返回

  my $lines = $config->{result};                # 获取命令执行结果

  return {success => 1, config => $lines};      # 返回成功及配置内容
}

#------------------------------------------------------------------------------
# ftpConfig: 将设备配置通过 FTP 备份
# 描述：连接到 FTP 服务器并上传配置文件
# 参数：
#   可选参数：$hostname - 可选的主机名,
#               $server - FTP 服务器地址,
#               $username - FTP 用户名,
#               $password - FTP 密码
# 返回：包含成功状态和上传结果的哈希引用
#------------------------------------------------------------------------------
sub ftpConfig {
  my ($self, $hostname, $server, $username, $password) = @_;

  # 从环境变量加载 FTP 服务器、用户名和密码（如果未提供）
  $server   //= $ENV{PDK_FTP_SERVER};
  $username //= $ENV{PDK_FTP_USERNAME};
  $password //= $ENV{PDK_FTP_PASSWORD};

  # 检查用户名和密码是否有效
  croak "请正确提供 FTP 服务器地址、账户和密码!" unless $username and $password and $server;

  # 从环境变量加载 PDK_FTP_PASSPHRASE（如果未提供）
  my $passphrase = $self->{passphrase} || $ENV{PDK_FTP_PASSPHRASE};

  # 生成 FTP 脚本组件
  my $host    = $self->{host};                      # 获取主机名
  my $command = "$self->{month}/$self->{date}/";    # 构建初始命令

  # 根据是否提供 hostname 生成文件名
  if ($hostname) {
    $command .= "${hostname}_$host.tar.gz";
  }
  else {
    $command .= "$host.tar.gz";
  }

  # 检查是否已经登录设备
  if (!$self->{exp}) {
    my $login = $self->login();
    croak $login->{reason} if $login->{success} == 0;    # 登录失败抛出异常
  }

  # 获取 Expect 对象实例
  my $exp    = $self->{exp};
  my $result = $exp ? $exp->match() : "";

  # 下发脚本并执行
  my $connector = "cfg/ptcfg ${server} -m -mansync";
  $exp->send("$connector\n");    # 连接到 FTP 服务器
  say "[debug] 执行FTP备份脚本[$connector]，备份到目标文件为 $command" if $self->{debug};

  # 处理 FTP 登录和配置备份
  my @ret = $exp->expect(
    15,
    [
      qr/hit return for automatic file name/i => sub {
        $result .= $exp->before() . $exp->match();    # 捕获输出
        $exp->send("$command\r");                     # 发送命令
        exp_continue;                                 # 继续处理
      }
    ],
    [
      qr/Enter username for FTP/i => sub {
        $result .= $exp->before() . $exp->match();    # 捕获输出
        $exp->send("$username\r");                    # 输入用户名
        exp_continue;                                 # 继续处理
      }
    ],
    [
      qr/Enter password for username/i => sub {
        $result .= $exp->before() . $exp->match();    # 捕获输出
        $exp->send("$password\r");                    # 输入密码
        exp_continue;                                 # 继续处理
      }
    ],
    [
      qr/(Display|Include) private keys/i => sub {
        $exp->send($passphrase ? "y\r" : "n\r");      # 根据是否提供 passphrase 发送确认
        $result .= $exp->before() . $exp->match();    # 捕获输出
        exp_continue;                                 # 继续处理
      }
    ],
    [
      qr/(Enter|Reconfirm) passphrase/i => sub {
        $exp->send("$passphrase\r");                  # 发送 passphrase
        $result .= $exp->before() . $exp->match();    # 捕获输出
        exp_continue;                                 # 继续处理
      }
    ],
    [
      qr/hit return for FTP server/i => sub {
        $result .= $exp->before() . $exp->match();    # 捕获输出
        $exp->send("\r");                             # 输入回车
      }
    ],
    [
      eof => sub {
        croak("执行[ftpConfig/登录FTP服务器]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {
        croak("执行[ftpConfig/登录FTP服务器]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ],
  );

  # 检查期望结果，处理错误
  croak($ret[3]) if defined $ret[1];    # 抛出异常

  # 检查是否正常登录
  @ret = $exp->expect(
    10,
    [
      qr/^Error: Illegal operation/mi => sub {
        croak("FTP 会话丢失: username or password is wrong!");
      }
    ],
    [
      qr/Current config successfully transferred/ => sub {
        $result .= $exp->before() . $exp->match();             # 捕获输出
        say "FTP 配置备份：文件 $command 上传成功!" if $self->{debug};    # 上传成功
      }
    ],
    [
      eof => sub {
        croak("执行[ftpConfig/检查备份任务是否成功]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {
        croak("执行[ftpConfig/检查备份任务是否成功]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ],
  );

  # 检查期望结果，处理错误
  croak($ret[3]) if defined $ret[1];    # 抛出异常
  $exp->send("cd\r");                   # 进入目标目录

  return {success => 1, config => $result};    # 返回成功及结果
}

# 标记类为不可变，以提高性能
__PACKAGE__->meta->make_immutable;

1;                                             # 返回真值以表示模块加载成功
