package PDK::Device::Cisco;

use 5.030;
use strict;
use warnings;

use Moose;
use Expect qw'exp_continue';
use Carp   qw'croak';
with 'PDK::Device::Base';
use namespace::autoclean;

# 定义提示符的正则表达式
has prompt => (
  is       => 'ro',
  required => 1,
  default  => '^\s*.*?[#>]\s*$',    # 默认提示符为类似于 'hostname# ' 或 'hostname> '
);

has enPrompt => (
  is       => 'ro',
  required => 0,
  default  => '^\s*.*?[>]\s*$',     # 特权模式提示符
);

has enCommand => (
  is       => 'ro',
  required => 0,
  default  => 'enable',             # 切换到特权模式的命令
);

#------------------------------------------------------------------------------
# errCodes: 返回可能的错误码列表
# 描述：这些错误码可以用于检查在执行命令时是否发生了错误。
# 例如：FTP 异常 %Error opening ftp://192.168.99.1/startup.cfg (Permission denied)
#------------------------------------------------------------------------------
sub errCodes {
  my $self = shift;

  return [
    qr/(Ambiguous|Incomplete|Unrecognized|Bad|not recognized)/i,
    qr/(Permission denied|syntax error|authorization failed)/i,
    qr/(Invalid (parameter|command|input)|Unknown command|Login invalid)/i,
  ];
}

#------------------------------------------------------------------------------
# waitfor: 期望匹配特定的提示符，自动交互式执行脚本
# 描述：此方法会等待设备输出，直到达到指定的提示符。
# N9K 存在的异常提示符：^[K | ^M | 配置备份进度条
#------------------------------------------------------------------------------
sub waitfor {
  my ($self, $prompt) = @_;

  my $buff = "";                  # 存储命令输出
  $prompt //= $self->{prompt};    # 使用默认提示符

  my $exp = $self->{exp};         # 获取 Expect 对象实例
  my @ret = $exp->expect(
    10,
    [
      qr/^.+more\s*.+$/mi => sub {
        $exp->send(" ");            # 发送空格以继续输出
        $buff .= $exp->before();    # 追加输出
        exp_continue;               # 继续监听
      }
    ],
    [
      qr/\[startup-config\]\?/i => sub {
        $exp->send("\r");                           # 发送回车
        $buff .= $exp->before() . $exp->match();    # 追加输出
        exp_continue;                               # 继续监听
      }
    ],
    [
      qr/Address or name of remote host/i => sub {
        $exp->send("\r");                           # 发送回车
        $buff .= $exp->before() . $exp->match();    # 追加输出
        exp_continue;                               # 继续监听
      }
    ],
    [
      qr/Destination filename \[/i => sub {
        $exp->send("\r");                           # 发送回车
        $buff .= $exp->before() . $exp->match();    # 追加输出
        exp_continue;                               # 继续监听
      }
    ],
    [
      qr/$prompt/m => sub {
        $buff .= $exp->before() . $exp->match();    # 追加输出和匹配
      }
    ],
    [
      eof => sub {
        croak("执行[waitfor/自动交互执行回显]，与设备 $self->{host} 会话丢失，连接被意外关闭！具体原因：\n" . $exp->before());
      }
    ],
    [
      timeout => sub {
        croak("执行[waitfor/自动交互执行回显]，与设备 $self->{host} 会话超时，请检查网络连接或服务器状态！具体原因：\n" . $exp->before());
      }
    ],
  );

  # 检查期望结果，处理错误
  croak($ret[3]) if defined $ret[1];    # 抛出异常

  # 规范化输出，修正内容
  $buff =~ s/\r\n|\n+\n/\n/g;           # 规范换行
  $buff =~ s/\x{08}+\s+\x{08}+//g;      # 移除退格字符

  # ^M 是一个控制字符，表示回车符（Carriage Return，CR）。在某些终端或设备中，它用于光标回到行首，常见于网络设备的命令行界面输出中。
  $buff =~ s/\x0D\[\s*#+\s*\]?\s*\d{1,2}%//g;    # 移除备份进度提示符，同时保留 100% 进度条
  $buff =~ s/\x1B\[K//g;                         # 替换所有的 ^[K 字符为空
  $buff =~ s/\x0D//g;                            # 替换所有的 ^M 字符为空

  return $buff;                                  # 返回命令输出
}

#------------------------------------------------------------------------------
# runCommands: 执行一系列命令，配置下发模式
# 描述：此方法检查命令列表，自动加载配置模式并在必要时添加保存配置的命令。
# 参数：
#   $commands - 命令列表，数组参考
# 返回：无
#------------------------------------------------------------------------------
sub runCommands {
  my ($self, $commands) = @_;

  # 确保命令列表是一个数组引用
  croak "执行[runCommands]，必须提供一组待下发脚本" unless ref $commands eq 'ARRAY';

  # 修正配置模块
  $self->{mode} = 'deployCommands';

  # 自动加载配置模式，如果命令列表中没有配置命令
  if ($commands->[0] !~ /conf/i) {
    unshift @$commands, 'configure terminal';
  }

  # 在命令列表末尾添加保存配置的命令（如果尚未存在）
  unless ($commands->[-1] =~ /(copy run|write)/i) {
    push @$commands, 'copy running-config startup-config';
  }

  # 执行命令
  $self->execCommands($commands);
}

#------------------------------------------------------------------------------
# getConfig: 获取设备的运行配置
# 返回：包含成功标志和配置内容的哈希引用
# 说明：terminal width (ios支持512，nxos支持511)，且 nxos 下不调整会出现断裂配置的问题
#------------------------------------------------------------------------------
sub getConfig {
  my $self = shift;

  # 定义要执行的命令
  my $commands = [
    "terminal width 511",          # 设置终端宽度为511
    "terminal length 0",           # 设置终端长度为0
    "show run | exclude !Time",    # 获取运行配置，排除时间信息
    "copy run start",              # 将运行配置复制到启动配置
  ];

  # 执行命令并获取结果
  my $config = $self->execCommands($commands);

  # 返回错误信息（如果有）
  if ($config->{success} == 0) {
    return $config;
  }
  else {
    my $lines = $config->{result};                       # 获取命令执行结果
    $lines =~ s/^\s*ntp\s+clock-period\s+\d+\s*$//mi;    # 移除特定行
        # $lines =~ s/.*?(\[\#*\s*\]\s*\d{1,3}%)[^[]*$/$1/;; # 移除备份进度提示符，同时保留 100% 进度条

    return {success => 1, config => $lines};    # 返回成功及配置
  }
}

#------------------------------------------------------------------------------
# ftpConfig: 将运行配置通过 FTP 备份
# 参数：$server - FTP 服务器地址, $hostname - 可选的主机名
# 返回：执行命令的结果
#------------------------------------------------------------------------------
sub ftpConfig {
  my ($self, $hostname, $server) = @_;

  # 从环境变量加载用FTP服务器
  $server ||= $ENV{PDK_FTP_SERVER};

  # 生成 FTP 脚本
  my $host    = $self->{host};                                                        # 获取主机名
  my $command = "copy running-config ftp://$server/$self->{month}/$self->{date}/";    # 构建命令

  # 根据是否提供 hostname 生成文件名
  if ($hostname) {
    $command .= $hostname . '_' . $host . '.cfg';                                     # 使用 hostname 和 host 作为文件名
  }
  else {
    $command .= $host . '.cfg';                                                       # 仅使用 host 作为文件名
  }

  # 执行命令并获取返回结果
  my $result = $self->execCommands([$command]);
  if ($result->{success} == 0) {
    croak "执行[ftpConfig/配置备份异常]，$result->{reason}";
  }
  else {
    return $result;
  }
}

# 标记类为不可变，以提高性能
__PACKAGE__->meta->make_immutable;
1;    # 返回真值以表示模块加载成功
