package Dua;
use strict;
use warnings;
use Dua::Parser;

our $VERSION = '0.02';

sub new {
    my $class = shift;

    bless {
        _ua     => shift || '',
        _parsed => 0,
        _result => {},
    }, $class;
}

sub ua { shift->{_ua} }

sub _result {
    my ($self, $value) = @_;

    if ($value) {
        $self->{_result} = $value;
        return $self;
    }
    else {
        $self->parse unless $self->{_parsed};
        return $self->{_result};
    }
}

sub parse {
    my $self = shift;

    if (ref $self eq __PACKAGE__) {
        $self->_parse;
    }
    elsif ($self eq __PACKAGE__ && scalar @_ == 1) {
        # my $dua_obj = Dua->parse('User-Agent String');
        my $dua = __PACKAGE__->new($_[0]);
        return $dua->_parse;
    }
    else {
        # my $dua_obj = Dua::parse('User-Agent String');
        my $dua = __PACKAGE__->new($self);
        return $dua->_parse;
    }
}

sub _parse {
    my $self = shift;

    $self->_result(Dua::Parser->parse($self));
    $self->{_parsed} = 1;

    return $self;
}

sub name {
    shift->_result->{name};
}

sub is_bot {
    shift->_result->{is_bot} ? 1 : 0;
}

sub is_ios {
    shift->_result->{is_ios} ? 1 : 0;
}

sub is_android {
    shift->_result->{is_android} ? 1 : 0;
}

sub is_linux {
    shift->_result->{is_linux} ? 1 : 0;
}

sub is_windows {
    shift->_result->{is_windows} ? 1 : 0;
}

1;

__END__

=encoding UTF-8

=head1 NAME

Dua - Detect User-Agent


=head1 SYNOPSIS

    use Dua;

    my $ua = 'Mozilla/5.0 (compatible; Googlebot/2.1; +http://www.google.com/bot.html)';

    my $dua = Dua->new($ua);
    $dua->is_bot
        and say $dua->name; # Googlebot

Or call as a function to parse immediately

    my $dua = Dua->parse($ua);
    $dua->is_bot
        and say $dua->name; # Googlebot


=head1 DESCRIPTION

Dua is a User-Agent detector.


=head1 METHODS

=head2 new($user_agent_string)

constructor

=head2 name

Get User-Agent name

=head2 is_bot

Return true value if the User-Agent is bot.

=head2 is_ios

Return true value if the User-Agent is iOS.

=head2 is_android

Return true value if the User-Agent is Android.

=head2 is_linux

Return true value if the User-Agent is Linux.

=head2 is_windows

Return true value if the User-Agent is Windows.

=head2 parse

Parse User-Agent string

=head2 ua

Return User-Agent string


=head1 REPOSITORY

=begin html

<a href="https://github.com/bayashi/Dua/blob/master/LICENSE"><img src="https://img.shields.io/badge/LICENSE-Artistic%202.0-GREEN.png"></a> <a href="https://github.com/bayashi/Dua/actions"><img src="https://github.com/bayashi/Dua/workflows/build/badge.svg?_t=1575757348&branch=master"/></a> <a href="https://coveralls.io/r/bayashi/Dua"><img src="https://coveralls.io/repos/bayashi/Dua/badge.png?_t=1575757348&branch=master"/></a>

=end html

Dua is hosted on github: L<http://github.com/bayashi/Dua>

I appreciate any feedback :D


=head1 AUTHOR

Dai Okabayashi E<lt>bayashi@cpan.orgE<gt>


=head1 LICENSE

C<Dua> is free software; you can redistribute it and/or modify it under the terms of the Artistic License 2.0. (Note that, unlike the Artistic License 1.0, version 2.0 is GPL compatible by itself, hence there is no benefit to having an Artistic 2.0 / GPL disjunction.) See the file LICENSE for details.

=cut
