package WWW::Scraper::ISBN::WHSmith_Driver;

use strict;
use warnings;

use vars qw($VERSION @ISA);
$VERSION = '0.05';

#--------------------------------------------------------------------------

=head1 NAME

WWW::Scraper::ISBN::WHSmith_Driver - Search driver for the WHSmith online book catalog.

=head1 SYNOPSIS

See parent class documentation (L<WWW::Scraper::ISBN::Driver>)

=head1 DESCRIPTION

Searches for book information from the WHSmith online book catalog

=cut

#--------------------------------------------------------------------------

###########################################################################
# Inheritence

use base qw(WWW::Scraper::ISBN::Driver);

###########################################################################
# Modules

use WWW::Mechanize;

###########################################################################
# Constants

use constant	REFERER	=> 'http://www.whsmith.co.uk';
use constant	SEARCH	=> 'http://www.whsmith.co.uk/pws/doAdvancedSearch.ice?page=1&results=60&advancedCategoryId=wc_dept_books&keywordCategoryId=wc_dept_books&advanceSearch=true&title=&contributor=&format_fg=&publisher=&series=&imprint=&isbn13=';
use constant    PRODUCT => '/products/[^/]+/product/';

#--------------------------------------------------------------------------

###########################################################################
# Public Interface

=head1 METHODS

=over 4

=item C<search()>

Creates a query string, then passes the appropriate form fields to the 
WHSmith server.

The returned page should be the correct catalog page for that ISBN. If not the
function returns zero and allows the next driver in the chain to have a go. If
a valid page is returned, the following fields are returned via the book hash:

  isbn          (now returns isbn13)
  isbn10        
  isbn13
  ean13         (industry name)
  author
  title
  book_link
  image_link
  description
  pubdate
  publisher
  binding       (if known)
  pages         (if known)
  weight        (if known) (in grammes)
  width         (if known) (in millimetres)
  height        (if known) (in millimetres)

The book_link and image_link refer back to the WHSmith website.

=cut

sub search {
	my $self = shift;
	my $isbn = shift;
	$self->found(0);
	$self->book(undef);

    # validate and convert into EAN13 format
    my $ean = $self->convert_to_ean13($isbn);
    return $self->handler("Invalid ISBN specified [$isbn]")   
        if(!$ean || (length $isbn == 13 && $isbn ne $ean)
                 || (length $isbn == 10 && $isbn ne $self->convert_to_isbn10($ean)));

    $isbn = $ean;
#print STDERR "\n# isbn=[\n$isbn\n]\n";

	my $mech = WWW::Mechanize->new();
    $mech->agent_alias( 'Linux Mozilla' );
    $mech->add_header( 'Accept-Encoding' => undef );
    $mech->add_header( 'Referer' => REFERER );

    my $search = SEARCH . $isbn;
#print STDERR "\n# search=[$search]\n";
    eval { $mech->get( $search ) };
    return $self->handler("the WHSmith website appears to be unavailable.")
	    if($@ || !$mech->success() || !$mech->content());

    my $html = $mech->content();
    my $product = PRODUCT . $isbn;
#print STDERR "\n# product=[$product]\n";
#print STDERR "\n# html=[\n$html\n]\n";
    my ($link) = $html =~ /($product)/si;
	return $self->handler("Failed to find that book on the WHSmith website. [$isbn]")
		unless($link);

#print STDERR "\n# link=[$link]\n";
    eval { $mech->get( REFERER . $link ) };
    return $self->handler("the WHSmith website appears to be unavailable.")
	    if($@ || !$mech->success() || !$mech->content());

  	# The Book page
    $html = $mech->content();
	return $self->handler("Failed to find that book on the WHSmith website. [$isbn]")
		if($html =~ m!Sorry, no products were found!si);

    my $url = $mech->uri();
	return $self->handler("Failed to find that book on the WHSmith website. [$isbn]")
		if($url =~ m!Error.aspx!si);

    $html =~ s/&amp;/&/g;
    $html =~ s/&#0?39;/'/g;
    $html =~ s/&nbsp;/ /g;

#print STDERR "\n# html=[\n$html\n]\n";

    my $data;
    ($data->{isbn13})           = $html =~ m!<li itemprop="ISBN13">\s*<strong>ISBN13:</strong>\s*(.*?)</li>!si;
    ($data->{isbn10})           = $html =~ m!<li itemprop="ISBN10">\s*<strong>ISBN10:</strong>\s*(.*?)</li>!si;
    ($data->{publisher})        = $html =~ m!<span class="bold ">Publisher:\s*</span><span><a href="[^"]+" style="text-decoration:underline;">([^<]+)</a></span>!si;
    ($data->{pubdate})          = $html =~ m!<li itemprop="publication date">\s*<strong>publication date:</strong>\s*(.*?)</li>!si;
    ($data->{title})            = $html =~ m!<h1 itemprop="name" id="product_title">([^<]*)</h1>!si;
    ($data->{binding})          = $html =~ m!<div id="product_title_container">.*?<em class="secondary">([^<]+)</em></div>!si;
    ($data->{binding})          = $html =~ m!<li itemprop="Format">\s*<strong>Format:</strong>\s*(.*?)</li>!si  unless($data->{binding});
    ($data->{pages})            = $html =~ m!<li itemprop="Number Of Pages">\s*<strong>Number Of Pages:</strong>\s*(.*?)</li>!si;
    ($data->{author})           = $html =~ m!<span class="secondary"><strong>By:</strong>(.*?)</span>!si;
    ($data->{image})            = $html =~ m!<img  id='main_image' src="([^"]+)" alt="[^"]*"\s*itemprop="image" />!si;
    ($data->{thumb})            = $html =~ m!<img  id='main_image' src="([^"]+)" alt="[^"]*"\s*itemprop="image" />!si;
    ($data->{description})      = $html =~ m!<meta name="description" content="([^"]*)" />!si;

    # currently not provided
    ($data->{width})            = $html =~ m!<span class="bold ">Width:\s*</span><span>([^<]+)</span>!si;
    ($data->{height})           = $html =~ m!<span class="bold ">Height:\s*</span><span>([^<]+)</span>!si;
    ($data->{weight})           = $html =~ m!<li itemprop="weight">\s*<strong>weight:</strong>\s*(.*?)</li>!s;

    $data->{width}  = int($data->{width})   if($data->{width});
    $data->{height} = int($data->{height})  if($data->{height});
    $data->{weight} = int($data->{weight})  if($data->{weight});

    if($data->{author}) {
        $data->{author} =~ s/<[^>]*>//g;
        $data->{author} =~ s/\(author\)/ /g;
        $data->{author} =~ s/\s+/ /g;
        $data->{author} =~ s/\s+,\s+/, /g;
    }

#use Data::Dumper;
#print STDERR "\n# data=" . Dumper($data);

	return $self->handler("Could not extract data from The WHSmith result page.")
		unless(defined $data);

	# trim top and tail
	foreach (keys %$data) { 
        next unless(defined $data->{$_});
        $data->{$_} =~ s/^\s+//;
        $data->{$_} =~ s/\s+$//;
    }

	my $bk = {
		'ean13'		    => $data->{isbn13},
		'isbn13'		=> $data->{isbn13},
		'isbn10'		=> $data->{isbn10},
		'isbn'			=> $data->{isbn13},
		'author'		=> $data->{author},
		'title'			=> $data->{title},
		'book_link'		=> $url,
		'image_link'	=> $data->{image},
		'thumb_link'	=> $data->{thumb},
		'description'	=> $data->{description},
		'pubdate'		=> $data->{pubdate},
		'publisher'		=> $data->{publisher},
		'binding'	    => $data->{binding},
		'pages'		    => $data->{pages},
		'weight'		=> $data->{weight},
		'width'		    => $data->{width},
		'height'		=> $data->{height},
        'html'          => $html
	};

#use Data::Dumper;
#print STDERR "\n# book=".Dumper($bk);

    $self->book($bk);
	$self->found(1);
	return $self->book;
}

=item C<convert_to_ean13()>

Given a 10/13 character ISBN, this function will return the correct 13 digit
ISBN, also known as EAN13.

=item C<convert_to_isbn10()>

Given a 10/13 character ISBN, this function will return the correct 10 digit 
ISBN.

=back

=cut

sub convert_to_ean13 {
	my $self = shift;
    my $isbn = shift;
    my $prefix;

    return  unless(length $isbn == 10 || length $isbn == 13);

    if(length $isbn == 13) {
        return  if($isbn !~ /^(978|979)(\d{10})$/);
        ($prefix,$isbn) = ($1,$2);
    } else {
        return  if($isbn !~ /^(\d{10}|\d{9}X)$/);
        $prefix = '978';
    }

    my $isbn13 = '978' . $isbn;
    chop($isbn13);
    my @isbn = split(//,$isbn13);
    my ($lsum,$hsum) = (0,0);
    while(@isbn) {
        $hsum += shift @isbn;
        $lsum += shift @isbn;
    }

    my $csum = ($lsum * 3) + $hsum;
    $csum %= 10;
    $csum = 10 - $csum  if($csum != 0);

    return $isbn13 . $csum;
}

sub convert_to_isbn10 {
	my $self = shift;
    my $ean  = shift;
    my ($isbn,$isbn10);

    return  unless(length $ean == 10 || length $ean == 13);

    if(length $ean == 13) {
        return  if($ean !~ /^(?:978|979)(\d{9})\d$/);
        ($isbn,$isbn10) = ($1,$1);
    } else {
        return  if($ean !~ /^(\d{9})[\dX]$/);
        ($isbn,$isbn10) = ($1,$1);
    }

	return  if($isbn < 0 or $isbn > 999999999);

	my ($csum, $pos, $digit) = (0, 0, 0);
    for ($pos = 9; $pos > 0; $pos--) {
        $digit = $isbn % 10;
        $isbn /= 10;             # Decimal shift ISBN for next time 
        $csum += ($pos * $digit);
    }
    $csum %= 11;
    $csum = 'X'   if ($csum == 10);
    return $isbn10 . $csum;
}

1;

__END__

=head1 REQUIRES

Requires the following modules be installed:

L<WWW::Scraper::ISBN::Driver>,
L<WWW::Mechanize>

=head1 SEE ALSO

L<WWW::Scraper::ISBN>,
L<WWW::Scraper::ISBN::Record>,
L<WWW::Scraper::ISBN::Driver>

=head1 BUGS, PATCHES & FIXES

There are no known bugs at the time of this release. However, if you spot a
bug or are experiencing difficulties that are not explained within the POD
documentation, please send an email to barbie@cpan.org or submit a bug to the
RT system (http://rt.cpan.org/Public/Dist/Display.html?Name=WWW-Scraper-ISBN-WHSmith_Driver).
However, it would help greatly if you are able to pinpoint problems or even
supply a patch.

Fixes are dependent upon their severity and my availability. Should a fix not
be forthcoming, please feel free to (politely) remind me.

=head1 AUTHOR

  Barbie, <barbie@cpan.org>
  Miss Barbell Productions, <http://www.missbarbell.co.uk/>

=head1 COPYRIGHT & LICENSE

  Copyright (C) 2010-2012 Barbie for Miss Barbell Productions

  This distribution is free software; you can redistribute it and/or
  modify it under the Artistic Licence v2.

=cut
