package P::Src::File;

use P qw[-class];
use Term::ANSIColor qw[:constants];

has action => ( is => 'ro', isa => Enum [qw[decompress compress obfuscate]], required => 1 );
has path => ( is => 'ro', isa => Str | InstanceOf ['P::Util::File::Path'], required => 1 );
has in_buffer => ( is => 'lazy', isa => ScalarRef, predicate => 1 );
has decode    => ( is => 'ro',   isa => Bool,      default   => 0 );
has dry_run   => ( is => 'ro',   isa => Bool,      default   => 0 );
has filter_args => ( is => 'ro', isa => HashRef );

has out_buffer  => ( is => 'lazy', isa => ScalarRef, init_arg => undef );
has is_binary   => ( is => 'lazy', isa => Bool,      init_arg => undef );
has was_changed => ( is => 'lazy', isa => Bool,      init_arg => undef );
has severity    => ( is => 'rw',   isa => Int,       default  => 0, init_arg => undef );
has severity_range => ( is => 'lazy', isa => Enum [ keys P::Src::File->cfg->{SEVERITY_RANGE}->%* ], init_arg => undef );

has _can_write => ( is => 'rw', isa => Bool, default => 0, init_arg => undef );
has _in_size  => ( is => 'lazy', isa => Int, init_arg => undef );
has _in_md5   => ( is => 'lazy', isa => Str, init_arg => undef );
has _out_size => ( is => 'lazy', isa => Int, init_arg => undef );
has _out_md5  => ( is => 'lazy', isa => Str, init_arg => undef );

# CLASS METHODS
sub cfg ($self) {
    state $cfg = P->cfg->load( $P->{SHARE_DIR} . 'src.perl' );

    return $cfg;
}

sub detect_filetype ( $self, $path, $buffer_ref = undef ) {
    $path = P->file->path($path);

    my $type;

    # detect file type by extension
    for my $t ( P::Src::File->cfg->{FILE_TYPE}->@* ) {
        if ( $path->filename =~ $t->{re} ) {
            $type = $t;

            last;
        }
    }

    # detect by shebang, only if filename has no extension
    if ( !$type && !$path->suffix ) {

        # read first 50 bytes
        if ( !$buffer_ref ) {
            P->file->read_bin(
                $path,
                buf_size => 50,
                cb       => sub {
                    $buffer_ref = $_[0] if defined $_[0];

                    return;
                }
            );
        }

        # if has shebang
        if ( $buffer_ref && index( $buffer_ref->$*, q[#!], 0 ) == 0 ) {
            my $shebang = substr $buffer_ref->$*, 0, index( $buffer_ref->$*, qq[\n], 0 );

            for my $t ( grep { exists $_->{shebang_re} } P::Src::File->cfg->{FILE_TYPE}->@* ) {
                if ( $shebang =~ $t->{shebang_re} ) {
                    $type = $t;

                    last;
                }
            }
        }
    }

    return $type;
}

# METHODS
sub _build_in_buffer ($self) {
    my $res;

    try {
        $res = P->file->read_bin( $self->path );

        $self->_can_write(1);
    }
    catch {
        $self->severity( P::Src::File->cfg->{SEVERITY}->{OPEN} );

        $self->_can_write(0);

        $res = \q[];
    };

    return $res;
}

sub _build_is_binary ($self) {
    if ( !$self->path ) {
        return 0;
    }
    else {
        return -B $self->path ? 1 : 0;
    }
}

sub _build_was_changed ($self) {
    return $self->_in_md5 eq $self->_out_md5 ? 0 : 1;
}

sub _build__in_size ($self) {
    return bytes::length( $self->in_buffer->$* );
}

sub _build__out_size ($self) {
    return bytes::length( $self->out_buffer->$* );
}

sub _build__in_md5 ($self) {
    return P->digest->md5_hex( $self->in_buffer );
}

sub _build__out_md5 ($self) {
    return P->digest->md5_hex( $self->out_buffer );
}

sub _build_out_buffer ($self) {
    my $buffer;

    if ( $self->has_in_buffer ) {
        $buffer = $self->in_buffer->$*;
    }
    else {
        # check if buffer is binary
        if ( $self->is_binary ) {
            $self->severity( P::Src::File->cfg->{SEVERITY}->{BINARY} );

            return $self->in_buffer;
        }

        # try to read file
        $buffer = $self->in_buffer->$*;

        # return, if has reading errors
        return $self->in_buffer if $self->severity;
    }

    if ( $self->decode ) {
        require Encode::Guess;

        # detect buffer encoding
        my $decoder = Encode::Guess::guess_encoding($buffer);

        $decoder = Encode::Guess::guess_encoding( $buffer, P::Src::File->cfg->{DEFAULT_GUESS_ENCODING}->@* ) unless ref $decoder;

        # appropriate encoding wasn't found
        unless ( ref $decoder ) {
            $self->severity( P::Src::File->cfg->{SEVERITY}->{ENCODING} );

            return $self->in_buffer;
        }

        # try to decode buffer
        try {
            $buffer = $decoder->decode( $buffer, Encode::FB_CROAK );
        }
        catch {
            my $e = shift;

            $self->severity( P::Src::File->cfg->{SEVERITY}->{ENCODING} );
        };

        return $self->in_buffer if $@;

        P->file->remove_bom($buffer);

        P->text->encode_utf8($buffer);
    }

    # detect filetype, require and run filter
    if ( my $type = $self->detect_filetype( $self->path, \$buffer ) ) {
        my $method = $self->action;

        my $filter_args = $type->{filter_args} // {};

        P->hash->merge( $filter_args, $self->filter_args ) if $self->filter_args;

        $self->severity( P->class->load( $type->{type}, ns => 'P::Src::Filter' )->new( { buffer => \$buffer } )->$method( $filter_args->%* ) );
    }

    if ( $self->action eq 'decompress' ) {

        # clean buffer
        P->text->decode_eol($buffer);    # decode CRLF to internal \n representation

        P->text->lcut_all($buffer);      # trim leading horizontal whitespaces

        P->text->rcut_all($buffer);      # trim trailing horizontal whitespaces

        P->text->rtrim_multi($buffer);   # right trim each line

        $buffer =~ s/\t/    /smg;        # convert tabs to spaces

        $buffer .= $LF;
    }

    return \$buffer;
}

sub _build_severity_range ($self) {
    for my $range ( reverse sort { P::Src::File->cfg->{SEVERITY_RANGE}->{$a} <=> P::Src::File->cfg->{SEVERITY_RANGE}->{$b} } keys P::Src::File->cfg->{SEVERITY_RANGE}->%* ) {
        if ( $self->severity >= P::Src::File->cfg->{SEVERITY_RANGE}->{$range} ) {
            return $range;
        }
    }

    return 'VALID';
}

sub severity_range_is ( $self, $range ) {
    return $self->severity_range eq uc $range;
}

sub run ($self) {
    $self->out_buffer;

    # write file, if it was physically read from disk
    if ( $self->_can_write && !$self->dry_run && $self->was_changed ) {
        P->file->write_bin( $self->path, $self->out_buffer );
    }

    return $self;
}

1;
## -----SOURCE FILTER LOG BEGIN-----
##
## PerlCritic profile "pcore-script" policy violations:
## ┌──────┬──────────────────────┬────────────────────────────────────────────────────────────────────────────────────────────────────────────────┐
## │ Sev. │ Lines                │ Policy                                                                                                         │
## ╞══════╪══════════════════════╪════════════════════════════════════════════════════════════════════════════════════════════════════════════════╡
## │    3 │ 17, 189, 212         │ References::ProhibitDoubleSigils - Double-sigil dereference                                                    │
## ├──────┼──────────────────────┼────────────────────────────────────────────────────────────────────────────────────────────────────────────────┤
## │    1 │ 64                   │ CodeLayout::ProhibitParensWithBuiltins - Builtin function called with parentheses                              │
## └──────┴──────────────────────┴────────────────────────────────────────────────────────────────────────────────────────────────────────────────┘
##
## -----SOURCE FILTER LOG END-----
__END__
=pod

=encoding utf8

=cut
