package Pcore::API::Google::Search;

use Pcore -class;
use Pcore::API::Google;

has max_threads => ( is => 'ro', isa => PositiveInt, default => 1 );
has antigate_key => ( is => 'ro', isa => Str );

has _ua => ( is => 'lazy', isa => InstanceOf ['Pcore::HTTP::Request'], init_arg => undef );
has _antigate => ( is => 'lazy', isa => Maybe [ InstanceOf ['Pcore::API::Antigate'] ], init_arg => undef );
has _threads => ( is => 'ro', isa => PositiveOrZeroInt, default => 0, init_arg => undef );
has _req_pool => ( is => 'ro', isa => ArrayRef, default => sub { [] }, init_arg => undef );
has _captcha_in_progress => ( is => 'ro', isa => Bool, default => 0, init_arg => undef );

sub _build__ua ($self) {
    return P->http->request(
        method     => 'GET',
        cookie_jar => 1,
        useragent  => 'Links (2.1; Linux 2.6.18-gentoo-r6 x86_64; 80x24)',
    );
}

sub _build__antigate ($self) {
    if ( $self->antigate_key ) {
        require Pcore::API::Antigate;

        return Pcore::API::Antigate->new( { api_key => $self->antigate_key } );
    }

    return;
}

sub search ( $self, $query, $cb ) {
    my $url = 'https://www.google.com/search?gws_rd=cr&num=100&q=' . P->data->to_uri($query);

    $self->_request(
        $url,
        sub ($res) {
            $cb->($res);

            return;
        }
    );

    return;
}

sub _request ( $self, $url, $cb ) {
    if ( $self->{_threads} >= $self->max_threads ) {
        push $self->{_req_pool}->@*, [ $url, $cb ];

        return;
    }

    $self->{_threads}++;

    my $on_finish = sub ($res) {
        $self->{_threads}--;

        $cb->($res);

        while (1) {
            last if $self->{_threads} >= $self->max_threads;

            if ( my $next_req = shift $self->{_req_pool}->@* ) {
                $self->_request( $next_req->@* );
            }
            else {
                last;
            }
        }

        return;
    };

    $self->_ua->run(
        url       => $url,
        on_finish => sub ($res) {
            if ( $res->status == 503 ) {    # captcha
                if ( !$self->_antigate ) {
                    $on_finish->($res);
                }
                else {

                    # store request in the pool if captcha is solved now
                    if ( $self->{_captcha_in_progress} ) {
                        $self->{_threads}--;

                        unshift $self->{_req_pool}->@*, [ $url, $cb ];
                    }
                    else {
                        $self->{_captcha_in_progress} = 1;

                        $self->_resolve_captcha(
                            $url, $res,
                            sub ($res) {
                                $self->{_captcha_in_progress} = 0;

                                $on_finish->($res);

                                return;
                            }
                        );
                    }
                }
            }
            else {
                $on_finish->($res);
            }

            return;
        }
    );

    return;
}

sub _resolve_captcha ( $self, $url, $res, $cb ) {
    my $base_url = $res->url;

    my ($id) = $res->body->$* =~ m[name="id" value="(\d+)"]sm;

    my ($image_url) = $res->body->$* =~ m[<img src="(/sorry/image.+?)"]sm;

    $image_url =~ s/&amp;/&/smg;

    # get captcha image
    $self->_ua->run(
        url       => P->uri( $image_url, base => $base_url ),
        on_finish => sub ($img_res) {

            # resolve captcha
          RESOLVE_CAPTCHA:
            $self->_antigate->resolve(
                captcha => $img_res->body,
                sub ($captcha) {
                    if ( !$captcha->is_success ) {
                        $self->_resolve_captcha( $url, $res, $cb );

                        return;
                    }

                    my $query = P->data->to_uri(
                        {   continue => $url,
                            id       => $id,
                            submit   => 'Submit',
                            captcha  => $captcha->result
                        }
                    );

                    # enter captchs
                    $self->_ua->run(
                        url       => P->uri( '/sorry/CaptchaRedirect?' . $query, base => $base_url ),
                        on_finish => sub ($res) {

                            # captcha recognized incorrectly
                            if ( $res->status == 503 ) {
                                $self->_antigate->report_failure($captcha);

                                $self->_resolve_captcha( $url, $res, $cb );
                            }

                            # captcha is valid
                            else {
                                $cb->($res);
                            }

                            return;
                        }
                    );

                    return;
                }
            );

            return;
        },
    );

    return;
}

1;
__END__
=pod

=encoding utf8

=head1 NAME

Pcore::API::Google::Search

=head1 SYNOPSIS

=head1 DESCRIPTION

=head1 ATTRIBUTES

=head1 METHODS

=head1 SEE ALSO

=cut
