#
# This file is part of Dist-Zilla-Plugin-SubmittingPatches
#
# This software is Copyright (c) 2010 by Caleb Cushing <xenoterracide@gmail.com>.
#
# This is free software, licensed under:
#
#   The Artistic License 2.0
#
use strict;
use warnings;

package Dist::Zilla::Plugin::SubmittingPatches;
BEGIN {
  $Dist::Zilla::Plugin::SubmittingPatches::VERSION = '0.01';
}
use Moose;
use namespace::autoclean;
extends 'Dist::Zilla::Plugin::InlineFiles';
with 'Dist::Zilla::Role::TextTemplate';

around add_file => sub {
	my ( $orig, $self, $file ) = @_;
	return $self->$orig(
		Dist::Zilla::File::InMemory->new({
			name => $file->name,
			content => $self->fill_in_string(
				$file->content,
			),
		}),
	);
};

__PACKAGE__->meta->make_immutable;
1;
# ABSTRACT: Add SubmittingPatches documentation




=pod

=head1 NAME

Dist::Zilla::Plugin::SubmittingPatches - Add SubmittingPatches documentation

=head1 VERSION

version 0.01

=head1 SYNOPSIS

In C<dist.ini>:

	[SubmittingPatches]

=head1 DESCRIPTION

add a patch submission policy to your repository similar to that of git.git

=head1 BUGS

Please report any bugs or feature requests on
L<http://github.com/xenoterracide/Dist-Zilla-Plugin-SubmittingPatches/issues>
as I'm not fond of RT.

=head1 SUBMITTING PATCHES

Please read the SubmittingPatches file included with this Distribution. Patches
that are of sufficient quality, within the goals of the project and pass the
checklist will probably be accepted.

=head1 AUTHOR

  Caleb Cushing <xenoterracide@gmail.com>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2010 by Caleb Cushing <xenoterracide@gmail.com>.

This is free software, licensed under:

  The Artistic License 2.0

=cut


__DATA__
__[ SubmittingPatches ]__
Checklist (and a short version for the impatient):

	Commits:

	- make commits of logical units
	- check for unnecessary whitespace with "git diff --check"
	  before committing
	- do not check in commented out code or unneeded files
	- the first line of the commit message should be a short
	  description and should skip the full stop
	- the body should provide a meaningful commit message, which:
		- uses the imperative, present tense: "change",
		  not "changed" or "changes".
		- includes motivation for the change, and contrasts
		  its implementation with previous behaviour
	- if you want your work included in the main repository, add a
	  "Signed-off-by: Your Name <you@example.com>" line to the
	  commit message (or just use the option "-s" when
	  committing) to confirm that you agree to the Developer's
	  Certificate of Origin
	- make sure that you have tests for the bug you are fixing
	- make sure that the test suite passes after your commit

	Patch:

	- use "git format-patch -M" to create the patch
	- do not PGP sign your patch
	- be careful doing cut & paste, not to corrupt whitespaces.
	- provide additional information (which is unsuitable for
	  the commit message) between the "---" and the diffstat
	- if you change, add, or remove any features or
	  make some other user interface change, the associated
	  documentation should be updated as well.
	- if your name is not writable in ASCII, make sure that
	  you send the patch in the correct encoding.

Long version:

I started reading over the SubmittingPatches document for git,
primarily because I wanted to have a document similar to it for
my projects to make sure people understand what they are doing
when they write "Signed-off-by" line.

But the patch submission requirements are a lot more relaxed
here on the technical/contents front, because my projects are
thousand times smaller ;-).  So here is only the relevant bits.

(0) Decide what to base your work on.

In general, always base your work on the oldest branch that your
change is relevant to.

 - A bugfix should be based on 'maint' in general. If the bug is not
   present in 'maint', base it on 'master'. For a bug that's not yet
   in 'master', find the topic that introduces the regression, and
   base your work on the tip of the topic.

 - A new feature should be based on 'master' in general. If the new
   feature depends on a topic that is in 'pu', but not in 'master',
   base your work on the tip of that topic.

 - Corrections and enhancements to a topic not yet in 'master' should
   be based on the tip of that topic. If the topic has not been merged
   to 'next', it's alright to add a note to squash minor corrections
   into the series.

 - In the exceptional case that a new feature depends on several topics
   not in 'master', start working on 'next' or 'pu' privately and send
   out patches for discussion. Before the final merge, you may have to
   wait until some of the dependent topics graduate to 'master', and
   rebase your work.

To find the tip of a topic branch, run "git log --first-parent
master..pu" and look for the merge commit. The second parent of this
commit is the tip of the topic branch.

(1) Make separate commits for logically separate changes.

Unless your patch is really trivial, you should not be sending
out a patch that was generated between your working tree and
your commit head.  Instead, always make a commit with complete
commit message and generate a series of patches from your
repository.  It is a good discipline.

Describe the technical detail of the change(s).

If your description starts to get too long, that's a sign that you
probably need to split up your commit to finer grained pieces.
That being said, patches which plainly describe the things that
help reviewers check the patch, and future maintainers understand
the code, are the most beautiful patches.  Descriptions that summarise
the point in the subject well, and describe the motivation for the
change, the approach taken by the change, and if relevant how this
differs substantially from the prior version, can be found on Usenet
archives back into the late 80's.  Consider it like good Netiquette,
but for code.

Oh, another thing.  I am picky about whitespaces.  Make sure your
changes do not trigger errors with the sample pre-commit hook shipped
in templates/hooks--pre-commit.  To help ensure this does not happen,
run git diff --check on your changes before you commit.

(2) Generate your patch using git tools out of your commits.

git based diff tools (git, Cogito, and StGIT included) generate
unidiff which is the preferred format.

You do not have to be afraid to use -M option to "git diff" or
"git format-patch", if your patch involves file renames.  The
receiving end can handle them just fine.

Please make sure your patch does not include any extra files
which do not belong in a patch submission.  Make sure to review
your patch after generating it, to ensure accuracy.  Before
sending out, please make sure it cleanly applies to the "master"
branch head.  If you are preparing a work based on "next" branch,
that is fine, but please mark it as such.


(3) Sending your patches.

People need to be able to read and comment on the changes you are
submitting.  Do not cut-n-paste your patch; you can lose
tabs that way if you are not careful.

It is a common convention to prefix your subject line with
[PATCH].  This lets people easily distinguish patches from other
e-mail discussions.  Use of additional markers after PATCH and
the closing bracket to mark the nature of the patch is also
encouraged.  E.g. [PATCH/RFC] is often used when the patch is
not ready to be applied but it is for discussion, [PATCH v2],
[PATCH v3] etc. are often seen when you are sending an update to
what you have previously sent.

You often want to add additional explanation about the patch,
other than the commit message itself.  Place such "cover letter"
material between the three dash lines and the diffstat.

Do not PGP sign your patch, at least for now.  Most likely, your
maintainer or other people on the list would not have your PGP
key and would not bother obtaining it anyway.  Your patch is not
judged by who you are; a good patch from an unknown origin has a
far better chance of being accepted than a patch from a known,
respected origin that is done poorly or does incorrect things.

If you really really really really want to do a PGP signed
patch, format it as "multipart/signed", not a text/plain message
that starts with '-----BEGIN PGP SIGNED MESSAGE-----'.  That is
not a text/plain, it's something else.

Unless your patch is a very trivial and an obviously correct one,
first send it with "To:" set to the mailing list, with "cc:" listing
people who are involved in the area you are touching (the output from
"git blame $path" and "git shortlog --no-merges $path" would help to
identify them), to solicit comments and reviews.  After the list
reached a consensus that it is a good idea to apply the patch, re-send
it with "To:" set to the maintainer and optionally "cc:" the list for
inclusion.  Do not forget to add trailers such as "Acked-by:",
"Reviewed-by:" and "Tested-by:" after your "Signed-off-by:" line as
necessary.


(4) Sign your work

To improve tracking of who did what, we've borrowed the
"sign-off" procedure from the Linux kernel project on patches
that are being emailed around.  Although this project is a lot
smaller it is a good discipline to follow it.

The sign-off is a simple line at the end of the explanation for
the patch, which certifies that you wrote it or otherwise have
the right to pass it on as a open-source patch.  The rules are
pretty simple: if you can certify the below:

        Developer's Certificate of Origin 1.1

        By making a contribution to this project, I certify that:

        (a) The contribution was created in whole or in part by me and I
            have the right to submit it under the open source license
            indicated in the file; or

        (b) The contribution is based upon previous work that, to the best
            of my knowledge, is covered under an appropriate open source
            license and I have the right under that license to submit that
            work with modifications, whether created in whole or in part
            by me, under the same open source license (unless I am
            permitted to submit under a different license), as indicated
            in the file; or

        (c) The contribution was provided directly to me by some other
            person who certified (a), (b) or (c) and I have not modified
            it.

        (d) I understand and agree that this project and the contribution
            are public and that a record of the contribution (including all
            personal information I submit with it, including my sign-off) is
            maintained indefinitely and may be redistributed consistent with
            this project or the open source license(s) involved.

then you just add a line saying

	Signed-off-by: Random J Developer <random@developer.example.org>

This line can be automatically added by git if you run the git-commit
command with the -s option.

Notice that you can place your own Signed-off-by: line when
forwarding somebody else's patch with the above rules for
D-C-O.  Indeed you are encouraged to do so.

Also notice that a real name is used in the Signed-off-by: line. Please
don't hide your real name.

Some people also put extra tags at the end.

"Acked-by:" says that the patch was reviewed by the person who
is more familiar with the issues and the area the patch attempts
to modify.  "Tested-by:" says the patch was tested by the person
and found to have the desired effect.

------------------------------------------------
An ideal patch flow

Here is an ideal patch flow for this project the current maintainer
suggests to the contributors:

 (0) You come up with an itch.  You code it up.

 (1) Send it to the bug tracker and cc people who may need to know about
     the change.

     The people who may need to know are the ones whose code you
     are butchering.  These people happen to be the ones who are
     most likely to be knowledgeable enough to help you, but
     they have no obligation to help you (i.e. you ask for help,
     don't demand).  "git log -p -- $area_you_are_modifying" would
     help you find out who they are.

 (2) You get comments and suggestions for improvements.  You may
     even get them in a "on top of your change" patch form.

 (3) Polish, refine, and re-send to the the people who spend their
     time to improve your patch.  Go back to step (2).

 (4) A topic branch is created with the patch and is merged to 'next',
     and cooked further and eventually graduates to 'master'.

In any time between the (2)-(3) cycle, the maintainer may pick it up
from the list and queue it to 'pu', in order to make it easier for
people play with it without having to pick up and apply the patch to
their trees themselves.

------------------------------------------------
Know the status of your patch after submission

* You can use Git itself to find out when your patch is merged in
  master. 'git pull --rebase' will automatically skip already-applied
  patches, and will let you know. This works only if you rebase on top
  of the branch in which your patch has been merged (i.e. it will not
  tell you if your patch is merged in pu if you rebase on top of
  master).
