package List::MoreUtils;

use 5.00503;
use strict;

require Exporter;
require DynaLoader;
use vars qw($VERSION @ISA @EXPORT_OK %EXPORT_TAGS);
@ISA = qw(Exporter DynaLoader);

%EXPORT_TAGS = ( 
    all => [ qw(any all none notall true false firstidx lastidx last_index insert_after insert_after_string apply
		after after_incl before before_incl indexes firstval first_value lastval last_value each_array
		pairwise natatime mesh zip uniq) ],
);

@EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );

$VERSION = '0.07';

eval {
    local $ENV{PERL_DL_NONLAZY} = 0 if $ENV{PERL_DL_NONLAZY};
    bootstrap List::MoreUtils $VERSION;
    1;
};


eval <<'EOP' if not defined &any;

sub any (&@) {
    my $f = shift;
    return if ! @_;
    for (@_) {
	return 1 if $f->();
    }
    return 0;
}
    
sub all (&@) {
    my $f = shift;
    return if ! @_;
    for (@_) {
	return 0 if ! $f->();
    }
    return 1;
}

sub none (&@) {
    my $f = shift;
    return if ! @_;
    for (@_) {
	return 0 if $f->();
    }
    return 1;
}

sub notall (&@) {
    my $f = shift;
    return if ! @_;
    for (@_) {
	return 1 if ! $f->();
    }
    return 0;
}

sub true (&@) {
    my $f = shift;
    my $count = 0;
    for (@_) {
	$count++ if $f->();
    }
    return $count;
}

sub false (&@) {
    my $f = shift;
    my $count = 0;
    for (@_) {
	$count++ if ! $f->();
    }
    return $count;
}

sub firstidx (&@) {
    my $f = shift;
    local *_;
    for my $i (0 .. $#_) {
	$_ = $_[$i];	
	return $i if $f->();
    }
    return -1;
}

sub lastidx (&@) {
    my $f = shift;
    local *_;
    for my $i (reverse 0 .. $#_) {
	$_ = $_[$i];
	return $i if $f->();
    }
    return -1;
}

sub insert_after (&$\@) {
    my ($code, $val, $list) = @_;
    my $c = -1;
    local *_;
    for my $i (0 .. $#$list) {
	$_ = $list->[$i];
	$c = $i, last if $code->();
    }
    @$list = (@{$list}[0..$c], $val, @{$list}[$c+1..$#$list]) and return 1 if $c != -1;
    return 0;
}

sub insert_after_string ($$\@) {
    my ($string, $val, $list) = @_;
    my $c = -1;
    for my $i (0 .. $#$list) {
	local $^W = 0;
	$c = $i, last if $string eq $list->[$i];
    }
    @$list = (@{$list}[0..$c], $val, @{$list}[$c+1..$#$list]) and return 1 if $c != -1;
    return 0;
}

sub apply (&@) {
    my $action = shift;
    &$action for my @values = @_;
    wantarray ? @values : $values[-1];
}

sub after (&@)
{
    my $test = shift;
    my $started;
    my $lag;
    grep $started ||= do { my $x=$lag; $lag=$test->(); $x},  @_;
}

sub after_incl (&@)
{
    my $test = shift;
    my $started;
    grep $started ||= $test->(), @_;
}

sub before (&@)
{
    my $test = shift;
    my $keepgoing=1;
    grep $keepgoing &&= !$test->(),  @_;
}

sub before_incl (&@)
{
    my $test = shift;
    my $keepgoing=1;
    my $lag=1;
    grep $keepgoing &&= do { my $x=$lag; $lag=!$test->(); $x},  @_;
}

sub indexes (&@)
{
    my $test = shift;
    local *_;
    grep {local $_=$_[$_]; $test->()} 0..$#_;
}

sub lastval (&@)
{
    my $test = shift;
    local *_;
    my $ix;
    for ($ix=$#_; $ix>=0; $ix--)
    {
        $_ = $_[$ix];
        my $testval = $test->();
        $_[$ix] = $_;    # simulate $_ as alias
        return $_ if $testval;
    }
    return undef;
}

sub firstval (&@)
{
    my $test = shift;
    foreach (@_)
    {
        return $_ if $test->();
    }
    return undef;
}

sub pairwise(&\@\@)
{
    my $op = shift;
    use vars qw/@A @B/;
    local (*A, *B) = @_;    # syms for caller's input arrays

    # Localise $a, $b
    my ($caller_a, $caller_b) = do
    {
        my $pkg = caller();
        no strict 'refs';
        \*{$pkg.'::a'}, \*{$pkg.'::b'};
    };

    my $limit = $#A > $#B? $#A : $#B;    # loop iteration limit

    local(*$caller_a, *$caller_b);
    map    # This map expression is also the return value.
    {
        # assign to $a, $b as refs to caller's array elements
        (*$caller_a, *$caller_b) = \($A[$_], $B[$_]);
        $op->();    # perform the transformation
    }  0 .. $limit;
}

sub each_array (\@;\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@)
{
    return each_arrayref(@_);
}

sub each_arrayref
{
    my @arr_list  = @_;     # The list of references to the arrays
    my $index     = 0;      # Which one the caller will get next
    my $max_num   = 0;      # Number of elements in longest array

    # Get the length of the longest input array
    foreach (@arr_list)
    {
        unless (ref($_) eq 'ARRAY')
        {
            require Carp;
            Carp::croak "each_arrayref: argument is not an array reference\n";
        }
        $max_num = @$_  if @$_ > $max_num;
    }

    # Return the iterator as a closure wrt the above variables.
    return sub
    {
        if (@_)
        {
            my $method = shift;
            if ($method eq 'index')
            {
                # Return current (last fetched) index
                return undef if $index == 0  ||  $index > $max_num;
                return $index-1;
            }
            else
            {
                require Carp;
                Carp::croak "each_array: unknown argument '$method' passed to iterator.";
            }
        }

        return if $index >= $max_num;     # No more elements to return
        my $i = $index++;
        return map $_->[$i], @arr_list;   # Return ith elements
    }
}

sub natatime ($@)
{
    my $n = shift;
    my @list = @_;

    return sub
    {
        return splice @list, 0, $n;
    }
}

sub mesh (\@\@;\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@\@) {
    my $max = -1;
    $max < $#$_  &&  ($max = $#$_)  for @_;

    map { my $ix = $_; map $_->[$ix], @_; } 0..$max; 
}
EOP

*first_index = \&firstidx;
*last_index = \&lastidx;
*first_value = \&firstval;
*last_value = \&lastval;
*zip = \&mesh;

1;
__END__

=head1 NAME

List::MoreUtils - Provide the stuff missing in List::Util

=head1 SYNOPSIS

    use List::MoreUtils qw(any all none notall true false firstidx 
                           lastidx insert_after insert_after_string
			   apply after after_incl before before_incl
			   indexes lastval firstval pairwise each_array
			   natatime mesh);

=head1 DESCRIPTION

C<List::MoreUtils> provides some trivial but commonly needed functionality on lists
which is not going to go into C<List::Util>.

All of the below functions are implementable in only a couple of lines of Perl
code. Using the functions from this module however should give slightly better
performance as everything is implemented in C. The pure-Perl implementation of
these functions only serves as a fallback in case the C portions of this module
couldn't be compiled on this machine.

=over 4

=item any BLOCK LIST

Returns a true value if any item in LIST meets the criterion given through
BLOCK. Sets C<$_> for each item in LIST in turn:

    print "At least one value undefined"
        if any { !defined($_) } @list;

Returns false otherwise, or C<undef> if LIST is empty.

=item all BLOCK LIST

Returns a true value if all items in LIST meet the criterion given through
BLOCK. Sets C<$_> for each item in LIST in turn:

    print "All items defined"
        if all { defined($_) } @list;

Returns false otherwise, or C<undef> if LIST is empty.

=item none BLOCK LIST

Logically the negation of C<any>. Returns a true value if no item in LIST meets the
criterion given through BLOCK. Sets C<$_> for each item in LIST in turn:

    print "No value defined"
        if none { defined($_) } @list;

Returns false otherwise, or C<undef> if LIST is empty.

=item notall BLOCK LIST

Logically the negation of C<all>. Returns a true value if not all items in LIST meet
the criterion given through BLOCK. Sets C<$_> for each item in LIST in turn:

    print "Not all values defined"
        if notall { defined($_) } @list;

Returns false otherwise, or C<undef> if LIST is empty.

=item true BLOCK LIST

Counts the number of elements in LIST for which the criterion in BLOCK is true. Sets C<$_> for 
each item in LIST in turn:

    printf "%i item(s) are defined", true { defined($_) } @list;

=item false BLOCK LIST

Counts the number of elements in LIST for which the criterion in BLOCK is false. Sets C<$_> for
each item in LIST in turn:

    printf "%i item(s) are not defined", false { defined($_) } @list;

=item firstidx BLOCK LIST

=item first_index BLOCK LIST

Returns the index of the first element in LIST for which the criterion in BLOCK is true. Sets C<$_>
for each item in LIST in turn:

    my @list = (1, 4, 3, 2, 4, 6);
    printf "item with index %i in list is 4", firstidx { $_ == 4 } @list;
    __END__
    item with index 1 in list is 4
    
Returns C<-1> if no such item could be found.

C<first_index> is an alias for C<firstidx>.

=item lastidx BLOCK LIST

=item list_index BLOCK LIST

Returns the index of the last element in LIST for which the criterion in BLOCK is true. Sets C<$_>
for each item in LIST in turn:

    my @list = (1, 4, 3, 2, 4, 6);
    printf "item with index %i in list is 4", lastidx { $_ == 4 } @list;
    __END__
    item with index 4 in list is 4

Returns C<-1> if no such item could be found.

C<last_index> is an alias for C<lastidx>.

=item insert_after BLOCK VALUE LIST

Inserts VALUE after the first item in LIST for which the criterion in BLOCK is true. Sets C<$_> for
each item in LIST in turn.

    my @list = qw/This is a list/;
    insert_after { $_ eq "a" } "longer" => @list;
    print "@list";
    __END__
    This is a longer list

=item insert_after_string STRING VALUE LIST

Inserts VALUE after the first item in LIST which is equal to STRING. 

    my @list = qw/This is a list/;
    insert_after_string "a", "longer" => @list;
    print "@list";
    __END__
    This is a longer list

=item apply BLOCK LIST

Applies BLOCK to each item in LIST and returns a list of the values after BLOCK
has been applied. In scalar context, the last element is returned.  This
function is similar to C<map> but will not modify the elements of the input
list:

    my @list = (1 .. 4);
    my @mult = apply { $_ *= 2 } @list;
    print "\@list = @list\n";
    print "\@mult = @mult\n";
    __END__
    @list = 1 2 3 4
    @mult = 2 4 6 8

Think of it as syntactic sugar for

    for (my @mult = @list) { $_ *= 2 }

=item after BLOCK LIST

Returns a list of the values of LIST after (and not including) the point
where CODE returns a true value. Sets C<$_> for each element in LIST in turn.

    @x = after { $_ % 5 == 0 } (1..9);    # returns 6, 7, 8, 9

=item after_incl BLOCK LIST

Same as C<after> but also inclues the element for which BLOCK is true.

=item before BLOCK LIST

Returns a list of values of LIST upto (and not including) the point where CODE
returns a true value. Sets C<$_> for each element in LIST in turn.

=item before_incl BLOCK LIST

Same as C<before> but also includes the element for which BLOCK is true.

=item indexes BLOCK LIST

Evaluates BLOCK for each element in LIST (assigned to C<$_>) and returns a list
of the indices of those elements for which BLOCK returned a true value. This is
just like C<grep> only that it returns indices instead of values:

    @x = indexes { $_ % 2 == 0 } (1..10);   # returns 1, 3, 5, 7, 9

=item firstval BLOCK LIST

=item first_val BLOCK LIST

Returns the first element in LIST for which BLOCK evaluates to true. Each
element of LIST is set to C<$_> in turn. Returns C<undef> if no such element
has been found.

C<first_val> is an alias for C<firstval>.

=item lastval BLOCK LIST

=item last_val BLOCK LIST

Returns the last value in LIST for which BLOCK evaluates to true. Each element
of LIST is set to C<$_> in turn. Returns C<undef> if no such element has been
found.

C<last_val> is an alias for C<lastval>.

=item pairwise BLOCK ARRAY1 ARRAY2

Evaluates BLOCK for each pair of elements in ARRAY1 and ARRAY2 and returns a
new list consisting of BLOCK's return values. The two elements are set to C<$a>
and C<$b>.  Note that those two are aliases to the original value so changing
them will modify the input arrays.

    @a = (1 .. 5);
    @b = (10 .. 15);
    @x = pairwise { $a + $b } @a, @b;	# returns 11, 13, 15, 17, 20

    # mesh with pairwise
    @a = qw/a b c/;
    @b = qw/1 2 3/;
    @x = pairwise { ($a, $b) } @a, @b;	# returns a, 1, b, 2, c, 3

=item each_array ARRAY1 ARRAY2 ...

Creates an array iterator to return the elements of the list of arrays ARRAY1,
ARRAY2 throughout ARRAYn in turn.  That is, the first time it is called, it
returns the first element of each array.  The next time, it returns the second
elements.  And so on, until all elements are exhausted.

This is useful for looping over more than one array at once:

    my $ea = each_array(@a, @b, @c);
    while ( ($a,$b,$c) = $ea->() )   { .... }

The iterator returns the empty list when it reached the end of all arrays.

If the iterator is passed an argument of 'C<index>', then it retuns
the index of the last fetched set of values, as a scalar.

=item natatime BLOCK LIST

Creates an array iterator, for looping over an array in chunks of
C<$n> items at a time.  (n at a time, get it?).  An example is
probably a better explanation than I could give in words.

Example:

    my @x = ('a' .. 'g');
    my $it = natatime 3, @x;
    while (my @vals = $it->())
    {
        print "@vals\n";
    }

This prints

    a b c
    d e f
    g

=item mesh ARRAY1 ARRAY2 [ ARRAY3 ... ]

=item zip ARRAY1 ARRAY2 [ ARRAY3 ... ]

Returns a list consisting of the first elements of each array, then
the second, then the third, etc, until all arrays are exhausted.

Examples:

    @x = qw/a b c d/;
    @y = qw/1 2 3 4/;
    @z = mesh @x, @y;	    # returns a, 1, b, 2, c, 3, d, 4

    @a = ('x');
    @b = ('1', '2');
    @c = qw/zip zap zot/;
    @d = mesh @a, @b, @c;   # x, 1, zip, undef, 2, zap, undef, undef, zot

C<zip> is an alias for C<mesh>.

=back

=head1 EXPORTS

Nothing by default. To import all of this module's symbols, do the conventional

    use List::MoreUtils qw/:all/;

It may make more sense though to only import the stuff your program actually needs:

    use List::MoreUtils qw/any firstidx/;

=head1 VERSION

This is version 0.07.

=head1 BUGS

No known ones.

If you have a functionality that you could imagine being in this module, please
drop me a line. This module's policy will be less strict than C<List::Util>'s when
it comes to additions as it isn't a core module.

=head1 THANKS

Credits go to a number of people: Steve Purkis for giving me namespace advice
and James Keenan and Terrence Branno for their effort of keeping the CPAN
tidier by making List::Utils obsolete. 

Brian McCauley suggested the includsion of C<apply> and provided the pure-Perl
implementation for it.

Eric J. Roode asked me to add all functions from his module C<List::MoreUtil>
into this one. With minor modifications, the pure-Perl implementations of those
are by him.

=head1 SEE ALSO

L<List::Util>

=head1 AUTHOR

Tassilo von Parseval, E<lt>tassilo.von.parseval@rwth-aachen.deE<gt>

=head1 COPYRIGHT AND LICENSE

Copyright (C) 2004 by Tassilo von Parseval

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself, either Perl version 5.8.4 or,
at your option, any later version of Perl 5 you may have available.

=cut
