/*
 *  Copyright 1999-2001 Vizdom Software, Inc. All Rights Reserved.
 * 
 *  This program is free software; you can redistribute it and/or 
 *  modify it under the same terms as the Perl Kit, namely, under 
 *  the terms of either:
 *
 *      a) the GNU General Public License as published by the Free
 *      Software Foundation; either version 1 of the License, or 
 *      (at your option) any later version, or
 *
 *      b) the "Artistic License" that comes with the Perl Kit.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See either
 *  the GNU General Public License or the Artistic License for more 
 *  details.
 */

package com.vizdom.util;

/** 
 * A public toolbox of assorted methods that don't belong anywhere else.
 *
 * @author John Lacey
 * @version $Revision: 1.26 $
 */
public class Toolbox
{
    /** 
     * Checks whether a string is uppercase.
     *
     * @param aString a string to be checked
     * @return true if the string has no lower-case characters
     */
    public static final boolean isUpperCase(String aString)
    {
        for (int i = 0; i < aString.length(); ++i) 
        {
            char ch = aString.charAt(i);
            if (ch != Character.toUpperCase(ch)) 
                return false;
        }
        return true;
    }


    /** 
     * Splits a string into substrings and returns the pieces as the
     * elements of a string array. The string is split on a specified
     * delimiter character, which is not included in the returned strings.
     *
     * @param aString the string to be split 
     * @param anElementCount the number of strings to be returned
     * @param aDelimiter the character which separates substrings
     * @return an array of strings formed from substrings of
     *      the original string
     */
    public static final String[] toStringArray(String aString, 
        int anElementCount, char aDelimiter)
    {
        // Performance tests show that using a while loop over a copy
        // of the character array is a little faster than using indexOf 
        // on the String, which in turn is faster than using a while loop
        // with charAt. Using substring rather than creating a new String
        // (as was originally done) is 40% faster. We lose some of the time
        // gained by doing an index check in the while loop (so that the
        // String doesn't require a terminating delimiter), but the result
        // is still 30% faster.

        char[] chars = aString.toCharArray();
        String[] strings = new String[anElementCount];
        int charIndex = 0;
        for (int i = 0; i < anElementCount; i++)
        {
            int startIndex = charIndex;
            while (charIndex < chars.length && chars[charIndex] != aDelimiter)
                charIndex++;
            strings[i] = aString.substring(startIndex, charIndex);
            charIndex++;
        }
        return strings;
    }


    /**
     * Splits a string at each occurrence of a given character. Two
     * delimiter characters in a row will cause the splitting to stop;
     * all remaining characters will be discarded.
     *
     * @param aString a string to be split
     * @param aDelimiter the delimiter character to use
     * @return an array of substrings of the original string
     */
    public static final String[] toStringArray(String aString, char aDelimiter)
    {
        char[] chars = aString.toCharArray();
        java.util.Vector stringVector = new java.util.Vector();
        int charIndex = 0;
        while (chars[charIndex] != aDelimiter)
        {
            int startIndex = charIndex;
            while (chars[charIndex] != aDelimiter)
                charIndex++;
            String s = aString.substring(startIndex, charIndex);
            stringVector.addElement(s);
            charIndex++;
        }
        String[] strings = new String[stringVector.size()];
        stringVector.copyInto(strings);
        return strings;
    }


    /**
     * Splits a string at each occurrence of a given string. Two
     * delimiter strings in a row will cause an empty string to be
     * added to the resulting array.
     *
     * @param aString a string to be split
     * @param aDelimiter the delimiter string to use
     * @return an array of substrings of the original string
     */
    public static final String[] toStringArray(String aString, 
        String aDelimiter)
    {
        java.util.Vector stringVector = new java.util.Vector();
        int startIndex = 0;
        while (true)
        {
            int charIndex = aString.indexOf(aDelimiter, startIndex);
            if (charIndex == -1)
                break;

            String s = aString.substring(startIndex, charIndex);
            stringVector.addElement(s);
            startIndex = charIndex + aDelimiter.length();
        }
        stringVector.addElement(aString.substring(startIndex));
        String[] strings = new String[stringVector.size()];
        stringVector.copyInto(strings);
        return strings;
    }


    /**
     * Returns a new string resulting from replacing all occurrences of 
     * <code>anOldString</code> in <code>aString</code> with 
     * <code>aNewString</code>. 
     * <p>
     * If the string <code>anOldString</code> does not occur in the character 
     * sequence represented by <code>aString</code>, then a reference to 
     * <code>aString</code> is returned. Otherwise, a new String object is 
     * created that represents a character sequence identical to the character 
     * sequence represented by <code>aString</code>, except that every 
     * occurrence of <code>anOldString</code> is replaced by an occurrence of 
     * <code>aNewString</code>. 
     *
     * @param aString the string to operate on
     * @param anOldString the old string
     * @param aNewString the new string
     * @return a string derived from this string by replacing every occurrence
     *     of <code>anOldString</code> with <code>aNewString</code>
     */
    public static final String replace(String aString, String anOldString, 
        String aNewString)
    {
        // Should we make a better guess than this???
        StringBuffer buffer = new StringBuffer(aString.length());
        int startIndex = 0;
        while (true)
        {
            int charIndex = aString.indexOf(anOldString, startIndex);
            if (charIndex == -1)
                break;

            buffer.append(aString.substring(startIndex, charIndex));
            buffer.append(aNewString);
            startIndex = charIndex + anOldString.length();
        }

        if (startIndex == 0)
            return aString;
        else
        {
            buffer.append(aString.substring(startIndex));
            return buffer.toString();
        }
    }


    /**
     * Splits a string into substrings and returns the pieces as the
     * elements of an int array. The string is split on a specified
     * delimiter character.
     *
     * @param aString the string to be split 
     * @param anElementCount the number of integers to be returned
     * @param aDelimiter the character which separates the integers
     * @return an array of ints formed from substrings of
     *      the original string
     */
    public static final int[] toIntArray(String aString, 
        int anElementCount, char aDelimiter)
    {
        char[] chars = aString.toCharArray();
        int[] ints = new int[anElementCount];
        int charIndex = 0;
        for (int i = 0; i < anElementCount; i++)
        {
            int startIndex = charIndex;
            while (charIndex < chars.length && chars[charIndex] != aDelimiter)
                charIndex++;
            ints[i] = 
                Integer.parseInt(aString.substring(startIndex, charIndex));
            charIndex++;
        }
        return ints;
    }


    /**
     * Joins the elements of the given array into a string, 
     * separated by the delimiter character.
     *
     * @param aList an array of integers
     * @param aDelimiter the character to use to separate the integers
     *      in the resulting string
     * @return a string containing the elements of the given array,
     *      separated by the delimiter character; null if the array is 
     *      null; the empty string if the array has no elements
     */
    public static final String toString(int[] aList, char aDelimiter) 
    {
        return toString(aList, String.valueOf(aDelimiter));
    }


    /**
     * Joins the elements of the given array into a string, 
     * separated by the delimiter character.
     *
     * @param aList an array of integers
     * @param aDelimiter the character to use to separate the integers
     *      in the resulting string
     * @param aSentinel the sentinel value is appended to the
     *      constructed string before it is returned
     * @return a string containing the elements of the given array,
     *      separated by the delimiter character; null if the array is 
     *      null; the empty string if the array has no elements
     */
    public static final String toString(int[] aList, char aDelimiter,
        String aSentinel) 
    {
        return toString(aList, String.valueOf(aDelimiter), 
            String.valueOf(aSentinel));
    }


    /**
     * Joins the elements of the given array into a string, 
     * separated by the delimiter character.
     *
     * @param aList an array of strings
     * @param aDelimiter the character to use to separate the strings
     *      in the resulting string
     *
     * @return a string containing the elements of the given array,
     *      separated by the delimiter character; null if the array is 
     *      null; the empty string if the array has no elements
     */
    public static final String toString(String[] aList, char aDelimiter) 
    {
        return toString(aList, String.valueOf(aDelimiter));
    }


    /**
     * Joins the elements of the given array into a string, 
     * separated by the delimiter character.
     *
     * @param aList an array of strings
     * @param aDelimiter the character to use to separate the strings
     *      in the resulting string
     * @param aSentinel the sentinel value is appended to the
     *      constructed string before it is returned
     * @return a string containing the elements of the given array,
     *      separated by the delimiter character; null if the array is 
     *      null; the empty string if the array has no elements
     */
    public static final String toString(String[] aList, char aDelimiter,
        char aSentinel) 
    {
        return toString(aList, String.valueOf(aDelimiter), 
            String.valueOf(aSentinel));
    }


    /**
     * Joins the elements of the given array into a string, 
     * separated by the delimiter string.
     *
     * @param aList an array of integers
     * @param aDelimiter the string to use to separate the integers
     *      in the resulting string
     * @return a string containing the elements of the given array,
     *      separated by the delimiter string; null if the array is 
     *      null; the empty string if the array has no elements
     */
    public static final String toString(int[] aList, String aDelimiter) 
    {
        return toString(aList, aDelimiter, "");
    }


    /**
     * Joins the elements of the given array into a string, 
     * separated by the delimiter string.
     *
     * @param aList an array of integers
     * @param aDelimiter the string to use to separate the integers
     *      in the resulting string
     * @param aSentinel the sentinel value is appended to the
     *      constructed string before it is returned
     * @return a string containing the elements of the given array,
     *      separated by the delimiter string; null if the array is 
     *      null; the empty string if the array has no elements
     */
    public static final String toString(int[] aList, String aDelimiter, 
        String aSentinel) 
    {
        String value;
        if (aList == null) 
            value = null;
        else if (aList.length == 0) 
            value = "";
        else 
        {
            StringBuffer buffer = new StringBuffer();
            buffer.append(Integer.toString(aList[0]));
            for (int i = 1; i < aList.length; i++)
            {
                buffer.append(aDelimiter);
                buffer.append(Integer.toString(aList[i]));
            }
            buffer.append(aSentinel);
            value = buffer.toString();
        }
        return value;
    }

    
    /**
     * Joins the elements of the given array into a string, 
     * separated by the delimiter string.
     *
     * @param aList an array of strings
     * @param aDelimiter the string to use to separate the strings
     *      in the resulting string
     * @return a string containing the elements of the given array,
     *      separated by the delimiter string; null if the array is 
     *      null; the empty string if the array has no elements
     */
    public static final String toString(String[] aList, String aDelimiter) 
    {
        return toString(aList, aDelimiter, "");
    }


    /**
     * Joins the elements of the given array into a string, 
     * separated by the delimiter string.
     *
     * @param aList an array of strings
     * @param aDelimiter the string to use to separate the strings
     *      in the resulting string
     * @param aSentinel the sentinel value is appended to the
     *      constructed string before it is returned
     * @return a string containing the elements of the given array,
     *      separated by the delimiter string; null if the array is 
     *      null; the empty string if the array has no elements
     */
    public static final String toString(String[] aList, String aDelimiter,
        String aSentinel) 
    {
        String value;
        if (aList == null)
            value = null;
        else if (aList.length == 0)
            value = "";
        else
        {
            StringBuffer buffer = new StringBuffer();
            buffer.append(aList[0]);
            for (int i = 1; i < aList.length; i++)
            {
                buffer.append(aDelimiter);
                buffer.append(aList[i]);
            }
            buffer.append(aSentinel);
            value = buffer.toString();
        }
        return value;
    }


    /**
     * Treats the given string as a C-style null-terminated string,
     * and returns a Java string without the null byte at the end.
     * This method is safe if there is no null character at the end,
     * much like Perl's chomp.
     * 
     * @param aString a string, possibly null-terminated
     * @return a Java string without the null byte at the end
     */
    public static final String toJavaString(String aString)
    {
        int length = aString.length();
        if (length > 0 && aString.charAt(length - 1) == '\u0000')
        {
            // ??? remove all trailing null characters, or just one?
            return aString.substring(0, length - 1);
        }
        else
            return aString;
    }


    /**
     * Returns a null-terminated copy of the string. A null character
     * is added even if the given string is already null-terminated.
     * 
     * @param aString a string, possibly null-terminated
     * @return a null-terminated copy of the string
     */
    public static final String toCString(String aString)
    {
        StringBuffer buffer = new StringBuffer(aString.length() + 1);
        buffer.append(aString);
        buffer.append('\u0000');
        return buffer.toString();
    }


    /**
     * Treats the string argument as a string hex digits and
     * converts each pair into a byte.
     * <p>
     * For example, "1a4bff043287" becomes {26, 75, -1, 4, 50, -121}.
     *
     * @param aHexString a string of hex digits; this string must have
     *      an even length
     * @return a byte array in which each byte corresponds to two
     *      of the characters of the original string
     * @exception NumberFormatException if the string contains non-hex
     *      characters
     */
    public static final byte[] hexToBytes(String aHexString)
    {
        int nybbleCount = aHexString.length();
        Debug.assertTrue((nybbleCount % 2) == 0);
        byte[] bytes = new byte[nybbleCount / 2];
        for (int i = 0; i < nybbleCount - 1; i += 2)
        {
            int high = Character.digit(aHexString.charAt(i), 16);
            int low = Character.digit(aHexString.charAt(i + 1), 16);
            bytes[i / 2] = (byte) ((high << 4) + low);
        }
        return bytes;
    }


    /**
     * Converts each byte to a pair of hexadecimal characters.
     * <p>
     * For example, {26, 75, -1, 4, 50, -121} becomes "1a4bff043287" 
     *
     * @param aValue an array of bytes
     * @return a string encoding the bytes as hexadecimal characters
     */
    public static final String bytesToHex(byte[] aValue)
    {
        String hexString;
        if (aValue.length == 0)
            hexString = "";
        else
        {
            char[] hexChars = new char[aValue.length * 2];
            int hexIndex = 0;
            for (int i = 0; i < aValue.length; ++i)
            {
                // Convert each byte to an unsigned int.
                int b = aValue[i] & 0xFF;

                // Each byte is converted to two hex characters.
                // It's convenient to convert the low 4 bits into the 
                // second character first.
                hexChars[hexIndex + 1] = Character.forDigit(b & 0x0F, 16);
                b >>>= 4;
                hexChars[hexIndex] = Character.forDigit(b, 16);
                hexIndex += 2;
            }
            if (Debug.ASSERT)
                Debug.assertTrue(hexIndex == hexChars.length);
            hexString = new String(hexChars);
        }
        return hexString;
    }


    /**
     * Loads properties from the given input stream and returns the
     * corresponding properties object.
     *
     * @param in an input stream whose content is in properties file
     *     format; this stream will be closed
     * @return a Properties object constructed from the given input stream
     * @exception java.io.IOException if an error occurs reading the file
     */
    private static java.util.Properties gGetProperties(java.io.InputStream in)
        throws java.io.IOException 
    {
        try
        {
            java.util.Properties properties = new java.util.Properties();
            properties.load(in);
            return properties;
        }
        finally
        {
            try { in.close(); } catch (Exception e) {}
        }
    }


    /**
     * Loads a properties file with the given name and returns the
     * corresponding properties object. The file must be located
     * on the current classpath, and it must be named 
     * &lt;aName&gt;<code>.properties</code>
     *
     * @param aName the base portion (no extension) of the properties filename
     * @return a Properties object constructed from the given file
     * @exception java.io.IOException if the properties file is not
     *     found, or an error occurs reading the file,
     */
    public static final java.util.Properties getProperties(String aName) 
        throws java.io.IOException 
    {
        String name = aName + ".properties";
        java.io.InputStream is = ClassLoader.getSystemResourceAsStream(name);
        if (is == null) 
            throw new java.io.FileNotFoundException(name);
        return gGetProperties(is);
    }


    /**
     * Loads a properties file from the give pathname and returns the
     * corresponding properties object.
     *
     * @param aFile the <code>java.io.File</code> specifying the file's path
     * @return a Properties object constructed from the given file
     * @exception java.io.IOException if the properties file is not
     *     found, or an error occurs reading the file,
     */
    public static final java.util.Properties getProperties(java.io.File aFile) 
        throws java.io.IOException 
    {
        java.io.FileInputStream is = new java.io.FileInputStream(aFile);
        return gGetProperties(is);
    }


    /** Instances of Toolbox are not permitted. */
    private Toolbox()
    {
        throw new UnreachableCodeException();
    }
}
